/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestJavaUtil.java,v 1.8 2005/03/15 14:52:19 dguilbaud Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.java.internal.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaModel;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.ui.JavaElementLabelProvider;
import org.eclipse.jdt.ui.StandardJavaElementContentProvider;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;
import org.eclipse.ui.dialogs.ISelectionStatusValidator;
import org.eclipse.ui.help.WorkbenchHelp;

import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.test.common.TestCommonPlugin;
import org.eclipse.hyades.test.common.internal.dialog.InvokedTestSelectionDialog;
import org.eclipse.hyades.test.common.internal.util.ContextIds;
import org.eclipse.hyades.test.java.TestJavaPlugin;

/**
 * @author marcelop
 * @since 0.3.0
 */
public class TestJavaUtil
{	

	public static ElementTreeSelectionDialog createPackageDialog(Shell parent, final boolean defaultPackageIsValid)
	{
		ViewerFilter viewerFilter = new ViewerFilter()
		{
			public boolean select(Viewer viewer, Object parentElement, Object element)
			{
				if(element instanceof IJavaProject)
					return true;
					
				if(element instanceof IPackageFragmentRoot)
				{
					IPackageFragmentRoot packageFragmentRoot = (IPackageFragmentRoot)element;
					if(packageFragmentRoot.isArchive())
						return false;
						
					if(packageFragmentRoot.isExternal())
						return false;
						
					return true;
				} 
					
				if(!defaultPackageIsValid)
				{
					if(element instanceof IPackageFragment)
						return !((IPackageFragment)element).isDefaultPackage();
				}
				
				return false;
			}
		};
		
		StandardJavaElementContentProvider contentProvider = new StandardJavaElementContentProvider()
		{
			private StructuredViewer structuredViewer;
			
			public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
			{
				if(viewer != structuredViewer)
				{
					structuredViewer = null;
					if(viewer instanceof StructuredViewer)
						structuredViewer = (StructuredViewer)viewer;
				}
				
				super.inputChanged(viewer, oldInput, newInput);
			}
			
			public boolean hasChildren(Object element)
			{
				if(element instanceof IPackageFragment)
					return false;
				
				try
				{
					if(element instanceof IPackageFragmentRoot)
						return ((IPackageFragmentRoot)element).getChildren().length > 1;
				}
				catch(JavaModelException e)
				{
				}
					
				return super.hasChildren(element);
			}
		
			public Object[] getChildren(Object parentElement)
			{
				try
				{
					if(parentElement instanceof IJavaModel) 
						return concatenate(this.getJavaProjects((IJavaModel)parentElement), getNonJavaProjects((IJavaModel)parentElement));
					
					return super.getChildren(parentElement);
				}
				catch(JavaModelException e)
				{
					TestJavaPlugin.logError(e);
					return new Object[0];
				}
			}
			
			private Object[] getNonJavaProjects(IJavaModel model)
			throws JavaModelException
			{
				return model.getNonJavaResources();
			}
		};
	
		ISelectionStatusValidator validator = new ISelectionStatusValidator()
		{
			public IStatus validate(Object[] selection)
			{
				if((selection.length == 1) && (selection[0] instanceof IPackageFragment))
					return new Status(IStatus.OK, TestJavaPlugin.getID(), IStatus.OK, "", null);
					
				return new Status(IStatus.ERROR, TestJavaPlugin.getID(), IStatus.OK, "", null);
			}
		};
		
		ElementTreeSelectionDialog dialog = new ElementTreeSelectionDialog(parent, new JavaElementLabelProvider(JavaElementLabelProvider.SHOW_DEFAULT), contentProvider);
		dialog.addFilter(viewerFilter);
		dialog.setInput(JavaCore.create(ResourcesPlugin.getWorkspace().getRoot()));
		dialog.setValidator(validator);
		dialog.setAllowMultiple(false);
		dialog.setDoubleClickSelects(true);
		
		return dialog;
	}	
	
	public static ElementTreeSelectionDialog createSourceFolderDialog(Shell parent, IJavaProject javaProject)
	{
		ViewerFilter viewerFilter = new ViewerFilter()
		{
			public boolean select(Viewer viewer, Object parentElement, Object element)
			{
				if(element instanceof IJavaProject)
					return true;
					
				if(element instanceof IPackageFragmentRoot)
				{
					IPackageFragmentRoot packageFragmentRoot = (IPackageFragmentRoot)element;
					if(packageFragmentRoot.isArchive())
						return false;
						
					if(packageFragmentRoot.isExternal())
						return false;
						
					return true;
				} 
				
				return false;
			}
		};
		
		StandardJavaElementContentProvider contentProvider = new StandardJavaElementContentProvider()
		{
			private StructuredViewer structuredViewer;
			
			public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
			{
				if(viewer != structuredViewer)
				{
					structuredViewer = null;
					if(viewer instanceof StructuredViewer)
						structuredViewer = (StructuredViewer)viewer;
				}
				
				super.inputChanged(viewer, oldInput, newInput);
			}
			
			public boolean hasChildren(Object element)
			{
				if(element instanceof IJavaProject)
					return hasSourceFolders((IJavaProject)element);
				
				return super.hasChildren(element);
			}		
		};
				
		ISelectionStatusValidator validator = new ISelectionStatusValidator()
		{
			public IStatus validate(Object[] selection)
			{
				if(selection.length == 1)
				{
					if(selection[0] instanceof IPackageFragmentRoot)
						return new Status(IStatus.OK, TestJavaPlugin.getID(), IStatus.OK, "", null);
					
					if((selection[0] instanceof IJavaProject) && !hasSourceFolders((IJavaProject)selection[0]))
						return new Status(IStatus.OK, TestJavaPlugin.getID(), IStatus.OK, "", null);
				}
					
				return new Status(IStatus.ERROR, TestJavaPlugin.getID(), IStatus.OK, "", null);
			}
		};

		ElementTreeSelectionDialog dialog = new ElementTreeSelectionDialog(parent, new JavaElementLabelProvider(JavaElementLabelProvider.SHOW_DEFAULT), contentProvider);
		dialog.addFilter(viewerFilter);
		dialog.setValidator(validator);
		if(javaProject != null)
			dialog.setInput(javaProject);
		else
			dialog.setInput(JavaCore.create(ResourcesPlugin.getWorkspace().getRoot()));
		dialog.setAllowMultiple(false);
		dialog.setDoubleClickSelects(true);
		
		return dialog;
	}	
	
	public static IJavaProject[] getJavaProjects()
	{
		try
		{
			return JavaCore.create(ResourcesPlugin.getWorkspace().getRoot()).getJavaProjects();
		}
		catch(Exception e)
		{			
		}
		
		return new IJavaProject[0];
	}
	
	public static boolean hasSourceFolders(IJavaProject javaProject)
	{
		IClasspathEntry[] entries = null;
		try
		{
			entries = javaProject.getRawClasspath();
		}
		catch (JavaModelException e)
		{
			return false;
		}
					
		for (int i = 0, maxi = entries.length; i < maxi; i++)
		{
			if((entries[i].getEntryKind() == IClasspathEntry.CPE_SOURCE) && (!javaProject.getPath().equals(entries[i].getPath())))
				return true;
		}
		return false;		
	}
	
	public static List getSourceFolders(IJavaProject javaProject)
	{
		IClasspathEntry[] entries = null;
		try
		{
			entries = javaProject.getRawClasspath();
		}
		catch (JavaModelException e)
		{
			return Collections.EMPTY_LIST;
		}
			
		List sourceFolders = new ArrayList();		
		for (int i = 0, maxi = entries.length; i < maxi; i++)
		{
			if((entries[i].getEntryKind() == IClasspathEntry.CPE_SOURCE) && (!javaProject.getPath().equals(entries[i].getPath())))
			{
				IResource resource = ResourcesPlugin.getWorkspace().getRoot().findMember(entries[i].getPath());
				if (resource != null) //NullPointerException
				{
					if(resource.exists() && (resource.getType() == IResource.FOLDER))
					sourceFolders.add(resource);
				}
			}
		}
		return sourceFolders;		
	}	
}
