package org.eclipse.hyades.logging.commons.sample;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import org.apache.commons.logging.LogFactory;
import org.eclipse.hyades.logging.commons.Logger;
import org.eclipse.hyades.logging.events.ICommonBaseEvent;
import org.eclipse.hyades.logging.events.IComponentIdentification;
import org.eclipse.hyades.logging.events.ISimpleEventFactory;
import org.eclipse.hyades.logging.events.SimpleEventFactoryImpl;

/** 
 * CLASS: HyadesLoggingCommonsSample.java
 * 
 * 
 * DESCRIPTION:  Sample class to demonstrate the usage of the Hyades support for logging Common Base Event log records in Apache Commons logging APIs.
 * 
 * 
 * ASSUMPTIONS:  1) The required JAR files must be specified in the JVM's classpath system variable.
 *               2) JDK must be specified in the PATH environment variable.
 *               3) When launching this application from the Profiling Perspective of the Workbench, logging is enabled in the Profiling and Logging preferences (Window --> Preferences --> Profiling and Logging --> Enable logging) 
 *                  and the 'org.eclipse.hyades.tests.logging.java.JavaLoggingTest' logging agent is added to the Profiling and Logging preferences (Window --> Preferences --> Profiling and Logging --> Logging Agents --> New agent --> Add agent).
 *               4) When either launching this application from the Profiling Perspective of the Workbench or attaching to a logging agent using the Workbench, the Agent Controller is installed and running.
 *               5) When launching this application from the Profiling Perspective of the Workbench, the Agent Controller is configured during installation to use JDK 1.4.0 or above.
 *
 * 
 * DEPENDENCIES: The following JAR files are required to compile and execute JavaLoggingTest.java:
 * 
 * 		         hexr.jar				-		org.eclipse.hyades.execution.remote
 * 		         hlcore.jar				-		org.eclipse.hyades.logging.core
 * 		         hlevents.jar			-		org.eclipse.hyades.logging.core
 *               commons-logging.jar	-		org.eclipse.hyades.logging.commons\runtime
 * 		         hlcommons.jar			-		org.eclipse.hyades.logging.commons
 *  
 * 
 * @author		Paul Slauenwhite
 * @version	May 29, 2003
 * @see		org.apache.commons.logging.Log
 * @see		org.apache.commons.logging.LogFactory
 * @see		org.eclipse.hyades.logging.commons.Logger
 * @see		org.eclipse.hyades.logging.events.ICommonBaseEvent 
 * @see		org.eclipse.hyades.logging.events.SimpleEventFactoryImpl
 */

public class HyadesLoggingCommonsSample {

    //Amount of time (seconds) to wait to allow time to attach to the logging agent:
    private static final int WAITTIME = 30;

    //The name of the Apache Commons logger:
    private static final String LOGGERS_NAME = "org.eclipse.hyades.logging.commons.sample.HyadesLoggingCommonsSample";

    public static void main(String[] args) {

        try {

            //Set the Apache Common's logger:
            System.setProperty("org.apache.commons.logging.Log", "org.eclipse.hyades.logging.commons.Logger");

            //Retrieve the log factory based on the current Apache Commons configuration (org.apache.commons.logging.impl.LogFactoryImpl by default):
            LogFactory logFactory = LogFactory.getFactory();

            //Create a logger named 'org.eclipse.hyades.logging.commons.sample.HyadesLoggingCommonsSample':
            //NOTE: The retrieved logger must be down-casted to an org.eclipse.hyades.logging.commons.Logger.
            Logger logger = ((Logger) (logFactory.getInstance(LOGGERS_NAME)));

            //Set the logger to log warning or lower messages:
            logger.setLevel(Logger.WARN_LEVEL);

            //Retrieve the instance of the Simple Event Factory:
            ISimpleEventFactory simpleEventFactory = SimpleEventFactoryImpl.getInstance();

            //Create a new instance of a Source Component:
            IComponentIdentification sourceComponentId = simpleEventFactory.createComponentIdentification();
            sourceComponentId.setComponent("Test Source Component");
            sourceComponentId.setSubComponent("Test Source Component Subcomponent");
            sourceComponentId.setComponentIdType("Test Source Component ID Type");
            sourceComponentId.setLocation("Test Source Component Location");
            sourceComponentId.setLocationType("Test Source Component Location Type");

            //Create a new instance of a Common Base Event:
            ICommonBaseEvent commonBaseEvent = simpleEventFactory.createCommonBaseEvent();
            commonBaseEvent.setSourceComponentId(sourceComponentId);
            commonBaseEvent.setCreationTime(System.currentTimeMillis());
            commonBaseEvent.setSeverity(((short) (10)));
            commonBaseEvent.setMsg("Apache Commons Common Base Event log message");

            //Log the Common Base Event log record:
            if (logger.isErrorEnabled())
                logger.error(commonBaseEvent);

            //Attempt to log a message at a higher logging level:
            if (logger.isDebugEnabled())
                logger.debug("Apache Commons invalid log message.");

            //Wait for 2 seconds until all log messages are written to the SimpleLog logger (i.e. standard error):
            try {
                Thread.sleep(2000);
            }
            catch (Throwable t) {
            }

            System.out.println("Please attach to the 'org.eclipse.hyades.logging.commons.sample.HyadesLoggingCommonsSample' logging agent in the next " + WAITTIME + " second" + (WAITTIME == 1 ? "" : "s") + ":");
            System.out.println("NOTE: When attaching to a logging agent using the Workbench, please remember to enable logging in the Profiling and Logging preferences (Window --> Preferences --> Profiling and Logging --> Enable logging) and the Agent Controller is running.");

            //Create the timer scale:
            System.out.print("I");

            for (int counter = 0; counter < WAITTIME; counter++)
                System.out.print("-");

            System.out.print("I\nI");

            //Increment the timer:
            try {

                for (int counter = 0; counter < WAITTIME; counter++) {

                    System.out.print("-");

                    Thread.sleep(1000);
                }
            }
            catch (Throwable t) {
            }

            System.out.println("I");

            System.out.println("HyadesLoggingCommonsSample successfully ended!");
        }
        catch (Throwable t) {
            System.out.println("ERROR - HyadesLoggingCommonsSample unsuccessfully ended!");
            System.out.println("REASON: " + t.getMessage());
        }
    }
}
