/**
 * A simple correlation engine created to illustrate how one can make correlations
 * between log records.
 */

import org.eclipse.emf.common.util.*;
import org.eclipse.hyades.logc.extensions.*;
import org.eclipse.hyades.logs.correlators.*;
import org.eclipse.hyades.models.cbe.*;


public class MyCorEngine implements ILogRecordCorrelationEngine 
{

		/* The name and the correlation engine for simple correlation engine */
		String correlationName;
		CBEAssociationEngine correlationEngine;


		/* Makes the necessary correlation between the log records based up on a number
		 * of factor(s). Currently the factor that is used to correlate log records is 
		 * the ID of each record included in the log file associated with 'Simple Parser 
		 * V1.0'
		 *
		 * @param list - A listing of the processes (i.e. the log files) for which 
		 * correlation are to be made made.
		 */
		public void correlate(EList logFiles) 
		{
				correlationName = "Simple Correlation Engine V1.0";
				correlationEngine=null;
                
				/* Traverse through each of the log files that a correlation needs to be made */
				for(int i = 0; i < logFiles.size(); i++)
				{
						/* For each of the existing log file, traverse through its log records and make the necessary correlations */
						if(logFiles.get(i) != null)
						{
								/* Store the list corresponding to the log records of the i-th logFile */
								EList recordList = ((RecordList)logFiles.get(i)).getList(); 

								/* Make the necessary correlations */
								makeCorrelations (recordList, logFiles, i);
						}
				}// End of for-loop
		}// End of correlator (EList)



		/* A helper method to 'correlate(EList)' that is used to make the necessary 
		 * correlations between the log records 
		 * 
		 * @param recordList - A list of log records for which correlations are made 
		 * logFiles - 'logRecord' belongs to a log file listed under this parameter
		 * logFileIndex - The index of 'logFiles' identifying the log file that 
		 * logRecord belongs to 
		 */
		private void makeCorrelations (EList recordList, EList logFiles, int logFileIndex)
		{
				/* Traverse through each of the log records and make the necessary correlations */
				for(int j = 0; j < recordList.size(); j++)
				{
						/* As long as the log records exist, reset their correlator list if needed before
						 * making any correlations */
						if(recordList.get(j) != null)
						{
								/* Store the correlator list */
								EList correlators = ((CBECommonBaseEvent)
								recordList.get(j)).getAssociatedEvents();

								/* Reset the correlator list */
								resetCorrelators (correlators);
						}
                
						/* Make the correlation for the j-th log record */
						setPartners (recordList.get(j), logFiles, logFileIndex);
				}// End of for-loop
		}// End of makeCorrelations (EList)



		/* Resets the correlator list only if it is necessary 
		 * 
		 * @param correlators - The correlator list that is reseted
		 */
		private void resetCorrelators (EList correlators)
		{
				/* Traverse through each of the elements of the correlators list and reset it if necessary */
				for (int k = 0; k < correlators.size(); k++)
				{
						/* Reset the element only if it is necessary */
						if((correlators.get(k) != null) && (((CBEAssociatedEvent) correlators.get(k)).getAssociationEngine() != null) && 
								(((CBEAssociatedEvent)correlators.get(k)).getAssociationEngine().getName() != null) && 
								((CBEAssociatedEvent) correlators.get(k)).getAssociationEngine().getName().equals(correlationName))
						{
								correlators.remove(k);
						}
				} // End of the for-loop
		}// End of resetCorrelators (EList)



		/* A helper method to 'makeCorrelations (EList, EList, int)' used to set the partners 
		 * 
		 * @param logRec                - The log record for which partners are set 
		 *                logFiles              - 'logRecord' belongs to a log file listed under this parameter
		 *                logFileIndex  - The index of 'logFiles' identifying the log file that logRecord belongs to 
		 */ 
		private void setPartners (Object logRec, EList logFiles, int logFileIndex)
		{
				/* The log records are mapped to a Common Base Event */
				CBECommonBaseEvent logRecord = (CBECommonBaseEvent) logRec;

				/* The list of records that will be checked */
				EList recordList = null;

				/* The correlators list and the correlated records of 'logRecord' */
				EList correlators = null;
				CBEAssociatedEvent correlationType = null;

				/* Get the record ID of the passed log record */
				String recordID = getRecordID (logRecord);

				/* Traverse through the logFiles (starting from index 'logFileIndex') and make the proper correlations*/
				for(int i = logFileIndex; i < logFiles.size(); i++)
				{
                        
						/* The record list of the i-th logFile */
						recordList = ((RecordList)logFiles.get(i)).getList();

						/* Traverse through the record list of the i-th log file and make the proper correlations */
						if(recordList != null)
						{
								for(int j = 0; j < recordList.size(); j++)      
								{
										if(recordList.get(j) != null && recordList.get(j)!=logRec && 
												getRecordID(recordList.get(j)).equals(recordID) && recordList.get(j)!=logRec)
										{
												correlators = logRecord.getAssociatedEvents();
												correlationType = createCorType (logRecord, correlators);
                                                
												/* If this j-th record is not already in the correlationType, then include it */
												if(!correlationType.getResolvedEvents().contains((CBECommonBaseEvent)recordList.get(j)))        
														correlationType.getResolvedEvents().add((CBECommonBaseEvent)recordList.get(j));
										} 
								} // End of the j-th loop 
						}
				} // End of the i-th loop
		} // End of setPartners (Object, EList, int)



		/* Creates the proper correlationType
		 * 
		 * @param logRecord - A log record
		 *                correlators - The list of correlators associated 'logRecord'
		 * 
		 * @return AssociateEvent - The list of correlation types associated with a 'correlators' list
		 */
		private CBEAssociatedEvent createCorType (CBECommonBaseEvent logRecord, EList correlators)
		{
				/* The correlation type that will be returned */
				CBEAssociatedEvent corType = null;

				/* Traverse through the passed correlators list and create the proper 'correlationType' object */ 
				for (int k = 0; k < correlators.size(); k++)
				{
						if (correlators.get(k) != null)
								corType = ((CBEAssociatedEvent)correlators.get(k));
                
						/* If a correlation type already exists, don't create it again */ 
						if(corType.getAssociationEngine()!=null && corType.getAssociationEngine().getType()!=null && 
						  corType.getAssociationEngine().getName().equals(correlationName))
						  return corType;
				}

				/* Create the correlation type */ 
                
				/* An instance of 'CBEFactory' is used to create the proper correlation type to indicate 
				the correlated records. */
				CBEFactory artifactsFactory = CBEFactory.eINSTANCE; 

				corType = artifactsFactory.createCBEAssociatedEvent();

				/* Create the correlation engine if it has already not been created */
				if (correlationEngine == null) 
				{
						correlationEngine=artifactsFactory.createCBEAssociationEngine();
						correlationEngine.setType("Correlated");
						correlationEngine.setName(correlationName);                     
						logRecord.getAgent().eResource().getContents().add(correlationEngine);  
				}

				/* Set the engine on the correlation type */ 
				corType.setAssociationEngine(correlationEngine);

				/* Add the correlation type to its respective correlators list */
				correlators.add(corType);
                
				return corType;

		} // End of createCorType (CBECommonBaseEvent, EList)



		/* Returns the ID associated with a log record from the 'syslog' log file (i.e. The log file that is associated 
		 * with the 'Simple Parser V1.0' */
		private String getRecordID (Object logRecord)
		{ 
				String textField = ((CBECommonBaseEvent)logRecord).getMsg();
				return textField.substring(textField.lastIndexOf(' '), textField.lastIndexOf('.')); 
		} 
        
}// End of MyCorEngine class
