/**
 * A simple parser created to illustrate how one can create their own
 * parsers for a given log file.
 */
import org.eclipse.hyades.logging.core.ILogger;
import org.eclipse.hyades.logging.parsers.LogParserException;
import org.eclipse.hyades.logging.parsers.Parser;


import java.util.Date;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Hashtable;


public class MyParser extends Parser {


		/* The version of the software used to generate the log file */
		private String prodVersion = null;

        
		/* The release (eg. Alpha, Beta, etc..) of the software used to generate the log file */
		private String prodReal = null;


		/* Keys for the product version & the product release */
		private final static String VER_KEY = "PRODUCT_VERSION";
		private final static String REL_KEY = "PRODUCT_RELEASE";

		/* The source ID (which is simply the concatenation of the release and the version of the 
		   software used to generate the log file) */
		private String sourceID = null;

		/* The creation time, unique ID, severity, and the text field of a log record */
		private StringBuffer creationTime = new StringBuffer ("");
		private StringBuffer uniqueID = new StringBuffer ("");
		private String textField = null;
		private String severity = null;

		/* The following field holds the creationTime of the previous log record */
		private String prevCreationTime = null;

		/* Store the message count */
		int messageCounter = 0;

		/* The following method is invoked right after the user clicks on the 'finish' button in the 
		 * import wizard page. This method sets the appropriate fields based on the user's input */
        
		 public void setUserInput(Hashtable table) throws LogParserException 
		{
				/* The following line of code sets file_path to the path that has been chosen by the user */
				super.setUserInput(table);

				/* Store the product version and the product release */
				prodVersion = (String) table.get (VER_KEY);
				prodReal = (String) table.get (REL_KEY);

				/* Create the sourceID */
				sourceID = prodReal + prodVersion;
		}



		/* The following method is invoked right after setUserInput(Hashtable) in order to parse the log file */
		public void parse(ILogger argLogger) throws LogParserException 
		{
				/* The following call causes the initialization of a few generalized fields */
				super.parse(argLogger);

				/* Attempting to parse the log file */
				try 
				{ 
						/* Read the log file line by line */
						while (curLine != null)
						{ 
								/* Go to the next line if the current line does not begin with "##" */
								if (!isChar (curLine, 0, '#') || !isChar (curLine, 1, '#'))
								{
										curLine = readLine();
										continue;
								}

								/* Initialize the Common Base Event. */
								messages[arrayIndex].init();
								messages[arrayIndex].setSourceComponentId(eventFactory.createComponentIdentification());
								messages[arrayIndex].getSourceComponentId().init();
								messages[arrayIndex].setMsgDataElement(eventFactory.createMsgDataElement());
								messages[arrayIndex].getMsgDataElement().init();

								messages[arrayIndex].getSourceComponentId().setComponent("myComponent");
								messages[arrayIndex].getSourceComponentId().setLocation(localHostId);
								messages[arrayIndex].getSourceComponentId().setLocationType(localHostIdFormat);
								messages[arrayIndex].getSourceComponentId().setSubComponent("Unknown");
								messages[arrayIndex].getMsgDataElement().setMsgIdType("Not Applicable");
								messages[arrayIndex].getMsgDataElement().setMsgId("None");

								/* Assertion: The current line begins with '##' and needs to be parsed */
								curLine = curLine.substring(3);

								/* Find the severity of the log record */
								severity = curLine.substring (1, curLine.indexOf('>'));
								if (severity.trim().equals ("Information"))
										messages[arrayIndex].setSeverity((short)10);
								else if (severity.trim().equals ("Error"))
										messages[arrayIndex].setSeverity((short)50);
								else if (severity.trim().equals ("Warning"))
										messages[arrayIndex].setSeverity((short)30);

								/* Find the creation time of the log record */
								findCreationTime ();
								messages[arrayIndex].setCreationTime(creationTime.toString());

                                
								/* Find the text body of the log record */
								textField = curLine.substring (curLine.indexOf ('>') + 2);
								messages[arrayIndex].setMsg(textField);

								/* Find the message count and the unique ID of the log record */
								if (prevCreationTime.trim().equals(creationTime.toString().trim()))
										messageCounter++;
								else
										messageCounter = 0;

								messages[arrayIndex].setRepeatCount((short)messageCounter); 
								generateUniqueID ();
								messages[arrayIndex].setGlobalInstanceId(uniqueID.toString());

								/* Write 'messages' to the logger and read the next line */
								argLogger.write(messages[arrayIndex]);
								messages[arrayIndex].init();
								curLine = readLine (); 
						}
				}
				catch (Throwable throwable)
				{
						LogParserException lpe = new LogParserException("Error parsing error log.", throwable);
						lpe.fillInStackTrace();
						throw lpe;
				}
		}

		/* Store the creation time in the desirable format */
		private void findCreationTime ()
		{
				SimpleDateFormat myFormat;
                
				/* This is the format found in the log file */
				myFormat = new SimpleDateFormat("MM.d.yy.kk.mm.ss");
				ParsePosition parseLoc = new ParsePosition(0);

				prevCreationTime = creationTime.toString();
				creationTime = new StringBuffer ();
				curLine = curLine.substring(curLine.indexOf('>') + 2);
				creationTime.append(curLine.substring (1, curLine.indexOf ('>')));
				Date date = myFormat.parse(creationTime.toString(), parseLoc);

				/* This is the desirable format that we would like creation time to be in
				 * (The creation time has to be in this format) */
				myFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
				creationTime = new StringBuffer(myFormat.format(date));
				creationTime.replace(10, 11, "T");
				creationTime.append(".000000");
		}

		/* Generates a unique ID for each of the log record */
		private void generateUniqueID ()
		{
				uniqueID = new StringBuffer ("");
                
				if (localHostName != null)
						uniqueID.append(localHostName).append('_');

				if (file_path != null)
						uniqueID.append(file_path).append('_');

				uniqueID.append (creationTime.toString ().trim ()).append ('_'). append(messages[arrayIndex].getRepeatCount());
		}

		/* Returns the name of this parser */
		public String getName ()
		{ return "Simple Parser"; }

		/* Returns the version of this parser */ 
		public String getVersion ()
		{ return "V1.0"; }

}
