import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Hashtable;

import org.apache.commons.logging.Log;
import org.eclipse.hyades.logging.events.IComponentIdentification;
import org.eclipse.hyades.logging.events.IReportSituation;
import org.eclipse.hyades.logging.events.ISituation;
import org.eclipse.hyades.logging.parsers.LogParserException;
import org.eclipse.hyades.logging.parsers.Parser;
import org.eclipse.hyades.logging.parsers.ParserUtilities;

/**
 * A simple parser created to illustrate how one can create their own
 * parsers for a given log file.
 */
public class MyParser extends Parser {

	/* The version of the software used to generate the log file */
	private String productVersion = null;

	/* The release (eg. Alpha, Beta, etc..) of the software used to generate the log file */
	private String productRelease = null;

	/* Keys for the product version & the product release */
	private final static String PRODUCT_VERSION_KEY = "PRODUCT_VERSION";
	private final static String PRODUCT_RELEASE_KEY = "PRODUCT_RELEASE";

	/* The source ID (which is simply the concatenation of the release and the version of the 
	   software used to generate the log file) */
	private String sourceID = null;

	/* The following field holds the creationTime of the previous log record */
	//NOTE:  MUST be initialized to an empty string (e.g. ""):
	private String previousCreationTime = "";

	/* Store the message count */
	int messageCounter = 0;

	/* The following method is invoked right after the user clicks on the 'finish' button in the 
	 * import wizard page. This method sets the appropriate fields based on the user's input */

	public void setUserInput(Hashtable table) throws LogParserException {

		/* The following line of code sets file_path to the path that has been chosen by the user */
		super.setUserInput(table);

		/* Store the product version and the product release */
		productVersion = ((String) (table.get(PRODUCT_VERSION_KEY)));
		productRelease = ((String) (table.get(PRODUCT_RELEASE_KEY)));

		/* Create the sourceID */
		if (productRelease != null) {

			if (productVersion != null) {
				sourceID = productRelease.concat(" ").concat(productVersion);
			}
			else {
				sourceID = productRelease;
			}
		}
	}

	/* The following method is invoked right after setUserInput(Hashtable) in order to parse the log file */
	public void parse(Log log) throws LogParserException {

		/* The following call causes the initialization of a few generalized fields */
		super.parse(log);

		/* Attempting to parse the log file */
		try {
			/* Read the log file line by line */
			while (curLine != null) {

				curLine = curLine.trim();

				/* Go to the next line if the current line does not begin with "##" */
				if ((curLine.length() > 0) && (curLine.charAt(0) == '#') && (curLine.charAt(1) == '#')) {

					/* Initialize the Common Base Event. */
					messages[arrayIndex].init();

					//Create a new instance of a Source Component:
					IComponentIdentification sourceComponentId = eventFactory.createComponentIdentification();
					sourceComponentId.setLocation(localHostId);
					sourceComponentId.setLocationType(localHostIdFormat);
					sourceComponentId.setExecutionEnvironment("Java");
					sourceComponentId.setComponent(sourceID);
					sourceComponentId.setSubComponent("MyParser");
					sourceComponentId.setComponentIdType("Application");
					sourceComponentId.setComponentType("HyadesLoggingSamples");

					//Create a new instance of a report situation:
					IReportSituation reportSituation = eventFactory.createReportSituation();
					reportSituation.setReasoningScope("INTERNAL");
					reportSituation.setReportCategory("LOG");

					//Create a new instance of a situation:
					ISituation situation = eventFactory.createSituation();
					situation.setCategoryName("ReportSituation");
					situation.setSituationType(reportSituation);

					messages[arrayIndex].setSituation(situation);
					messages[arrayIndex].setSourceComponentId(sourceComponentId);

					int lessThanIndex = curLine.indexOf('<');
					int greaterThanIndex = curLine.indexOf('>', lessThanIndex);

					/* Find the severity of the log record */
					String severity = curLine.substring((lessThanIndex + 1), greaterThanIndex).trim();

					if (severity.equals("Information"))
						messages[arrayIndex].setSeverity(((short) (10)));
					else if (severity.equals("Error"))
						messages[arrayIndex].setSeverity(((short) (50)));
					else if (severity.equals("Warning"))
						messages[arrayIndex].setSeverity(((short) (30)));

					lessThanIndex = curLine.indexOf('<', greaterThanIndex);
					greaterThanIndex = curLine.indexOf('>', lessThanIndex);

					/* Find the creation time of the log record */
					String creationTime = findCreationTime(lessThanIndex, greaterThanIndex);
					messages[arrayIndex].setCreationTime(creationTime);

					/* Find the message count and the unique ID of the log record */
					if (previousCreationTime.equals(creationTime)) {
						messages[arrayIndex].setSequenceNumber(((short) (messageCounter++)));
					}
					else {
						messageCounter = 0;
						previousCreationTime = creationTime;
					}

					/* Find the text body of the log record */
					messages[arrayIndex].setMsg(curLine.substring(greaterThanIndex + 1).trim());

					/* Write 'messages' to the logger and read the next line */
					log.trace(messages[arrayIndex]);
				}

				curLine = readLine();
			}
		}
		catch (Throwable throwable) {
			ParserUtilities.exceptionHandler(throwable, curLineNumber, curLine, "Error parsing error log.");
		}
	}

	/* Store the creation time in the desirable format */
	private String findCreationTime(int startIndex, int stopIndex) {

		if ((curLine.charAt(startIndex) == '<') && (curLine.charAt(stopIndex) == '>')) {

			/* This is the format found in the log file */
			SimpleDateFormat dateFormat = new SimpleDateFormat("MM.d.yy.kk.mm.ss");
			Date date = dateFormat.parse(curLine.substring((startIndex + 1), stopIndex), new ParsePosition(0));

			/* This is the desirable format that we would like creation time to be in
			 * (The creation time has to be in this format) */
			dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

			StringBuffer creationTime = new StringBuffer(dateFormat.format(date));
			creationTime.replace(10, 11, "T");
			creationTime.append(".000000");

			return (creationTime.toString().trim());
		}

		return null;
	}

	/* Returns the name of this parser */
	public String getName() {
		return "Simple Parser";
	}

	/* Returns the version of this parser */
	public String getVersion() {
		return "V1.0";
	}
}
