
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.analysis.engine.ISymptomDatabase;
import org.eclipse.hyades.analysis.engine.Incident;
import org.eclipse.hyades.analysis.engine.Solution;
import org.eclipse.hyades.models.internal.sdb.SDBFactory;
import org.eclipse.hyades.models.internal.sdb.SDBMatchPattern;
import org.eclipse.hyades.models.internal.sdb.SDBRuntime;
import org.eclipse.hyades.models.internal.sdb.SDBSolution;
import org.eclipse.hyades.models.internal.sdb.SDBSymptom;
import org.eclipse.hyades.models.internal.sdb.impl.SDBResourceSetImpl;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Sample class to demonstrate the capability of creating a symptom database.
 * <p>
 * A symptom database is used by an analysis engine to analyze log records
 * within a log file.
 * <p>
 * 
 * 
 * @author Paul E. Slauenwhite
 * @version September 15, 2004
 * @since September 15, 2004
 * @see org.eclipse.hyades.analysis.engine.IAnalysisEngine
 */
public class SymptomDatabaseImpl implements ISymptomDatabase {

    private SDBRuntime symptomDatabase = null;

    private String symptomDatabasePath = null;

    /**
     * @see org.eclipse.hyades.analysis.engine.ISymptomDatabase#load()
     */
    public boolean load() {

        unload();

        symptomDatabase = loadDatabase(symptomDatabasePath);

        return (symptomDatabase != null);
    }

    /**
     * @see org.eclipse.hyades.analysis.engine.ISymptomDatabase#unload()
     */
    public boolean unload() {

        ResourceSet resourceSet = SDBResourceSetImpl.getInstance();

        if ((symptomDatabase != null) && (resourceSet != null)) {
            resourceSet.getResources().remove(symptomDatabase.eResource());
        }

        symptomDatabase = null;

        return true;
    }

    /**
     * @see org.eclipse.hyades.analysis.engine.ISymptomDatabase#replace(java.lang.String)
     */
    public boolean replace(String symptomDatabasePath) {

        this.symptomDatabasePath = symptomDatabasePath;

        if (symptomDatabasePath == null) {
            return (unload());
        }

        return (load());
    }

    /**
     * @see org.eclipse.hyades.analysis.engine.ISymptomDatabase#merge(java.lang.String)
     */
    public boolean merge(String symptomDatabasePath) {

        SDBRuntime newDatabase = loadDatabase(symptomDatabasePath);

        if (newDatabase == null) {
            return false;
        }

        if (symptomDatabase == null) {
            symptomDatabase = newDatabase;
        }

        else {

            List symptoms = newDatabase.getSymptoms();
            SDBSymptom symptom = null;
            String symptomDescription = null;
            List symptomDatabaseSymptoms = null;

            for (int counter = 0; counter < symptoms.size(); counter++) {

                boolean found = false;
                symptom = ((SDBSymptom) (symptoms.get(counter)));
                symptomDescription = symptom.getDescription();
                symptomDatabaseSymptoms = symptomDatabase.getSymptoms();

                for (int index = 0; index < symptomDatabaseSymptoms.size(); index++) {

                    if (symptomDescription.equals(((SDBSymptom) (symptomDatabaseSymptoms.get(index))).getDescription())) {
                        found = true;
                        break;
                    }
                }

                if (!found) {

                    SDBSymptom newSymptom = SDBFactory.eINSTANCE.createSDBSymptom();
                    newSymptom.setDescription(symptom.getDescription());
                    symptomDatabase.getSymptoms().add(newSymptom);
                    List patterns = symptom.getPatterns();
                    SDBMatchPattern pattern = null;
                    SDBMatchPattern newPattern = null;

                    for (int index = 0; index < patterns.size(); index++) {

                        pattern = ((SDBMatchPattern) (patterns.get(index)));

                        newPattern = SDBFactory.eINSTANCE.createSDBMatchPattern();
                        newPattern.setName(pattern.getName());
                        newPattern.setValue(pattern.getValue());

                        newSymptom.getPatterns().add(newPattern);
                    }

                    List solutions = symptom.getSolutions();
                    SDBSolution solution = null;
                    SDBSolution newSolution = null;

                    for (int index = 0; index < solutions.size(); index++) {

                        solution = (SDBSolution) solutions.get(index);

                        newSolution = SDBFactory.eINSTANCE.createSDBSolution();
                        newSolution.setDescription(solution.getDescription());

                        newSymptom.getSolutions().add(newSolution);

                        symptomDatabase.getSolutions().add(newSolution);
                    }
                }
            }
        }

        return true;
    }

    /**
     * @see org.eclipse.hyades.analysis.engine.ISymptomDatabase#getPath()
     */
    public String getPath() {
        return symptomDatabasePath;
    }

    /**
     * @see org.eclipse.hyades.analysis.engine.ISymptomDatabase#getDirectives(org.eclipse.hyades.analysis.engine.Incident)
     */
    public Object[] getDirectives(Incident incident) {

        ArrayList directives = new ArrayList();

        if (symptomDatabase != null) {

            String incidentId = incident.getMessageId();

            if ((incidentId != null) && (!incidentId.trim().equals(""))) {

                List symptoms = symptomDatabase.getSymptoms();
                SDBSymptom symptom = null;
                StringTokenizer tokens = null;
                boolean match = false;

                for (int counter = 0; counter < symptoms.size(); counter++) {

                    symptom = (SDBSymptom) symptoms.get(counter);

                    if (symptom != null) {

                        List patterns = symptom.getPatterns();

                        if (patterns.size() == 0) {

                            match = true;

                            if ((symptom.getDescription().trim().equals("")) || (incidentId.indexOf(symptom.getDescription()) == -1)) {

                                match = false;
                                continue;
                            }
                        } else {

                            int nullStringCount = 0;
                            match = (patterns.size() > 0);
                            SDBMatchPattern pattern = null;

                            for (int index = 0; index < patterns.size(); index++) {

                                pattern = (SDBMatchPattern) patterns.get(index);

                                if (pattern.getValue().equals("")) {
                                    nullStringCount++;
                                } 
                                else if (incidentId.indexOf(pattern.getValue()) == -1) {
                                    match = false;
                                    break;
                                }
                            }
                            if (nullStringCount == patterns.size()) {

                                match = true;

                                if (incidentId.indexOf(symptom.getDescription()) == -1) {
                                    match = false;
                                    continue;
                                }
                            }
                        }

                        if (match) {
                            directives.add(symptom);
                        }
                    }
                }
            }
        }

        return (directives.toArray());
    }

    /**
     * @see org.eclipse.hyades.analysis.engine.ISymptomDatabase#getSolutions(org.eclipse.hyades.analysis.engine.Incident)
     */
    public Solution[] getSolutions(Incident incident) {
        
        Object[] directives = getDirectives(incident);
        ArrayList solutions = new ArrayList();
        
        for (int counter = 0; counter < directives.length; counter++) {            
            solutions.addAll(((SDBSymptom)(directives[counter])).getSolutions());
        }
        
        return ((Solution[])(solutions.toArray(new Solution[solutions.size()])));
    }

    private SDBRuntime loadDatabase(String path) {

        Resource resource = null;
        ResourceSet resourceSet = SDBResourceSetImpl.getInstance();

        if (resourceSet != null) {

            if (path.startsWith("platform:/resource")) {
                resource = resourceSet.getResource(URI.createURI(path), true);
            } else {
                resource = resourceSet.getResource(URI.createFileURI(path), true);
            }
        }

        Iterator iterator = resource.getContents().iterator();
        Object object = null;

        while (iterator.hasNext()) {

            object = iterator.next();

            if (object instanceof SDBRuntime) {
                return ((SDBRuntime) (object));
            }
        }

        return null;
    }
}