/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.trace;

import org.eclipse.hyades.loaders.trace.TraceUtils.InvocationInfo;
import org.eclipse.hyades.loaders.util.HierarchyContext;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.loaders.util.LookupServiceExtensions;
import org.eclipse.hyades.models.trace.TRCFullMethodInvocation;
import org.eclipse.hyades.models.trace.TRCFullTraceObject;


/**
 * @author slavescu
 *
 */
public class XMLmethodExitLoader extends TraceMethodBaseLoader {
    //~ Static fields/initializers -----------------------------------------------------------------

    //  protected static final String TICKET = "ticket";
    //	protected static final String STACK_DEPTH = "stackDepth";
    protected static final String OVERHEAD = "overhead";

    //~ Instance fields ----------------------------------------------------------------------------

    private double overhead;
    private long sequenceCounter;

    //~ Methods ------------------------------------------------------------------------------------

    public void addAttribute(String name, String value) {
        if (!loadToModel) {
            return;
        }

        switch (LoadersUtils.getHashCode(name)) {
        case TraceConstants.OVERHEAD_int:
            overhead = Double.parseDouble(value);

            break;

        case TraceConstants.STACK_DEPTH_int:
            stackDepth = Short.parseShort(value);

            break;

        case TraceConstants.TICKET_int:
            ticket = Long.parseLong(value);

            break;

        default:
            super.addAttribute(name, value);

            break;
        }
    }

    public void addYourselfInContext() {
        if (!loadToModel) {
            return;
        }

        cs = (CallStackPerThread) LookupServiceExtensions.getInstance().locate(context, CallStackPerThread.class, LoadersUtils.getLookUpKey(threadIdRef));

        if ((cs == null) || cs.isEmpty()) {
            return;
        }

        invocationPool = cs.invocationPool;

        theProcess = getProcess();

        dispatchProcessMode(ProcessSteps.ALL);
    }

    public void initialize(HierarchyContext context, String name) {
        loadToModel = context.isLoadToModel();

        if (!loadToModel) {
            return;
        }

        super.initialize(context, name);
        overhead = 0;
        stackDepth = 0;
        ticket = 0;
        sequenceCounter = 0;
    }

    protected void processEF(int step) {
        super.processEF(step);

        fullInvocation = ((InvocationInfo) cs.peek()).getMethodInvocation();

        if (fullInvocation != null) {
            theThread = fullInvocation.getThread();
            fullInvocation.setExitTime(createDeltaTime());
            fullInvocation.setOverhead(overhead);

            updateStatisticalInfo();
            invocationPool.release((InvocationInfo) cs.pop());
        } else {
            // ignore method exit if there is no method entry
        }
    }

    protected void processENI(int step) {
        super.processENI(step);
        processEF(step);
    }

    protected void processES(int step) {
        super.processES(step);

        //		theThread = getThreadByIdRef(theProcess);
        //		theMethod = fullInvocation.getMethod();
        //		theObject = fullInvocation.getOwningObject();
        //		theClass = theMethod.getDefiningClass();
        updateStatisticalInfoOnly();
    }

    private void updateStatisticalInfo() {
        try {
            int size = fullInvocation.getInvokes().size();

            if (size > 0) {
                //this "invocation" has children
                lastChildExitTime = ((TRCFullMethodInvocation) fullInvocation.getInvokes().get(size - 1)).getExitTime();
                deltaBaseTime = fullInvocation.getExitTime() - lastChildExitTime - fullInvocation.getOverhead();
            } else {
                //this "invocation" doesn't have children
                deltaBaseTime = fullInvocation.getExitTime() - fullInvocation.getEntryTime() - fullInvocation.getOverhead();
            }

			invokerMethod = fullInvocation.getMethod();
			invokerObject = (TRCFullTraceObject) fullInvocation.getOwningObject();
			invokerClass = fullInvocation.getMethod().getDefiningClass();

			invokerObjectClass = getExtendedClass(invokerObject,invokerClass);
			
            //update base and cumulative using deltaBaseTime, invokerObject and invokerClass
            updateTimeStatistics();

            //			p.setCurrentTime(createTimeDouble());
        } catch (Exception e) {
			LoadersUtils.log(e);
        }
    }

    private void updateStatisticalInfoOnly() {
        try {
            InvocationInfo currentInvocation = (InvocationInfo) cs.peek();
            double exitTime = createDeltaTime();

            lastChildExitTime = currentInvocation.getLastChildExitTime();

            if (lastChildExitTime > 0) {
                deltaBaseTime = exitTime - lastChildExitTime - overhead;
            } else {
                deltaBaseTime = exitTime - currentInvocation.getEntryTime() - overhead;
            }

            invokerObject = currentInvocation.getObject();

            invokerClass = currentInvocation.getTheClass();

            invokerObjectClass = currentInvocation.getObjectClass();

            invokerMethod = currentInvocation.getMethod();

            //update base and cumulative time using deltaBaseTime, invokerObject and invokerClass
            updateTimeStatistics();

            //remove current invocation
            invocationPool.release((InvocationInfo) cs.pop());

            if (!cs.isEmpty()) {
                //update current invocation parent 
                InvocationInfo invoker = (InvocationInfo) cs.peek();

                if (invoker != null) {
                    invoker.setLastChildExitTime(exitTime);
                }
            }

            theProcess.setLastEventTime(exitTime);
        } catch (Exception e) {
			LoadersUtils.log(e);
        }
    }
}
