package org.eclipse.hyades.internal.config.generator;

import java.io.FileWriter;
import java.io.PrintWriter;
import java.lang.reflect.Method;
import java.util.Hashtable;
import java.util.Stack;

import org.apache.xerces.dom.DOMImplementationImpl;
import org.apache.xerces.dom.DocumentImpl;
import org.apache.xerces.dom.DocumentTypeImpl;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;
import org.xml.sax.helpers.XMLReaderFactory;

public class ConfigFileImpl extends DefaultHandler implements ConfigFile {
	protected static String TAG = ConfigUtility.getString("Config.AgentControllerConfig.Tag");

	protected DocumentImpl doc = null;
	protected AgentControllerConfig acConfig = null;
	protected Element holder = null;

	private int size = 100;
	private Hashtable hash = new Hashtable(size);

	private Stack stack = null;
	private String parserName = "org.apache.xerces.parsers.SAXParser";
	private String sr = PlatformObject.sr;

	public ConfigFileImpl() {
		createNewDocument();
	}

	public ConfigFileImpl(String fileName) {
		this();
		loadExistingDocument(fileName);
	}

	public void saveToFile() {
		String filename = new String(getValue("RASERVER_HOME") + sr + "config" + sr + "serviceconfig.xml");
		saveToFile(filename);
	}

	public void saveToFile(String fileName) {
		PrintWriter writer = null;

		try {
			writer = new PrintWriter(new FileWriter(fileName));
		}
		catch(Exception e) {
			Logger.err(ConfigUtility.getString("Config.Writer.Error.CannotWriteFile") + " " + fileName);
			Logger.err(e.getMessage());
		}

		if(writer != null) {
			writer.print(toString());
			writer.flush();
			writer.close();
		}
	}

	public String toString() {
		StringBuffer strbuf = new StringBuffer();

		//
		// XML header information
		//
		strbuf.append("<?xml");

		String encoding = PlatformObject.encoding; // Set a default value
		String version = "1.0"; // Set a default value

		// Bug 54730
		try {
			Class docCls = Class.forName("org.apache.xerces.dom.CoreDocumentImpl");
			Class arg[] = { };
			Method mGetEncoding = null;
			Method mGetVersion = null;
			try {
				// Try the previous declaration (pre-Xerces2) e.g. JDK 1.3.1
				mGetEncoding = docCls.getDeclaredMethod("getEncoding", arg);
				mGetVersion = docCls.getDeclaredMethod("getVersion", arg);
			} catch (SecurityException e1) {
				// Ignore this exception
			} catch (NoSuchMethodException e1) {
				try {
					// Try the new declaration (Xerces2) e.g. JDK 1.4.2
					mGetEncoding = docCls.getDeclaredMethod("getXmlEncoding", arg);
					mGetVersion = docCls.getDeclaredMethod("getXmlVersion", arg);
				} catch (SecurityException e2) {
					// Ignore this exception
				} catch (NoSuchMethodException e2) {
					// e2.printStackTrace();
				}
			}
			if((mGetEncoding != null) && (mGetVersion != null)) {
				encoding = (String)mGetEncoding.invoke(doc, arg);
				version = (String)mGetVersion.invoke(doc, arg);
			}
		} catch(Exception e) {
			e.printStackTrace();
		}

		strbuf.append(" version=\"" + version + "\"");
		strbuf.append(" encoding=\"" + encoding + "\"");

		strbuf.append("?>\n");

		//
		// DOCTYPE information
		//
		strbuf.append("<!DOCTYPE");
		strbuf.append(" " + doc.getDoctype().getName());
		strbuf.append(" " + doc.getDoctype().getSystemId());
		strbuf.append(" " + doc.getDoctype().getPublicId());
		strbuf.append(">\n");

		//
		// Print each child node
		//
		NodeList nodes = holder.getChildNodes();
		if(nodes != null) {
			for(int i = 0; i < nodes.getLength(); i++) {
				strbuf.append(ConfigUtility.print(nodes.item(i)));
			}
		}

		return strbuf.toString();
	}

	public void generateConfiguration() {
		generateHyadesConfiguration();
	}

	/**
	 * Generate the basic Hyades Data Collection Engine configuration
	 */
	public void generateHyadesConfiguration() {
		AgentControllerEnvironment acEnv;
		Application app;
		Variable var;
		Parameter param;
		Hosts hosts;
		Allow allow;
		Agent agent;
		Option option;
		String sr = PlatformObject.sr;

		//
		// Set the version to current
		//
		acConfig.setVersion(ConfigUtility.getString("Config.AgentController.Version"));

		//
		// Set the Jvm location
		//
		acConfig.setJvm(resolveJvmDll(getValue("JAVA_PATH")));

		//
		// Agent Controller Environment
		//
		acEnv = acConfig.addAgentControllerEnvironment("default");

		var = acEnv.addVariable();
		var.setName("JAVA_PATH");
		var.setValue(getValue("JAVA_PATH"));
		var.setPosition("replace");

		var = acEnv.addVariable();
		var.setName("RASERVER_HOME");
		var.setValue(getValue("RASERVER_HOME"));
		var.setPosition("replace");

		var = acEnv.addVariable();
		var.setName(PlatformObject.libEnv);
		var.setValue("%RASERVER_HOME%" + sr + PlatformObject.libPath);
		var.setPosition("prepend");

		var = acEnv.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "hcframe.jar");
		var.setPosition("prepend");

		var = acEnv.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "hexl.jar");
		var.setPosition("prepend");

		var = acEnv.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "hexr.jar");
		var.setPosition("prepend");

		var = acEnv.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "hexcore.jar");
		var.setPosition("prepend");

		var = acEnv.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "hlcore.jar");
		var.setPosition("prepend");

		var = acEnv.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "hlevents.jar");
		var.setPosition("prepend");

		//
		// Application: java.exe
		//
		app = acConfig.addApplication("default", "java.exe");
		app.setPath("%JAVA_PATH%");
		app.setLocation(PlatformObject.tempDir);

		//
		// Application: RemoteLogParserLoader
		//
		app = acConfig.addApplication("default", "RemoteLogParserLoader");
		app.setPath("%JAVA_PATH%");
		app.setLocation("%RASERVER_HOME%" + sr + "config");

		param = app.addParameter();
		param.setValue("org.eclipse.hyades.logging.parsers.RemoteLogParserLoader");
		param.setPosition("prepend");

		var = app.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "commons-logging.jar");
		var.setPosition("prepend");

		var = app.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "hlcommons.jar");
		var.setPosition("prepend");

		var = app.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "hparse.jar");
		var.setPosition("prepend");

		var = app.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "hglaconfig.jar");
		var.setPosition("prepend");

		var = app.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "hgla.jar");
		var.setPosition("prepend");

		var = app.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "jakarta-oro.jar");
		var.setPosition("prepend");

		var = app.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "xercesImpl.jar");
		var.setPosition("prepend");

		var = app.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "xmlParserAPIs.jar");
		var.setPosition("prepend");

		//
		// Application: HyadesTestingSession
		//
		app = acConfig.addApplication("default", "HyadesTestingSession");
		app.setPath("%JAVA_PATH%");
		app.setLocation(PlatformObject.tempDir);

		param = app.addParameter();
		param.setValue("&quot;-Dhyades.test.JVM=%JAVA_PATH%&quot;");
		param.setPosition("prepend");

		var = app.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "common.runner.jar");
		var.setPosition("append");

		var = app.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "java.runner.jar");
		var.setPosition("append");

		var = app.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "http.runner.jar");
		var.setPosition("append");

		var = app.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "junit.jar");
		var.setPosition("append");

		var = app.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "xerces.jar");
		var.setPosition("append");

		var = app.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "xercesImpl.jar");
		var.setPosition("append");

		var = app.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "manual.runner.jar");
		var.setPosition("append");

		var = app.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "hexrecr.jar");
		var.setPosition("append");

		var = app.addVariable();
		var.setName("CLASSPATH");
		var.setValue("%RASERVER_HOME%" + sr + "lib" + sr + "http.hexrecr.jar");
		var.setPosition("append");

		//
		// Application: PerfmonAgent
		//
		app = acConfig.addApplication("default", "PerfmonAgent");
		app.setPath("%RASERVER_HOME%" + sr + "bin" + sr + "PerfmonAgent" + PlatformObject.exeExt);
		app.setLocation("%RASERVER_HOME%" + sr + "bin");

		//
		// Hosts
		//
		hosts = acConfig.addHosts("default");

		if(getValue("ALLOW") != null) {
			if(getValue("ALLOW").toLowerCase().equals("custom")) {
				String[] allowedHostList = ConfigUtility.stringToArray(getValue("HOSTS"));
				if(allowedHostList.length > 0) {
					for(int i = 0; i < allowedHostList.length; i++) {
						allow = hosts.addAllow();
						if(allowedHostList[i].equalsIgnoreCase("all") || allowedHostList[i].equalsIgnoreCase("local")) {
							allow.setHost(allowedHostList[i].toUpperCase());
						}
						else {
							allow.setHost(allowedHostList[i]);
						}
					}
				}
				else {
					allow = hosts.addAllow();
					allow.setHost("LOCAL");
				}
			}
			else {
				allow = hosts.addAllow();
				allow.setHost(getValue("ALLOW").toUpperCase());
			}
		}
		else {
			allow = hosts.addAllow();
			allow.setHost("LOCAL");
		}

		//
		// Agents
		//
		agent = acConfig.addAgent("default", "Java Profiling Agent");
		agent.setType("Profiler");
		agent.setDataChannelSize(PlatformObject.dataSize);
		agent.setClient("DYNAMIC");

		//
		// Option
		//
		option = acConfig.addOption("Thread Analysis Agent");
		option.setType("Profiler");
		option.setValue("hcthread");
	}

	public void createNewDocument() {
		DOMImplementation dom = DOMImplementationImpl.getDOMImplementation();
		DocumentTypeImpl dtd = null;

		dtd = (DocumentTypeImpl)dom.createDocumentType(TAG, "\"../dtd/serviceconfig.dtd\"", "SYSTEM");
		doc = (DocumentImpl)dom.createDocument(null, TAG, dtd);

		// Bug 54730
		try {
			Class docCls = Class.forName("org.apache.xerces.dom.CoreDocumentImpl");
			Class arg[] = { new String().getClass() };
			Method mSetEncoding = null;
			Method mSetVersion = null;
			try {
				// Try the previous declaration (pre-Xerces2) e.g. JDK 1.3.1
				mSetEncoding = docCls.getDeclaredMethod("setEncoding", arg);
				mSetVersion = docCls.getDeclaredMethod("setVersion", arg);
			} catch (SecurityException e1) {
				// Ignore this exception
			} catch (NoSuchMethodException e1) {
				try {
					// Try the new declaration (Xerces2) e.g. JDK 1.4.2
					mSetEncoding = docCls.getDeclaredMethod("setXmlEncoding", arg);
					mSetVersion = docCls.getDeclaredMethod("setXmlVersion", arg);
				} catch (SecurityException e2) {
					// Ignore this exception
				} catch (NoSuchMethodException e2) {
					// e2.printStackTrace();
				}
			}
			if((mSetEncoding != null) && (mSetVersion != null)) {
				String[] encoding = { new String(PlatformObject.encoding) };
				String[] version = { new String("1.0") };
				mSetEncoding.invoke(doc, encoding);
				mSetVersion.invoke(doc, version);
			}
		} catch(Exception e) {
			e.printStackTrace();
		}

		holder = doc.createElement("ElementHolder");

		//
		// There can be only one AgentControllerConfiguration element
		//
		acConfig = new AgentControllerConfigImpl(doc);
		acConfig.setActiveConfiguration("default");
		acConfig.setLoggingDetail("LOW");
		acConfig.setLoggingLevel("WARNING");
		acConfig.setPort("10002");
		acConfig.setSecurePort("10003");
		acConfig.setVersion(ConfigUtility.getString("Config.AgentController.Version"));
		holder.appendChild(acConfig); // Add to the root holder
	}

	public void loadExistingDocument(String file) {
		XMLReader parser = null;
		stack = new Stack();
		try {
			parser = XMLReaderFactory.createXMLReader(parserName);
			parser.setContentHandler(this);
			parser.setErrorHandler(this);
		} catch (Exception e) {
			Logger.err(ConfigUtility.getString("Config.Parser.Error.CreateParser") + " " + parserName);
			Logger.err(e.getMessage());
		}

		if(parser != null) {
			try {
				parser.parse(file);
			} catch (Exception e) {
				Logger.err(ConfigUtility.getString("Config.Parser.Error.ParseFile") + " " + file);
				Logger.err(e.getMessage());
				if(e instanceof SAXException) {
					Exception e1 = ((SAXException)e).getException();
					if(e1 != null) {
						Logger.err(e1.getMessage());
					}
				}
			}
		}
	}

	public void startDocument() throws SAXException {
		createNewDocument();
	}

	public void startElement(String uri, String local, String raw, Attributes attrs) throws SAXException {
		//
		// AgentControllerConfiguration
		//
		if(local.equals(AgentControllerConfig.TAG)) {
			if(stack.empty()) {
				for(int i = 0; i < attrs.getLength(); i++) {
					if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.AgentControllerConfig.ActiveConfig.Tag"))) {
						acConfig.setActiveConfiguration(attrs.getValue(i));
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.AgentControllerConfig.Jvm.Tag"))) {
						acConfig.setJvm(attrs.getValue(i));
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.AgentControllerConfig.LoggingDetail.Tag"))) {
						acConfig.setLoggingDetail(attrs.getValue(i));
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.AgentControllerConfig.LoggingLevel.Tag"))) {
						acConfig.setLoggingLevel(attrs.getValue(i));
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.AgentControllerConfig.Port.Tag"))) {
						acConfig.setPort(attrs.getValue(i));
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.AgentControllerConfig.SecurePort.Tag"))) {
						acConfig.setSecurePort(attrs.getValue(i));
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.AgentControllerConfig.Version.Tag"))) {
						acConfig.setVersion(attrs.getValue(i));
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.AgentControllerConfig.Configuration.Tag"))) {
						acConfig.setConfiguration(attrs.getValue(i));
					}
					else {
						Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getLocalName(i));
					}
				}
				holder.appendChild(acConfig);
				stack.push(acConfig);
			}
			else {
				Logger.err(ConfigUtility.getString("Config.Parser.Error.OwnerNodeMismatch"));
			}
		}

		//
		// AgentControllerEnvironment
		//
		else if(local.equals(AgentControllerEnvironment.TAG)) {
			if(stack.peek() instanceof AgentControllerConfig) {
				AgentControllerEnvironment acEnv = null;
				String config = null;

				for(int i = 0; i < attrs.getLength(); i++) {
					if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.AgentControllerConfig.Configuration.Tag"))) {
						config = attrs.getValue(i);
					}
					else {
						Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getLocalName(i));
					}
				}

				if(config != null) {
					acEnv = ((AgentControllerConfig)(stack.peek())).addAgentControllerEnvironment(config);
				}

				if(acEnv != null) {
					stack.push(acEnv);
				}
				else {
					Logger.err(ConfigUtility.getString("Config.Parser.Error.MissingConfiguration") + " " + local);
				}
			}
			else {
				Logger.err(ConfigUtility.getString("Config.Parser.Error.OwnerNodeMismatch"));
			}
		}

		//
		// Application
		//
		else if(local.equals(Application.TAG)) {
			if(stack.peek() instanceof AgentControllerConfig) {
				Application app = null;
				String config = null;
				String exe = null;
				String location = null;
				String path = null;

				for(int i = 0; i < attrs.getLength(); i++) {
					if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.AgentControllerConfig.Configuration.Tag"))) {
						config = attrs.getValue(i);
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Application.Executable.Tag"))) {
						exe = attrs.getValue(i);
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Application.Location.Tag"))) {
						location = attrs.getValue(i);
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Application.Path.Tag"))) {
						path = attrs.getValue(i);
					}
					else {
						Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getLocalName(i));
					}
				}

				if((config != null) && (exe != null)) {
					app = ((AgentControllerConfig)(stack.peek())).addApplication(config, exe);
					app.setLocation(location);
					app.setPath(path);
				}

				if(app != null) {
					stack.push(app);
				}
				else {
					Logger.err(ConfigUtility.getString("Config.Parser.Error.MissingConfiguration") + " " + local);
				}
			}
			else {
				Logger.err(ConfigUtility.getString("Config.Parser.Error.OwnerNodeMismatch"));
			}
		}

		//
		// Hosts
		//
		else if(local.equals(Hosts.TAG)) {
			if(stack.peek() instanceof AgentControllerConfig) {
				Hosts hosts = null;
				String name = null;
				for(int i = 0; i < attrs.getLength(); i++) {
					if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.AgentControllerConfig.Configuration.Tag"))) {
						name = attrs.getValue(i);
					}
					else {
						Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getLocalName(i));
					}
				}

				if(name != null) {
					hosts = ((AgentControllerConfig)(stack.peek())).addHosts(name);
				}

				if(hosts != null) {
					stack.push(hosts);
				}
				else {
					Logger.err(ConfigUtility.getString("Config.Parser.Error.MissingConfiguration") + " " + local);
				}
			}
			else {
				Logger.err(ConfigUtility.getString("Config.Parser.Error.OwnerNodeMismatch"));
			}
		}

		//
		// Agent
		//
		else if(local.equals(Agent.TAG)) {
			if(stack.peek() instanceof AgentControllerConfig) {
				Agent agent = null;
				String config = null;
				String name = null;
				String dataSize = null;
				String type = null;

				for(int i = 0; i < attrs.getLength(); i++) {
					if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.AgentControllerConfig.Configuration.Tag"))) {
						config = attrs.getValue(i);
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Agent.DataChannelSize.Tag"))) {
						dataSize = attrs.getValue(i);
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Agent.Name.Tag"))) {
						name = attrs.getValue(i);
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Agent.Type.Tag"))) {
						type = attrs.getValue(i);
					}
					else {
						Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getLocalName(i));
					}
				}
				if((config != null) && (name != null)) {
					agent = ((AgentControllerConfig)(stack.peek())).addAgent(config, name);
					agent.setDataChannelSize(dataSize);
					agent.setType(type);
				}
				if(agent != null) {
					stack.push(agent);
				}
				else {
					Logger.err(ConfigUtility.getString("Config.Parser.Error.MissingConfiguration") + " " + local);
				}
			}
			else {
				Logger.err(ConfigUtility.getString("Config.Parser.Error.OwnerNodeMismatch"));
			}
		}

		//
		// Option
		//
		else if(local.equals(Option.TAG)) {
			if(stack.peek() instanceof Agent) {
				Option option = null;
				String name = null;
				String type = null;
				String value = null;

				for(int i = 0; i < attrs.getLength(); i++) {
					if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Option.Name.Tag"))) {
						name = attrs.getValue(i);
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Option.Type.Tag"))) {
						type = attrs.getValue(i);
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Option.Value.Tag"))) {
						value = attrs.getValue(i);
					}
					else {
						Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getLocalName(i));
					}
				}
				if((name != null) && (type != null) && (value != null)) {
					option = ((Agent)(stack.peek())).addOption();
					option.setName(name);
					option.setType(type);
					option.setValue(value);
				}
				if(option != null) {
					stack.push(option);
				}
				else {
					Logger.err(ConfigUtility.getString("Config.Parser.Error.MissingConfiguration") + " " + local);
				}
			}
			else {
				Logger.err(ConfigUtility.getString("Config.Parser.Error.OwnerNodeMismatch"));
			}
		}

		//
		// Security
		//
		else if(local.equals(Security.TAG)) {
			if(stack.peek() instanceof AgentControllerConfig) {
				Security security = null;
				String aclEnabled = null;
				String clientAuthenticate = null;
				String keystore = null;
				String keystoreManager = null;
				String keystorePassword = null;
				String provider = null;

				for(int i = 0; i < attrs.getLength(); i++) {
					if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Security.AclEnabled.Tag"))) {
						aclEnabled = attrs.getValue(i);
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Security.ClientAuthenticate.Tag"))) {
						clientAuthenticate = attrs.getValue(i);
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Security.Keystore.Tag"))) {
						keystore = attrs.getValue(i);
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Security.KeystoreManager.Tag"))) {
						keystoreManager = attrs.getValue(i);
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Security.KeystorePassword.Tag"))) {
						keystorePassword = attrs.getValue(i);
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Security.Provider.Tag"))) {
						provider = attrs.getValue(i);
					}
					else {
						Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getLocalName(i));
					}
				}

				if((aclEnabled != null) && (clientAuthenticate != null) && (keystore != null) && (keystoreManager != null) && (keystorePassword != null) && (provider != null)) {
					security = ((AgentControllerConfig)(stack.peek())).addSecurity();
					security.setAclEnabled(aclEnabled);
					security.setClientAuthenticate(clientAuthenticate);
					security.setKeystore(keystore);
					security.setKeystoreManager(keystoreManager);
					security.setKeystorePassword(keystorePassword);
					security.setProvider(provider);
					setValue("SECURITY", "true");
				}
				else {
					setValue("SECURITY", "false");
				}

				if(security != null) {
					stack.push(security);
				}
				else {
					Logger.err(ConfigUtility.getString("Config.Parser.Error.MissingConfiguration") + " " + local);
				}
			}
			else {
				Logger.err(ConfigUtility.getString("Config.Parser.Error.OwnerNodeMismatch"));
			}
		}

		//
		// UserDefinition
		//
		else if(local.equals(UserDefinition.TAG)) {
			if(stack.peek() instanceof Security) {
				UserDefinition userDefinition = null;
				String authenticationType = null;
				String name = null;

				for(int i = 0; i < attrs.getLength(); i++) {
					if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.UserDefinition.AuthenticationType.Tag"))) {
						authenticationType = attrs.getValue(i);
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.UserDefinition.Name.Tag"))) {
						name = attrs.getValue(i);
					}
					else {
						Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getLocalName(i));
					}
				}
				if((authenticationType != null) && (name != null)) {
					userDefinition = ((Security)(stack.peek())).addUserDefinition();
					userDefinition.setAuthenticationType(authenticationType);
					userDefinition.setName(name);

					if(getValue("USERS") == null) {
						setValue("USERS", name);
					}
					else {
						setValue("USERS", getValue("USERS") + "," + name);
					}

				}
				if(authenticationType != null) {
					stack.push(authenticationType);
				}
				else {
					Logger.err(ConfigUtility.getString("Config.Parser.Error.MissingConfiguration") + " " + local);
				}
			}
			else {
				Logger.err(ConfigUtility.getString("Config.Parser.Error.OwnerNodeMismatch"));
			}
		}

		//
		// Variable
		//
		else if(local.equals(Variable.TAG)) {
			Variable var = null;
			String name = null;
			String value = null;
			String position = null;

			for(int i = 0; i < attrs.getLength(); i++) {
				if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Variable.Name.Tag"))) {
					name = attrs.getValue(i);
				}
				else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Variable.Value.Tag"))) {
					value = attrs.getValue(i);
				}
				else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Variable.Position.Tag"))) {
					position = attrs.getValue(i);
				}
				else {
					Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getLocalName(i));
				}
			}

			if((name != null) && (value != null) && (position != null)) {
				if(stack.peek() instanceof AgentControllerEnvironment) {
					var = ((AgentControllerEnvironment)(stack.peek())).addVariable();
					var.setName(name);
					var.setValue(value);
					var.setPosition(position);
					//
					// Populate the hash table, do not add CLASSPATH or PATH
					//
					if(!name.equals("CLASSPATH") && !name.equals("PATH")) {
						setValue(name, value);
					}
				}
				else if(stack.peek() instanceof Application) {
					var = ((Application)(stack.peek())).addVariable();
					var.setName(name);
					var.setValue(value);
					var.setPosition(position);
				}
				else {
					Logger.err(ConfigUtility.getString("Config.Parser.Error.OwnerNodeMismatch"));
				}
			}

			if(var != null) {
				stack.push(var);
			}
		}

		//
		// Parameter
		//
		else if(local.equals(Parameter.TAG)) {
			Parameter param = null;
			String value = null;
			String position = null;

			if(stack.peek() instanceof Application) {
				for(int i = 0; i < attrs.getLength(); i++) {
					if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Parameter.Value.Tag"))) {
						value = attrs.getValue(i);
					}
					else if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Parameter.Position.Tag"))) {
						position = attrs.getValue(i);
					}
					else {
						Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getLocalName(i));
					}
				}

				if((value != null) && (position != null)) {
					param = ((Application)(stack.peek())).addParameter();
					param.setValue(value);
					param.setPosition(position);
				}

				if(param != null) {
					stack.push(param);
				}
			}
			else {
				Logger.err(ConfigUtility.getString("Config.Parser.Error.OwnerNodeMismatch"));
			}
		}

		//
		// Allow
		//
		else if(local.equals(Allow.TAG)) {
			Allow allow = null;
			String name = null;

			if(stack.peek() instanceof Hosts) {
				for(int i = 0; i < attrs.getLength(); i++) {
					if(attrs.getLocalName(i).equals(ConfigUtility.getString("Config.Allow.Host.Tag"))) {
						name = attrs.getValue(i);
					}
					else {
						Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getLocalName(i));
					}
				}

				if(name != null) {
					allow = ((Hosts)(stack.peek())).addAllow();
					allow.setHost(name);

					if(name.toLowerCase().equals("all") || name.toLowerCase().equals("local")) {
						setValue("ALLOW", name.toUpperCase());
						removeKey("HOSTS");
					}
					else {
						setValue("ALLOW", "CUSTOM");
						if(getValue("HOSTS") == null) {
							setValue("HOSTS", name);
						}
						else {
							setValue("HOSTS", getValue("HOSTS") + "," + name);
						}
					}
				}

				if(allow != null) {
					stack.push(allow);
				}
			}
			else {
				Logger.err(ConfigUtility.getString("Config.Parser.Error.OwnerNodeMismatch"));
			}
		}

		//
		// Unrecognized element name
		//
		else {
			Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedElement") + " " + local);
			stack.push(null); // push a null object for the pop()
		}
	}

	public void endElement(String uri, String local, String raw) throws SAXException {
		stack.pop();
	}

	public String resolveJvmDll(String string) {
		if(PlatformObject.name.equals("OS/400")) {
			return getValue("JAVA_PATH");
		}
		else {
			String str = string.substring(0, string.lastIndexOf("java"));
			String rc = null;

			try {
				rc = ConfigUtility.getJvmLib(str);
			}
			catch(Exception e) {
				Logger.out(ConfigUtility.getString("Config.Jvm.Warning.CannotFindJvmLibrary") + ": " + PlatformObject.jvmLib);
			}

			return rc;
		}
	}

	/**
	 * @return Returns the acConfig.
	 */
	public AgentControllerConfig getAcConfig() {
		return acConfig;
	}

	public String getValue(String tag) {
		return (String)hash.get(tag);
	}

	public void setValue(String tag, String value) {
		hash.put(tag, value);
	}

	public void removeKey(String tag) {
		hash.remove(tag);
	}

	public boolean isExist(String tag) {
		return hash.containsKey(tag);
	}
}
