package org.eclipse.hyades.internal.config.generator;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Hashtable;

public class SetConfig {
	protected static ConfigFileImpl conf = null;
	protected static boolean overwrite = false;
	protected static boolean silent = false;
	protected static boolean verify = true;
	private static int size = 100;
	private static String defaultConfigFile = null;
	private static Hashtable hash = new Hashtable(size);
	private static String sr = PlatformObject.sr;

	public static void main(String[] args) {
		doIt(args);
		if (conf != null) {
			conf.saveToFile();
		}
	}

	protected static void doIt(String[] args) {
		String[] hostList = null;

		populateHashtable(args);

		if (exist("-help") || exist("-h") || exist("-?")) {
			printHelp();
			return;
		}
		if (exist("-example") || exist("-ex")) {
			printExamples();
			return;
		}
		if (exist("-debug")) {
			Logger.setDebug(true);
		}
		if (exist("-overwrite") || exist("-ovr")) {
			overwrite = true;
		}
		if (exist("-silent") || exist("-s")) {
			silent = true;
		}
		if (exist("-noverify") || exist("-nov")) {
			verify = false;
		}
		//
		// Try to use the $PWD/.. directory as the RASERVER_HOME if none
		// specified
		//
		if (getValue("RASERVER_HOME") == null) {
			File pwd = new File("..");
			try {
				setValue("RASERVER_HOME", pwd.getCanonicalPath());
			} catch (IOException e) {
				// cannot resolve ".."
			}
		}
		//
		// Get the required parameters
		//
		if (!isValidRaserverHome(getValue("RASERVER_HOME"))) {
			setValue("RASERVER_HOME", promptRaserverHome(getValue("RASERVER_HOME")));
		}
		defaultConfigFile = new String(getValue("RASERVER_HOME") + sr + "config" + sr + "serviceconfig.xml");
		//
		// If input file is specified, load it if not overwriting
		//
		if (overwrite) {
			conf = new ConfigFileImpl();
		} else {
			File input = new File(defaultConfigFile);
			//
			// If default configuration file exist, load it
			//
			if (input.exists()) {
				conf = new ConfigFileImpl(input.getAbsolutePath());
			}
			//
			// Otherwise, create a new one
			//
			else {
				conf = new ConfigFileImpl();
			}
		}
		//
		// RASERVER_HOME
		//
		conf.setValue("RASERVER_HOME", getValue("RASERVER_HOME"));
		//
		// JAVA_PATH=
		//
		if(PlatformObject.name.equals("OS/400")) {
			conf.setValue("JAVA_PATH", "/QSYS.LIB/Q5BVAJVM.PGM");
		}
		else {
			if (getValue("JAVA_PATH") == null) { // JAVA_PATH not specified
				if(isValidJavaHome(conf.getValue("JAVA_PATH"))) {
					conf.setValue("JAVA_PATH", promptJavaPath(conf.getValue("JAVA_PATH")) + sr + "java" + PlatformObject.exeExt);
				}
				else {
					conf.setValue("JAVA_PATH", promptJavaPath(System.getProperty("java.home") + sr + "bin") + sr + "java" + PlatformObject.exeExt);
				}
			}
			else { // JAVA_PATH specified
				if(isValidJavaHome(getValue("JAVA_PATH"))) { // if valid
					conf.setValue("JAVA_PATH", getValue("JAVA_PATH") + sr + "java" + PlatformObject.exeExt);
				}
				else { // if not valid
					conf.setValue("JAVA_PATH", promptJavaPath(getValue("JAVA_PATH")) + sr + "java" + PlatformObject.exeExt);
				}
			}
		}
		//
		// ALLOW=
		//
		if(getValue("ALLOW") == null) { // ALLOW not specified
			if(isValidAllow(conf.getValue("ALLOW"))) {
				conf.setValue("ALLOW", promptAllow(conf.getValue("ALLOW")));
			}
			else {
				conf.setValue("ALLOW", promptAllow("LOCAL"));
			}
		}
		else {
			if(isValidAllow(getValue("ALLOW"))) { // if valid
				conf.setValue("ALLOW", getValue("ALLOW"));
			}
			else { // if not valid
				conf.setValue("ALLOW", promptAllow(getValue("ALLOW")));
			}
		}
		//
		// HOSTS=
		//
		if(conf.getValue("ALLOW").toLowerCase().equals("custom")) {
			if(getValue("HOSTS") == null) { // HOSTS not specified
				if(isValidHosts(conf.getValue("HOSTS"))) {
					conf.setValue("HOSTS", promptHosts(conf.getValue("HOSTS")));
				}
				else {
					conf.setValue("HOSTS", promptHosts("localhost"));
				}
			}
			else {
				if(isValidHosts(getValue("HOSTS"))) { // if valid
					conf.setValue("HOSTS", getValue("HOSTS"));
				}
				else { // if not valid
					conf.setValue("HOSTS", promptHosts(getValue("HOSTS")));
				}
			}
		}
		//
		// Generate the config
		//
		conf.generateConfiguration();
	}

	//
	// Retrieve the raserver home specified by the user
	//
	private static String promptRaserverHome(String defaultValue) {
		String rc = null;
		if (silent) {
			String dftVal = ConfigUtility.removeTrailings(defaultValue);
			if((dftVal == null) || dftVal.equals("")) {
				return new String("%RASERVER_HOME%");
			} else {
				return dftVal;
			}
		} else {
			System.out.println(ConfigUtility.getString("Config.Prompt.RASERVER_HOME.Tag"));
			System.out.print("  " + ConfigUtility.getString("Config.Prompt.Default") + "\"" + defaultValue + "\"");
			System.out.print(" (");
			System.out.print(ConfigUtility.getString("Config.Prompt.DefaultAccept"));
			System.out.println(")");
			do {
				System.out.print("  " + ConfigUtility.getString("Config.Prompt.NewValue"));
				rc = promptUserInput(defaultValue);
			} while (verify && !isValidRaserverHome(rc));
			return rc;
		}
	}

	//
	// Retrieve the java home specified by the user
	//
	private static String promptJavaPath(String defaultValue) {
		String rc = null;
		if (silent) {
			String dftVal = ConfigUtility.removeTrailings(defaultValue);
			if((dftVal == null) || dftVal.equals("")) {
				return new String("%JAVA_HOME%" + sr + "jre" + sr + "bin");
			} else {
				return dftVal;
			}
		} else {
			System.out.println(ConfigUtility.getString("Config.Prompt.JAVA_HOME.Tag"));
			System.out.print("  " + ConfigUtility.getString("Config.Prompt.Default") + "\"" + defaultValue + "\"");
			System.out.print(" (");
			System.out.print(ConfigUtility.getString("Config.Prompt.DefaultAccept"));
			System.out.println(")");
			do {
				System.out.print("  " + ConfigUtility.getString("Config.Prompt.NewValue"));
				rc = promptUserInput(defaultValue);
			} while (verify && !isValidJavaHome(rc));
			return rc;
		}
	}

	//
	// Retrieve the allowed hosts specified by the user
	//
	private static String promptAllow(String defaultValue) {
		if (silent) {
			String dftVal = ConfigUtility.removeTrailings(defaultValue);
			if((dftVal == null) || dftVal.equals("")) {
				return new String("LOCAL");
			} else {
				return dftVal;
			}
		} else {
			System.out.println(ConfigUtility.getString("Config.Prompt.Allow.Tag"));
			System.out.print("  " + ConfigUtility.getString("Config.Prompt.Default") + "\"" + defaultValue + "\"");
			System.out.print(" (");
			System.out.print(ConfigUtility.getString("Config.Prompt.DefaultAccept"));
			System.out.println(")");
			System.out.print("  " + ConfigUtility.getString("Config.Prompt.NewValue"));
			return promptUserInput(defaultValue);
		}
	}

	//
	// Retrieve the host names specified by the user
	//
	private static String promptHosts(String defaultValue) {
		if (silent) {
			String dftVal = ConfigUtility.removeTrailings(defaultValue);
			if((dftVal == null) || dftVal.equals("")) {
				return new String("localhost");
			} else {
				return dftVal;
			}
		} else {
			System.out.println(ConfigUtility.getString("Config.Prompt.Hosts.Tag"));
			System.out.print("  " + ConfigUtility.getString("Config.Prompt.Default") + "\"" + defaultValue + "\"");
			System.out.print(" (");
			System.out.print(ConfigUtility.getString("Config.Prompt.DefaultAccept"));
			System.out.println(")");
			System.out.print("  " + ConfigUtility.getString("Config.Prompt.NewValue"));
			return promptUserInput(defaultValue);
		}
	}

	//
	// Return the user's input
	//
	protected static String promptUserInput(String defaultValue) {
		if (silent) {
			return ConfigUtility.removeTrailings(defaultValue);
		} else {
			String input;
			BufferedReader bReader = new BufferedReader(new InputStreamReader(System.in));
			try {
				input = bReader.readLine();
			} catch (IOException e) {
				input = defaultValue;
			}
			if (input.equals("")) {
				input = defaultValue;
			}
			return ConfigUtility.removeTrailings(input);
		}
	}

	//
	// Check if RASREVER_HOME is valid
	//
	private static boolean isValidRaserverHome(String path) {
		boolean rc = false;

		if (path == null) {
			System.out.println(ConfigUtility.getString("Config.Prompt.InvalidValue"));
			return false;
		} else {
			File p = new File(path);
			// Make sure specified path is a directory and exist
			if (p.exists() && p.isDirectory()) {
				if(PlatformObject.name.equals("OS/400")) {
					// Check if SetConfig.sh exists
					File f = new File(p.getAbsolutePath() + File.separator + "bin" + File.separator + "SetConfig.sh");
					if (f.exists()) {
						rc = true;
					}
				}
				else {
					// Check if RAServer.exe or RAServer exists
					File f = new File(p.getAbsolutePath() + File.separator + "bin" + File.separator + "RAServer" + PlatformObject.exeExt);
					if (f.exists()) {
						rc = true;
					}
				}
			}

			if (!rc) {
				System.out.println(ConfigUtility.getString("Config.Prompt.InvalidValue"));
			}
			return rc;
		}
	}

	//
	// Check if JAVA_PATH is valid
	//
	private static boolean isValidJavaHome(String path) {
		boolean rc = false;
		if (path == null) {
			System.out.println(ConfigUtility.getString("Config.Prompt.InvalidValue"));
			return false;
		} else {
			File p = new File(path);
			// Make sure specified path is a directory and exist
			if (p.exists() && p.isDirectory()) {
				// Check if java.exe or java exists
				File f = new File(p.getAbsolutePath() + sr + "java" + PlatformObject.exeExt);
				if (f.exists()) {
					rc = true;
				}
			}
			if (!rc) {
				System.out.println(ConfigUtility.getString("Config.Prompt.InvalidValue"));
			}
			return rc;
		}
	}

	//
	// Check if ALLOW is valid
	//
	private static boolean isValidAllow(String allow) {
		if(allow != null) {
			return (allow.toLowerCase().equals("all") || allow.toLowerCase().equals("local") || allow.toLowerCase().equals("custom"));
		}
		else {
			return false;
		}
	}

	//
	// Check if HOSTS is valid
	//
	private static boolean isValidHosts(String hosts) {
		return (hosts != null);
	}

	//
	// Check if a parameter exist
	//
	protected static boolean exist(String tag) {
		return hash.containsKey(tag);
	}

	//
	// Return the value specified by the parameter
	//
	protected static String getValue(String tag) {
		if (hash.containsKey(tag)) {
			return (String) hash.get(tag);
		} else {
			return null;
		}
	}

	//
	// Set the value specified by the parameter
	//
	protected static void setValue(String tag, String value) {
		hash.put(tag, value);
	}

	protected static void removeKey(String tag) {
		hash.remove(tag);
	}

	//
	// Populate the hash table using the command line arguments
	//
	private static void populateHashtable(String[] args) {
		if (args != null) {
			for (int i = 0; i < args.length; i++) {
				int eq = args[i].indexOf('=');
				if (eq == -1) { // a flags, e.g. -help
					setValue(args[i], "");
				} else { // a parameter, e.g. name=value
					setValue(args[i].substring(0, eq), args[i].substring(eq + 1));
				}
			}
		}
	}

	//
	// Print the help menu
	//
	private static void printHelp() {
		if (PlatformObject.name.startsWith("Windows")) {
			System.out.println(ConfigUtility.getString("Config.Cmd.Usage.Windows"));
		} else {
			System.out.println(ConfigUtility.getString("Config.Cmd.Usage.Unix"));
		}
		System.out.println(ConfigUtility.getString("Config.Cmd.Flags"));
		System.out.println(ConfigUtility.getString("Config.Cmd.Help.Tag"));
		System.out.println("\t" + ConfigUtility.getString("Config.Cmd.Help.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.Silent.Tag"));
		System.out.println("\t" + ConfigUtility.getString("Config.Cmd.Silent.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.NoVerify.Tag"));
		System.out.println("\t" + ConfigUtility.getString("Config.Cmd.NoVerify.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.Examples.Tag"));
		System.out.println("\t" + ConfigUtility.getString("Config.Cmd.Examples.Msg"));
		System.out.println();
		System.out.println(ConfigUtility.getString("Config.Cmd.Params"));
		System.out.println(ConfigUtility.getString("Config.Cmd.RASERVER_HOME.Tag"));
		System.out.println("\t" + ConfigUtility.getString("Config.Cmd.RASERVER_HOME.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.JAVA_PATH.Tag"));
		System.out.println("\t" + ConfigUtility.getString("Config.Cmd.JAVA_PATH.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.ALLOW.Tag"));
		System.out.println("\t" + ConfigUtility.getString("Config.Cmd.ALLOW.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.HOSTS.Tag"));
		System.out.println("\t" + ConfigUtility.getString("Config.Cmd.HOSTS.Msg"));
	}

	//
	// Print the help menu
	//
	private static void printExamples() {
		System.out.println(ConfigUtility.getString("Config.Example.Tag"));
		System.out.println(ConfigUtility.getString("Config.Example.Example1.Tag"));
		if (PlatformObject.name.startsWith("Windows")) {
			System.out.println("\t" + ConfigUtility.getString("Config.Example.Example1.Msg.Windows"));
		} else {
			System.out.println("\t" + ConfigUtility.getString("Config.Example.Example1.Msg.Unix"));
		}
		System.out.println(ConfigUtility.getString("Config.Example.Example2.Tag"));
		if (PlatformObject.name.startsWith("Windows")) {
			System.out.println("\t" + ConfigUtility.getString("Config.Example.Example2.Msg.Windows"));
		} else {
			System.out.println("\t" + ConfigUtility.getString("Config.Example.Example2.Msg.Unix"));
		}
		System.out.println(ConfigUtility.getString("Config.Example.Example3.Tag"));
		if (PlatformObject.name.startsWith("Windows")) {
			System.out.println("\t" + ConfigUtility.getString("Config.Example.Example3.Msg.Windows"));
		} else {
			System.out.println("\t" + ConfigUtility.getString("Config.Example.Example3.Msg.Unix"));
		}
		System.out.println(ConfigUtility.getString("Config.Example.Example4.Tag"));
		if (PlatformObject.name.startsWith("Windows")) {
			System.out.println("\t" + ConfigUtility.getString("Config.Example.Example4.Msg.Windows"));
		} else {
			System.out.println("\t" + ConfigUtility.getString("Config.Example.Example4.Msg.Unix"));
		}
	}
}
