/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.log.ui.internal.navigator;

import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.hyades.log.ui.internal.LogUIPlugin;
import org.eclipse.hyades.log.ui.internal.util.ContextIds;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.trace.internal.ui.PDPerspective;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.hyades.trace.ui.IProfileEventListener;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.ui.internal.navigator.TreeNavigator;
import org.eclipse.hyades.ui.internal.provider.ResourceChangeUpdaterProvider;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.plugin.AbstractUIPlugin;

/**
 * @author marcelop
 * @since 1.1.0
 */
public class LogNavigator extends TreeNavigator implements IProfileEventListener
{
	public static final String ID = "org.eclipse.hyades.log.ui.logNavigator";

	private final String LOG_SECTION = "LogNavigator";
	private LogContentProvider contentProvider;
	private ResourceChangeUpdaterProvider resourceChangeUpdaterProvider;
	private LogNavigatorSynchronizer synchronizer;
	private int fFilterProperties;
	
	/**
	 * Constructor for LogNavigator
	 */
	public LogNavigator()
	{
		super(ID);
	}
	
	/**
	 * @see org.eclipse.ui.IWorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createPartControl(Composite parent)
	{
		super.createPartControl(parent);

		UIPlugin.getDefault().addSelectionProvider((ISelectionProvider)getViewer(), (IWorkbenchPart)this);		
		UIPlugin.getDefault().addProfileEventListener(this);
	}

	
	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.TreeNavigator#dispose()
	 */
	public void dispose()
	{
		UIPlugin.getDefault().removeSelectionProvider((ISelectionProvider)getViewer(), (IWorkbenchPart)this);		
		UIPlugin.getDefault().removeProfileEventListener(this);
		
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(resourceChangeUpdaterProvider);
		resourceChangeUpdaterProvider.dispose();
		resourceChangeUpdaterProvider = null;

		contentProvider.getResourceSet().eAdapters().remove(synchronizer);
		synchronizer.dispose();
		synchronizer = null;
				
		contentProvider.dispose();
		contentProvider = null;
		HyadesUtil.cleanupResources(PDPerspective.ID_LOG_NAVIGATOR_VIEW);
		
		super.dispose();
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.INavigator#getPlugin()
	 */
	public AbstractUIPlugin getPlugin()
	{
		return LogUIPlugin.getDefault();
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.INavigator#isShowingFolders()
	 */
	public boolean isShowingFolders()
	{
		return false;
	}

	public void handleProfileEvent(ProfileEvent event) {
			
		if(event.getSource()!=null && ((event.getSource() instanceof TRCAgentProxy && ((TRCAgentProxy)event.getSource()).getType().equals(TraceConstants.LOG_AGENT_TYPE))||(event.getSource() instanceof IWorkspaceRoot))){

			switch(event.getType())
			{
				case ProfileEvent.START_MONITOR:
			
				   getViewer().setSelection(new StructuredSelection(event.getSource()));
			   
				break;
			
				case ProfileEvent.ATTACH:
			
				getViewer().setSelection(new StructuredSelection(event.getSource()));
			   
				break;
			
				case ProfileEvent.CLEAN_UP:
				break;

				case ProfileEvent.REFRESH_LOG_NAVIGATOR:
			
				getContentProvider().refreshFromLocal();
				break;

				case ProfileEvent.REFRESH_PD_EXPLORER:
				break;
					
				default:
				getViewer().refresh(event.getSource());			
			
			}	

		}
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#createActions()
	 */
	protected void createActions()
	{
		setActionGroup(new LogNavigatorActionGroup(this));
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#createViewerInput()
	 */
	protected Object createViewerInput()
	{
		return ResourcesPlugin.getWorkspace().getRoot();
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#editorActivated(org.eclipse.ui.IEditorPart)
	 */
	protected boolean editorActivated(IEditorPart editor)
	{
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#getStoreSection()
	 */
	public String getStoreSection()
	{
		return LOG_SECTION;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#linkToEditor(org.eclipse.jface.viewers.IStructuredSelection)
	 */
	protected void linkToEditor(IStructuredSelection selection)
	{

	}

	/**
	 * @see org.eclipse.jface.viewers.IDoubleClickListener#doubleClick(org.eclipse.jface.viewers.DoubleClickEvent)
	 */
	public void doubleClick(DoubleClickEvent event)
	{
		if(event.getViewer() == getTreeViewer())
		{
			((LogNavigatorActionGroup)getActionGroup()).handleDoubleClick(event);
		}
	}
	


	public LogContentProvider getContentProvider()
	{
		return contentProvider;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.TreeNavigator#adjustTreeViewer(org.eclipse.jface.viewers.TreeViewer)
	 */
	protected void adjustTreeViewer(TreeViewer treeViewer)
	{
		contentProvider = new LogContentProvider(this);
		treeViewer.setContentProvider(contentProvider);
		treeViewer.setLabelProvider(new LogLabelProvider());
		
		resourceChangeUpdaterProvider = new ResourceChangeUpdaterProvider.UIUpdaterProvider();
		ResourcesPlugin.getWorkspace().addResourceChangeListener(resourceChangeUpdaterProvider, IResourceChangeEvent.POST_CHANGE);
		
		synchronizer = new LogNavigatorSynchronizer(this);
		synchronizer.register(resourceChangeUpdaterProvider, contentProvider.getResourceSet());		
	}
	
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.ISelectionListener#selectionChanged(org.eclipse.ui.IWorkbenchPart, org.eclipse.jface.viewers.ISelection)
	 */
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
	
	}


	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.internal.navigator.INavigator#setLinkingEnabled(boolean)
	 */
	public void setLinkingEnabled(boolean enabled) {
		
		super.setLinkingEnabled(enabled);
		if(enabled)
		{
			if(getViewer() != null)
				getViewer().setSelection(getViewer().getSelection());
		}
		
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#initContextMenu()
	 */
	protected void initContextMenu() {
		
		MenuManager menuMgr = new MenuManager("#LogNavigatorPopupMenu#");
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(this);

		Menu menu = menuMgr.createContextMenu(getViewer().getControl());
		getViewer().getControl().setMenu(menu);
		
		getSite().registerContextMenu(menuMgr, getViewer());
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.IPartListener#partActivated(org.eclipse.ui.IWorkbenchPart)
	 */
	public void partActivated(IWorkbenchPart part) {

		super.partActivated(part);
		if(part==this){
			getViewer().setSelection(((LogNavigator)part).getStructuredSelection());
		}		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.IPartListener#partBroughtToTop(org.eclipse.ui.IWorkbenchPart)
	 */
	public void partBroughtToTop(IWorkbenchPart part) {

		super.partBroughtToTop(part);
		if(part==this){
			getViewer().setSelection(((LogNavigator)part).getStructuredSelection());
		}		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#getHelpContextId()
	 */
	protected String getHelpContextId() {		
		return ContextIds.LOG_NAVIGATOR_VIEW;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#getSettings()
	 */
	public IDialogSettings getSettings() {
		
		return super.getSettings();
	}

}
