/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.log.ui.internal.util;

import java.util.ArrayList;
import java.util.Collections;

import org.eclipse.hyades.log.ui.internal.LogUIPlugin;
import org.eclipse.hyades.security.util.GridUtil;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnLayoutData;
import org.eclipse.jface.viewers.ColumnPixelData;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.help.WorkbenchHelp;

public class CBEAdvFilterUI extends LogAnalyzerUI {

		class EditDialog extends Dialog implements ModifyListener {
			private String _attribute;
			private String _operator;
			private String _value;
			private Combo _AttributeText;
			private Combo _OperatorText;
			private Text _ValueText;
			private String _title;
	
			public EditDialog(
				Shell shell,
				String title,
				String attribute,
				String operator,
				String value) {
				super(shell);
	
				_attribute = attribute;
				_operator = operator;
				_value = value;
				_title = title;
			}
	
			protected void configureShell(Shell shell) {
				super.configureShell(shell);
				shell.setText(_title);
			}
	
			public void attribute(String attribute) {
				_attribute = attribute;
			}
	
			public String attribute() {
				return _attribute;
			}
	
			public void operator(String operator) {
				_operator = operator;
			}
	
			public String operator() {
				return _operator;
			}
	
			public void value(String value) {
				_value = value;
			}
	
			public String value() {
				return _value;
			}
	
			protected Control createDialogArea(Composite parent) {
				Composite result = (Composite) super.createDialogArea(parent);
	
				GridLayout layout;
				GridData data;
				Label label;
	
				layout = new GridLayout();
				layout.numColumns = 2;
				result.setLayout(layout);
				data = GridUtil.createFill();
				data.widthHint = 400;
				result.setLayoutData(data);
	
				label = new Label(result, SWT.NULL);
				label.setText(LogUIPlugin.getResourceString("HEADER_ATTRIBUTE"));
				_AttributeText = new Combo(result, SWT.BORDER | SWT.READ_ONLY);
				_AttributeText.setLayoutData(GridUtil.createHorizontalFill());
				
				for (int i=0;i<fAttributeValues.length;i++)
				{
					_AttributeText.add(fAttributeValues[i]);	
				}
				
				label = new Label(result, SWT.NULL);
				label.setText(LogUIPlugin.getResourceString("HEADER_OPERATOR"));
				_OperatorText = new Combo(result, SWT.BORDER | SWT.READ_ONLY);
				_OperatorText.setLayoutData(GridUtil.createHorizontalFill());
	
				for (int i=0;i<fOperatorValues.length;i++)
				{
					_OperatorText.add(fOperatorValues[i]);	
				}
	
				label = new Label(result, SWT.NULL);
				label.setText(LogUIPlugin.getResourceString("HEADER_VALUE"));
				_ValueText = new Text(result, SWT.BORDER);
				_ValueText.setLayoutData(GridUtil.createHorizontalFill());
	
				_AttributeText.select(0);
				_OperatorText.select(0);
				_ValueText.addModifyListener(this);

				if (_attribute != null)
					_AttributeText.setText(_attribute);
				if (_operator != null)
					_OperatorText.setText(_operator);
				if (_value != null)
					_ValueText.setText(_value);
	
				_AttributeText.setFocus();

				return result;
			}
	
			public void modifyText(ModifyEvent e) {
				if (e.widget == _ValueText) {
					if (getButton(IDialogConstants.OK_ID) != null)
						getButton(IDialogConstants.OK_ID).setEnabled(
							_ValueText.getText().trim() != "");
				}
			}
	
			protected void okPressed() {
				_attribute = _AttributeText.getText().trim();
				_operator = _OperatorText.getText().trim();
				_value = _ValueText.getText();
				super.okPressed();
			}
		}

	private Composite result;
	
	private Button _addBtn;
	private Button _editBtn;
	private Button _removeBtn;
	private Button _upBtn;
	private Button _downBtn;
	
	private CheckboxTableViewer _tableViewer;
	private Table _table; 
	private FilterList _filterList;
	private static String[] tableColumnProperties =
		{
			FilterTableElement.P_ATTRIBUTE,
			FilterTableElement.P_OPERATOR,
			FilterTableElement.P_VALUE };
			
	public static String[] fAttributeValues;// = new String[] {"attributeA","attributeB","attributeC"};
	public static String[] fOperatorValues = new String[] {
							LogUIPlugin.getResourceString("STR_EQUAL"), 
							LogUIPlugin.getResourceString("STR_NOT_EQUAL"),
							LogUIPlugin.getResourceString("STR_LARGER_THAN"),
							LogUIPlugin.getResourceString("STR_SMALLER_THAN")
							};


	public CBEAdvFilterUI() {
		super();
		
	}
	
	public Control createControl(Composite parent)
	{
		return createControl(parent,LogUIPlugin.getResourceString("ADVANCE_FILTERS_LABEL"));
	}
	
	public Control createControl(Composite parent, String contentLabel) {
		GridData data;
		GridLayout layout;
		
		initializeAttribute();

		result = new Composite(parent, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 2;
		layout.verticalSpacing = 0;
		result.setLayout(layout);
		data = GridUtil.createFill();
		result.setLayoutData(data);

		Label _contentLabel = new Label(result, SWT.NULL);
		_contentLabel.setText(contentLabel);
		data = new GridData();
		data.horizontalSpan = 2;
		_contentLabel.setLayoutData(data);				
	
		Composite tableGroup = new Composite(result, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		tableGroup.setLayout(layout);
		data = GridUtil.createFill();
		data.heightHint = 180;
		tableGroup.setLayoutData(data);
		initializeTable(tableGroup);
	
		Composite btnGroup = new Composite(result, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		btnGroup.setLayout(layout);
		btnGroup.setLayoutData(GridUtil.createVerticalFill());

		Composite addGroup = new Composite(btnGroup, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		addGroup.setLayout(layout);
		addGroup.setLayoutData(GridUtil.createVerticalFill());

		_addBtn = new Button(addGroup, SWT.PUSH);
		_addBtn.setText(LogUIPlugin.getResourceString("ADD_FILTER_BTN"));
		_addBtn.setToolTipText(LogUIPlugin.getResourceString("ADD_FILTER__BTN_TOOLTIP_TEXT"));
		_addBtn.setLayoutData(GridUtil.createHorizontalFill());

		_editBtn = new Button(addGroup, SWT.PUSH);
		_editBtn.setText(LogUIPlugin.getResourceString("EDIT_FILTER_BTN"));
		_editBtn.setToolTipText(LogUIPlugin.getResourceString("EDIT_FILTER_BTN_TOOLTIP_TEXT"));
		_editBtn.setLayoutData(GridUtil.createHorizontalFill());

		_removeBtn = new Button(addGroup, SWT.PUSH);
		_removeBtn.setText(LogUIPlugin.getResourceString("REMOVE_FILTER_BTN"));
		_removeBtn.setToolTipText(LogUIPlugin.getResourceString("REMOVE_FILTER_BTN_TOOLTIP_TEXT"));
		_removeBtn.setLayoutData(GridUtil.createHorizontalFill());

		Composite moveGroup = new Composite(btnGroup, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		moveGroup.setLayout(layout);
		moveGroup.setLayoutData(GridUtil.createHorizontalFill());

		_upBtn = new Button(moveGroup, SWT.PUSH);
		_upBtn.setText(LogUIPlugin.getResourceString("MOVEUP_TEXT"));
		_upBtn.setToolTipText(LogUIPlugin.getResourceString("MOVEUP_TOOLTIP_TEXT"));
		_upBtn.setLayoutData(GridUtil.createHorizontalFill());

		_downBtn = new Button(moveGroup, SWT.PUSH);
		_downBtn.setText(LogUIPlugin.getResourceString("MOVEDOWN_TEXT"));
		_downBtn.setToolTipText(LogUIPlugin.getResourceString("MOVEDOWN_TOOLTIP_TEXT"));
		_downBtn.setLayoutData(GridUtil.createHorizontalFill());

		_editBtn.setEnabled(false);
		_removeBtn.setEnabled(false);
		_upBtn.setEnabled(false);
		_downBtn.setEnabled(false);
		
		_upBtn.addSelectionListener(this);
		_downBtn.addSelectionListener(this);
		_addBtn.addSelectionListener(this);
		_editBtn.addSelectionListener(this);
		_removeBtn.addSelectionListener(this);
		_table.addSelectionListener(this);

		WorkbenchHelp.setHelp(_addBtn, ContextIds.ACTLOG_VIEW_DIALOG_ADV_FILTER_ADD);
		WorkbenchHelp.setHelp(_editBtn, ContextIds.ACTLOG_VIEW_DIALOG_ADV_FILTER_EDIT);
		WorkbenchHelp.setHelp(_removeBtn, ContextIds.ACTLOG_VIEW_DIALOG_ADV_FILTER_REMOVE);
		WorkbenchHelp.setHelp(_upBtn, ContextIds.ACTLOG_VIEW_DIALOG_ADV_FILTER_UP);
		WorkbenchHelp.setHelp(_downBtn, ContextIds.ACTLOG_VIEW_DIALOG_ADV_FILTER_DOWN);
		WorkbenchHelp.setHelp(_table, ContextIds.ACTLOG_VIEW_DIALOG_ADV_FILTER_TABLE);
	

		return result;
	}
	
	protected void initializeTable(Composite parent) {
		createTable(parent);
		_tableViewer = new CheckboxTableViewer(_table);
		createColumns();

		_tableViewer.setContentProvider(new FilterListContentProvider(_tableViewer));
		_tableViewer.setLabelProvider(new FilterListLabelProvider());

		_filterList = new FilterList(_tableViewer);
		_tableViewer.setInput(_filterList);

		CellEditor editors[] = new CellEditor[_table.getColumnCount()];

		editors[0] = new ComboBoxCellEditor(_table,fAttributeValues);
		editors[1] = new ComboBoxCellEditor(_table,fOperatorValues);
		editors[2] = new TextCellEditor(_table);

		_tableViewer.setCellEditors(editors);

		FilterElementModifier modifier = new FilterElementModifier(_tableViewer);
		_tableViewer.setCellModifier(modifier);
		_tableViewer.setColumnProperties(tableColumnProperties);
	}

	private void createTable(Composite parent) {
		_table = new Table(
						parent,
						SWT.H_SCROLL
						| SWT.V_SCROLL
						| SWT.BORDER
						| SWT.MULTI
						| SWT.FULL_SELECTION);
		_table.setLinesVisible(true);
		_table.setLayoutData(GridUtil.createFill());
	}
	
	private void createColumns() {
		TableLayout layout = new TableLayout();
		_table.setLayout(layout);
		_table.setHeaderVisible(true);

		String headers[] =
			{	LogUIPlugin.getResourceString("HEADER_ATTRIBUTE"),
				LogUIPlugin.getResourceString("HEADER_OPERATOR"),
				LogUIPlugin.getResourceString("HEADER_VALUE")};
		ColumnLayoutData layouts[] =
			{	new ColumnPixelData(120, true),
				new ColumnPixelData(70, true),
				new ColumnPixelData(130, true)};

		final TableColumn columns[] = new TableColumn[headers.length];
		
		for (int i = 0; i < headers.length; i++) {
			layout.addColumnData(layouts[i]);
			TableColumn tc = new TableColumn(_table, SWT.NONE, i);
			tc.setResizable(layouts[i].resizable);
			tc.setText(headers[i]);
			columns[i] = tc;
		}
	}

	public void initializeFilters(String advFilters) {
		ArrayList _tableElements = FilterTableElement.createFilterTableElementFromString(advFilters);
		populateTable(_tableElements);
	}

	public void initializeValues(boolean defaultValue)
	{
		IPreferenceStore store = LogUIPlugin.getDefault().getPreferenceStore();
		String advFilters = defaultValue ? store.getDefaultString(LogUIConstants.PD_ADV_FILTER_OPTIONS):store.getString(LogUIConstants.PD_ADV_FILTER_OPTIONS);
		initializeFilters(advFilters);
		if(defaultValue){
			Event e = new Event();
			e.widget = result;
			e.type = SWT.Selection;
			notifyListeners(new SelectionEvent(e));	 			
		}
	}

	public void initializeAttribute() {
		IPreferenceStore store = LogUIPlugin.getDefault().getPreferenceStore();
		String pdFilters = store.getString(LogUIConstants.PD_FILTER_OPTIONS);
		ArrayList tempList = new ArrayList();

		ArrayList list = ColumnData.createColumnDataFromString(pdFilters);
			if(list!=null){
				for (int i=0;i<list.size();i++)
				{
					ColumnData data = (ColumnData)(list.get(i));
					tempList.add(data.name());	
						
				} 
			}
		Collections.sort(tempList);
		if (tempList.size()>0)
		{
			fAttributeValues = new String[tempList.size()];
			for (int i=0;i<tempList.size();i++)
			{
				fAttributeValues[i]=(String)(tempList.get(i));	 
			}
		}

	}
	
	public boolean storeValues(IPreferenceStore store,String preferenceProperty) {
		return FilterTableElement.saveFilters(_filterList.getFilters(),preferenceProperty,store);
	}
	
	public boolean storeValues(IPreferenceStore store) {
		return FilterTableElement.saveFilters(_filterList.getFilters(),LogUIConstants.PD_ADV_FILTER_OPTIONS,store);		
	}

	public Composite getControl() {
		return result;
	}

	protected void enableButtons() {
		int selection = _table.getSelectionIndex();
		int selectionCount = _table.getSelectionCount();

		_removeBtn.setEnabled(_filterList.getSize() > 0 && selection != -1);
		_editBtn.setEnabled(_filterList.getSize() > 0 && selection != -1);

		_upBtn.setEnabled(selection > 0 && selectionCount == 1);
		_downBtn.setEnabled(selection >= 0
				&& selection < _table.getItemCount() - 1
				&& selectionCount == 1);
	}
	
	protected void selectTableItem(Table table, int selection) {
		table.setFocus();
		table.setSelection(selection);
		table.showSelection();
	}
	
	private void showEditDialog(String title,String attribute,String operator,String value) {

		EditDialog dialog =	new EditDialog(result.getShell(), title, attribute, operator, value);
		dialog.open();

		if (dialog.getReturnCode() == Window.OK) {

			if (attribute == null) {

				//add filter
				FilterTableElement elem = new FilterTableElement(
						dialog.attribute(),
						dialog.operator(),
						dialog.value());

				_filterList.add(elem, _filterList.getSize());
				elem.setParent(_filterList);
				_tableViewer.refresh();
				selectTableItem(_table, _filterList.getSize() - 1);
			} else {
				int idx = _table.getSelectionIndex();
				if (idx != -1) {

					FilterTableElement sel =
						(FilterTableElement) _filterList.getElementAt(idx);

					sel.setAttribute(dialog.attribute());
					sel.setOperator(dialog.operator());
					sel.setValue(dialog.value());

					_tableViewer.refresh();
				}
			}
		}
	}

	public void widgetSelected(SelectionEvent event) {
		if (event.widget == _addBtn) {
			showEditDialog(
				LogUIPlugin.getResourceString("STR_ADD_FILTER_DLG"),
				null,
				null,
				null);
			enableButtons();
		} else if (event.widget == _editBtn) {
			int idx = _table.getSelectionIndex();
			if (idx != -1) {
				FilterTableElement sel =
					(FilterTableElement) _filterList.getElementAt(idx);
	
				String attribute = sel.getAtrribute();
				String operator = sel.getOperator();
				String value = sel.getValue();
	
				showEditDialog(
					LogUIPlugin.getResourceString("STR_EDIT_FILTER_DLG"),
					attribute,
					operator,
					value);
			}
		} else if (event.widget == _table) {
			enableButtons();
			return;
		} else if (event.widget == _removeBtn) {
			int selection = _table.getSelectionIndex();
	
			_filterList.removeElements(_table.getSelectionIndices());
	
			_tableViewer.refresh();
	
			int size = _filterList.getSize();
			if (size > 0) {
				if (selection > 0 && selection < size) {
					selectTableItem(_table, selection - 1);
				} else {
					selectTableItem(_table, 0);
				}
			}
			enableButtons();
		} else if (event.widget == _upBtn) {
			_tableViewer.refresh();
	
			int selection = _table.getSelectionIndex();
			if (selection > 0) {
				FilterTableElement sel =
					(FilterTableElement) _filterList.getElementAt(selection);
				FilterTableElement previous =
					(FilterTableElement) _filterList.getElementAt(
						selection - 1);
	
				String attribute = sel.getAtrribute();
				String operator = sel.getOperator();
				String value = sel.getValue();
	
				sel.setAttribute(previous.getAtrribute());
				sel.setOperator(previous.getOperator());
				sel.setValue(previous.getValue());
				
				previous.setAttribute(attribute);
				previous.setOperator(operator);
				previous.setValue(value);
		
				selectTableItem(_table, selection - 1);
			}
	
			_tableViewer.refresh();
			enableButtons();
		} else if (event.widget == _downBtn) {
			_tableViewer.refresh();
	
			int selection = _table.getSelectionIndex();
			if (selection < _table.getItemCount() - 1) {
				FilterTableElement sel =
					(FilterTableElement) _filterList.getElementAt(selection);
				FilterTableElement next =
					(FilterTableElement) _filterList.getElementAt(
						selection + 1);
	
				String attribute = sel.getAtrribute();
				String operator = sel.getOperator();
				String value = sel.getValue();
	
				sel.setAttribute(next.getAtrribute());
				sel.setOperator(next.getOperator());
				sel.setValue(next.getValue());
	
				next.setAttribute(attribute);
				next.setOperator(operator);
				next.setValue(value);
	
				selectTableItem(_table, selection + 1);
			}
	
			_tableViewer.refresh();
			enableButtons();
		}
	}

	protected void populateTable(ArrayList tableElems) {

		_filterList.removeAll();
		
		if (tableElems!=null) {
			for (int idx = 0; idx < tableElems.size(); idx++) {
				FilterTableElement elem = (FilterTableElement) tableElems.get(idx);
				_filterList.add(elem, _filterList.getSize());
				elem.setParent(_filterList);
			}
	
			if (tableElems.size() > 0) {
				_table.select(0);
			}
		}
		_tableViewer.refresh();
		enableButtons();	
	}

	protected void populateList() {
	}


	/**
	 * @return
	 */
	public Button getAddBtn() {
		return _addBtn;
	}

	/**
	 * @return
	 */
	public Button getRemoveBtn() {
		return _removeBtn;
	}

	/**
	 * @return
	 */
	public FilterList getFilterList() {
		return _filterList;
	}

}