/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.log.ui.internal.views;


import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.log.ui.internal.LogUIPlugin;
import org.eclipse.hyades.log.ui.internal.LogUIPluginImages;
import org.eclipse.hyades.log.ui.internal.actions.provider.ILogFilterProvider;
import org.eclipse.hyades.log.ui.internal.actions.provider.ILogFindProvider;
import org.eclipse.hyades.log.ui.internal.actions.provider.LogFilterCriteria;
import org.eclipse.hyades.log.ui.internal.actions.provider.LogFindCriteria;
import org.eclipse.hyades.log.ui.internal.util.ColumnsDialog;
import org.eclipse.hyades.log.ui.internal.util.FeatureNode;
import org.eclipse.hyades.log.ui.internal.util.FilterTableElement;
import org.eclipse.hyades.log.ui.internal.util.FiltersDialog;
import org.eclipse.hyades.log.ui.internal.util.LogRecordSearch;
import org.eclipse.hyades.log.ui.internal.util.LogUIConstants;
import org.eclipse.hyades.log.ui.internal.util.LogUtil;
import org.eclipse.hyades.log.ui.internal.util.RecordChangeEvent;
import org.eclipse.hyades.log.ui.internal.util.SearchRecordDialog;
import org.eclipse.hyades.log.ui.internal.util.SortDialog;
import org.eclipse.hyades.log.ui.internal.util.TerminalNode;
import org.eclipse.hyades.models.cbe.CBECommonBaseEvent;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.ui.HyadesConstants;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.TraceViewer;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.ViewSelectionChangedEvent;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.part.IPage;

public class LogViewer extends TraceViewer implements ILogFindProvider, ILogFilterProvider{
	protected final String _title =
		LogUIPlugin.getResourceString("STR_WAS_VIEW_TITLE");
	private LogPage logPage;
	private Action pageDown;
	private Action pageUp;
	private Action goToPage;
	
	private LogRecordSearch logRecordSearch = null;
	private Object lastMatch = null;
	private List cbeList;
	private Label lblMessage;
	
	public boolean handleRefreshEvent =false;
	public LogViewer() {
		super();
	}
	/**
	 * Creates a log viewer page
	 * Creation date: (08/01/2000 4:17:37 PM)
	 * @return TraceViewerPage
	 */
	public TraceViewerPage createPage(EObject mofObject) {
		logPage = new LogPage(mofObject, this);
		return logPage;
	}

	public void handleProfileEvent(ProfileEvent event) {

		if (((event.getType() == ProfileEvent.REFRESH_VIEWS)
			|| (event.getType() == ProfileEvent.UPDATE_MODEL)
			|| (event.getType() == ProfileEvent.UPDATE_DELTA_TIME))			
			&& (event.getSource() instanceof EObject)) {
				
			if(!fPartVisible)
			{		
				//ignore the events and keep track that the page need to be updated			
			   fRefreshView = true;		   
				return;
			}
			handleRefreshEvent=true;
			refreshPage(getObjectToView((EObject) event.getSource()));
			handleRefreshEvent=false;
		}
	}

	/**	 
	 * Disposes every cached page and 
	 * cleans up the hashtable that caches the log (trace) viewer pages
	 * @see TraceViewer#dispose
	 * @see IWorkbenchPart#dispose
	 */
	public void dispose() {
		super.dispose();
	}
	/**
	 * Creates actions for opening the filter and sort dialogs
	 * and adds them to the toolbar
	 * Creation date: (1/11/2001 2:54:24 PM)
	 */
	public void makeActions() {
		if (isInitializedMenu()) {
			return; //create the toolbar only once
		}

		initializedMenu(true);

		String STR_SORT = LogUIPlugin.getResourceBundle().getString("STR_SORT");
		Action sort = new Action(STR_SORT) {
			public void run() {
				SortDialog dialog =
					new SortDialog(
						getViewSite().getShell(),
						LogUIPlugin.getResourceString(
							"STR_ST_SORT_COL_SETTINGS"),
						null);
				dialog.open();
				if (dialog.getReturnCode() == Window.OK) {
					final IPage page = getCurrentPage();

					if (page != null && page instanceof LogPage) {

						org
							.eclipse
							.swt
							.custom
							.BusyIndicator
							.showWhile(null, new Runnable() {
							public void run() {

								((LogPage) page).getView().update();
							}
						});
					}
				}

			}
		};
		sort.setText(STR_SORT);
		LogUIPluginImages.setImageDescriptors(
			sort,
			LogUIPluginImages.T_LCL,
			LogUIPluginImages.IMG_SORT);
		sort.setDescription(STR_SORT);
		sort.setToolTipText(STR_SORT);

		String STR_FILTER =
			LogUIPlugin.getResourceBundle().getString("STR_FILTER");
		Action filter = new Action(STR_FILTER) {
			public void run() {
				ColumnsDialog dialog =
					new ColumnsDialog(
						getViewSite().getShell(),
						LogUIPlugin.getResourceString("STR_ST_COL_SETTINGS"),
						null);
				dialog.open();
			}
		};
		filter.setText(STR_FILTER);
		LogUIPluginImages.setImageDescriptors(
			filter,
			LogUIPluginImages.T_LCL,
			LogUIPluginImages.IMG_FILTER);
		filter.setDescription(STR_FILTER);
		filter.setToolTipText(STR_FILTER);

		String STR_ADV_FILTER =
			LogUIPlugin.getResourceBundle().getString("STR_ADV_FILTER");
		Action advfilter = new Action(STR_ADV_FILTER) {
			public void run() {
					filter();
			}
		};
		advfilter.setText(STR_ADV_FILTER);
		LogUIPluginImages.setImageDescriptors(
			advfilter,
			LogUIPluginImages.T_LCL,
			LogUIPluginImages.IMG_ADV_FILTER);
		advfilter.setDescription(STR_ADV_FILTER);
		advfilter.setToolTipText(STR_ADV_FILTER);

		Action searchRecord = createSearchButton();
		
		getViewSite().getActionBars().setGlobalActionHandler(IWorkbenchActionConstants.FIND,searchRecord);

		createPageUpButton();
		createPageDownButton();
		createGoToPageButton();
		
		IToolBarManager tbm = getViewSite().getActionBars().getToolBarManager();

		tbm.add(sort);
		tbm.add(filter);
		tbm.add(advfilter);
		tbm.add(searchRecord);
		tbm.add(pageUp);
		tbm.add(pageDown);
		tbm.add(goToPage);
		
		// requires update because the toolbar control has been created by this point,
		// but the toolbar manager does not update it automatically once it has been created
		getViewSite().getActionBars().updateActionBars();
	}
	private Action createSearchButton() {
		String STR_FIND_RECORD =
			LogUIPlugin.getResourceBundle().getString("STR_FIND_RECORD");
		Action searchRecord = new Action(STR_FIND_RECORD) {
			public void run() {
				find();
			}
		};
		searchRecord.setText(STR_FIND_RECORD);
		LogUIPluginImages.setImageDescriptors(
			searchRecord,
			LogUIPluginImages.T_LCL,
			LogUIPluginImages.IMG_SEARCH_RECORD);
		searchRecord.setDescription(STR_FIND_RECORD);
		searchRecord.setToolTipText(STR_FIND_RECORD);
		return searchRecord;
	}

	private Action createPageUpButton() {
		String STR_PAGE_UP =
			LogUIPlugin.getResourceBundle().getString("STR_PAGE_UP");
		pageUp = new Action(STR_PAGE_UP) {
			public void run() {
				final IPage page = getCurrentPage();
				
				if (page != null && page instanceof LogPage) {

					org
					.eclipse
					.swt
					.custom
					.BusyIndicator
					.showWhile(null, new Runnable() {
						public void run() {

							LogContentProvider logContentProvider = (LogContentProvider)((LogPage) page).getView().getViewer().getContentProvider();
							pageUp.setEnabled(logContentProvider.pageUp());
							((LogPage) page).getView().update();
						}
					});
				}
			}
		};
		
		getViewSite().getActionBars().setGlobalActionHandler(ActionFactory.PREVIOUS.getId(),pageUp);
		
		pageUp.setText(STR_PAGE_UP);
		LogUIPluginImages.setImageDescriptors(
			pageUp,
			LogUIPluginImages.T_LCL,
			LogUIPluginImages.IMG_PAGE_UP);
		pageUp.setDescription(STR_PAGE_UP);
		pageUp.setToolTipText(STR_PAGE_UP);
		return pageUp;
	}

	private Action createGoToPageButton() {
		String STR_GO_TO_PAGE =
			LogUIPlugin.getResourceString("STR_GO_TO_PAGE");
		goToPage = new Action(STR_GO_TO_PAGE) {
			public void run() {
				final IPage page = getCurrentPage();
				final LogContentProvider logContentProvider = (LogContentProvider)((LogPage) page).getView().getViewer().getContentProvider();
				final int totalPages = logContentProvider.getTotalPages();
				if(totalPages<2)
				{	
					goToPage.setEnabled(false);
					return;
				}
					
				String[] variables = new String[2];
				variables[0]="1";
				variables[1]=""+totalPages;
				String dialogTitle = LogUIPlugin.getResourceString("GO_TO_PAGE_DIALOG_TITLE");
				String dialogMessage = LogUIPlugin.getResourceString("GO_TO_PAGE_DIALOG_MESSAGE",variables);
				InputDialog inputDialog = new InputDialog(getSite().getShell(),dialogTitle,dialogMessage,null,new IInputValidator() {
					public String isValid(String newText) {
						if(newText==null || newText.length()==0)
							return "";
						try {
							int pageNr = Integer.parseInt(newText);
							if(pageNr<1 || pageNr>totalPages)
								return LogUIPlugin.getResourceString("GO_TO_PAGE_VNIR");
						} catch (Exception e) {
							return LogUIPlugin.getResourceString("GO_TO_PAGE_VNAN");
						}
						return null;
					}
				});
				inputDialog.open();
				if(inputDialog.getReturnCode() != InputDialog.OK)
					return;
				final int pageNr = Integer.parseInt(inputDialog.getValue());
				if (page != null && page instanceof LogPage) {

					org
					.eclipse
					.swt
					.custom
					.BusyIndicator
					.showWhile(null, new Runnable() {
						public void run() {
							
							boolean res = logContentProvider.goToPage(pageNr);
							if(pageNr==1)
								pageUp.setEnabled(false);
							goToPage.setEnabled(res);
							((LogPage) page).getView().update();
						}
					});
				}
			}
		};
		
		goToPage.setText(STR_GO_TO_PAGE);
		getViewSite().getActionBars().setGlobalActionHandler(IWorkbenchActionConstants.M_NAVIGATE+"/goToPage",goToPage);
		
		LogUIPluginImages.setImageDescriptors(
			goToPage,
			LogUIPluginImages.T_LCL,
			LogUIPluginImages.IMG_GO_TO_PAGE);
		goToPage.setDescription(STR_GO_TO_PAGE);
		goToPage.setToolTipText(STR_GO_TO_PAGE);
		return goToPage;
	}
	
	private Action createPageDownButton() {
		String STR_PAGE_DOWN =
			LogUIPlugin.getResourceBundle().getString("STR_PAGE_DOWN");
		pageDown = new Action(STR_PAGE_DOWN) {
			public void run() {
				final IPage page = getCurrentPage();
				
				if (page != null && page instanceof LogPage) {

					org
					.eclipse
					.swt
					.custom
					.BusyIndicator
					.showWhile(null, new Runnable() {
						public void run() {
							
							LogContentProvider logContentProvider = (LogContentProvider)((LogPage) page).getView().getViewer().getContentProvider();
							pageDown.setEnabled(logContentProvider.pageDown());
							((LogPage) page).getView().update();
						}
					});
				}
			}
		};
		
		getViewSite().getActionBars().setGlobalActionHandler(ActionFactory.NEXT.getId(),pageDown);
		
		pageDown.setText(STR_PAGE_DOWN);
		LogUIPluginImages.setImageDescriptors(
			pageDown,
			LogUIPluginImages.T_LCL,
			LogUIPluginImages.IMG_PAGE_DOWN);
		pageDown.setDescription(STR_PAGE_DOWN);
		pageDown.setToolTipText(STR_PAGE_DOWN);
		return pageDown;
	}
	
	/**
	 * @see WorkbenchPart#setFocus 
	 * Creation date: (10/5/2001 11:42:53 AM)
	 */
	public void setFocus() {

		if (getCurrentPage() != null)
			getCurrentPage().setFocus();
	}

	/**
	 * @return true if the object can be viewed using this type of view 
	 * 
	 */
	public boolean isValidObject(Object mofObject) {
		if (mofObject == null)
			return false;

		if (mofObject instanceof TRCAgent
			&& ((TRCAgent) mofObject).getType().equals(
				HyadesConstants.LOG_AGENT_TYPE)) {
			return true;
		}

		if (mofObject instanceof TRCProcessProxy) {
			return hasProcessLogAgents((TRCProcessProxy)mofObject);
		}
		
		if(mofObject instanceof TRCNode){
			return hasNodeLogAgents((TRCNode)mofObject);
		}
		
		if(mofObject instanceof TRCMonitor){
			return hasMonitorLogAgents((TRCMonitor)mofObject);
		}
		

		return true;
	}

	/**
	 * @return the object that should be viewed
	 * For example, if the process node is selected, the view should display
	 * the profiling data if trace collects profiling data
	 * Returns the agent if there is only one log agent
	 * 
	 */

	public EObject getObjectToView(EObject selObject) {

		if (selObject == null)
			return selObject;

		List list =null;
		
		if(selObject instanceof TRCMonitor){

            return selObject; //selection model must be at the monitor level
			//list = getLogObjectInMonitor((TRCMonitor)selObject);
			
		}else if(selObject instanceof TRCNode){
			
			return selObject;//selection model must be at the monitor level
			//list = getLogObjectInNode((TRCNode)selObject);
		
		}else if (selObject instanceof TRCProcessProxy) {

			list = getLogAgentInProcess((TRCProcessProxy)selObject); 
			
		}
		if(list!=null && list.size()==1){
			return (EObject)list.get(0);
		}
		return selObject;
	}

	private List getLogObjectInMonitor(EObject selObject){

		EList nodes = ((TRCMonitor)selObject).getNodes();
		TRCNode node = null;

		int size = nodes.size();
		int nrOfNodes = 0;
		List object_list = new ArrayList();
		List list;		

		for(int i=0; i < size && nrOfNodes < 2;i++){		
			node = (TRCNode)nodes.get(i);
			if(node==null){
				continue;
			}else{	
				
				list = getLogObjectInNode(node); 
				if(list.size()>=1){
					nrOfNodes++;					
					object_list.add(node);
				}
			}
			
		}
		
		if(object_list.size()==1){
			list = getLogObjectInNode((TRCNode)object_list.get(0));
			if(list.size()==1){
				object_list.clear();
				object_list.add(list.get(0));
			}
		}
		
		return object_list;
	}
	
	private List getLogObjectInNode(EObject selObject){

		EList processes = ((TRCNode)selObject).getProcessProxies();
		TRCProcessProxy process = null;
		int nrOfProcesses = 0;
		int size = processes.size();
		List list;
		List object_list = new ArrayList();

		for(int i=0; i < size && nrOfProcesses < 2;i++){		
			process = (TRCProcessProxy)processes.get(i);
			if(process!=null){
				list = getLogAgentInProcess(process);
				if(list.size()>=1){
					nrOfProcesses++;
					object_list.add(process);
				}
			}else{
				continue;								
			}
		}
		
		if(object_list.size()==1){
			list = getLogAgentInProcess((TRCProcessProxy)object_list.get(0));
			if(list.size()==1){
				object_list.clear();
				object_list.add(list.get(0));
			}
			
		}
		return object_list;
	}
	
	private List getLogAgentInProcess(EObject selObject){

		int nrOfAgents = 0;
		EList agents = ((TRCProcessProxy)selObject).getAgentProxies();
		int size = agents.size();
		TRCAgentProxy a = null;
		List list =  new ArrayList();
		
		for (int idx = 0; idx < size && nrOfAgents < 2; idx++) {
			a = (TRCAgentProxy) agents.get(idx);
			if (a == null || a.eIsProxy())
				continue;
			if (a.getType().equals(HyadesConstants.LOG_AGENT_TYPE)) {
				nrOfAgents++;
				list.add(a);
			}
		}
		
		return list;
	
	}
	
	
	private boolean hasMonitorLogAgents(TRCMonitor monitor){
		
		EList nodes = monitor.getNodes();
		TRCNode node = null;
		int size = nodes.size();
		boolean ret = false;		

		for(int i=0; i < size  && !ret;i++){		
			node = (TRCNode)nodes.get(i);
			if(node!=null){
				ret = hasNodeLogAgents(node);
			}
			
		}
		
		return ret;
		
	}
	
	private boolean hasNodeLogAgents(TRCNode node){

		EList processes = node.getProcessProxies();
		TRCProcessProxy process = null;
		boolean ret = false;
		
		int size = processes.size();

		for(int i=0; i < size && !ret;i++){
			process = (TRCProcessProxy)processes.get(i);
			if(process!=null){
				ret = hasProcessLogAgents(process);
			}
		}
		
		return ret;
		
	}
	
	private boolean hasProcessLogAgents(TRCProcessProxy process){
		
		EList agents = process.getAgentProxies();
		int size = agents.size();
		TRCAgentProxy agent = null;		
		
		for (int idx = 0; idx < size; idx++) {
			agent = (TRCAgentProxy) agents.get(idx);
			if (agent == null || agent.eIsProxy())
				continue;
			if (agent.getType().equals(HyadesConstants.LOG_AGENT_TYPE)) {
				return true;
			}
		}
		
		return false;
	}
	
	
	/**
	 * Initial view title 
	 */
	public String getViewTitle() {
		return _title;
	}

	public void setRecordSelection(EObject record, EObject treeSel) {
		if (getPage(treeSel) == null)
			return;

		LogViewerUI ui = ((LogPage) getPage(treeSel)).getView();
	
		if (ui != null && ui.getViewer() != null) {
			LogContentProvider logContentProvider = (LogContentProvider)ui.getViewer().getContentProvider();
			if(logContentProvider.revealObject(record))
				ui.update();
			ui.getViewer().setSelection(new StructuredSelection(record), true);
		}
	}

	protected void setViewTitle(Object selection) {
		if (selection != null) {
			/*if (selection instanceof EObject)
				selection = getObjectToView((EObject) selection);*/

			if (selection instanceof TRCAgentProxy) {
				TRCAgentProxy a = (TRCAgentProxy) selection;

				String status = HyadesUtil.getAgentLabel(a);

				if(a.getProcessProxy()!=null)
					setTitle
						((new StringBuffer())
							.append(_title)
							.append(" - ")
							.append(a.getName())
							.append(" ")
							.append(a.getProcessProxy().getName())
							.toString());
				return;

			}
		}
		setTitle(getViewTitle());
	}

	/**
	 * @return Returns the pageDown.
	 */
	public Action getPageDownAction() {
		return pageDown;
	}

	/**
	 * @return Returns the pageUp.
	 */
	public Action getPageUpAction() {
		return pageUp;
	}
	/**
	 * @return Returns the goToPage.
	 */
	public Action getGotPageAction() {
		return goToPage;
	}
	
	public void find(){
		SearchRecordDialog dialog =	new SearchRecordDialog(
											getViewSite().getShell(),
											LogUIPlugin.getResourceString("STR_ST_FIND_SETTINGS"),
											null,
											this);
		dialog.open();
	}
	
	public void filter(){
		FiltersDialog dialog = new FiltersDialog(
										getViewSite().getShell(),
										LogUIPlugin.getResourceString("STR_ST_FILTER_SETTINGS"),
										null,
										this,
										false);
		dialog.open();
		
	}
	
	public boolean findRecord(LogFindCriteria criteria){
		lblMessage = criteria.getFindMessage();
		Object sel = getRecordSearchEngine().search(criteria.getFilters(),criteria.isForwardDirection()?1:-1,lastMatch);
		setSelection(sel);
		return true;
	}

	public boolean filterRecord(LogFilterCriteria criteria){
		LogUIPlugin.getDefault().fireRecordChangeNotification(new RecordChangeEvent(RecordChangeEvent.FILTER_ADV_CHANGE, this));
		return true;
	}
	
	public LogRecordSearch getRecordSearchEngine(){
		if(logRecordSearch==null){
			logRecordSearch = new LogRecordSearch();
		}
		LogViewerUI ui = logPage.getView();
		if (ui != null && ui.getViewer() != null) {
			LogContentProvider logContentProvider = (LogContentProvider)ui.getViewer().getContentProvider();
			cbeList = logContentProvider.getCachedElements();
		}
	
		Object start = getStartNode();
		logRecordSearch.initialize(cbeList, start);
		return logRecordSearch;	 
	}
	
	private void setSelection(Object sel){	
		if(sel==null){
			lblMessage.setText(LogUIPlugin.getResourceString("STR_STRING_NOT_FOUND"));
			return;
		}
		
		EObject mofObject = logPage.getTraceViewer().getObjectToView(HyadesUtil.getMofObject());
		if(sel!=null && sel instanceof EObject)
			logPage.getTraceViewer().setRecordSelection((EObject)sel, mofObject);	            	

		lastMatch = sel;
		logPage.getView().getViewer().reveal(lastMatch);
		notifySelectionChanged((EObject)sel);
	}
	
	private void notifySelectionChanged(EObject sel) {
		LogUtil.notifyViewSelectionChanged(logPage.getView(),sel);
//		UIPlugin.getDefault().getSelectionModel(HyadesUtil.getMofObject()).add(sel);
//			
//		ViewSelectionChangedEvent event = UIPlugin.getDefault().getViewSelectionChangedEvent();
//		event.setSource(this);
//		//event.setSource(e.item.getData());
//		UIPlugin.getDefault().notifyViewSelectionChangedListener(event);
	}
	
	private Object getStartNode(){
			IStructuredSelection sel = (IStructuredSelection)((logPage).getView().getViewer().getSelection());
	
				Object object = sel.getFirstElement();;
				if (object instanceof CBECommonBaseEvent) {
					return object;
				}
				else if (object instanceof EObject) {
					return getCBENode(object);
				}
				else if (object instanceof FeatureNode) {
					return getCBENode(((FeatureNode)object).getElement());
				}
				else if (object instanceof TerminalNode) {
					return getCBENode(((TerminalNode)object).getTreeParent());
				}

			return null;	
	}
	
	private Object getCBENode(Object object)
	{
		if (object instanceof CBECommonBaseEvent){
			return object;
		}
		else if (object instanceof EObject) {
			return getCBENode(((EObject)object).eContainer());
		}
		return null;
	}
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.log.ui.internal.actions.provider.ILogFindProvider#storeFindOptions(org.eclipse.hyades.log.ui.internal.actions.provider.LogFindCriteria)
	 */
	public boolean storeFindOptions(LogFindCriteria criteria) {
		FilterTableElement.saveFilters(criteria.getFilters(),LogUIConstants.PD_SEARCH_FILTER_OPTIONS,LogUIPlugin.getDefault().getPreferenceStore());
		LogUIPlugin.getDefault().getPreferenceStore().setValue(LogUIConstants.PD_SEARCH_DIRECTION_OPTIONS, criteria.isForwardDirection()?"1":"-1");
		return true;
	}
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.log.ui.internal.actions.provider.ILogFindProvider#loadFindPreferenceString()
	 */
	public String loadFindPreferenceString() {
		IPreferenceStore store = LogUIPlugin.getDefault().getPreferenceStore();
		return store.getString(LogUIConstants.PD_SEARCH_FILTER_OPTIONS);
	}
	
	public boolean loadFindDirection() {
		return LogUIPlugin.getDefault().getPreferenceStore().getString(LogUIConstants.PD_SEARCH_DIRECTION_OPTIONS).equals("1");
	}

	public LogPage getLogPage() {
		return logPage;
	}
}
