/*******************************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.ui.internal.navigator.action;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Path;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.internal.extension.NavigatorFilterSet;
import org.eclipse.hyades.ui.internal.extension.NavigatorExtensionUtil;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.dialogs.ListSelectionDialog;
import org.eclipse.ui.model.BaseWorkbenchContentProvider;
import org.eclipse.ui.model.IWorkbenchAdapter;
import org.eclipse.ui.model.WorkbenchAdapter;
import org.eclipse.ui.model.WorkbenchLabelProvider;

/**
 * The "Filters..." item that appears in Hyades navigator
 * view menus. Opens a standard dialog allowing the user to
 * select the filter sets to use for the navigator.
 * 
 * @since 3.0
 */
public class FiltersAction extends Action {
	private ImageDescriptor _imageDescriptor;
	private INavigator _navigator;
	private String _label;
	
	/**
	 * Constucts the item.
	 * 
	 * @param navigator the navigator whose view menu will contain
	 *    this item.
	 */
	public FiltersAction(INavigator navigator) {
		_imageDescriptor = ImageDescriptor.createFromURL(HyadesUIPlugin.getInstance().find(new Path("icons/full/elcl16/filters.gif")));
		_label = HyadesUIPlugin.getString("LBL_FILTERS");
		_navigator = navigator;
	}
	
	/**
	 * @see org.eclipse.jface.action.Action#getText()
	 */
	public String getText() {
		return _label;
	}
	
	/**
	 * @see org.eclipse.jface.action.Action#getImageDescriptor()
	 */
	public ImageDescriptor getImageDescriptor() {
		return _imageDescriptor;
	}
	
	/**
	 * @see org.eclipse.jface.action.Action#run()
	 */
	public void run() {
		Shell shell = HyadesUIPlugin.getInstance().getWorkbench().getActiveWorkbenchWindow().getShell();
		final NavigatorFilterSet[] sets = NavigatorExtensionUtil.getFilterSetElements(_navigator.getID());

		// remember what was selected before
		Set before = new HashSet(sets.length);
		for (int i=0;i<sets.length;++i)
			if (sets[i].isEnabled())
				before.add(sets[i]);
		
		// the ListSelectionDialog's input (all filter sets)
		IAdaptable input = new IAdaptable() {
			public Object getAdapter(Class adapter) {
				if (IWorkbenchAdapter.class.equals(adapter))
					return new WorkbenchAdapter() {
						public Object[] getChildren(Object o) {
							return sets;
						}
					};
				return null;
			}
		};
		
		String msg = HyadesUIPlugin.getString("DLG_FILTERS_MSG");
		ListSelectionDialog dialog = new ListSelectionDialog(shell, input, new BaseWorkbenchContentProvider(), new WorkbenchLabelProvider(), msg);
		dialog.setTitle(HyadesUIPlugin.getString("DLG_FILTERS_TITLE"));
		dialog.setInitialSelections(before.toArray());

		if (dialog.open() == Window.OK) {
			Set after = new HashSet(sets.length);
			after.addAll(Arrays.asList(dialog.getResult()));
			
			// look for the differences before and after
			for (int i=0;i<sets.length;++i)
				if (sets[i].isEnabled() != after.contains(sets[i]))
					sets[i].setEnabled(!sets[i].isEnabled());
			
			_navigator.getViewer().refresh();
		}
	}
}
