/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.navigator.action;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.actions.ActionContext;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.views.framelist.BackAction;
import org.eclipse.ui.views.framelist.ForwardAction;
import org.eclipse.ui.views.framelist.FrameList;
import org.eclipse.ui.views.framelist.GoIntoAction;
import org.eclipse.ui.views.framelist.UpAction;

import org.eclipse.hyades.ui.internal.navigator.ITreeNavigator;
import org.eclipse.hyades.ui.internal.navigator.TreeNavigator;

/**
 * This is the action group for the frame list actions for tree navigators.
 * 
 *  * <p>This implementation is based on the 
 * {@link org.eclipse.ui.views.navigator.GotoActionGroup} class.
 * 
 * @author marcelop
 * @since 0.2.0
 */
public class FrameListActionGroup 
extends NavigatorActionGroup
{
	private BackAction backAction;
	private ForwardAction forwardAction;
	private GoIntoAction goIntoAction;
	private UpAction upAction;
	private IResourceChangeListener resourceChangeListener;

	/**
	 * Constructor for GotoActionGroup
	 * @param navigator
	 */
	public FrameListActionGroup(ITreeNavigator treeNavigator)
	{
		super(treeNavigator);

		// Listen for project open/close changes. Fixes bug 5958
		resourceChangeListener = new IResourceChangeListener()
		{
			public void resourceChanged(IResourceChangeEvent event)
			{
				updateActionBars();
			}
		};
		ResourcesPlugin.getWorkspace().addResourceChangeListener(resourceChangeListener);
	}

	/**
	 * @see org.eclipse.ui.actions.ActionGroup#dispose()
	 */
	public void dispose()
	{
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(resourceChangeListener);
		resourceChangeListener = null;
		
		super.dispose();		
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup#createActions()
	 */
	protected void createActions()
	{
		FrameList frameList = ((TreeNavigator)getNavigator()).getFrameList();
		goIntoAction = new GoIntoAction(frameList);
		backAction = new BackAction(frameList);
		forwardAction = new ForwardAction(frameList);
		upAction = new UpAction(frameList);
	}
	
	/**
	 * @see org.eclipse.ui.actions.ActionGroup#fillContextMenu(org.eclipse.jface.action.IMenuManager)
	 */
	public void fillContextMenu(IMenuManager menu)
	{
		IStructuredSelection structuredSelection = (IStructuredSelection) getContext().getSelection();
		if(structuredSelection.size() == 1)
		{
			if(canBeFrameRoot(structuredSelection.getFirstElement()))
				menu.add(goIntoAction);
		}
	}
	/**
	 * @see org.eclipse.ui.actions.ActionGroup#fillActionBars(org.eclipse.ui.IActionBars)
	 */
	public void fillActionBars(IActionBars actionBars)
	{
		actionBars.setGlobalActionHandler(IWorkbenchActionConstants.GO_INTO, goIntoAction);
		actionBars.setGlobalActionHandler(ActionFactory.BACK.getId(), backAction);
		actionBars.setGlobalActionHandler(ActionFactory.FORWARD.getId(), forwardAction);
		actionBars.setGlobalActionHandler(IWorkbenchActionConstants.UP,	upAction);
			
		IToolBarManager toolBar = actionBars.getToolBarManager();
		toolBar.add(backAction);
		toolBar.add(forwardAction);
		toolBar.add(upAction);
	}

	/**
	 * @see org.eclipse.ui.actions.ActionGroup#updateActionBars()
	 */
	public void updateActionBars()
	{
		ActionContext context = getContext();
		boolean enable = false;
		if (context != null)
		{
			IStructuredSelection structuredSelection =(IStructuredSelection) context.getSelection();
			if(structuredSelection.size() == 1)
				enable = canBeFrameRoot(structuredSelection.getFirstElement());
		}	
		goIntoAction.setEnabled(enable);
	}

	/**
	 * Returns whether a given element can be the root of a frame.
	 * @param element
	 * @return boolean
	 */
	protected boolean canBeFrameRoot(Object element)
	{
		StructuredViewer viewer = getNavigator().getViewer();
		if(viewer instanceof TreeViewer)
			return ((TreeViewer)viewer).isExpandable(element);	

		if((element instanceof IProject) && !((IProject)element).isOpen())
			return false;
		
		if(element instanceof IContainer)
			return true;
						
		return false;		
	}
}
