/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.provider;

import org.eclipse.core.resources.IResource;

import org.eclipse.hyades.ui.util.IDisposable;

/**
 * Instances of this class are invoked to perform the updates required due to 
 * resource changes.
 *  
 * @author marcelop
 * @since 0.0.1 
 */
public interface IResourceChangeUpdater
extends IDisposable
{
	/**
	 * Invoked at the beginning of the change notification.
	 */
	public void started();

	/**
	 * Invoked at the end of the change notification.
	 */
	public void ended();
	
	/**
	 * Invoked after the resources in <code>affectedResources</code> were
	 * added to <code>parent</code>.
	 * @param parent
	 * @param affectedResources
	 * @return <code>true</code> if there was a full refresh of the resource
	 * or <code>false</code> otherwise.  Normally no more resource change analysis 
	 * if this method returns <code>true</code>. 
	 */
	public boolean add(IResource parent, IResource[] affectedResources);
	
	/**
	 * Invoked after the resources in <code>affectedResources</code> were
	 * removed from <code>parent</code>.
	 * @param parent
	 * @param affectedResources
	 * @return <code>true</code> if there was a full refresh of the resource
	 * or <code>false</code> otherwise.  Normally no more resource change analysis 
	 * if this method returns <code>true</code>. 
	 */
	public boolean remove(IResource parent, IResource[] affectedResources);

	/**
	 * Invoked after the <code>resource</code> was replaced - for example when it is
	 * deleted and recreated
	 * @param affectedResource
	 * @return <code>true</code> if there was a full refresh of the resource
	 * or <code>false</code> otherwise.  Normally no more resource change analysis 
	 * if this method returns <code>true</code>. 
	 */
	public boolean replaced(IResource affectedResource);

	/**
	 * Invoked after some properties of the <code>resource</code> were changed.
	 * @param affectedResource
	 * @return <code>true</code> if there was a full refresh of the resource
	 * or <code>false</code> otherwise.  Normally no more resource change analysis 
	 * if this method returns <code>true</code>. 
	 */
	public boolean updateProperties(IResource affectedResource);
	
	/**
	 * Invoked after at least one of the child of the <code>resource</code> had their 
	 * type changed.
	 * @param affectedResource
	 * @return <code>true</code> if there was a full refresh of the resource
	 * or <code>false</code> otherwise.  Normally no more resource change analysis 
	 * if this method returns <code>true</code>. 
	 */
	public boolean updateChildrenType(IResource affectedResource);
	
	/**
	 * Invoked after a content change in the specified resource.
	 * @param parent resource
	 * @param affectedResource
	 * @return <code>true</code> if there was a full refresh of the resource
	 * or <code>false</code> otherwise.  Normally no more resource change analysis 
	 * if this method returns <code>true</code>. 
	 */
	public boolean updateContent(IResource parent, IResource affectedResource);
	
	/**
	 * Sets whether this resource change updater is active. This setting has no effect
	 * during a change notification.
	 * @param enable
	 */
	public void setActive(boolean enable); 
	
	/**
	 * Returns whether this resource change updater is active.
	 * @return boolean
	 */
	public boolean isActive();
}
