/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.widgets.zoomslider.internal;

import java.util.Vector;

import org.eclipse.hyades.statistical.ui.EditorPlugin;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.*;

public class ZoomControlBar extends Object
{
    protected ZoomSlider	slider;
    protected double		value;
    protected Rectangle	area;
    protected Color		color;
    protected int 		size = 11;
    protected boolean 	enabled = true;
    protected Image 		image;
    
    int 	indent = 3;
    double	oldValue;
    
    Vector listeners = new Vector();
    
    public ZoomControlBar(ZoomSlider slider)
    {
        this.slider = slider;
        
		setColor(slider.getDisplay().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
        
        value = slider.getMinVisible();
        
        slider.addZoomControlBar(this);
    }
    
    public ZoomControlBar(ZoomSlider slider, double value)
    {
        this(slider);
        
        setValue(value);
        
        slider.updateScale();
    }
    
    public ZoomControlBar(ZoomSlider slider, Color color)
    {
        this(slider);
        
        setColor(color);
        
        slider.updateScale();
    }
    
    public ZoomControlBar(ZoomSlider slider, double value, Color color)
    {
        this(slider, value);
        
        setColor(color);
        
        slider.updateScale();
    }
    
    public ZoomControlBar(ZoomSlider slider, int size)
    {
        this(slider);
        
        setSize(size);
        
        slider.updateScale();
    }
    
    public ZoomControlBar(ZoomSlider slider, double value, int size)
    {
        this(slider, value);
        
        setSize(size);
        
        slider.updateScale();
    }    
    
    public ZoomControlBar(ZoomSlider slider, Color color, int size)
    {
        this(slider, color);
        
        setSize(size);
        
        slider.updateScale();
    }
    
    public ZoomControlBar(ZoomSlider slider, double value, Color color, int size)
    {
        this(slider, value, color);
        
        setSize(size);
        
        slider.updateScale();
    }    

    public void setZoomSlider(ZoomSlider slider)
    {
        this.slider = slider;
    }
    
    public ZoomSlider getZoomSlider()
    {
        return slider;
    }    

    String tooltip = null;
    public void setToolTipText(String text) {
    	tooltip = text;
    }
    public String getToolTipText() {
    	return tooltip;	
    }
    
    public void setValue(double d)
    {
        oldValue = value;
        
        if (d > slider.getMaxLimit()) d = slider.getMaxLimit();
		if (d < slider.getMinLimit()) d = slider.getMinLimit();
        
        value = d;
    }
    
    public double getValue()
    {
        return value;
    }
    
    public void setColor(Color color)
    {
        this.color = color;
    }    
    
    public Color getColor()
    {
        return color;
    }        
        
    public void setEnabled(boolean b)
    {
        enabled = b;
    }
    
    public boolean isEnabled()
    {
        return enabled;
    }    

    public void setSize(int size)
    {
        this.size = size;
    }
    
    public int getSize()
    {
        return size;
    }    
        
    public void hide(Image backBuffer, GC g)
    {
        g.setClipping(getDrawingArea(oldValue));
        g.drawImage(backBuffer, 0, 0);
        g.setClipping(0, 0, slider.getBounds().width, slider.getBounds().height);
    }    
    
    public void paint(GC g)
    {
        area = getDrawingArea(value);
        
        // Create the ZoomControlBar's Image
        if (image == null)
        {
        	// Draw the ZoomControlBar on a temporary Image, and get its ImageData
        	Image tempImage = new Image(slider.getDisplay(), area.width, area.height);
			draw3DBox(new GC(tempImage), 0, 0, area.width, area.height, color);
        	ImageData imageData = tempImage.getImageData();
        	tempImage.dispose();
        	
	        // Set the Alpha blending on the ImageData
	        for (int i = 0; i < area.width; i++)
	        { 
                for (int j = 0; j < area.height; j++) 
                { 
                        imageData.setAlpha(i, j, 180); 
                } 
	        } 

			// Use the ImageData to create the required transparent Image
        	image = new Image(slider.getDisplay(), imageData);
        }

		// Draw the ZoomControlBar's Image on the Graphics Context        
        g.drawImage(image, area.x, area.y);

		// Draw 2 little lines from the edges of the Image to the edges of the ZoomSlider        
        g.setForeground(slider.getForeground());
        if (slider.getOrientation() == ZoomSlider.HORIZONTAL)
        {
            g.drawLine(area.x + area.width/2, 1, area.x + area.width/2, indent);
            g.drawLine(area.x + area.width/2, slider.getBounds().height - indent, area.x + area.width/2, slider.getBounds().height - 1);
        }
        else
        {
            g.drawLine(1, area.y + area.height/2, indent, area.y + area.height/2);
            g.drawLine(slider.getBounds().width - indent, area.y + area.height/2, slider.getBounds().width - 1, area.y + area.height/2);
        }            
    }
    
    protected Rectangle getDrawingArea(double d)
    {
        if (slider.getOrientation() == ZoomSlider.VERTICAL)
        {
            return new Rectangle(indent, (int)(slider.value2Pixel(d)) - size/2, slider.getBounds().width - indent * 2, size);
        }
        else
        {
            return new Rectangle((int)(slider.value2Pixel(d)) - size/2, indent, size, slider.getBounds().height - indent * 2);
        }    
    }
    
    public boolean inside(int x, int y)
    {
        if (area == null)
        {
            return false;
        }
        
        if (area.contains(x,y) && enabled)
        {
            return true;
        }
        return false;
    }
    
	public void sprayZoomControlBarEvent(boolean finished)
	{
        for (int i=0; i<listeners.size(); i++)
        {
            ((ZoomControlBarListener)(listeners.elementAt(i))).zoomControlBarChanged(new ZoomControlBarEvent(this, value, finished));
        }
	}
	
    public void addZoomControlBarListener(ZoomControlBarListener listener)
    {
        listeners.addElement(listener);
    }

    public void removeZoomControlBarListener(ZoomControlBarListener listener)
    {
        listeners.removeElement(listener);
    }

    protected void draw3DBox(GC g, int x, int y, int w, int h, Color boxcolor)
    {
        Color darkColor = null, darkerColor = null, lightColor = null;
        double dark = 0.5, darker = 0.1, light = 3;
        
		//if (!enabled) boxcolor = slider.getBackground();


if (boxcolor == null) {
	EditorPlugin.DBG.warning("3d box colour is null!");
}
        RGB rgb = boxcolor.getRGB();
		rgb.red = (int)(rgb.red * dark);
		rgb.blue = (int)(rgb.blue * dark);
		rgb.green = (int)(rgb.green * dark);				
		darkColor = new Color(slider.getDisplay(), rgb);
		
        rgb = boxcolor.getRGB();
		rgb.red = (int)(rgb.red * darker);
		rgb.blue = (int)(rgb.blue * darker);
		rgb.green = (int)(rgb.green * darker);				
		darkerColor = new Color(slider.getDisplay(), rgb);

        rgb = boxcolor.getRGB();
		rgb.red = (int)(rgb.red * light);
		rgb.blue = (int)(rgb.blue * light);
		rgb.green = (int)(rgb.green * light);				
		if (rgb.red > 255) rgb.red = 255;
		if (rgb.blue > 255) rgb.blue = 255;
		if (rgb.green > 255) rgb.green = 255;		
		lightColor = new Color(slider.getDisplay(), rgb);

        g.setBackground(darkerColor);
        g.fillRectangle(x, y, w, h);
        
        g.setBackground(boxcolor);
        g.fillRectangle(x, y, w-1, h-1);
        
        g.setBackground(darkColor);
        g.fillRectangle(x+1, y+1, w-2, h-2);
        
        g.setBackground(lightColor);
        g.fillRectangle(x+1, y+1, w-3, h-3);
        
        g.setBackground(boxcolor);
        g.fillRectangle(x+2, y+2, w-4, h-4);
        
        darkColor.dispose();
        darkerColor.dispose();
        lightColor.dispose();
   }
   
   public void dispose()
   {
		if (image != null) image.dispose();
		if (color != null) color.dispose();
   }
}


            
            
