/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.perfmon;

import org.eclipse.hyades.statistical.ui.variableloader.internal.*;

import org.eclipse.hyades.statistical.ui.editor.internal.*;
import org.eclipse.hyades.model.statistical.*;
import org.eclipse.hyades.models.hierarchy.*;
import org.eclipse.hyades.models.hierarchy.util.*;

import org.eclipse.ui.dialogs.*;

import org.eclipse.core.runtime.*;

import org.eclipse.emf.common.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.emf.ecore.resource.*;

import org.eclipse.emf.common.util.URI;

import org.eclipse.hyades.statistical.ui.widgets.zoomslider.internal.*;

import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;

import java.io.*;
import java.text.*;
import java.util.*;

public class PerfmonModuleTrace implements AgentChangeListener, SDVariableModifierProxy {

//@TODO this should be FALSE when in hyades
private static final boolean USE_CUSTOM_LOADER = false;

ImageManager img = PerfmonPlugin.img;

StatConInterface statcon;
	
String PERFMON_NODE = "Perfmon Trace";

String ACTION_STOP_TRACE = "Stop Trace";
String ACTION_UPDATE_TREE = "Update Counter Tree";

String ACTION_START_TRACING = "Start Tracing this Counter";
String ACTION_STOP_TRACING = "Stop Tracing this Counter";

String ACTION_SETSMODELPATH = "Set Statistical Model save path...";

String tracename;

SimpleDateFormat smodel_sdf = new SimpleDateFormat("yyyyMMdd-HHmmss");

TRCMonitor monitor;
TRCAgent agent;
PerfmonLoader loader;

VariablePadder var_padder;

SDModifiableVariableRepresentation freq_var;
double frequency = 1.0;	//seconds

URI smodelpath;

	public void unloadTrace() {
		var_padder.die = true;

		try {
			loader.shutdown();
		} catch (Exception e) {
			PerfmonPlugin.DBG.warning("failed to shut down Perfmon loader");
		}

		try {
			statcon.setConstantUpdate("PERFMON UPDATE "+loader.hashCode(),false,1000);
			statcon.setAgentImage(agent,img.getImage(ImageManager.IMG_PERFMON_STOP));
			
			statcon.removeAgentAction(agent,ACTION_SETSMODELPATH);
			statcon.removeAgentAction(agent,ACTION_STOP_TRACE);
			statcon.removeAgentAction(agent,ACTION_UPDATE_TREE);
		} catch (Exception e) {
			PerfmonPlugin.DBG.warning("failed to remove stuff from statcon");
		}
		
		statcon.removeAgentChangeListener(this);
	}

	public void agentAdded(TRCAgent dead_agent) {
	}

	public void agentRemoved(TRCAgent dead_agent) {
		if (agent == dead_agent) {
			unloadTrace();	
		}
	}

	public void requestValue(String unique_id, int i) throws ClassCastException {
		requestValue(unique_id,(double)i);
	}
	public void requestValue(String unique_id, long i) throws ClassCastException {
		requestValue(unique_id,(double)i);
	}
	public void requestValue(String unique_id, float i) throws ClassCastException {
		requestValue(unique_id,(double)i);
	}
	public void requestValue(String unique_id, double d) throws ClassCastException {
		if (unique_id.equals("PERFMON#FREQUENCY")) {
			if (d < 0.5) d = 0.5;
			int i = (int)(d * 1000.0d);
			try {
				loader.setFrequency(i);
				frequency = d;
				freq_var.setMeasuredValue(System.currentTimeMillis(),d);
			} catch (IOException e) {
				PerfmonPlugin.DBG.logVisibleError(e,"Error setting perfmon agent frequency to "+i,false);
			}
		}
	}
	public void requestValue(String unique_id, Object newval) throws ClassCastException {
		requestValue(unique_id,((Number)newval).doubleValue());
	}
	

	public PerfmonModuleTrace(StatConInterface statcon, String rac_host, String reg_host, URI smodelpath) {
		this.statcon = statcon;
		this.smodelpath = smodelpath;
		init(rac_host, reg_host);	
	}
	
	private String getStatModelFileName() {
		return tracename;
	}
	
	private void removeRepresentations(SDDescriptor descriptor) {
		if (descriptor == null) return;
		if (descriptor instanceof SDMemberDescriptor) {
			SDMemberDescriptor mem = (SDMemberDescriptor)descriptor; 
			mem.setRepresentation(null);
			EList obslist = mem.getSnapshotObservation();
			for (int i = 0; i < obslist.size(); i++) {
				SDSnapshotObservation obs = (SDSnapshotObservation)obslist.get(i);
				if (obs.getWindow() == null) {
					System.out.println("NULL OBSERVATION WINDOW UNDER "+descriptor.getName()+" ("+descriptor.getDescription()+")");	
				}	
			}	
		}
		EList list = descriptor.getChildren();
		for (int i = 0; i < list.size(); i++) {
			removeRepresentations((SDDescriptor)list.get(i));	
		}
	}
	
	private void saveStatisticalModel() throws IOException {

		EList descriptors = agent.getDescriptor();
		for (int i = 0; i < descriptors.size(); i++) {
			removeRepresentations((SDDescriptor)descriptors.get(i));	
		}

		URI baseStatisticalModelURI = smodelpath;

		IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();			

		IPath path = new Path(baseStatisticalModelURI+"/"+getStatModelFileName()+".trcmxmi"); 
		URI monitorURI = URI.createPlatformResourceURI(workspaceRoot.getFile(path).getProjectRelativePath().toString());
						
		path = new Path(baseStatisticalModelURI+"/"+getStatModelFileName()+".trcnxmi"); 
		URI nodeURI = URI.createPlatformResourceURI(workspaceRoot.getFile(path).getProjectRelativePath().toString());
						
		path = new Path(baseStatisticalModelURI+"/"+getStatModelFileName()+".trcpxmi"); 
		URI processURI = URI.createPlatformResourceURI(workspaceRoot.getFile(path).getProjectRelativePath().toString());

		path = new Path(baseStatisticalModelURI+"/"+getStatModelFileName()+".trcaxmi"); 
		URI agentURI = URI.createPlatformResourceURI(workspaceRoot.getFile(path).getProjectRelativePath().toString());				

		ResourceSet resourceSet = HierarchyResourceSetImpl.getInstance();

		Resource monitorResource = resourceSet.createResource(monitorURI);
		Resource nodeResource = resourceSet.createResource(nodeURI);
		Resource processResource = resourceSet.createResource(processURI);
		Resource agentResource = resourceSet.createResource(agentURI);
		
		monitorResource.getContents().add(monitor);
		nodeResource.getContents().add(agent.getAgentProxy().getProcessProxy().getNode());
		processResource.getContents().add(agent.getAgentProxy().getProcessProxy());
		agentResource.getContents().add(agent);

		monitorResource.save(Collections.EMPTY_MAP);
		nodeResource.save(Collections.EMPTY_MAP);
		processResource.save(Collections.EMPTY_MAP);
		agentResource.save(Collections.EMPTY_MAP);		
		
		statcon.setAgentFile(agent,agentURI);
	}

	private void configureSliders() throws Exception {
		TimeZoomSlider xslider = statcon.getGraphWindow().getXSlider(0);
		double minvis = xslider.getMinVisible();
		double maxvis = xslider.getMaxVisible();
		double min = xslider.getMinLimit();
		double max = xslider.getMaxLimit();
		
		double ctime = System.currentTimeMillis();
		
		minvis = ctime;
		maxvis = ctime + (1000 * 60);
		min = Math.min(min,ctime - (double)(1000 * 60 * 60));
		max = Math.max(max,ctime + (double)(1000 * 60 * 60 * 24));

		xslider.configure(min,max,minvis,maxvis,xslider.getResolution());		
		
		ZoomSlider yslider = statcon.getGraphWindow().getYSlider(0);
		minvis = yslider.getMinVisible();
		maxvis = Math.max(yslider.getMaxVisible(),100.0d);
		min = yslider.getMinLimit();
		max = Math.max(yslider.getMaxLimit(),maxvis*10);
		
		yslider.configure(min,max,minvis,maxvis,yslider.getResolution());			
	}

	private void createTRCAgent(String tracename) {
		HierarchyFactory hfactory = HierarchyFactory.eINSTANCE;

		TRCMonitor monitorroot;
		TRCNode noderoot;
		TRCProcessProxy pproxyroot;
		TRCAgentProxy aproxyroot;
		TRCAgent agentroot;

		//TRCMonitor setup			
		monitorroot = hfactory.createTRCMonitor();
		monitorroot.setName(tracename);

		//timestamps filled in when monitor is saved
		
		//TRCNode setup			
		noderoot = hfactory.createTRCNode();
		noderoot.setMonitor(monitorroot);
		noderoot.setName(tracename);
		noderoot.setRuntimeId("unknown");
		noderoot.setTimezone(0);
		
		//TRCProcessProxy setup			
		pproxyroot = hfactory.createTRCProcessProxy();
		pproxyroot.setNode(noderoot);
		noderoot.getProcessProxies().add(pproxyroot);
		pproxyroot.setActive(true);
		pproxyroot.setPid(-1);
		pproxyroot.setName("Scapa Engine");
		
		//TRCAgentProxy setup		
		aproxyroot = hfactory.createTRCAgentProxy();
		aproxyroot.setProcessProxy(pproxyroot);
		pproxyroot.getAgentProxies().add(aproxyroot);
		aproxyroot.setActive(true);

		//TRCAgent setup
		agentroot = hfactory.createTRCAgent();
		agentroot.setAgentProxy(aproxyroot);
		agentroot.setName(tracename);

		HierarchyResourceSetImpl resourceSet = HierarchyResourceSetImpl.getInstance();

		URI project_uri = statcon.getCurrentProjectURI();

		IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();			

		IPath dummyPath = new Path(project_uri+"/"+"agent-"+Math.random()+"-"+System.currentTimeMillis()+".trcaxmi"); 
		URI dummyURI = URI.createPlatformResourceURI(workspaceRoot.getFile(dummyPath).getProjectRelativePath().toString());

		Resource dummyResource = resourceSet.createResource(dummyURI);
		dummyResource.getContents().add(agentroot);

		agent = agentroot;
		monitor = monitorroot;
	}

	private void redoActions() {
		statcon.addAgentAction(agent,ACTION_SETSMODELPATH,PerfmonPlugin.getString("ACTION_SETSMODELPATH")+" ("+PerfmonStatConModule.uriToHumanReadable(smodelpath)+")", img.getImage(ImageManager.IMG_PERFMON_SAVE), new PerfmonSetPath());
	}	

	private void init(String rac_host, String reg_host) {


		try {
			if (reg_host.equalsIgnoreCase(rac_host)) {
				reg_host = "null";	
			}

			HierarchyFactory hfactory = HierarchyFactory.eINSTANCE;
		
			long started = System.currentTimeMillis();

			String host = rac_host;
			int port = 10002;
			String reg_host_name = reg_host;

			if (reg_host_name.equalsIgnoreCase("null")) {
				reg_host_name = host;	
			} else {
				if (!reg_host.startsWith("\\\\")) {
					reg_host = "\\\\" + reg_host;	
				}
			}
			
			tracename = "Perfmon Trace - "+reg_host_name+" ("+host+") "+smodel_sdf.format(new Date(started));
			String modelname = "Perfmon Host Trace - "+smodel_sdf.format(new Date(started));

			createTRCAgent(tracename);

			PerfmonPlugin.DBG.info("Creating variable loader");

			SDVariableLoader varloader = new SDVariableLoader(agent,"Perfmon Controls");

ProfileEvent event = UIPlugin.getDefault().getProfileEvent();
event.setSource(agent.getAgentProxy());
event.setType(ProfileEvent.UNSPECIFIED);
UIPlugin.getDefault().notifyProfileEventListener(event);

event = UIPlugin.getDefault().getProfileEvent();
event.setSource(agent.getAgentProxy());
event.setType(ProfileEvent.REFRESH_VIEWS);
UIPlugin.getDefault().notifyProfileEventListener(event);

event = UIPlugin.getDefault().getProfileEvent();
event.setSource(agent.getAgentProxy());
event.setType(ProfileEvent.COLLECTING);
UIPlugin.getDefault().notifyProfileEventListener(event);

//			agent = hfactory.createTRCAgent();
//			agent.setName(tracename);

			AgentUpdateThread agentupdate = new AgentUpdateThread(agent);
			agentupdate.start(); 
			

			if (USE_CUSTOM_LOADER) {
				loader = new PerfmonCustomLoader(started,agent,modelname,host,port,reg_host,(int)(1000.0*frequency));
			} else {
				loader = new PerfmonGenericLoader(started,agent,modelname,host,port,reg_host,(int)(1000.0*frequency));
			}

			loader.addLoaderListener(agentupdate);

			statcon.setConstantUpdate("PERFMON UPDATE "+loader.hashCode(),true,1000);

			statcon.addAgent(agent);
			statcon.addAgentAction(agent,ACTION_STOP_TRACE,PerfmonPlugin.getString("STOP_TRACE"),img.getImage(ImageManager.IMG_PERFMON_STOP),new StopTraceAction(loader,agent));
			statcon.addAgentAction(agent,ACTION_UPDATE_TREE,PerfmonPlugin.getString("UPDATE_TREE"),img.getImage(ImageManager.IMG_PERFMON_UPDATE_TREE),new UpdateTreeAction(loader,agent));
			statcon.addAgentAction(agent,ACTION_SETSMODELPATH,PerfmonPlugin.getString("ACTION_SETSMODELPATH")+" ("+PerfmonStatConModule.uriToHumanReadable(smodelpath)+")", img.getImage(ImageManager.IMG_PERFMON_SAVE), new PerfmonSetPath());

			statcon.addModelAction(agent,ACTION_START_TRACING,PerfmonPlugin.getString("START_TRACING"),img.getImage(ImageManager.IMG_PERFMON_START),new CounterON(loader));
			statcon.addModelAction(agent,ACTION_STOP_TRACING,PerfmonPlugin.getString("STOP_TRACING"),img.getImage(ImageManager.IMG_PERFMON_STOP),new CounterOFF(loader));

			configureSliders();

//			PerfmonPlugin.DBG.info("Getting statistical model from loader");
//			while (loader.getModel() == null) {
//				Thread.sleep(500);	
//			} 
//			PerfmonPlugin.DBG.info("got statistical model from loader");
			
//			SDVariableLoader varloader = new SDVariableLoader(loader.getModel());

			PerfmonPlugin.DBG.info("Creating frequency variable");
			freq_var = varloader.createSDModifiableVariable(
										"PERFMON#FREQUENCY",
										"Frequency (sec)",
										"The period between data updates in seconds",
										null,
										false,
										SDMeasuredVariable.DOUBLE,
										SDModifiableVariable.DOUBLE,
										"Frequency",
										"Perfmon Controls",
										this,
										0.5d,
										Double.MAX_VALUE
										);
										
			PerfmonPlugin.DBG.info("Setting requested frequency to "+frequency);
			freq_var.setRequestedValue(System.currentTimeMillis(),frequency);
			

			
//				SampleEventGenerator gen = new SampleEventGenerator(loader,agent);
//				gen.start();
/*
				FileInputStream fin = new FileInputStream("C:\\Scapa\\Eclipse-Hyades\\org.eclipse.hyades.perfmon\\sampletrace.xml");
				ByteArrayOutputStream bout = new ByteArrayOutputStream();
				
				byte[] buf = new byte[1024];
				
				int n = 0;
				while (n != -1) {
					n = fin.read(buf);	
					if (n > 0) {
						bout.write(buf,0,n);	
					}	
				}

System.out.println("GOT XML");
				String xml = bout.toString();				
	
System.out.println("POPULATING LOADER");

				long t = System.currentTimeMillis()+5000;
				int count = 0;
				while (System.currentTimeMillis() < t) {
					for (int z = 0; z < 10; z++) { 
						loader.incoming(xml);
					}
					count+=10;
				}
System.out.println("LOADER = "+(count/5)+" messages per second");

*/
				
			PerfmonPlugin.DBG.info("setting up trcagent + variable padder");
			statcon.addAgentChangeListener(this);
				
			statcon.setAgentImage(agent,img.getImage(ImageManager.IMG_PERFMON_START));
//			statcon.updateAgent(agent);
				
			var_padder = new VariablePadder();
			var_padder.start();	
			
		} catch (Throwable t) {
			PerfmonPlugin.DBG.logVisibleError(t,PerfmonPlugin.getString("ERROR_STARTING_TRACE"),true);	
		}
	}

	class VariablePadder extends Thread {
		boolean die = false;
		public void run() {
			while (!die) {
				try {
					Thread.sleep(750);	
				} catch (Exception e) {
				}
				try {
					freq_var.padRequestedValue(System.currentTimeMillis());
					freq_var.setMeasuredValue(System.currentTimeMillis(),freq_var.getLastMeasuredValue());	
				} catch (Exception e) {
					PerfmonPlugin.DBG.warning("unable to pad variable requested and measured - "+e);
				}
			}
		}
	}

	class UpdateTreeAction implements Runnable {
		PerfmonLoader loader;
		TRCAgent agent;
		public UpdateTreeAction(PerfmonLoader loader, TRCAgent agent) {
			this.loader = loader;
			this.agent = agent;
		}
		public void run() {
			try {
				loader.getUpdatedTree();
			} catch (IOException e) {
				PerfmonPlugin.DBG.logVisibleError(e,PerfmonPlugin.getString("UPDATE_TREE_ERROR"),false);	
			}
		}	
	}

	class StopTraceAction implements Runnable {
		PerfmonLoader loader;
		TRCAgent agent;
		public StopTraceAction(PerfmonLoader loader, TRCAgent agent) {
			this.loader = loader;
			this.agent = agent;
		}
		public void run() {
			unloadTrace();
			
			try {
				saveStatisticalModel();
			} catch (Exception e) {
				PerfmonPlugin.DBG.error("problem saving statistical model",e);
			}
		}	
	}


	class CounterON implements Runnable {
		PerfmonLoader loader;
		public CounterON(PerfmonLoader loader) {
			this.loader = loader;	
		}
		public void run() {
			//this should recursively turn on / off subchildren too
			//and it should set the images in StatCon
			try {
				//turn it and all its children on
				applyDown(statcon.getSelectedDescriptor());
				//turn on its parents
				applyUp(statcon.getSelectedDescriptor());
			} catch (Exception e) {
				PerfmonPlugin.DBG.error("unable to set counter to ON",e);	
			}
		}
		void apply(SDDescriptor descriptor) throws IOException {
			try {
				loader.counterON(descriptor);
			} catch (NullPointerException e) {}	//descriptor not found (root)
			statcon.setDescriptorImage(descriptor,img.getImage(ImageManager.IMG_PERFMON_START));
		}
		void applyUp(SDDescriptor descriptor) throws IOException {
			SDDescriptor parent = descriptor.getParent();
			if (parent != null) {
				apply(parent);
				applyUp(parent);					
			}	
		}
		void applyDown(SDDescriptor descriptor) throws IOException {
			apply(descriptor);
			List children = descriptor.getChildren();
			for (int i = 0; i < children.size(); i++) {
				applyDown((SDDescriptor)children.get(i));
			}
		}
	}
	
	class CounterOFF implements Runnable {
		PerfmonLoader loader;
		public CounterOFF(PerfmonLoader loader) {
			this.loader = loader;	
		}
		public void run() {
			try {
				apply(statcon.getSelectedDescriptor());
			} catch (Exception e) {
				PerfmonPlugin.DBG.error("unable to set counter to ON",e);	
			}
		}
		void apply(SDDescriptor descriptor) throws IOException {
			loader.counterOFF(descriptor);
			statcon.setDescriptorImage(descriptor,img.getImage(ImageManager.IMG_PERFMON_STOP));
			List children = descriptor.getChildren();
			for (int i = 0; i < children.size(); i++) {
				apply((SDDescriptor)children.get(i));
			}
		}
	}

	class PerfmonSetPath implements Runnable {
		public void run() {
			PerfmonPlugin.DBG.info("set test model save path");

			IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
			Path container_savepath = null;
			IContainer container = null;
			try {
				container_savepath = new Path(""+smodelpath);
				container = workspaceRoot.getFolder(container_savepath);
			} catch (Throwable e) {
				PerfmonPlugin.DBG.warning("could not find valid IContainer for model save path");
			}

			PerfmonPlugin.DBG.info("showing container selection dialog");
			//pop up a dialog	
			ContainerSelectionDialog dialog = null;
			dialog = new ContainerSelectionDialog(statcon.getGraphWindow().getShell(), container, false, PerfmonPlugin.getString("SMODEL_SELECT_PATH_DESCRIPTION"));
			dialog.open();
			Object[] result = dialog.getResult();

			if (result != null) {			
				if (result.length > 0) {

					Path container_path = (Path)result[0];
					URI savepath = URI.createPlatformResourceURI(container_path.toString());

					smodelpath = savepath;

					PerfmonPlugin.DBG.info("updating menuitems to reflect new model save path");
					
					redoActions();
					statcon.setDirty(true);
				}
			}
		}
			
	}

	class ListUpdate extends Thread {
		ArrayList items;
		public ListUpdate(ArrayList items) {
			this.items = items;
		}
		public void run() {
			try {
				for (int i = 0; i < items.size(); i++) {
					Object o = items.get(i);
					if (o instanceof SDDescriptor) {
						statcon.updateModel((SDDescriptor)o);
					} else if (o instanceof TRCAgent) {
						statcon.updateAgent((TRCAgent)o);
						//we exit here because updating the agent updates everything
						return;
					}
				}
				
			} catch (Throwable t) {
				PerfmonPlugin.DBG.warning("couldnt update perfmon agent "+t);	
			}	
		}	
	}

	class AgentUpdateThread extends Thread implements StatisticalLoaderListener {
		public boolean die;
		TRCAgent agent;
		
		Object list_lock = new Object();
		ArrayList descriptor_updates = new ArrayList();
		HashMap descriptor_map = new HashMap();
		ArrayList agent_updates = new ArrayList();
		
		public AgentUpdateThread(TRCAgent trcagent) {
			this.agent = trcagent;
		}
		public void modelChanged(SDDescriptor des) {
//			SDDescriptor parent = des.getParent();
			SDDescriptor parent = des;
			synchronized(list_lock) {

				if (parent == null) {
					if (agent_updates.size() == 0) {
						agent_updates.add(agent);
					}
				} else {
					
					if (descriptor_map.get(parent) != null) {
						return;	
					}
					
					SDDescriptor tmp = parent.getParent();
					while (tmp != null) {
						if (descriptor_map.get(tmp) != null) {
//						if (descriptor_updates.contains(tmp)) {
							//its parent is there - no need to update it
							return;
						}
						tmp = tmp.getParent();
					}
					
					descriptor_updates.add(parent);
					descriptor_map.put(parent,parent);
				}

			}
		}
		
		private void collapse() {
			synchronized(list_lock) {
				
				if (agent_updates.size() > 0) {
//System.out.println("WHOLE AGENT UPDATE");					
					return;	
				}
				
				//lots of changes - might as well update the whole tree
				if (descriptor_updates.size() > 350) {
//System.out.println(descriptor_updates.size()+" DESCRIPTORS - WHOLE AGENT UPDATE");					
					agent_updates.add(agent);
					return;
				}

//System.out.println(descriptor_updates.size()+" DESCRIPTORS - COLLAPSING");					
				
				for (int i = 0; i < descriptor_updates.size(); i++) {
					SDDescriptor descriptor = (SDDescriptor)descriptor_updates.get(i);
	
					SDDescriptor tmp = descriptor.getParent();
					while (tmp != null) {
						if (descriptor_map.get(tmp) != null) {
//						if (descriptor_updates.contains(tmp)) {
							//its parent is there - no need to update it
							descriptor_updates.remove(i--);
							descriptor_map.remove(tmp);
							break;
						}
						tmp = tmp.getParent();
					}
				}	

//System.out.println(descriptor_updates.size()+" DESCRIPTORS - UPDATING");					
				
			}
		}
		
		public void run() {
			while (!die) {
				try {
					Thread.sleep(1000);	
				} catch (Exception e) {
				}

				synchronized(list_lock) {

					collapse();

//@@@agent_updates.add(agent);

					//if we are updating the agent, dont bother with its models
					if (agent_updates.size() > 0) {
						statcon.getGraphWindow().getDisplay().syncExec(new ListUpdate(agent_updates));
	
					//update bits of the agent's models
					} else if (descriptor_updates.size() > 0) {
						statcon.getGraphWindow().getDisplay().syncExec(new ListUpdate(descriptor_updates));
					}
					descriptor_updates = new ArrayList();
					descriptor_map = new HashMap();
					agent_updates = new ArrayList();
					
				}	
			}	
		}	
	}
		
}