/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.ui.load;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.operation.IRunnableWithProgress;

/**
 * An interface to the background thread for loading sequence
 * diagrams. Ensures that only one thread is running, and that
 * only one sequence diagram is being loaded at any time. A call
 * to newTask() overrides the current task, canceling it. 
 *
 * @author Curtis d'Entremont
 * @since 1.3
 */
public class BackgroundLoader {
	
	private static BackgroundLoader instance;
	private Thread thread;
	private IProgressMonitor monitor = new CancelMonitor();
	private IRunnableWithProgress nextTask;
	
	/**
	 * This is a singleton class. If the instance does not yet
	 * exist, it is created.
	 * 
	 * @return the single instance.
	 */
	public synchronized static BackgroundLoader getInstance() {
		if (instance == null)
			instance = new BackgroundLoader();
		return instance;
	}
	
	/**
	 * Schedules a new task (load diagram, refresh, etc). If the
	 * thread is currently busy, the current task is canceled and
	 * the task is scheduled as the next task. If not, the task
	 * starts immediately.
	 * 
	 * Note: The task is responsible for polling the progress
	 * monitor for cancelation.
	 * 
	 * @param task the task to be performed.
	 */
	public synchronized void newTask(final IRunnableWithProgress task) {
		nextTask = task;
		if (thread == null || !thread.isAlive()) {
			thread = new Thread(new Runnable() {
				public void run() {
					while (nextTask != null) {
						try {
							IRunnableWithProgress current;
							synchronized (BackgroundLoader.this) {
								current = nextTask;
								monitor.setCanceled(false);
							}
							
							current.run(monitor);
							
							synchronized (BackgroundLoader.this) {
								if (current == nextTask)
									nextTask = null;
							}
						}
						catch (Exception e) {
							//e.printStackTrace();
						}
						monitor.setCanceled(false);
					}
				}
			});
			thread.setName("Sequence diagram loader"); //$NON-NLS-1$
			thread.start();
		}
		else {
			monitor.setCanceled(true);
		}
	}

	/**
	 * Cancels the current task. This will only take effect the
	 * next time the task checks for cancelation.
	 */
	public synchronized void cancelTask() {
		monitor.setCanceled(true);
		if (thread.isAlive()) {
			thread.destroy();
		}
	}
	
	public static void shutdown() {
		if (instance != null && instance.monitor != null && !instance.monitor.isCanceled()) {
			instance.monitor.setCanceled(true);
		}
	}
	
	private BackgroundLoader() {
	}
	
	/*
	 * This monitor is for cancelation monitoring only. It is used to
	 * signal the running task that the user canceled. 
	 */
	private static class CancelMonitor implements IProgressMonitor {
		private volatile boolean canceled = false;
		
		public boolean isCanceled() {
			return canceled;
		}

		public void setCanceled(boolean canceled_) {
			canceled = canceled_;
		}
		
		public void beginTask(String name, int totalWork) {}
		public void done() {}
		public void internalWorked(double work) {}
		public void setTaskName(String name) {}
		public void subTask(String name) {}
		public void worked(int work) {}
	}
}
