/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.core;

import java.util.*;
import java.util.List;

import org.eclipse.debug.core.*;
import org.eclipse.hyades.internal.execution.local.common.Options;
import org.eclipse.hyades.security.util.GridUtil;
import org.eclipse.hyades.trace.internal.ui.PDPluginImages;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.launcher.*;
import org.eclipse.hyades.trace.ui.internal.util.*;
import org.eclipse.hyades.trace.ui.launcher.*;
import org.eclipse.jface.dialogs.*;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.*;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.*;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.IWorkbench;

/*
 * CONTEXT_ID pfpp0001 for Trace profilings table in Profile profilings ui page
 */
/*
 * CONTEXT_ID pfpp0002 for New button in Profile profilings ui page
 */
/*
 * CONTEXT_ID pfpp0003 for Edit button in Profile profilings ui page
 */
/*
 * CONTEXT_ID pfpp0004 for Delete button in Profile profilings ui page
 */
/*
 * CONTEXT_ID pfpp0005 for Up button in Profile profilings ui page
 */
/*
 * CONTEXT_ID pfpp0006 for Down button in Profile profilings ui page
 */
/*
 * CONTEXT_ID pfpp0007 for Package entry field in Profile profilings add/edit dialog page
 */
/*
 * CONTEXT_ID pfpp0008 for Method Name entry field in Profile profilings for Package entry field in Profile profilings add/edit dialog page
 */
/*
 * CONTEXT_ID pfpp0009 for Rule combobox in Profile profilings for Package entry field in Profile profilings add/edit dialog page
 */

/*
 * CONTEXT_ID pfps0001 for Trace profilings Set table in Profile profilings ui page
 */

/*
 * CONTEXT_ID pfps0002 for New Set button in Profile profilings ui page
 */
/*
 * CONTEXT_ID pfps0003 for Edit Set button in Profile profilings ui page
 */
/*
 * CONTEXT_ID pfps0004 for Delete Set button in Profile profilings ui page
 */

public class TraceProfileOverviewUI extends AbstractChangeable implements SelectionListener, ICheckStateListener {
	private Composite _result;

	protected ArrayList _profilingList;
	protected ProfilingSetList _profilingSetList;
	
	private Button _editBtn;

	protected Table _table;
	protected CheckboxTableViewer _tableViewer;

	private Table _profilingSetTable;
	private CheckboxTableViewer _profilingSetTableViewer;
	private Button _addSetBtn;
	private Button _renameSetBtn;
	private Button _removeSetBtn;

	private Label _contentLabel;
	
	private Label profilingSetLabel;	
	
	private ProfileTab parentTab;
	
	private ProfilingSetsManagerCopy managerCopy;
	
	private Listener _listener = new Listener() {
		public void handleEvent(Event event) {
			changed();
		}
	};
	
	public class ProfilingSetLabelProvider extends LabelProvider
	implements ITableLabelProvider {

		public ProfilingSetLabelProvider() {
			super();
		}
		public Image getColumnImage(Object element, int columnIndex) {
			return PDPluginImages.getImage(PDPluginImages.IMG_FILTER);
		}
		public Image getColumnImage(
				Viewer viewer,
				Object element,
				int columnIndex) {
			return PDPluginImages.getImage(PDPluginImages.IMG_FILTER);
		}
		public String getColumnText(Object element, int columnIndex) {
			return ((IProfilingSet) element).getName();
		}
		public String getColumnText(
				Viewer viewer,
				Object element,
				int columnIndex) {
			
			return ((IProfilingSet) element).getName();
		}
	}

	class ProfilingSetContentProvider implements IStructuredContentProvider {
		private CheckboxTableViewer fViewer;
		//protected ProfilingSetElement fChangedElement;
		protected IProfilingSet fChangedElement;

		/**
		 * The constructor.
		 */
		public ProfilingSetContentProvider(CheckboxTableViewer tableViewer) {
			fViewer = tableViewer;
		}
		/**
		 * The visual part that is using this content provider is about
		 * to be disposed. Deallocate all allocated SWT resources.
		 */
		public void dispose() {
		}
		/**
		 * 
		 */
		public void elementChanged(
				IProfilingSet changedElement,
				String changedProperty) {
			fChangedElement = changedElement;
			fViewer.getControl().getDisplay().asyncExec(new Runnable() {
				public void run() {

					fViewer.refresh(fChangedElement);
				}
			});

		}
		/**
		 * Returns all the markers that should be shown for
		 * the current settings.
		 */

		public Object[] getElements(Object parent) {
			return ((ProfilingSetList) parent).elements();
		}
		public void inputChanged(
				Viewer viewer,
				Object oldInput,
				Object newInput) {
			fViewer = (CheckboxTableViewer) viewer;

			// Disconnect from old input.
			if (oldInput != null) {
				if (oldInput instanceof ProfilingList) {
					((ProfilingSetList) oldInput).removeElementChangedListener(
							this);

				}
			}

			// Connect to new input.
			if (newInput != null) {
				if (newInput instanceof ProfilingList)
					((ProfilingSetList) newInput).addElementChangedListener(this);
			}

		}
		public boolean isDeleted(Object element) {
			return false;
		}
	}

	public class ProfilingSetList implements IBasicPropertyConstants {
		ArrayList profilings = new ArrayList();
		TableViewer fViewer;

		private ArrayList listeners;

		public ProfilingSetList(TableViewer viewer) {
			fViewer = viewer;
		}
		public void add(IProfilingSet element) {
			add(element, -1);
		}

		public void add(IProfilingSet element, int index) {
			if (index < 0)
				profilings.add(element);
			else
				profilings.add(index, element);
		}
		/**
		 * 
		 */
		public void addElementChangedListener(ProfilingSetContentProvider newListener) {

			if (newListener == null)
				return;
			if (listeners == null)
				listeners = new ArrayList();
			if (!listeners.contains(newListener))
				listeners.add(newListener);

		}
		/**
		 * Insert the method's description here.
		 * Creation date: (08/23/2000 11:34:27 AM)
		 * @return java.util.Enumeration
		 */
		public Object[] elements() {
			return profilings.toArray();
		}
		/**
		 * 
		 */
		public void fireElementChanged(
				IProfilingSet changedElement,
				String changedProperty) {
			// forward change to content providers.
			if (listeners != null) {
				Iterator e = listeners.iterator();
				while (e.hasNext()) {
					ProfilingSetContentProvider l =
						(ProfilingSetContentProvider) e.next();
					l.elementChanged(changedElement, changedProperty);
				}
			}
		}
		public Object getElementAt(int index) {
			return profilings.get(index);
		}
		/**
		 * Insert the method's description here.
		 * Creation date: (11/13/2000 12:53:52 PM)
		 * @return java.util.Vector
		 */
		public ArrayList getprofilings() {
			return (ArrayList) profilings.clone();
		}
		/**
		 * Insert the method's description here.
		 * Creation date: (09/18/2000 4:02:44 PM)
		 * @return int
		 */
		public int getSize() {
			return profilings.size();
		}
		/**
		 * Insert the method's description here.
		 * Creation date: (08/22/2000 4:53:57 PM)
		 */
		public void removeAll() {
			profilings.clear();
		}
		/**
		 * Insert the method's description here.
		 * Creation date: (08/22/2000 3:48:34 PM)
		 * @param idx int
		 */
		public Object removeElementAt(int idx) {
			Object element = profilings.get(idx);

			if (element == null)
				return null;

			profilings.remove(element);

			return element;
		}
		/**
		 * Removes a listener for selection changes.
		 */
		public void removeElementChangedListener(ProfilingSetContentProvider listener) {

			if (listeners != null)
				listeners.remove(listener);
		}
		/**
		 * Insert the method's description here.
		 * Creation date: (08/23/2000 11:34:27 AM)
		 * @return java.util.Enumeration
		 */
		public Object[] uiElements() {
			return profilings.toArray();
		}
	}
    
	/**
	 * Simple list content provider
	 */
	class ListContentProvider implements IStructuredContentProvider {
		ArrayList contents;

		public ListContentProvider() {
		}
		/**
		 * Implements IStructuredContentProvider.
		 * 
		 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(Object)
		 */
		public Object[] getElements(Object input) {
			if (contents != null && contents == input) {
				return contents.toArray();
			}
			return new Object[0];
		}
		/**
		 * Implements IContentProvider.
		 * 
		 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(Viewer, Object, Object)
		 */
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			if (newInput instanceof ArrayList) {
				contents = (ArrayList) newInput;
			}
			else {
				contents = null;
			}
		}
		/**
		 * Implements IContentProvider.
		 * 
		 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
		 */
		public void dispose() {
		}
	}
	
    public class EditWizard extends Wizard {
		
		private EditWizardPage1 wizPage1;
		private EditWizardPage2 wizPage2;
		protected IStructuredSelection selection;
		protected IWorkbench workbench;
		private String parentName;
		
		public EditWizard(String psName)
		{
			super();
			parentName = psName;
		}
		
		public void setParent (String parent) {
			
			parentName = parent;
		}


		public void init(IWorkbench workbench, IStructuredSelection selection) {
			this.workbench = workbench;
			this.selection = selection;

		}

		public void addPages() {
			super.addPages();
			
			wizPage1 = new EditWizardPage1("profilingTypes", parentName);			
			wizPage1.setTitle(this.getWindowTitle());
			wizPage1.setDescription(this.getWindowTitle());
			addPage(wizPage1);
			
			wizPage2 = new EditWizardPage2("filters");
			wizPage2.setTitle(this.getWindowTitle());
			wizPage2.setDescription(this.getWindowTitle());
			addPage(wizPage2);
			
		}

		public boolean performFinish() 
		{
			boolean finish =((wizPage1.performFinish()) && (wizPage2.performFinish()));
			parentTab.update();			
			
			return finish;
		}
	}
	
	public class EditWizardPage2 extends WizardPage {
	
		private ProfileFiltersTab _filterTab;
	
		public EditWizardPage2(String pageId) {
			super(pageId);
			
			_filterTab = new ProfileFiltersTab("name");
		}
		
		public void createControl(Composite parent) {

			Composite composite = _filterTab.createControl(parent);
			_filterTab.addListener(_listener);
			
			
			Collection set = managerCopy.getFilterSets().values();
			FilterSetElement[] array = new FilterSetElement[set.size()];			
			
			_filterTab.activateFilters((FilterSetElement[])set.toArray(array));
			setControl(composite);
		}
		
		public boolean performFinish() {
			
			//save filters to manager copy
			Map profilingFilters = managerCopy.getFilterSets();
			profilingFilters.clear();
			ArrayList filters = _filterTab.getFilterSet();			
			for(int idx=0; idx<filters.size(); idx++)
			{
				FilterSetElement  elem = (FilterSetElement) filters.get(idx);
				profilingFilters.put(elem.getName(), elem);
				
				if(elem.getEnabled())
					managerCopy.setDefaultFilterSet(elem);
			}
			
			IProfilingSet profSet = managerCopy.getDefaultSet();
			Map profAttrs = profSet.getAttributes();
			
			//apply attributes to the manager copy
			FilterSetElement set = managerCopy.getDefaultFilterSet();			
			ArrayList attrs = set.getChildren();
			StringBuffer filterStr = new StringBuffer();
			for(int idx=0; idx<attrs.size(); idx++)
			{
				FilterTableElement elem = (FilterTableElement) attrs.get(idx);
				filterStr.append(elem.getText());
				if(elem.getMethod().length() > 0)
					filterStr.append(":").append(elem.getMethod());
				filterStr.append(":").append(elem.getVisibility());
				filterStr.append("&");
			}
			
			String key = UIPlugin.getPluginId()+".SETFILTER";
			profAttrs.put(key, new ProfilingAttribute(key, filterStr.toString()));
			
			return true;
		}
	}
	
	public class EditWizardPage1 extends WizardPage implements ICheckStateListener, ISelectionChangedListener {
		
		private boolean canFlipToNext;
		private PageContent dc;
		private String profilingSetName;
		private CheckboxTreeViewer treeViewer;
		private TreeContentProvider  provider;
		
		public EditWizardPage1(String pageId, String psName) {			
			super(pageId);
			profilingSetName = psName;
		}
				
		public boolean isPageComplete() {
			return true;
		}
		
		public boolean canFlipToNextPage() {
			return canFlipToNext;
		}
		
		public void dispose()
		{
			dc.dispose();
			super.dispose();			
		}
		
		public void createControl(Composite parent) {
			
			Composite result = (Composite) new Composite(parent, SWT.NONE);
			
			GridLayout layout;
			GridData data;

			layout = new GridLayout();
			layout.numColumns = 2;
			result.setLayout(layout);
			data = GridUtil.createFill();
			result.setLayoutData(data);
			
			data = GridUtil.createFill();
			
			GridLayout layout2 = new GridLayout();
			layout2.numColumns = 1;
			
			provider = new TreeContentProvider();
			treeViewer = new CheckboxTreeViewer(result);
			treeViewer.setContentProvider(provider);
			
			Object[] profileTypeGroups = getManagerCopy().getProfilingTypesGroups().values().toArray();			
			ImageDescriptor[] descriptorList = new ImageDescriptor[profileTypeGroups.length];
			for (int i = 0; i < profileTypeGroups.length; i ++) {
				descriptorList[i] = ((IProfilingSetTypeGroup)profileTypeGroups[i]).getIcon();
			}
			
			treeViewer.setLabelProvider(new TreeLabelProvider(descriptorList));

			treeViewer.setInput("");
			treeViewer.getTree().setLayout(layout2);
			treeViewer.getTree().setLayoutData(GridUtil.createVerticalFill());
			
			IProfilingSetType[] array = new ProfilingSetType[getManagerCopy().getProfilingTypes().size()];
			dc = new PageContent(result, SWT.NONE);
			dc.createContent((IProfilingSetType[])getManagerCopy().getProfilingTypes().values().toArray(array));
			dc.setVisible(true);
			
			initializeSelection();
			
			treeViewer.addCheckStateListener(this);
			treeViewer.addSelectionChangedListener(this);
			
			setControl(result);
		
			setPageComplete(isPageComplete());
		}
		
		protected void initializeSelection()
		{
			IProfilingSet set = managerCopy.getDefaultSet();
			if(set == null)
			    return;
			
			List types = set.getProfilingTypes();
			for(int idx=0; idx<types.size(); idx++)
			{
				String type = types.get(idx).toString();
				
				Object ptype = managerCopy.getProfilingTypes().get(type);
				if(type != null)
					setChecked((IProfilingSetType)ptype);
			}
		}
		
		private void setChecked(IProfilingSetType ptype)
		{
			if(ptype == null)
				return;
			
			treeViewer.expandToLevel(ptype, 0);
			treeViewer.setChecked(ptype, true);
			
			// Get all of this ProfileType's parent's children. If all of them are now checked,
			// the check our parent.
			Object parent = provider.getParent(ptype);
			Object[] parentsChildren = provider.getChildren(parent);
			boolean foundUnchecked = false;
			for(int i = 0; i < parentsChildren.length; i ++) {
				if (treeViewer.getChecked(parentsChildren[i]) == false) foundUnchecked = true;
			}
			
			if (foundUnchecked == false) treeViewer.setChecked(parent, true);
		}
		
		public void checkStateChanged(CheckStateChangedEvent event){
			Object checkedElement = event.getElement();

			if (checkedElement instanceof IProfilingSetType) {

				Object parent = provider.getParent(checkedElement);
				Object[] parentsChildren = provider.getChildren(parent);
				boolean foundChecked = false;
				for(int i = 0; i < parentsChildren.length; i ++) {
					if (treeViewer.getChecked(parentsChildren[i]) == true) foundChecked = true;
				}
				
				treeViewer.setChecked(parent, foundChecked);

			}
			else if (checkedElement instanceof IProfilingSetTypeGroup) {
				
				Object[] parentsChildren = provider.getChildren(checkedElement);
				for(int i = 0; i < parentsChildren.length; i ++) {
					treeViewer.setChecked(parentsChildren[i], event.getChecked());
				}
			}
			updateWizardButtons();
		}

		public void selectionChanged(SelectionChangedEvent event) {
			updateWizardButtons();
			IStructuredSelection selection = (IStructuredSelection) event.getSelection();
			if (selection.getFirstElement() instanceof IProfilingSetType) {
				updateGui((IProfilingSetType)selection.getFirstElement());
			} else {
				dc.showEmptyDetails();
			}
		}
		

		private void updateGui(IProfilingSetType newProfileType) {
			Control newControl = newProfileType.getProfilingType().getControl(profilingSetName);
			Control currentControl = dc.getCurrentlyDisplayingControl();
			if (newControl.equals(currentControl) == false) {
				dc.showDetailsFor(newControl);
			}
		}
		

		/**
		 * Updates the Next/Back buttons on the wizard.
		 * The buttons will be enabled if one of the following conditions are true:
		 * - if at least one of the checked elements use filters.
		 * - if the first selected valid element uses filters.
		 *
		 * Otherwise, the buttons will be disabled.
		 */
		private void updateWizardButtons() {
			Object[] checkedElementList = treeViewer.getCheckedElements();

			// Get the list of checked elements, then filter out all of the ones that aren't
			// ProfileType objects. For each of the ones that ARE profile type objects, if any
			// of them have filters turned on, then enable the next button and return.
			for(int i = 0; i < checkedElementList.length; i ++) {
				if (checkedElementList[i] instanceof IProfilingSetType) {
					IProfilingSetType nextProfile = (IProfilingSetType) checkedElementList[i];
					if (nextProfile.isUseFilters()) {
						canFlipToNext = nextProfile.isUseFilters();
						getWizard().getContainer().updateButtons();
						return;
					}
				}
			}

			// If none of the checked elements have filters turned on, then check that the currently
			// selected element has filters turned on. If it does, then enable the next button. Otherwise,
			// disable the next button.
			IStructuredSelection selection = (IStructuredSelection) treeViewer.getSelection();
			Iterator i = selection.iterator();
			while(i.hasNext() == true) {
				Object currentIteration = i.next();
				if (currentIteration instanceof IProfilingSetType) {
					IProfilingSetType firstSelectedProfile = (IProfilingSetType) currentIteration;
					canFlipToNext = firstSelectedProfile.isUseFilters();
					getWizard().getContainer().updateButtons();
					return;
				}
			}
			
			// If none of the conditions hold, disable the buttons.
			canFlipToNext = false;
			getWizard().getContainer().updateButtons();
			
		}		
		
		public boolean performFinish()
		{
			//update profiling types
			IProfilingSet set = managerCopy.getDefaultSet();
			
			set.getProfilingTypes().clear();
			Object[] selections = treeViewer.getCheckedElements();
			for(int idx=0; idx<selections.length; idx++)
			{
				Object type = selections[idx];
				if(type != null && type instanceof IProfilingSetType)
				{
					set.getProfilingTypes().add(((IProfilingSetType)type).getId());
				}
			}
			
			return dc.performFinish();
		}
		
	}
	
	// This class is responsible for showing the appropriate control when the user clicks on
	// an entry in the TreeViewer.
	public class PageContent extends Composite {
	
		private StackLayout detailsLayout;
		private Control[] controls;
		private IProfilingSetType[] profilingTypes;
		private Control currentControl;

		// Constructor.
		public PageContent(Composite composite, int style) {
			super(composite, style);
			currentControl = null;
		}

		public Composite createContent(IProfilingSetType[] types) {

			// Create a new stack layout.
			detailsLayout = new StackLayout();
		    this.setLayout(detailsLayout);
	    	this.setLayoutData(new GridData(GridData.FILL_BOTH));

			controls = new Control[types.length + 1];
			profilingTypes = types;

			for(int i = 0; i < types.length; i ++) {
				controls[i] = types[i].getProfilingType().createControl(this, managerCopy);
				controls[i].setVisible(false);
			}

			controls[controls.length - 1] = new Composite(this, SWT.NONE);
			controls[controls.length - 1].setVisible(false);

			return null;
		}

		public boolean performFinish()
		{
			IProfilingSet set = managerCopy.getDefaultSet();
			if(set == null)
				return true;
			
			StringBuffer errorMsg = new StringBuffer();			
			List types = set.getProfilingTypes();
			for(int i = 0; i < profilingTypes.length; i++) {
				
				if(types.contains(profilingTypes[i].getId()))
				{
					IProfilingType type = profilingTypes[i].getProfilingType();
					String msg = type.validateConfiguration(managerCopy);
					if(msg != null)
					{
						errorMsg.append(msg).append("\n");
					}
					
				}
			}
			
			if(errorMsg.length() > 0)// conflicts for the current configuration
			{
				String msg = UIPlugin.getResourceString("ERROR_DLG_CONFLICTS_MSG");												
				if(!MessageDialog.openQuestion(getShell(), UIPlugin.getResourceString("ERROR_DLG_CONFLICTS_TITLE"), msg+errorMsg.toString()))
				{
					return false;
				}										
			}
			
			List optionsList = new ArrayList();
			for(int i = 0; i < profilingTypes.length; i++) {
				
				if(types.contains(profilingTypes[i].getId()))
				{
					IProfilingType type = profilingTypes[i].getProfilingType();
					ProfilingAttribute[] options = type.getAttributes();
					for(int idx=0; idx<options.length; idx++)
					{
						addProfilingOption(optionsList, options[idx]);
					}
//					System.out.println("perform apply " + profilingTypes[i].getName());
				}
			}
			
			//apply attributes to the manager copy
			set.getAttributes().clear();
			Map attrs = set.getAttributes();
			for(int idx=0; idx<optionsList.size(); idx++)
			{
				ProfilingAttribute option = (ProfilingAttribute) optionsList.get(idx);
				attrs.put(option.getName(), option);
			}
			
			return true;
		}
				
		// Show the control specified on the screen.
		public void showDetailsFor(Control detail) {
			detailsLayout.topControl = detail;
			currentControl = detail;
			detail.setVisible(true);
   			layout();
		}

		public void showEmptyDetails() {
			detailsLayout.topControl = controls[controls.length - 1];
			currentControl = controls[controls.length - 1];
			controls[controls.length - 1].setVisible(true);
			layout();
		}

		// Dispose all of the controls.
		public void dispose() {
			currentControl = null;
			profilingTypes = null;
			for(int i = 0; i < controls.length; i++) {
				if (controls[i] != null) {
					controls[i].dispose();
				}
			}
			
			controls = null;
			
		}
		
		/**
		 * Returns the control that is currently being displayed in the StackLayout.
		 * Returns null if no control has been displayed.
		 * 
		 * @return the control object currently being displayed.
		 */
		public Control getCurrentlyDisplayingControl() {
			return currentControl;
		}

		public void selectionChanged(SelectionChangedEvent event) {

				}
	}

	/**
	 * This class is responsible for providing the text labels and the icon graphics
	 * that is displayed in the tree view.
	 *  
	 * @author sshiah
	 */
	public class TreeLabelProvider extends LabelProvider {

		// This hashtable maps each ImageDescriptor object (provided by the ProfileType
		// object) to the corresponding Image object.
		private Hashtable iconList;

		private ImageDescriptor memoryDescriptor;
		private ImageDescriptor coverageDescriptor;
		private ImageDescriptor probesDescriptor;

		/**
		 * Constructor for TreeLabelProvider.
		 * 
		 * @param descList list of ImageDescriptors of the images that will be displayed in the tree.
		 */
		public TreeLabelProvider(ImageDescriptor[] descList) {

			// Create a new hashtable to store the ImageDescriptor to Image mappings.
			iconList = new Hashtable();

			// Iterate through the list of mappings already in the hashtable. Create a
			// new image (and mapping) only if the ImageDescriptor does not already exist.
			// This allows elements to share icon images if they use the same image. 
			for(int i = 0; i < descList.length; i ++) {
				if (iconList.containsKey(descList[i]) == false) {
					iconList.put(descList[i], descList[i].createImage());
				}
			}
		}

		/**
		 * Retrieves the label that should appear for the given tree element. Returns
		 * the given string if the element is one of the categories, or returns the
		 * name of the Profile Type if the given element is a ProfileType object.
		 * 
		 * @return the name of the element.
		 */
		public String getText(Object element) {
			if (element instanceof IProfilingSetType) {
				return ((IProfilingSetType)element).getName();
			} else if(element instanceof IProfilingSetTypeGroup){
				return ((IProfilingSetTypeGroup)element).getName();
			}
			
			return element.toString();
		}

		/**
		 * Retrieves the icon for the given Tree element.
		 * 
		 * @return the icon if the element has one. Returns null if this element does not have an icon.
		 */
		public Image getImage(Object element) {
			
			ImageDescriptor imageDesc = null;
			if (element instanceof IProfilingSetType) {
				imageDesc = ((IProfilingSetType)element).getImage();
			} else if(element instanceof IProfilingSetTypeGroup){
				imageDesc = ((IProfilingSetTypeGroup)element).getIcon();
			}			
			if(imageDesc == null)
				return null;
			
			return (Image)iconList.get(imageDesc); 
		}

		/**
		 * Disposes of all of the images currently stored within this class, and de-references
		 * all class variables.
		 */
		public void dispose() {
			
			Iterator list = iconList.keySet().iterator();
			while(list.hasNext()) {
				((Image)iconList.get(list.next())).dispose();
			}
			iconList.clear();
			iconList = null;
	    }
	}

	/**
	 * This class is responsible for providing the structure (i.e. by specifying the children
	 * parent relationship) of each element that should appear in the tree view.
	 *  
	 * @author sshiah
	 */
	public class TreeContentProvider implements ITreeContentProvider {

		// Contains a list of all of the ProfileType objects, each of which encapsulates
		// all of the information stored in the extension.
//		private ProfileType[] profileType;
//		private ProfilingTypeGroup[] profileType;

		/**
		 * Constructor for TreeContentProvider.
		 * 
		 * @param type the list of ProfileType objects containing all of the extensions.
		 */
		public TreeContentProvider() {
//			profileType = type;
		} 

		/**
		 * Retrieves the list of ProfileType objects that belong to the given level. 
		 * If a ProfileType object was specified, an empty array is returned.
		 * 
		 * This list that is returned determines which ProfileType object goes under
		 * which level in the tree view.
		 * 
		 * @return the list of children that belongs to the given element. 
		 */
		public Object[] getChildren(Object element) {

			if(element instanceof IProfilingSetTypeGroup)
			{
				return getManagerCopy().getProfilingTypesForGroup(((IProfilingSetTypeGroup)element).getId());
			}	
			
			return new Object[0];
		}

		/**
		 * Retrieves the list of levels that each profile type must belong to.
		 * 
		 * This returned list will determine the root elements in the tree. 
		 * 
		 * @return the list of upper-most elements in the tree. 
		 */
		public Object[] getElements(Object element) {
			return getManagerCopy().getProfilingTypesGroups().values().toArray();
		}

		/**
		 * This method will return false if the given element is a ProfileType object (as 
		 * ProfileType objects don't have children in the tree). Otherwise, this method
		 * returns true if the given level has at least 1 Profile Type belonging to it.
		 * 
		 * @return true if the given element has at least 1 children in the tree. Otherwise, false is returned.
		 */
		public boolean hasChildren(Object element) {
			
			return getChildren(element).length > 0;
		}

		/**
		 * This method will return a null string (e.g. "") if the given element is the label
		 * of a level. Otherwise (i.e. the given element is a ProfileType object), this method
		 * will return the level that this profile type belongs to.
		 * 
		 * @return The level that the profile type belongs to. Returns null string if the given element is a level.
		 */
		public Object getParent(Object element) {
			
			if(element instanceof IProfilingSetType)
			{
				String group = ((IProfilingSetType)element).getGroup();
				if(group != null)
				{
					return getManagerCopy().getProfilingTypesGroups().get(group);					
				}				
			}
			
			return "";
		}

		/**
		 * This method will de-reference the class variables.
		 */
		public void dispose() {
		}

		public void inputChanged(Viewer viewer, Object old_input, Object new_input){}
	}


	/*
	 * 
	 */
	class EditWSetDialog extends Dialog implements ModifyListener {
		private String _name;
		private Text _nameText;
		private String _title;
		private String _description;
		private String _id;
		private Text _descText;

		public EditWSetDialog(Shell shell, String title, String name, String desc, String id) {
			super(shell);

			_name = name;
			_title = title;
			_description = desc;
			_id = id;
		}

		protected void configureShell(Shell shell) {
			super.configureShell(shell);
			shell.setText(_title);
		}

		protected Control createDialogArea(Composite parent) {
			Composite result = (Composite) super.createDialogArea(parent);

			GridLayout layout;
			GridData data;
			Label label1;
			Label label2;

			layout = new GridLayout();
			layout.numColumns = 2;
			result.setLayout(layout);
			data = GridUtil.createFill();
			data.widthHint = 400;
			result.setLayoutData(data);

			label1 = new Label(result, SWT.NULL);
			label1.setText(UIPlugin.getResourceString("STR_PROFILING_SET_NAME"));
			_nameText = new Text(result, SWT.BORDER);
			_nameText.setLayoutData(GridUtil.createHorizontalFill());
			
			label2 = new Label(result, SWT.NULL);
			label2.setText(UIPlugin.getResourceString("STR_PROFILING_SET_DESCRIPTION"));
			_descText = new Text(result, SWT.BORDER);
			_descText.setLayoutData(GridUtil.createHorizontalFill());

			_nameText.addModifyListener(this);
			_descText.addModifyListener(this);

			if(_name == null)
				_name = "NewSet";
			_nameText.setText(_name);
				
			
			if (_description == null)
				_description = "New description";
			_descText.setText(_description);
			
			return result;
		}

		public void modifyText(ModifyEvent e) {
			if (e.widget == _nameText) {
				if (getButton(IDialogConstants.OK_ID) != null)
					getButton(IDialogConstants.OK_ID).setEnabled(
							_nameText.getText().trim() != "");
			}
			if (e.widget == _descText) {
				if (getButton(IDialogConstants.OK_ID) != null)
					getButton(IDialogConstants.OK_ID).setEnabled(
							_descText.getText().trim() != "");
			}
		}

		protected void okPressed() {

			_name = _nameText.getText().trim();
			_description = _descText.getText().trim();
			
			//update manager copy
			if(_id == null)//new set
			{
				IProfilingSet set = new ProfilingSet(_name, _name, _description);
				managerCopy.setDefaultSet(set);
				managerCopy.getProfilingSets().put(_name, set);
				_profilingSetList.add(set);
			}
			else
			{//existing filter
				IProfilingSet set = (IProfilingSet)managerCopy.getProfilingSets().get(_id);
				if(set == null)
				{
					set = new ProfilingSet(_id, _name, _description);
					managerCopy.setDefaultSet(set);
					managerCopy.getProfilingSets().put(_id, set);
					_profilingSetList.add(set);			
				}
				else
				{
					set.setName(_name);
					set.setDescription(_description);
				}				
			}
			
			super.okPressed();
		}

	}


	
	public TraceProfileOverviewUI(ProfileTab pt)
	{
		super();
		parentTab = pt;
	}

	private void createColumns() {
		TableLayout layout = new TableLayout();
		_table.setLayout(layout);
		_table.setHeaderVisible(true);

		String headers[] =
			{
				UIPlugin.getResourceString("HEADER_PROFILING_TYPE")};
		ColumnLayoutData layouts[] =
			{
				new ColumnPixelData(300, true),
		};

		final TableColumn columns[] = new TableColumn[headers.length];

		for (int i = 0; i < headers.length; i++) {
			layout.addColumnData(layouts[i]);
			TableColumn tc = new TableColumn(_table, SWT.NONE, i);
			tc.setResizable(layouts[i].resizable);
			tc.setText(headers[i]);
			columns[i] = tc;
		}
	}
	public Composite createControl(Composite parent) {
		GridLayout layout;

		_result = new Composite(parent, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 2;
		layout.verticalSpacing = 0;
		_result.setLayout(layout);
		GridData data = GridUtil.createFill();
		_result.setLayoutData(data);
			
		createProfilingSetGroup(_result);
		createprofilingContents(_result);

		data = new GridData();
		data.horizontalSpan = 2;
		
		return _result;
	}
	
	/*
	 * 
	 */
	private void createprofilingContents(Composite parent) {
		GridData data;
		GridLayout layout;
		

		Label label = new Label(parent, SWT.NULL);
		data = new GridData();
		data.horizontalSpan = 2;
		label.setLayoutData(data);

		_contentLabel = new Label(parent, SWT.NULL);
		_contentLabel.setText(
				UIPlugin.getResourceString("STR_MODIFY_PROFILINGSET_CONTENTS"));
		data = new GridData();
		data.horizontalSpan = 2;
		_contentLabel.setLayoutData(data);

		Composite tableGroup = new Composite(parent, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		tableGroup.setLayout(layout);
		data = GridUtil.createFill();
		data.heightHint = 180;
		tableGroup.setLayoutData(data);
		initializeTypesTable(tableGroup);

		Composite btnGroup = new Composite(parent, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		btnGroup.setLayout(layout);
		btnGroup.setLayoutData(GridUtil.createVerticalFill());

		Composite addGroup = new Composite(btnGroup, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		addGroup.setLayout(layout);
		addGroup.setLayoutData(GridUtil.createVerticalFill());
		
		_editBtn = new Button(addGroup, SWT.PUSH);
		_editBtn.setText(UIPlugin.getResourceString("EDIT_PROFILINGSETCONTENTS_TEXT"));
		_editBtn.setToolTipText(
				UIPlugin.getResourceString("EDIT_FILTER_TOOLTIP_TEXT"));
		_editBtn.setLayoutData(GridUtil.createHorizontalFill());


		Composite moveGroup = new Composite(btnGroup, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		moveGroup.setLayout(layout);
		moveGroup.setLayoutData(GridUtil.createHorizontalFill());

		_editBtn.setEnabled(false);

		label = new Label(parent, SWT.NULL);
		label.setLayoutData(GridUtil.createHorizontalFill());

		//add listeners
		_editBtn.addSelectionListener(this);
		_table.addSelectionListener(this);

		org.eclipse.ui.help.WorkbenchHelp.setHelp(
				_table,
				UIPlugin.getPluginId() + ".pfpp0002");
		/*
		 org.eclipse.ui.help.WorkbenchHelp.setHelp(
		 _addBtn,
		 UIPlugin.getPluginId() + ".pfpp0002");
		 org.eclipse.ui.help.WorkbenchHelp.setHelp(
		 _editBtn,
		 UIPlugin.getPluginId() + ".pfpp0003");
		 org.eclipse.ui.help.WorkbenchHelp.setHelp(
		 _removeBtn,
		 UIPlugin.getPluginId() + ".pfpp0004");
		 org.eclipse.ui.help.WorkbenchHelp.setHelp(
		 _upBtn,
		 UIPlugin.getPluginId() + ".pfpp0005");
		 org.eclipse.ui.help.WorkbenchHelp.setHelp(
		 _downBtn,
		 UIPlugin.getPluginId() + ".pfpp0006");
		 */
	}

	/*
	 * 
	 */
	private void createProfilingSetGroup(Composite parent) {
		GridData data;
		GridLayout layout;
		
		Label label = new Label(parent, SWT.NULL);
		label.setText(UIPlugin.getResourceString("STR_SELECT_PROFILING_SET"));
		data = new GridData();
		data.horizontalSpan = 2;
		label.setLayoutData(data);
		
		Composite tableGroup = new Composite(parent, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		tableGroup.setLayout(layout);
		data = GridUtil.createFill();
		data.heightHint = 100;

		tableGroup.setLayoutData(data);
		_profilingSetTable =
			new Table(
					tableGroup,
					SWT.V_SCROLL
					| SWT.BORDER
					| SWT.SINGLE
					| SWT.FULL_SELECTION
					| SWT.CHECK);
		_profilingSetTable.setLinesVisible(false);
		_profilingSetTable.setLayoutData(GridUtil.createFill());
	//	_profilingSetTable.addListener(SWT.Selection, _listener);

		_profilingSetTableViewer = new CheckboxTableViewer(_profilingSetTable);
		_profilingSetTableViewer.setContentProvider(
				new ProfilingSetContentProvider(_profilingSetTableViewer));
		_profilingSetTableViewer.setLabelProvider(new ProfilingSetLabelProvider());
		
		_profilingSetList = new ProfilingSetList(_profilingSetTableViewer);
		
		Object[] sets = order(getManagerCopy().getProfilingSets().values().toArray());
		for (int i = 0; i < sets.length; i++)
		{
			IProfilingSet set = (IProfilingSet)sets[i];
			_profilingSetList.add(set);
		}
						
		_profilingSetTableViewer.setInput(_profilingSetList);
		_profilingSetTableViewer.addCheckStateListener(this);

		TableLayout layoutt = new TableLayout();
		_profilingSetTable.setLayout(layoutt);

		ColumnLayoutData l = new ColumnPixelData(420, true);
		layoutt.addColumnData(l);

		TableColumn tc = new TableColumn(_profilingSetTable, SWT.NONE, 0);
		tc.setResizable(l.resizable);

		Composite btnGroup = new Composite(parent, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		btnGroup.setLayout(layout);
		btnGroup.setLayoutData(GridUtil.createVerticalFill());

		Composite addGroup = new Composite(btnGroup, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		addGroup.setLayout(layout);
		addGroup.setLayoutData(GridUtil.createVerticalFill());

		_addSetBtn = new Button(addGroup, SWT.PUSH);
		_addSetBtn.setText(UIPlugin.getResourceString("ADD_PROFILINGSET_TEXT"));
		_addSetBtn.setToolTipText(
				UIPlugin.getResourceString("ADD_FILTER_TOOLTIP_TEXT"));
		_addSetBtn.setLayoutData(GridUtil.createHorizontalFill());

		_renameSetBtn = new Button(addGroup, SWT.PUSH);
		_renameSetBtn.setText(UIPlugin.getResourceString("RENAME_PROFILINGSET_TEXT"));
		_renameSetBtn.setToolTipText(
				UIPlugin.getResourceString("EDIT_SET_TOOLTIP_TEXT"));
		_renameSetBtn.setLayoutData(GridUtil.createHorizontalFill());

		_removeSetBtn = new Button(addGroup, SWT.PUSH);
		_removeSetBtn.setText(UIPlugin.getResourceString("REMOVE_PROFILINGSET_TEXT"));
		_removeSetBtn.setToolTipText(
				UIPlugin.getResourceString("REMOVE_TOOLTIP_TEXT"));
		_removeSetBtn.setLayoutData(GridUtil.createHorizontalFill());

		_renameSetBtn.setEnabled(false);
		_removeSetBtn.setEnabled(false);

		//add listeners
		_removeSetBtn.addSelectionListener(this);
		_addSetBtn.addSelectionListener(this);
		_renameSetBtn.addSelectionListener(this);
		_profilingSetTable.addSelectionListener(this);

		org.eclipse.ui.help.WorkbenchHelp.setHelp(
				_profilingSetTable,
				UIPlugin.getPluginId() + ".pfps0001");
		
		profilingSetLabel = new Label(parent, SWT.WRAP);
		data = new GridData();
		data.widthHint = 375;
		profilingSetLabel.setLayoutData(data);
		/*
		 org.eclipse.ui.help.WorkbenchHelp.setHelp(
		 _addSetBtn,
		 UIPlugin.getPluginId() + ".pfps0002");
		 org.eclipse.ui.help.WorkbenchHelp.setHelp(
		 _editSetBtn,
		 UIPlugin.getPluginId() + ".pfps0003");
		 org.eclipse.ui.help.WorkbenchHelp.setHelp(
		 _removeSetBtn,
		 UIPlugin.getPluginId() + ".pfps0004");
		 */
	}
	
	private Object[] order(Object[] sets)
	{
		int counter = 0;
		IProfilingSet tempSet = null;
		Object[] newProfilingSets = new Object[sets.length];
		
		IProfilingSet defaultSet = getManagerCopy().getDefaultSet();
		
		//first check if first is the default
		tempSet = (IProfilingSet)sets[0];
		if (tempSet.getName().equals(defaultSet.getName())) return sets;
		
		//put defaultSet at index 0
		newProfilingSets[counter] = defaultSet;
		
		for (int j = 0; j < sets.length; j++)
		{
			tempSet = (IProfilingSet)sets[j];
			if (tempSet.getId() != defaultSet.getId())
			{
				counter++;
				newProfilingSets[counter] = tempSet;
			}
		}
		return newProfilingSets;
	}

	/**
	 * createPartControl method comment.
	 */
	private void createTable(Composite parent) {
		_table =
			new Table(
					parent,
					SWT.H_SCROLL
					| SWT.V_SCROLL
					| SWT.BORDER
					| SWT.MULTI
					| SWT.FULL_SELECTION);
		_table.setLinesVisible(false);
		_table.setLayoutData(GridUtil.createFill());

	}
	/**
	 * Insert the method's description here.
	 * Creation date: (08/22/2000 10:16:27 AM)
	 */
	public void enable(boolean enable) {
		_addSetBtn.setEnabled(enable);
		_removeSetBtn.setEnabled(enable);
		_renameSetBtn.setEnabled(enable);

		_profilingSetTable.setEnabled(enable);
		_profilingSetTableViewer.getTable().setEnabled(enable);

		_editBtn.setEnabled(enable);

		if (!enable) {
			_tableViewer.setCellEditors(null);
			_tableViewer.setCellModifier(null);
		}

	}
	/**
	 * Insert the method's description here.
	 * Creation date: (08/22/2000 10:16:27 AM)
	 */
	protected void enableButtons() {

		int selection = _table.getSelectionIndex();
		int selectionCount = _table.getSelectionCount();

		selection = _profilingSetTable.getSelectionIndex();

		_removeSetBtn.setEnabled(
		    _profilingSetList.getSize() > 1 && selection != -1);
		
		
		_renameSetBtn.setEnabled(_profilingSetList.getSize() > 0 && selection != -1);
	
		_editBtn.setEnabled(_profilingSetList.getSize() > 0 && selection != -1);
		
		if (selection != -1)
		{
			IProfilingSet set = (IProfilingSet) _profilingSetList.getElementAt(selection);
		 	profilingSetLabel.setText(set.getDescription());
		}
		else profilingSetLabel.setText("");
	}
	

	
	/**
	 * Insert the method's description here.
	 * Creation date: (11/13/2000 1:19:41 PM)
	 * @return boolean
	 */
	/**
	 * Insert the method's description here.
	 * Creation date: (08/21/2000 7:12:31 PM)
	 */
	protected void initializeTypesTable(Composite parent) {
		
		createTable(parent);
		_tableViewer = new CheckboxTableViewer(_table);
		createColumns();

		_tableViewer.setContentProvider(
				new ListContentProvider());
		_tableViewer.setLabelProvider(new LabelProvider());

		_profilingList = new ArrayList();
		_tableViewer.setInput(_profilingList);
	}


	/**
	 * Insert the method's description here.
	 * Creation date: (4/30/2001 1:12:16 PM)
	 */
	public void reset() {
		_table.removeAll();
		_profilingList.clear();
		_profilingSetList.removeAll();
		_profilingSetTableViewer.refresh();
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (08/22/2000 10:27:08 AM)
	 * @param selection int
	 */
	protected void selectTableItem(Table table, int selection) {

		//ZN
		table.setFocus();
		
		for(int idx=0; idx<table.getItemCount(); idx++)
		{
			table.getItem(idx).setChecked(idx==selection);
			
		}
		table.setSelection(selection);
		table.showSelection();
		
		
		if (table == _profilingSetTable) {
			showProfilingTypeDetails(
					(IProfilingSet) _profilingSetList.getElementAt(selection));
		}
		//ZN
		enableButtons();
	}

	/*
	 * 
	 */
	private void showProfilingTypeDetails(IProfilingSet elem) {
		
		_profilingList.clear();
		List types = elem.getProfilingTypes();
		for(int idx=0; idx<types.size(); idx++)
		{
			String type = types.get(idx).toString();
			
			IProfilingSetType ptype = (IProfilingSetType)managerCopy.getProfilingTypes().get(type);
			if(ptype != null)
				_profilingList.add(ptype.getName());
		}

		_tableViewer.refresh();
	}

	/**
	 * Insert the method's description here.
	 * Creation date: (08/21/2000 7:49:33 PM)
	 * @param event com.ibm.swt.events.SelectionEvent
	 */
	public void widgetDefaultSelected(SelectionEvent event) {
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (08/21/2000 7:49:33 PM)
	 * @param event com.ibm.swt.events.SelectionEvent
	 */
	public void widgetSelected(SelectionEvent event) {
		
		if (event.widget == _addSetBtn) {
			showWSetDialog(UIPlugin.getResourceString("STR_ADD_SET_DLG"), null, null, null);
			enableButtons();
			return;

		} else if (event.widget == _renameSetBtn) {
			
			renameProfilingSet();
			
		} else if (event.widget == _removeSetBtn) {
			
			int selection = _profilingSetTable.getSelectionIndex();
			if (selection != -1) {			
				removeProfilingSet(selection);
			}
			
		} else if (event.widget == _editBtn) {


			int selection = _profilingSetTable.getSelectionIndex();
			if (selection != -1) {
				IProfilingSet element = (IProfilingSet) _profilingSetList.getElementAt(selection);
				//element.getChildren().get(0);
				String name = element.getName();
				
				showEditDialog (
					UIPlugin.getResourceString("STR_EDIT_PROFILINGSET_DLG"),
					name,
					" "
				);
				
				showProfilingTypeDetails(element);
			}

		} else if (event.widget == _table) {
			enableButtons();

		} else if (event.widget == _profilingSetTable) {
			int idx = _profilingSetTable.getSelectionIndex();
			if (idx != -1) {
				// Set corresponding checkbox.
				IProfilingSet elem1 =
					(IProfilingSet) _profilingSetList.getElementAt(idx);
				
				managerCopy.setDefaultSet(elem1);

				showProfilingTypeDetails(elem1);
				_profilingSetTableViewer.setChecked(elem1, true);

				// Clear all other checkboxes.
				for (int kdx = 0; kdx < _profilingSetList.getSize(); kdx++) {
					if (kdx != idx) {
						IProfilingSet elem2 =
							(IProfilingSet) _profilingSetList.getElementAt(kdx);
						_profilingSetTableViewer.setChecked(elem2, false);
					}
				}
				_profilingSetTableViewer.refresh();

			}
			enableButtons();
			changed();
			return;
		}
	}

	private void showEditDialog(
			String title,
			String parentName,
			String profilingType) {

		EditWizard wizard = new EditWizard(parentName);

		WizardDialog dialog = new WizardDialog(new Shell(Display.getDefault()), wizard);

		wizard.init(org.eclipse.ui.PlatformUI.getWorkbench(), null);
		wizard.setParent(parentName);
		wizard.setWindowTitle(title);
		dialog.create();
		dialog.open();

	}

	private void showWSetDialog(String title, String name, String description, String id) {
		
		EditWSetDialog dialog =
			new EditWSetDialog(_result.getShell(), title, name, description, id);
		dialog.open();

		if (dialog.getReturnCode() == Window.OK) {
			
			_profilingSetTableViewer.refresh();
			for (int idx = 0; idx < _profilingSetList.getSize(); idx++) {
				IProfilingSet el =
					(IProfilingSet) _profilingSetList.getElementAt(idx);
				if (el.getName().equals(dialog._name)) {
					selectTableItem(_profilingSetTable, idx);
					showProfilingTypeDetails(el);					
					break;
				}
			
			changed();
		  }
		}

	}


	public void checkStateChanged(CheckStateChangedEvent event) {
		
		IProfilingSet elem = (IProfilingSet) event.getElement();
		boolean enable = true;
		//update any previous selected set  	 		
		for (int i = 0; i < _profilingSetList.getSize(); i++) {
			IProfilingSet data =
				(IProfilingSet) _profilingSetList.getElementAt(i);
			if (data == elem) {
				// Set corresponding selection.
				if (_profilingSetTableViewer.getChecked(data))
					_profilingSetTable.setSelection(i);
				continue;
			}

			//data.setEnabled(enable);
			_profilingSetTableViewer.setChecked(data, enable);
		}

		_profilingSetTableViewer.refresh();
		managerCopy.setDefaultSet(elem);
		
	}
	
	/**
	 * 
	 *
	 */
    protected void renameProfilingSet()
	{
		int idx = _profilingSetTable.getSelectionIndex();
		if (idx != -1) {
			
			IProfilingSet set = (IProfilingSet)_profilingSetList.getElementAt(idx);
			showWSetDialog(
					UIPlugin.getResourceString("STR_EDIT_SET_DLG"),
					set.getName(), set.getDescription(), set.getId());
			
			_profilingSetTable.getItem(idx).setText(set.getName());
			profilingSetLabel.setText(set.getDescription());
			
			showProfilingTypeDetails(set);
			changed();

		}
    }
	
	/**
	 * 
	 */
	protected void removeProfilingSet(int selection) {
		
	
		boolean select = false;
		if (_profilingSetTable.getItem(selection).getChecked())
			select = true;

		_profilingSetList.removeElementAt(selection);
		if (select) {
			IProfilingSet elem =
				(IProfilingSet) _profilingSetList.getElementAt(0);
			_profilingSetTableViewer.setChecked(elem, true);
		}
		_profilingSetTableViewer.refresh();

		selectTableItem(_profilingSetTable, 0);	
		showProfilingTypeDetails((IProfilingSet)_profilingSetList.getElementAt(0));					
		
		enableButtons();
	
		managerCopy.setDefaultSet(((IProfilingSet)_profilingSetList
		    .getElementAt(_profilingSetTable.getSelectionIndex())));
		changed();
	}

	/**
	 * Initialize ui content using selected configuration attributes
	 * @param conf
	 */
	public void initializeFrom(ILaunchConfiguration conf)
	{
//		System.out.println("initializeFrom config");
		getManagerCopy().initializeFrom(conf);
		
		int idx = -1;
		IProfilingSet set = getManagerCopy().getDefaultSet();
		if(set != null)
		{
			for(int i=0; i<_profilingSetList.getSize(); i++)
			{
				if(((IProfilingSet)_profilingSetList.getElementAt(i)).getId().equals(set.getId()))
				{
					idx = i;
					break;
				}
			}
		}
		if(idx != -1)
		{
			selectTableItem(_profilingSetTable, idx);
		}
		
	}
	
	public void performApply(ILaunchConfigurationWorkingCopy wc)
	{
//		System.out.println("apply profilingSets ");		
		managerCopy.performApply(wc);
		managerCopy.applyChanges();
	}
	
	public boolean isValid(ILaunchConfiguration conf)
	{
		//return _pot.isValid(conf);
		return true;
	}
	
	/**
	 * @return Returns the managerCopy.
	 */
	public ProfilingSetsManagerCopy getManagerCopy() {
		
		if(managerCopy == null)
			managerCopy = new ProfilingSetsManagerCopy();
		
		return managerCopy;
	}
	
	/**
	 * Add a profiling option to the set of existing options. If the option already exists,
	 * update the existing option value if necessary
	 * @param options
	 * @param option
	 */
	protected void addProfilingOption(List options, ProfilingAttribute option)
	{
		for(int idx=0; idx<options.size(); idx++)
		{
			ProfilingAttribute pOption = (ProfilingAttribute)options.get(idx);
			if(pOption.getName().equals(option.getName()))
			{
				if(pOption.getName().equals(Options.OPTION_STACK_INFORMATION))
				{
					//OPTION_VALUE_NORMAL take precedence to OPTION_VALUE_NONE
					//OPTION_VALUE_BOUNDARY take precedence to OPTION_VALUE_NORMAL
					if(pOption.getValue().equals(Options.OPTION_VALUE_BOUNDARY))
						 return; //higher precedence
					
					if(!option.getValue().equals(Options.OPTION_VALUE_NONE))
						 pOption.setValue(option.getValue());
				}
				else if(pOption.getName().equals(Options.OPTION_TRACE_MODE))
				{
					if(pOption.getValue().equals(Options.OPTION_VALUE_NOOBJECTCORRELATION))
						pOption.setValue(option.getValue()); // override no object correlation value with any other value				
				}
				else if(pOption.getValue().equals(Options.OPTION_VALUE_FALSE))
				{
					//false value is overridden by any other value
					pOption.setValue(option.getValue());
				}
				
				
				return;
			}
		}
		
		options.add(option);
	}	
}
