/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.launcher;

import java.io.*;
import java.util.*;
import org.eclipse.core.runtime.*;
import org.eclipse.debug.core.*;
import org.eclipse.debug.core.model.ILaunchConfigurationDelegate;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.HyadesConstants;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.jdt.launching.ExecutionArguments;
import org.eclipse.pde.internal.core.*;
import org.eclipse.pde.internal.ui.PDEPlugin;
import org.eclipse.pde.internal.ui.launcher.ILauncherSettings;
import org.eclipse.pde.internal.ui.launcher.LauncherUtils;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.pde.core.plugin.IPluginModelBase;

public class ProfileRuntimeWorkbenchDelegate implements ILaunchConfigurationDelegate, ILauncherSettings {
	
	private static final String ECLIPSE_MAIN = "org.eclipse.core.launcher.Main";

	// These variables came from WorkbenchLaunchConfigurationDelegate.java (see comment below).
	private static final String KEY_BAD_FEATURE_SETUP = "WorkbenchLauncherConfigurationDelegate.badFeatureSetup";
	private static final String KEY_NO_STARTUP = "WorkbenchLauncherConfigurationDelegate.noStartup";
	private File fConfigFile = null;
	
	public void launch(ILaunchConfiguration configuration, String mode, ILaunch launch, IProgressMonitor monitor) throws CoreException {

		try {
			monitor.beginTask("", 3);

			String workspace = getWorkspace(configuration);
			LauncherUtils.clearWorkspace(configuration, workspace);
			monitor.worked(1);

			// Build up the trace argument information.
			TraceArguments traceArguments = new TraceArguments(ECLIPSE_MAIN);
			traceArguments.setClassPath(getClassPath(configuration));
			traceArguments.setParameters(getProgramArgs(configuration));
			traceArguments.setVMArguments(getVMArgs(configuration));
			traceArguments.setEnvironmentVariable(getEnvironment(configuration));
			traceArguments.setAutoMonitoring(isAutoMonitoring(configuration));
			traceArguments.setHostName(getHostName(configuration));
			traceArguments.setPortNumber(getPortNumber(configuration));
			traceArguments.setProfileFile(getProfileFile(configuration));
			monitor.worked(1);

			ProfilingSetsManager manager = ProfilingSetsManager.instance();
			
			// Get all of the profiling-related information.
			ArrayList filters = manager.getFilters(configuration);
			Vector options = manager.getOptions(configuration);
			String projectName = getProjectName(configuration);
			String monitorName = getMonitorName(configuration);
			monitor.worked(1);

			// Launch the process and start the trace.
			PDCoreUtil.launchTrace(traceArguments, 
					 filters, options, 
					 projectName, monitorName);

		} catch (CoreException e) {
			monitor.setCanceled(true);
			throw e;
		}
	}

	/**
	 * Returns true if the user wants to auto-monitor the workbench after it has been launched.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return true if user specified to auto-montor. Defaults to true.
	 * @throws CoreException
	 */
	private boolean isAutoMonitoring(ILaunchConfiguration configuration) throws CoreException {
		return configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_AUTO_MONITORING, true);
	}
	
	/**
	 * Returns the hostname of where to launch the workbench.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return hostname of where to launch the workbench.
	 * @throws CoreException
	 */
	private String getHostName(ILaunchConfiguration configuration) throws CoreException {
		return configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, "localhost");
	}
	
	/**
	 * Returns the port number of where to launch the workbench.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return port number of the machine to launch the workbench.
	 * @throws CoreException
	 */
	private int getPortNumber(ILaunchConfiguration configuration) throws CoreException {
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
		return configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, store.getInt(HyadesConstants.LOCALHOST_PORT));
	}
	
	/**
	 * Returns the project name to store all of the profiled data.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return name of the project to use to hold the profiled data.
	 * @throws CoreException
	 */
	private String getProjectName(ILaunchConfiguration configuration) throws CoreException {
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
		return configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_PROJECT, store.getString(TraceConstants.TRACE_PROJECT_NAME));
	}
	
	/**
	 * Returns the monitor name.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return name of the monitor to use.
	 * @throws CoreException
	 */
	private String getMonitorName(ILaunchConfiguration configuration) throws CoreException {
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
		return configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_MONITOR, store.getString(TraceConstants.TRACE_MONITOR_NAME));
	}
	
	/**
	 * Returns the path to the workspace in which this workbench should use.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return path to the workspace to use.
	 * @throws CoreException
	 */
	// We should synchonize this with the call used in getProgramArguments (that comes from WorkbenchLaunchConfigurationDelegate.java).
	// *Ideally*, an API should be provided for us to retrieve this information, so we wouldn't even have to worry about sync'ing these
	// two.
	private String getWorkspace(ILaunchConfiguration configuration) throws CoreException {
		return configuration.getAttribute(LOCATION + "0", LauncherUtils.getDefaultPath().append("runtime-workbench-workspace").toOSString());
	}
	
	/**
	 * Returns the path to the profile file.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return path to the profile, or null if the user wishes to profile to display.
	 * @throws CoreException
	 */
	private String getProfileFile(ILaunchConfiguration configuration) throws CoreException {
		return configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_FILE, (String) null);
	}

	/**
	 * Returns an ArrayList of the environment variables.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return list of environment variables.
	 * @throws CoreException
	 */
	// This method uses the same call found in WorkbenchLaunchConfigurationDelegate.java to get this information
	// then transforms it into a format that we can use.
	private ArrayList getEnvironment(ILaunchConfiguration configuration) throws CoreException {
		String[] environment = DebugPlugin.getDefault().getLaunchManager().getEnvironment(configuration);
		
		ArrayList envList = new ArrayList();
		for(int i = 0; environment != null && i < environment.length; i ++) {
			envList.add(environment[i]);
		}
		return envList;
	}
	
	/**
	 * Returns a String of the classpath to use.  
	 * 
	 * @return the classpath.
	 * @throws CoreException
	 */
	// This method uses the same call found in WorkbenchLaunchConfigurationDelegate.java to get this information
	// then transforms it into a format that we can use.
	private String getClassPath(ILaunchConfiguration conf) throws CoreException {
		String[] classpath = LauncherUtils.constructClasspath(conf);
		if (classpath == null) {
			String message = PDEPlugin.getResourceString(KEY_NO_STARTUP);
			throw new CoreException(LauncherUtils.createErrorStatus(message));
		}

		if (classpath.length == 0) return null;

		String classpathList = "";
		for(int i = 0; i < classpath.length - 1; i ++) {
			classpathList = classpathList + classpath[i] + " ";
		}
		classpathList = classpathList + classpath[classpath.length - 1];
		
		return classpathList;
	}
	
	/**
	 * Returns the list of VM Arguments to use.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return list of VM arguments.
	 * @throws CoreException
	 */
	// This method uses the same call found in WorkbenchLaunchConfigurationDelegate.java to get this information
	// then transforms it into a format that we can use.
	private String getVMArgs(ILaunchConfiguration configuration) throws CoreException {
		String[] vmArgs = new ExecutionArguments(configuration.getAttribute(VMARGS,""),"").getVMArgumentsArray();
		if (vmArgs.length == 0) return null;
		
		String vmArgList = "";
		for(int i = 0; i < vmArgs.length - 1; i ++) {
			vmArgList = vmArgList + vmArgs[i] + " ";
		}
		vmArgList = vmArgList + vmArgs[vmArgs.length - 1];

		return vmArgList;
	}

	/**
	 * Returns an ArrayList of the environment variables.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return list of environment variables.
	 * @throws CoreException
	 */
	// This method calls the methods below (also found in WorkbenchLaunchConfigurationDelegate.java) to get 
	// this information then transforms it into a format that we can use.
	private String getProgramArgs(ILaunchConfiguration configuration) throws CoreException {
		String[] programArgs = getProgramArguments(configuration);
		if (programArgs.length == 0) return null;

		String programArgList = ""; 
		for(int i = 0; i < programArgs.length - 1; i ++) {
			programArgList = programArgList + programArgs[i] + " ";
		}
		programArgList = programArgList + programArgs[programArgs.length - 1];

		return programArgList;
	}
	
	/*
	 * Comment by sshiah:
	 * 
	 * All of the code below came from WorkbenchLaunchConfigurationDelegate.java, in org.eclipse.pde.internal.ui.launcher.
	 * This code processes all of the arguments provided by the user in the non-profiling tabs and builds the invocation 
	 * string that should be used. Since these helper methods are private within WorkbenchLaunchConfigurationDelegate.java, 
	 * a decision was made to re-use the existing code. 
	 * 
	 * A request has been made (bugzilla #53124) to allow access to these methods (or another solution in which we could
	 * access these processed arguments). When such access has been provided, the code below should be removed and replaced
	 * with the appropriate API call(s). Also, this class should not implement ILauncherSettings.
	 * 
	 * reference: bugzilla bugs #51979, #53124
	 */
	
	private String[] getProgramArguments(ILaunchConfiguration configuration) throws CoreException {
		ArrayList programArgs = new ArrayList();
		
		// specify the application to launch, only if it is not the default.
		String appName = configuration.getAttribute(APPLICATION, (String)null);
		if (appName != null && appName.length() > 0) {
			programArgs.add("-application");
			programArgs.add(appName);
		}
		
		// specify the workspace location for the runtime workbench
		String targetWorkspace =
			configuration.getAttribute(LOCATION + "0", LauncherUtils.getDefaultPath().append("runtime-workbench-workspace").toOSString());
		programArgs.add("-data");
		programArgs.add(targetWorkspace);
		
		boolean isOSGI = PDECore.getDefault().getModelManager().isOSGiRuntime();
		if (configuration.getAttribute(USEFEATURES, false)) {
			validateFeatures();
			IPath installPath = PDEPlugin.getWorkspace().getRoot().getLocation();
			File installDir = installPath.removeLastSegments(1).toFile();
			programArgs.add("-install");
			programArgs.add("file:" + installDir.getPath() + File.separator);
			programArgs.add("-update");
		} else {
			TreeMap pluginMap = LauncherUtils.getPluginsToRun(configuration);
			if (pluginMap == null) 
				return null;
				
			String primaryFeatureId = LauncherUtils.getPrimaryFeatureId();
			fConfigFile =
				TargetPlatform.createPlatformConfigurationArea(
					pluginMap,
					new Path(targetWorkspace),
					primaryFeatureId,
					LauncherUtils.getAutoStartPlugins(configuration));
			programArgs.add("-configuration");
			programArgs.add("file:" + fConfigFile.getPath());
			
			if (!isOSGI) {
				if (primaryFeatureId != null) {
					programArgs.add("-feature");
					programArgs.add(primaryFeatureId);					
				}
				IPluginModelBase bootModel = (IPluginModelBase)pluginMap.get("org.eclipse.core.boot");
				String bootPath = LauncherUtils.getBootPath(bootModel);
				if (bootPath != null && !bootPath.endsWith(".jar")) {
					programArgs.add("-boot");
					programArgs.add("file:" + bootPath);
				}
			}
		}
		
		// add the output folder names
		programArgs.add("-dev");
		String devEntry = LauncherUtils.getBuildOutputFolders();
		programArgs.add(configuration.getAttribute(CLASSPATH_ENTRIES, devEntry));

		// add tracing, if turned on
		if (configuration.getAttribute(TRACING, false)
				&& !TRACING_NONE.equals(configuration.getAttribute(
					TRACING_CHECKED, (String) null))) {
			if (fConfigFile == null) {
				fConfigFile =
					TargetPlatform.createWorkingDirectory(new Path(targetWorkspace));
			}
			String directoryName = fConfigFile.isDirectory() ? fConfigFile.toString() : fConfigFile.getParent();
			programArgs.add("-debug");
			programArgs.add(
				LauncherUtils.getTracingFileArgument(
					configuration,
					directoryName + Path.SEPARATOR + ".options"));
		}

		// add the program args specified by the user
		StringTokenizer tokenizer =
			new StringTokenizer(configuration.getAttribute(PROGARGS, ""));
		while (tokenizer.hasMoreTokens()) {
			programArgs.add(tokenizer.nextToken());
		}
		
		// show splash only if we are launching the default application
		boolean showSplash = true;
		int index = programArgs.indexOf("-application");
		if (index != -1 && index <= programArgs.size() - 2) {
			if (!programArgs.get(index + 1).equals(LauncherUtils.getDefaultApplicationName())) {
				showSplash = false;
			}
		}
		if (showSplash && !programArgs.contains("-nosplash")) {
			programArgs.add("-showsplash");
			programArgs.add(computeShowsplashArgument());
		}
		return (String[])programArgs.toArray(new String[programArgs.size()]);
	}
	
	private String[] getVMArguments(ILaunchConfiguration configuration) throws CoreException {
		return new ExecutionArguments(configuration.getAttribute(VMARGS,""),"").getVMArgumentsArray();
	}
			
	private void validateFeatures() throws CoreException {
		IPath installPath = PDEPlugin.getWorkspace().getRoot().getLocation();
		String lastSegment = installPath.lastSegment();
		boolean badStructure = lastSegment == null;
		if (!badStructure) {
			IPath featuresPath = installPath.removeLastSegments(1).append("features");
			badStructure = !lastSegment.equalsIgnoreCase("plugins")
					|| !featuresPath.toFile().exists();
		}
		if (badStructure) {
			throw new CoreException(LauncherUtils.createErrorStatus(PDEPlugin
					.getResourceString(KEY_BAD_FEATURE_SETUP)));
		} else {
			// Ensure important files are present
			ensureProductFilesExist(getProductPath());
		}
	}
	
	private IPath getInstallPath() {
		return PDEPlugin.getWorkspace().getRoot().getLocation();
	}
	
	private IPath getProductPath() {
		return getInstallPath().removeLastSegments(1);
	}

	private String computeShowsplashArgument() {
		IPath eclipseHome = ExternalModelManager.getEclipseHome();
		IPath fullPath = eclipseHome.append("eclipse");
		return fullPath.toOSString() + " -showsplash 600";
	}

	private void ensureProductFilesExist(IPath productArea) {
		File productDir = productArea.toFile();
		File marker = new File(productDir, ".eclipseproduct");
		File ini = new File(productDir, "install.ini");
		if (marker.exists() && ini.exists()) return;
		IPath eclipsePath = ExternalModelManager.getEclipseHome();
		if (!marker.exists()) 
			copyFile(eclipsePath, ".eclipseproduct", marker);
		if (!ini.exists())
			copyFile(eclipsePath, "install.ini", ini);
	}

	private void copyFile(IPath eclipsePath, String name, File target) {
		File source = new File(eclipsePath.toFile(), name);
		if (source.exists() == false)
			return;
		FileInputStream is = null;
		FileOutputStream os = null;
		try {
			is = new FileInputStream(source);
			os = new FileOutputStream(target);
			byte[] buf = new byte[1024];
			long currentLen = 0;
			int len = is.read(buf);
			while (len != -1) {
				currentLen += len;
				os.write(buf, 0, len);
				len = is.read(buf);
			}
		} catch (IOException e) {
		} finally {
			try {
				if (is != null)
					is.close();
				if (os != null)
					os.close();
			} catch (IOException e) {
			}
		}
	}
}
