/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.ui.internal.util;
import java.io.IOException;
import java.net.InetAddress;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.IStreamListener;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.debug.core.model.IStreamMonitor;
import org.eclipse.debug.core.model.IStreamsProxy;
import org.eclipse.hyades.internal.execution.local.common.Console;
import org.eclipse.hyades.internal.execution.local.common.DataProcessor;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.internal.execution.local.control.ProcessListener;
/**
 * An adapter from <code>Process</code> (Hyades) to <code>IProcess</code>
 * (Eclipse). Used to add a process to an <code>ILaunch</code> by the launch
 * delegate, in order to provide I/O to the console view, get the process's
 * status (running, terminated), and to terminate the process.
 */
public class ProcessAdapter implements IProcess {
	private static final String DATE_FORMAT = "dd/MM/yy h:mm a"; //$NON-NLS-1$
	private Map _attributes = new HashMap();
	private boolean _isAlive = false;
	private String _label = null;
	private Process _process;
	private IOProxy _proxy;
	private ILaunch _launch;
	/**
	 * Creates an adapter using the given <code>Process</code> and <code>
	 * ILaunch</code>.
	 */
	public ProcessAdapter(Process process, ILaunch launch) {
		DateFormat format = new SimpleDateFormat(DATE_FORMAT);
		_label = process.getExecutable() + " (" + format.format(new Date(System.currentTimeMillis())) + ")"; //$NON-NLS-1$
		_process = process;
		_launch = launch;
		_process.addProcessListener(new ProcessListener() {
			public void processLaunched(Process p) {
				_isAlive = true;
			}
			public void processExited(Process p) {
				_isAlive = false;
			}
		});
	}
	/**
	 * @see IAdaptable#getAdapter()
	 */
	public Object getAdapter(Class adapter) {
		if (DataProcessor.class.equals(adapter))
			return _proxy;
		return null;
	}
	/**
	 * @see IProcess#getAttribute()
	 */
	public String getAttribute(String key) {
		return (String) _attributes.get(key);
	}
	/**
	 * @see IProcess#setAttribute()
	 */
	public void setAttribute(String key, String value) {
		_attributes.put(key, value);
	}
	/**
	 * @see IProcess#getExitValue()
	 */
	public int getExitValue() throws DebugException {
		return 0;
	}
	/**
	 * @see IProcess#getLabel()
	 */
	public String getLabel() {
		return _label;
	}
	/**
	 * @see IProcess#getLaunch()
	 */
	public ILaunch getLaunch() {
		return _launch;
	}
	/**
	 * @see IProcess#getStreamsProxy()
	 */
	public IStreamsProxy getStreamsProxy() {
		if (_proxy == null)
			_proxy = new IOProxy(_process);
		return _proxy;
	}
	/**
	 * @see ITerminate#canTerminate()
	 */
	public boolean canTerminate() {
		return _isAlive;
	}
	/**
	 * @see ITerminate#isTerminated()
	 */
	public boolean isTerminated() {
		return !_isAlive;
	}
	/**
	 * @see ITerminate#terminate()
	 */
	public void terminate() throws DebugException {
		try {
			_process.kill(0);
		} catch (Exception e) {
			e.printStackTrace();
		}
		_isAlive = false;
	}
	public void setLaunch(ILaunch launch) {
		_launch = launch;
	}
}
/**
 * A proxy to the streams provided by the <code>Process</code>. Needed by
 * the console view.
 */
class IOProxy implements IStreamsProxy, IStreamMonitor, DataProcessor {
	private List _listeners = new ArrayList();
	private StringBuffer _buffer = new StringBuffer();
	private Console _console;
	/**
	 * Creates a new proxy.
	 */
	public IOProxy(Process process) {
		_console = process.getConsole();
		_console.setDataProcessor(this);
	}
	/**
	 * @see IStreamsProxy#getErrorStreamMonitor()
	 */
	public IStreamMonitor getErrorStreamMonitor() {
		return null;
	}
	/**
	 * @see IStreamsProxy#getOutputStreamMonitor()
	 */
	public IStreamMonitor getOutputStreamMonitor() {
		return this;
	}
	/**
	 * @see IStreamsProxy#write()
	 */
	public void write(String input) throws IOException {
		_console.write(input);
	}
	/**
	 * @see IStreamMonitor#addListener()
	 */
	public void addListener(IStreamListener listener) {
		synchronized (_listeners) {
			_listeners.add(listener);
		}
	}
	/**
	 * @see IStreamMonitor#removeListener()
	 */
	public void removeListener(IStreamListener listener) {
		synchronized (_listeners) {
			_listeners.remove(listener);
		}
	}
	/**
	 * @see IStreamMonitor#getContents()
	 */
	public String getContents() {
		return _buffer.toString();
	}
	/**
	 * @see IDataProcessor#incommingData()
	 */
	public void incommingData(byte[] buffer, int length, InetAddress peer) {
		String str = new String(buffer, 0, length);
		_buffer.append(str);
		synchronized (_listeners) {
			Iterator i = _listeners.iterator();
			while (i.hasNext())
				((IStreamListener) i.next()).streamAppended(str, this);
		}
	}
	/**
	 * @see IDataProcessor#incommingData()
	 */
	public void incommingData(char[] buffer, int length, InetAddress peer) {
		byte[] buf = new String(buffer).getBytes();
		incommingData(buf, buf.length, peer);
	}
	/**
	 * @see IDataProcessor#invalidDataType()
	 */
	public void invalidDataType(byte[] arg0, int arg1, InetAddress arg2) {
	}
	/**
	 * @see IDataProcessor#waitingForData()
	 */
	public void waitingForData() {
	}
}
