/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.launcher;

import java.util.*;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.*;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.trace.ui.internal.launcher.*;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.hyades.trace.ui.internal.util.*;
import org.eclipse.hyades.trace.ui.internal.util.FilterSetElement;

/**
 * 
 * @author znensi
 *
 * This class is used to store changes to the LCD, before the user hits apply.
 * Necessary for the case when user reverts, or cancel.
 * During a save action, this manager copy instance gets sent to the manager.
 * During a cancel action, the manager copy info is not persisted.
 */
public class ProfilingSetsManagerCopy extends ProfilingSetsManager {
	
		
	/**
	 * This class will be used to keep temporary information ragarding profiling sets
	 */
	public ProfilingSetsManagerCopy() {
		super();
		
		 initialize();
		 setDefaultSet(ProfilingSetsManager.instance.getDefaultSet());
		
	}
	
	public void initializeFrom(ILaunchConfiguration conf)
	{
		try {
			String defaultSet = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILINGSET, "");
			String sets = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILINGSETS, "");
			String filters = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILINGFILTERS, "");
			
			readSets(sets, defaultSet);	
			readFilters(filters);
		}
		catch(CoreException exc)
		{
			exc.printStackTrace();
		}
		
	}
	
	public static ProfilingSetsManager instance()
	{
		//this method is not supposed to be called on a copy instance
		return null;
	}
	
	
	/**
	 * Apply any canghes to the main profiling set manager
	 *
	 */
	public void applyChanges()
	{
//		System.out.println("apply config");
		
		//update profiling set manager
		ProfilingSetsManager manager = ProfilingSetsManager.instance();
		manager.getProfilingSets().clear();
		manager.getProfilingSets().putAll(getProfilingSets());
		
		manager.getProfilingTypes().clear();
		manager.getProfilingTypes().putAll(getProfilingTypes());
		
		manager.getProfilingTypesGroups().clear();
		manager.getProfilingTypesGroups().putAll(getProfilingTypesGroups());
		
		manager.getFilterSets().clear();
		manager.getFilterSets().putAll(getFilterSets());
		
		manager.setDefaultSet(getDefaultSet());
		manager.setDefaultFilterSet(getDefaultFilterSet());
		
		writeSetsToPreferences();				
	}
		
	protected void initializeProfilingTypesGroups()
	{
		profilingTypesGroups = new HashMap();
		
		Map groups = ProfilingSetsManager.instance().getProfilingTypesGroups();
		Iterator keys = groups.keySet().iterator();
		while(keys.hasNext())
		{
			String key = keys.next().toString();
			
			IProfilingSetTypeGroup group = (IProfilingSetTypeGroup) groups.get(key);
			profilingTypesGroups.put(key, new ProfilingSetTypeGroup(group.getId(), group.getName(), group.getIcon()));
		}		
	}
	
	protected void initializeProfilingTypes()
	{
		profilingTypes = new HashMap();
		
		Map types = ProfilingSetsManager.instance().getProfilingTypes();
		Iterator keys = types.keySet().iterator();
		while(keys.hasNext())
		{
			String key = keys.next().toString();
			
			IProfilingSetType type = (IProfilingSetType) types.get(key);
			profilingTypes.put(key, 
					    new ProfilingSetType(type.getId(), type.getName(), type.getDescription()
					    		, type.getImage(), type.isUseFilters(), type.getGroup()
								, type.getProfilingType()));
		}		
		
	}

	protected void initializeProfilingSets()
	{
		profilingSets = new HashMap();
		
		Map sets = ProfilingSetsManager.instance().getProfilingSets();
		Iterator keys = sets.keySet().iterator();
		while(keys.hasNext())
		{
			String key = keys.next().toString();
			
			IProfilingSet set = (IProfilingSet) sets.get(key);
			
			IProfilingSet copy = new ProfilingSet(set.getId(), set.getName(), set.getDescription());
            copy.getProfilingTypes().addAll(set.getProfilingTypes());
            copy.getAttributes().putAll(set.getAttributes());
			
			profilingSets.put(key, copy);
		}				
	}
	
	protected void initializeFilterSets()
	{
		filterSets = new HashMap();
		
		Map sets = ProfilingSetsManager.instance().getFilterSets();
		Iterator keys = sets.keySet().iterator();
		while(keys.hasNext())
		{
			String key = keys.next().toString();
			
			FilterSetElement set = (FilterSetElement) sets.get(key);
			
			FilterSetElement copy = new FilterSetElement(set.getName(), set.getEnabled());
			ArrayList children = set.getChildren();
			ArrayList ccopy = new ArrayList();
			for(int idx=0; idx<children.size(); idx++)
			{
				FilterTableElement child = (FilterTableElement)children.get(idx);
				ccopy.add(new FilterTableElement(child.getText(), child.getMethod(), child.getVisibility()));
				
			}
			copy.setChildren(ccopy);
			
			filterSets.put(key, copy);
		}		
		
	}
	
	
	public void performApply(ILaunchConfigurationWorkingCopy wc)
	{
		//remove all previous attributes from the launch configuration
		try {
			Iterator wcIterator = wc.getAttributes().keySet().iterator();
			while(wcIterator.hasNext())
			{
				String key = wcIterator.next().toString();
				if(key.startsWith("SETFILTER_") || key.startsWith("SETOPTION_"))
				{
					wc.setAttribute(key, (String)null);
				}
			}
		}
		catch(Exception exc)
		{
			exc.printStackTrace();
		}
		//set profiling options attributes
		IProfilingSet set = getDefaultSet();
		if(set == null)
		  return;
		
		Iterator attrs = set.getAttributes().values().iterator();
		while(attrs.hasNext())
		{
			ProfilingAttribute option = (ProfilingAttribute)attrs.next();
			wc.setAttribute(option.getName(), option.getValue());
		}
		
		wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILINGSET, set.getId());
		wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILINGSETS, getProfilingSetsString().toString());
		wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILINGFILTERS, getProfilingFiltersString().toString());
		
	}	
	
}
