/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.java.runner;

import java.io.OutputStream;
import java.io.PrintStream;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.StringTokenizer;

import junit.extensions.RepeatedTest;
import junit.framework.Test;

import org.eclipse.hyades.execution.core.IControlMessage;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.remote.AgentControllerUnavailableException;
import org.eclipse.hyades.internal.execution.remote.CustomCommandHandler;
import org.eclipse.hyades.internal.execution.remote.RemoteComponentSkeleton;
import org.eclipse.hyades.test.common.agent.ComptestAgent;
import org.eclipse.hyades.test.common.event.ExecutionEvent;
import org.eclipse.hyades.test.common.event.MessageEvent;
import org.eclipse.hyades.test.common.junit.DefaultTestArbiter;
import org.eclipse.hyades.test.common.junit.HyadesTestRunner;
import org.eclipse.hyades.test.common.junit.HyadesTestSuite;
import org.eclipse.hyades.test.common.junit.HyadesTestUtil;
import org.eclipse.hyades.test.common.util.BaseString;

public class HyadesJUnitRunner 
extends HyadesTestRunner
{
	private boolean isOKToStart = false;
	protected ComptestAgent comptestAgent;
	public static final String LOADTESTSUITEPREFIX = "_@USERGROUP_";	
	private AgentConsoleStream stdout = null;
	private AgentConsoleStream stderr = null;
	protected RemoteComponentSkeleton agent;
	public HashMap agentCollection = new HashMap();
	
	
	public class Commandhandler implements CustomCommandHandler 
	{
		public void handleCommand(CustomCommand command) 
		{
			String cmd = command.getData();
					
			if (cmd.equals(IControlMessage.START)) 
			{
				setOKToStart(true);
			
			} 
			else if(cmd.startsWith("AgentIDs:"))
			{
				String dpString = new String(command.getDataBinary());
				dpString = dpString.substring(dpString.indexOf(':')+1,dpString.length());
				StringTokenizer dpTokenizer = new StringTokenizer(dpString,",");
				while(dpTokenizer.hasMoreTokens())
				{
					String id = dpTokenizer.nextToken();
					RemoteComponentSkeleton agent = new RemoteComponentSkeleton("Executor$" + id, "tester");
					agent.addCommandListener
					(
						new CustomCommandHandler()
						{
							public void handleCommand(CustomCommand command) 
							{
								String cmd = command.getData();
							}
						}
					);
					try
					{
						agent.initialize();
						agentCollection.put(id,agent);
					}
					catch (AgentControllerUnavailableException e)
					{
						e.printStackTrace();
					}
				
				}
			}
			else 
			{
					handleCustomCommand(command);		
			
			}
		}
	

	}
	
	/**
	 * This class provides a OutputStrema that can be used to override stdout and stderr.
	 * @author rduggan
	 */
	
	private static class AgentConsoleStream 
	extends OutputStream
	{
		/* The types of streams we may have */
		public static final int OUT = 0;
		public static final int ERR = 1;

		/* Our mode (type) */
		private int mode;

		/* Agent to log the message to */
		private RemoteComponentSkeleton agent;
		private String hierarchyId;
		
		/* Buffer to append data to until we get a newline character */
		private StringBuffer buffer = new StringBuffer();

		public AgentConsoleStream(RemoteComponentSkeleton agent, int mode, String hierarchyId)
		{
			this.mode = mode;
			this.agent = agent;
			this.hierarchyId = hierarchyId;
		}

		/**
		 * Provide the write function.  We will be invoked by the PrintWriter here.
		 * RKD:  Does this need to be synchonized?  Will we be invoked by the PrintWriter in
		 * a thread-safe manner?  I will assume so but this may need to change.
		 * @see java.io.OutputStream#write(int)
		 */
		public void write(int b)
		{
			buffer.append((char)b);
			if(b == '\n')
			{
				flush();
			}
		}
		
		public void flush()
		{
			if ( buffer.length() > 0)
			{
				MessageEvent messageEvent = new MessageEvent();
				messageEvent.setText(buffer.toString());
				messageEvent.setSeverity(MessageEvent.INFO);
				//messageEvent.setOwnerId(hierarchyId);
				
				String parentID = HyadesTestRunner.peekParentEventID();
				messageEvent.setParentId(parentID);
	
				if (mode == OUT)
				{
					messageEvent.setText("System.out:\n" + buffer.toString());
				}
				else if (mode == ERR)
				{
					messageEvent.setText("System.err:\n" + buffer.toString());
				}
	
				/* Log the message */
				agent.logMessageUTF8(messageEvent.toString());
	
				/* Create the next line */
				buffer = new StringBuffer();
			}
		}
	}

	
	public static void main(String[] args)
	{
		
		HyadesJUnitRunner runner = new HyadesJUnitRunner(args);
		

		try
		{
			/* We need to locate the testsuite for our IDs */
			Class c = Class.forName(args[0], true, HyadesJUnitRunner.class.getClassLoader());
			Method m = c.getMethod("suite", null);
			Test test = (Test)m.invoke(null, null);

			/* Setup stdout and stder */
			HyadesTestSuite suite = (HyadesTestSuite)test;
			String hierarchyId = HyadesTestUtil.appendToHierarchyId(HyadesTestUtil.getHierarchyIds(suite), null, 1);
			runner.stdout = new AgentConsoleStream(runner.agent, AgentConsoleStream.OUT, hierarchyId);
			runner.stderr = new AgentConsoleStream(runner.agent, AgentConsoleStream.ERR, hierarchyId);
			System.setOut(new PrintStream(runner.stdout));
			System.setErr(new PrintStream(runner.stderr));

			/* Is this a LoadTest? */
			int instances = 1;
			String sInstances = System.getProperty("hyades.loadtest.nusers");
			if (sInstances != null) {
				try {
					instances = Integer.valueOf(sInstances).intValue();
				}
				catch (Exception e) {
					instances = 1;
				}
				
			}
			
			if (instances <= 1) {
				/* Run the testcase */
				runner.run(test);
			}
			else {
				/* This is a LoadTest. Use the ThreadRunner to run multiple
				 * instances simultaneously.
				 */
				runner.agent.sendMessageToAttachedClient("Running a " + instances + " instance test", 0);
				
				// Create a loadTestSuite
				HyadesTestSuite loadTest = createLoadTestSuite(suite, instances);
				
				// Run the loadtest Suite
				runner.run(loadTest);
			}

			runner.agent.sendMessageToAttachedClient("Testcase completed successfuly", 0);
		}
		catch(Throwable e)
		{
			/* We could not find the class (or something else, report the error */
			MessageEvent messageEvent = new MessageEvent();
			messageEvent.setText(BaseString.getStackTrace(e));
			messageEvent.setSeverity(MessageEvent.ERROR);
			messageEvent.setOwnerId(args[1]);
			messageEvent.setId("ROOT");
			runner.agent.logMessageUTF8(messageEvent.toString());
		}
	}
	
	
	protected void handleCustomCommand(CustomCommand command){}
	/**
	 * Flush any data that is cached in the sysout and syserr consoles
	 */
	protected void flushConsoles()
	{
		if ( stdout != null )
			stdout.flush();
		
		if ( stderr != null )
			stderr.flush();
	}

	/**
	 * Constructor for HyadesJUnitRunner
	 * @param agent
	 */
	public HyadesJUnitRunner(RemoteComponentSkeleton agent)
	{
		super();
		comptestAgent = new ComptestAgent(agent);
	}
	
	public HyadesJUnitRunner(String[] args)
	{
		super();
		
				
		/* Create our agent and wait for start command */
		agent = new RemoteComponentSkeleton("Executor$" + args[args.length-1], "tester");
		
		comptestAgent = new ComptestAgent(agent);
		
		// Add a command listener that can okay the starting of the test.
		agent.addCommandListener(new Commandhandler());

		/* Handshake with the Agent controller */
		try
		{
			agent.initialize();
		}
		catch (Throwable e)
		{
		}
			

		/* Wait for our start.  We will never wait more that 3 minutes */
		while (!isOKToStart())
		{
			synchronized (this)
			{
				try
				{
					wait(3 * 60000);
					if (!isOKToStart())
					{
						/* Try and tell the client we are timing out */
						agent.sendMessageToAttachedClient("Test was not started after 3 minutes.  Exiting testcase.", 0);

						/* We have timed out.  exit */
						System.exit(-1);

					}
				}
				catch (InterruptedException e)
				{

				}
			}
		}
	}

	/**
	 * Is it okay to continue starting the test?
	 * @return <code>true</code> if it is okay to start the test,
	 * <code>false</code> otherwise
	 */
	public synchronized boolean isOKToStart()
	{
		return isOKToStart;
	}

	// Set the "okay to start" flag and notify any threads that are waiting
	// on this object's monitor.
	protected synchronized void setOKToStart(boolean isOK)
	{
		isOKToStart = isOK;
		notifyAll();
	}

	/**
	 * @see org.eclipse.hyades.test.common.junit.HyadesTestRunner#runnerStarted()
	 */
	protected void runnerStarted()
	{
		comptestAgent.write("<EXECUTION>");
		super.runnerStarted();
	}

	/**
	 * @see org.eclipse.hyades.test.common.junit.HyadesTestRunner#runnerExit(boolean, long)
	 */
	protected void runnerExit(boolean runnerStopped, long elapsedTime)
	{
		super.runnerExit(runnerStopped, elapsedTime);
		comptestAgent.write("</EXECUTION>");
	}

	/**
	 * @see org.eclipse.hyades.test.common.junit.HyadesTestRunner#writeEvent(org.eclipse.hyades.test.common.event.ExecutionEvent)
	 */
	public void writeEvent(ExecutionEvent executionEvent)
	{
		flushConsoles();
		
		if((comptestAgent != null) && (executionEvent != null))
			comptestAgent.write(executionEvent.toString());
	}
	
	/**
	 * @see org.eclipse.hyades.test.common.junit.HyadesTestRunner#getLastEventText(boolean)
	 */
	protected String getLastEventText(boolean runnerStopped, long elapsedTime)
	{
		String[] args = new String[]
		{
			new Integer(getSucceedTests().size() + getFailureTests().size() + getErrorTests().size()).toString(), 
			new Integer(getFailureTests().size()).toString(), 
			new Integer(getErrorTests().size()).toString(),
			new Long(elapsedTime).toString()
		};

		if(runnerStopped)
			return JUnitResourceBundle.getInstance().getString("execution.TestStoped", args);
		return JUnitResourceBundle.getInstance().getString("execution.TestsFinished", args);
	}
	
	protected static HyadesTestSuite createLoadTestSuite(HyadesTestSuite suite, int instances) {
		
		String id = suite.getId();
//		HyadesTestSuite root = new HyadesTestSuite("root");
//		root.setArbiter(DefaultTestArbiter.INSTANCE).setId(id).setSynchronous(false);
//		
//		HyadesTestSuite usergroup = new HyadesTestSuite("usergroup");
//
//		root.addTest(new RepeatedTest(suite, instances));
//
//		return root;
		
		HyadesTestSuite root = new HyadesTestSuite("root");
		root.setArbiter(DefaultTestArbiter.INSTANCE).setId(id + "HTTP_ROOTID");

		HyadesTestSuite usergroup = new HyadesTestSuite(LOADTESTSUITEPREFIX + instances);
		root.addTest(new RepeatedTest(usergroup, instances));
		usergroup.setId(id + "_HTTP_USERGROUP").setSynchronous(false);
		usergroup.addTest(suite);
		
		return root;
	
	}
	/**
	 * @param key
	 * @return
	 */
	public RemoteComponentSkeleton getAgent(Object key)
	{
		if(key==null)
			return agent;
		return (RemoteComponentSkeleton)agentCollection.get(key);
	}

}