/*******************************************************************************
 * Copyright (c) 2003, 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.datapool.internal.control;

import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.edit.datapool.IDatapoolCell;
import org.eclipse.hyades.edit.datapool.IDatapoolEquivalenceClass;
import org.eclipse.hyades.edit.datapool.IDatapoolFactory;
import org.eclipse.hyades.edit.datapool.IDatapoolListener;
import org.eclipse.hyades.edit.datapool.IDatapoolRecord;
import org.eclipse.hyades.edit.datapool.IDatapoolSuggestedType;
import org.eclipse.hyades.edit.datapool.IDatapoolVariable;
import org.eclipse.hyades.test.ui.datapool.DatapoolPlugin;
import org.eclipse.hyades.test.ui.datapool.internal.action.DeleteColumnAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.DeleteRowAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.DeleteRowGroupAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.EditColumnAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.EditRowAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.EditRowGroupAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.InsertColumnAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.InsertRowAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.InsertRowGroupAction;
import org.eclipse.hyades.test.ui.datapool.internal.dialog.DataPoolColumnDialog;
import org.eclipse.hyades.test.ui.datapool.internal.dialog.DataPoolRowDialog;
import org.eclipse.hyades.test.ui.datapool.internal.dialog.DataPoolRowGroupDialog;
import org.eclipse.hyades.test.ui.datapool.internal.interfaces.IDatapoolPart;
import org.eclipse.hyades.test.ui.datapool.internal.util.TypeChecker;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.ColumnPixelData;
import org.eclipse.jface.viewers.ICellEditorListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ControlEditor;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.custom.TableCursor;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;

/**
 * A Datapool Table is a Excel like table editor that wraps the JFace 
 * TableViewer and SWT Table widget and edits the DPLDatapool object.  It 
 * provides the ability to insert, delete, and edit variables, equivalence 
 * classes, and records.  A varible is represented as a column in the 
 * table.  An equivalence class is a grouping of rows with the same name,
 * and a record is a single row in the table.  The table enforces that an 
 * equivalence class always has one or more records. 
 * 
 * @author psun
 */
public class DatapoolTable implements IMenuListener, SelectionListener, IDatapoolListener {

	private static final String START_LITERAL = "<Literal>"; //$NON-NLS-1$
	private static final String END_LITERAL = "</Literal>"; //$NON-NLS-1$
	private static final String TAG_DATA = "data"; //$NON-NLS-1$
	private static final String TAG_RECORD = "record"; //$NON-NLS-1$
	private static final String TAG_VARIABLE = "variable"; //$NON-NLS-1$
	private static final String TAG_VARIABLE_INDEX = "variableIndex"; //$NON-NLS-1$
	private static final String TAG_EQUIVALENCE_CLASS = "equivalenceClass"; //$NON-NLS-1$
	
	private static final int MAX_TABLE_WIDTH = 60000;
	private static final int MAX_TABLE_HEIGHT = 120000;
		 
	private IDatapool datapool = null;
	private Table table = null;
	private DatapoolTableUtil tableUtil = null;
	private TableViewer viewer = null;
	private Composite parent = null;
	private Menu contextMenu = null;
	private TableCursor cursor = null;
	private ControlEditor controlEditor = null;
	private IDatapoolPart datapoolPart = null;
	private IDatapoolFactory datapoolFactory = null;
	private Text textEditor = null;
	boolean ignoreECNotification = false;

	private FocusListener focusListener;
	private ICellEditorListener	cellEditorListener;

	/*
	 * Handles the selection of the header area.
	 * <p>Clicking on the header area bring up the dialog to change and/or 
	 * move the selected variable/column.
	 */
	private SelectionListener headerListener = new SelectionAdapter()
	{
		public void widgetSelected(SelectionEvent event)
		{
			if(datapoolPart.isReadOnly())
				return;
			if(textEditor != null && !textEditor.isDisposed())
				textEditor.dispose();
			TableColumn tableColumn = (TableColumn) event.widget;
			editColumnAux(tableColumn);
		}
	};

	/*
	 * Handles the resizing of scroll bars when the ScrollableComposite 
	 * containing the table is resized.  The datapool table uses the 
	 * composite's scroll bars. 
	 */
	private ControlListener resizeListener = new ControlAdapter()
	{
		public void controlMoved(ControlEvent e) 
		{
		}

		public void controlResized(ControlEvent e) 
		{
			resetScrollBarSize(true);
		}				
	};

	/*
	 * Handles the resizing of scroll bars when columns in the table are 
	 * resized.  
	 */
	private ControlListener resizeColumnListener = new ControlAdapter()
	{
		public void controlMoved(ControlEvent e) 
		{
		}

		public void controlResized(ControlEvent e) 
		{
			if(cursor != null && !cursor.isDisposed())
			{
				if(table.getSelectionIndex() >= 0)
				{
					cursor.setSelection(table.getSelectionIndex(), cursor.getColumn());
				}
			}
			resetScrollBarSize(false);
		}				
	};


	/*
	 * Handles the focus of the cell editor.
	 */
	class TableCellFocusAdaptor implements FocusListener
	{
		private Text text = null;
		private TableItem row = null;
		private int rowIndex = -1;
		private int column = -1;
		
		public TableCellFocusAdaptor(Text text, TableItem row, int column)
		{
			this.text = text;
			this.row = row;
			this.column = column;
			if(row != null)
				this.rowIndex = table.indexOf(row);
		}
		
		public void focusGained(FocusEvent e) 
		{
		}

		public void focusLost(FocusEvent e) 
		{
			row.setText(column, text.getText());
			applyEditingValue(text.getText());
			text = null;
		}
	}

	/*
	 * Handles the keystrokes of the cell editor when it is in edit mode.
	 */
	class TableCellKeyAdaptor implements KeyListener
	{
		private Text text = null;
		private TableItem row = null;
		private int rowIndex = -1;
		private int column = -1;
		
		public TableCellKeyAdaptor(Text text, TableItem row, int column)
		{
			this.text = text;
			this.row = row;
			this.column = column;
			if(row != null)
				this.rowIndex = table.indexOf(row);
		}
		
		/* 
		 * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
		 */		
		public void keyPressed(KeyEvent e) 
		{
		}

		/* 
		 * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
		 */
		public void keyReleased(KeyEvent e) 
		{
			if(datapoolPart.isReadOnly())
				return;
			// Close the text editor and copy the data over when the user hits
			// "ENTER" or any of the arrow keys.
			if (e.character == SWT.CR ||
				e.keyCode == SWT.ARROW_DOWN ||
				e.keyCode == SWT.ARROW_UP ||
				e.keyCode == SWT.ARROW_LEFT ||
				e.keyCode == SWT.ARROW_RIGHT) 
			{
				row.setText(column, text.getText());
				applyEditingValue(text.getText());
			}
			// Move the cursor if any of the arrow keys are hit.  This is to
			// simulate excel behavior.
			if(cursor != null && !cursor.isDisposed())
			{
				if(e.keyCode == SWT.ARROW_DOWN)
				{
					if(rowIndex + 1 < table.getItemCount())
					{
						cursor.setSelection(rowIndex + 1, column);	
						table.setSelection(rowIndex + 1);
						setScrollBars();
					}
				}
				if(e.keyCode == SWT.ARROW_UP)
				{
					if(rowIndex > 0)
					{
						cursor.setSelection(rowIndex - 1, column);	
						table.setSelection(rowIndex - 1);						
						setScrollBars();
					}
				}
				if(e.keyCode == SWT.ARROW_RIGHT)
				{
					if(column + 1 < table.getColumnCount())
					{
						cursor.setSelection(rowIndex, column + 1);
						setScrollBars();							
					}
				}
				if(e.keyCode == SWT.ARROW_LEFT)
				{
					if(column > 0)
					{
						cursor.setSelection(rowIndex, column - 1);
						setScrollBars();							
					}
				}
			}
			// Close the text editor when the user hits "ESC"
			if (e.character == SWT.ESC) {
				text.dispose();
			}			
		}		
		
	}

	/*
	 * Default constructor for the Datapool Table.  Registers various key, 
	 * menu, and selection listeners.  Creates context menu and populates 
	 * the table.
	 */
	public DatapoolTable(Composite parent, 
						 IDatapoolPart datapoolPart, 
						 IDatapool datapool, 
						 IDatapoolFactory datapoolFactory)
	{
		if(parent == null || datapool == null || datapoolPart == null || datapoolFactory == null)
			return;
		this.datapool = datapool;
		this.parent = parent;
		this.datapoolFactory = datapoolFactory;
		
		setWaitCursor();
		
		this.datapoolPart = datapoolPart;
		createTable(parent);
		tableUtil = new DatapoolTableUtil(table);

		MenuManager menuMgr= new MenuManager("#DatapoolEditor"); //$NON-NLS-1$
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(this);
		
		contextMenu = menuMgr.createContextMenu(table);
		table.setMenu(contextMenu);
		table.addSelectionListener(this);

		makeActions();

		datapool.addDatapoolListener(this);
		
		try
		{
			parent.getParent().getParent().addControlListener(resizeListener);
		}
		catch(Exception e)
		{
			// do nothing
		}
		
		unsetWaitCursor();
	}

	/* 
	 * Creates the table widget and viewer.  Populates the table from the datapool.
	 */
	private void createTable(Composite parent)
	{					
		table = new Table(parent, SWT.BORDER | SWT.FULL_SELECTION);
		viewer = new TableViewer(table);

		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		GridData gd = new GridData();
		gd.horizontalAlignment = GridData.FILL;
		gd.grabExcessHorizontalSpace = true;
		gd.verticalAlignment = GridData.FILL;
		gd.grabExcessVerticalSpace = true;
		gd.widthHint = MAX_TABLE_WIDTH;
		gd.heightHint = MAX_TABLE_HEIGHT;
		table.setLayoutData(gd);
		
		TableLayout layout = new TableLayout();
		layout.addColumnData(new ColumnPixelData(100, false));
		table.setLayout(layout);
				
		// Populate table		
		if(isWithoutColumns())
			makeHeaderColumn();	
		else
			makeColumns();
		
		if(!isWithoutRows())
			makeRows();

		createCursor();
			
		// Show the TableCursor when the user releases the "SHIFT" or "CTRL" key.
		// This signals the end of the multiple selection task.
		table.addKeyListener(new KeyAdapter() {
			
			public void keyReleased(KeyEvent e) {
				if (e.keyCode == SWT.CONTROL && (e.stateMask & SWT.SHIFT) != 0) return;
				if (e.keyCode == SWT.SHIFT && (e.stateMask & SWT.CONTROL) != 0) return;
				if (e.keyCode != SWT.CONTROL && (e.stateMask & SWT.CONTROL) != 0) return;
				if (e.keyCode != SWT.SHIFT && (e.stateMask & SWT.SHIFT) != 0) return;
			
				if(table.getItemCount() != 0)
				{
					TableItem[] selection = table.getSelection();
					TableItem row = (selection.length == 0) ? table.getItem(table.getTopIndex()) : selection[0];
					table.showItem(row);
					if(cursor != null && !cursor.isDisposed())
					{
						cursor.setSelection(row, 0);
						cursor.setVisible(true);
						cursor.setFocus();
					}
				}
			}
		});
	}

	/*
	 * Creates the table cursor and its assicated event listeners.
	 */
	private void createCursor() {
		if(datapool.getEquivalenceClassCount() == 0)
			return;
		cursor = new TableCursor(table, SWT.NONE);
		// Create an editor to edit the cell when the user hits "ENTER" 
		// while over a cell in the table
		controlEditor = new ControlEditor(cursor);
		controlEditor.grabHorizontal = true;
		controlEditor.grabVertical = true;
		
		cursor.setMenu(contextMenu);
		
		cursor.addSelectionListener(new SelectionAdapter() {
			// When the TableEditor is over a cell, select the corresponding row in 
			// the table.
			public void widgetSelected(SelectionEvent e) 
			{
				if(datapoolPart.isReadOnly())
					return;
				if(textEditor != null && !textEditor.isDisposed())
					textEditor.dispose();
				table.setSelection(new TableItem[] {cursor.getRow()});
				setScrollBars();
			}
			
			// Create an editor when the user hits "ENTER".
			public void widgetDefaultSelected(SelectionEvent e){
				if(datapoolPart.isReadOnly())
					return;
				textEditor = new Text(cursor, SWT.NONE);
				TableItem row = cursor.getRow();
				int column = cursor.getColumn();


				// No editing of equivalence class/record name;
				if(column == 0)
				{
					editRowAux(cursor.getRow());
					return;					
				}
				textEditor.setText(row.getText(column));				
				textEditor.addKeyListener(new TableCellKeyAdaptor(textEditor, row, column));
				textEditor.addFocusListener(new TableCellFocusAdaptor(textEditor, row, column));
				controlEditor.setEditor(textEditor);
				textEditor.setFocus();
			}
		});

		// Handles keyboard shortcuts for most operations.  Kicks the cell into
		// editing mode if the user hits a non-excluded key.
		cursor.addKeyListener(new KeyAdapter() {
			
			public void keyPressed(KeyEvent e)
			{
				return;
			}
			
			public void keyReleased(KeyEvent e) 
			{
				if(datapoolPart.isReadOnly())
					return;
				if (e.keyCode == SWT.INSERT && (e.stateMask & SWT.CTRL) != 0)
				{
					int column = cursor.getColumn();
					if(column == 0)
						insertRowGroup();
					if(column > 0)
						insertColumn();
					return;
				}
				if (e.keyCode == SWT.DEL && (e.stateMask & SWT.CTRL) != 0)
				{
					int column = cursor.getColumn();
					if(column == 0)
						deleteRowGroup();
					if(column > 0)
						deleteColumn();
					return;
				}
				if(e.keyCode == SWT.ESC || 
					e.keyCode == SWT.CR ||
					e.keyCode == SWT.ALT ||
					e.keyCode == SWT.PAGE_DOWN ||
					e.keyCode == SWT.PAGE_UP ||
					e.keyCode == SWT.HOME ||
					e.keyCode == SWT.END ||
					e.keyCode == SWT.CTRL ||
					e.keyCode == SWT.SHIFT ||
					((e.stateMask & SWT.CTRL) != 0) ||
					((e.stateMask & SWT.ALT) != 0) ||
					e.keyCode == SWT.F1 ||
					(e.keyCode >= SWT.F3 && e.keyCode <= SWT.F12))
				{
					return;
				}
				if(e.keyCode == SWT.INSERT)
				{
					int column = cursor.getColumn();
					if(column == 0)
						insertRow();
					if(column <= 0)
						return;
				}
				if(e.keyCode == SWT.DEL)
				{
					int column = cursor.getColumn();
					if(column == 0)
						deleteRow();
					if(column > 0)
						clearCell();
					return;					
				}
				if(e.keyCode == SWT.ARROW_DOWN ||
					e.keyCode == SWT.ARROW_UP ||
					e.keyCode == SWT.ARROW_LEFT ||
					e.keyCode == SWT.ARROW_RIGHT)
				{
					if(textEditor != null && !textEditor.isDisposed())
						textEditor.dispose();
					int selectedColumnIndex = cursor.getColumn();
					int selectedRowIndex = table.getSelectionIndex();
					// Code to prevent the cursor from moving too far to the
					// right.  This is to accomidate the fact that when a 
					// table column deleted, it is really hidden as a zero sized
					// column to the right
					if(e.keyCode == SWT.ARROW_RIGHT)
					{
						int visibleColumnCount = tableUtil.getColumnCount();
						if(selectedColumnIndex > visibleColumnCount - 1)
							cursor.setSelection(selectedRowIndex, selectedColumnIndex - 1);
					}
					return;	
				}
				else 
				{
					char inputCharacter = e.character;
					textEditor = new Text(cursor, SWT.NONE);
					TableItem row = cursor.getRow();
					int column = cursor.getColumn();

					// No editing of equivalence class/record name;
					if(column == 0)
					{
						editRowAux(cursor.getRow());
						return;					
					}
					if(e.keyCode == SWT.BS)
						textEditor.setText(new String());
					else if(e.keyCode == SWT.F2)
						textEditor.setText(row.getText(column));
					else
						textEditor.setText(String.valueOf(inputCharacter));
					textEditor.setSelection(textEditor.getText().length() + 1);						
					textEditor.addKeyListener(new TableCellKeyAdaptor(textEditor, row, column));
					textEditor.addFocusListener(new TableCellFocusAdaptor(textEditor, row, column));
					controlEditor.setEditor(textEditor);
					textEditor.setFocus();
				}
			}
		});
		
		cursor.addMouseListener(new MouseListener() {
			
			// Double clicking in a cell creates an editor and begins editing 
			// for that cell
			public void mouseDoubleClick(MouseEvent event) 
			{
				if(datapoolPart.isReadOnly())
					return;				
				textEditor = new Text(cursor, SWT.NONE);
				TableItem row = cursor.getRow();
				int column = cursor.getColumn();


				
				// No editing of equivalence class/record name;
				if(column == 0)
				{
					editRowAux(row);
					return;					
				}
				textEditor.setText(row.getText(column));
				textEditor.setSelection(textEditor.getText().length() + 1);						
				textEditor.addKeyListener(new TableCellKeyAdaptor(textEditor, row, column));
				textEditor.addFocusListener(new TableCellFocusAdaptor(textEditor, row, column));
				controlEditor.setEditor(textEditor);
				textEditor.setFocus();												
			}
		
			public void mouseDown(MouseEvent event) 
			{
				// Bring up the context menu if the user hits the right mouse
				// button on a cursor.
				if(event.button == 3)
				{
					contextMenu.setEnabled(true);
					contextMenu.setVisible(true);
				}
				if(event.button != 1)				
					return;
				if(textEditor != null && !textEditor.isDisposed())
					textEditor.dispose();
			}
			
			public void mouseUp(MouseEvent event) 
			{
			}
		});

		if(table.getItemCount() > 0)
		{
			table.select(0);
			cursor.setSelection(0,0);
		}
	}
	
	/*
	 * Creates and displays a wait cursor.
	 */
	private void setWaitCursor()
	{
		if(parent == null)
			return;
		Display display = parent.getShell().getDisplay();
		Cursor waitCursor = new Cursor(display, SWT.CURSOR_WAIT);
		setDisplayCursor(waitCursor);		
	}

	/*
	 * Cancels the display of the wait cursor.
	 */
	private void unsetWaitCursor()
	{
		setDisplayCursor(null);
	}

	/*
	 * Creates and sets the cursor to c.
	 */
	private void setDisplayCursor(Cursor c) {
		if(parent == null)
			return;
		Display display = parent.getShell().getDisplay();
		Shell[] shells = display.getShells();
		for (int i = 0; i < shells.length; i++)
			shells[i].setCursor(c);
	}

	/*
	 * Takes the value text from the editor and sets it to the table item as
	 * well as the datapool cell.
	 */
	private void applyEditingValue(String text)
	{
		if (cursor != null && cursor.getColumn() > 0)
		{
			TableItem selectedTableItem = cursor.getRow();
			int selectedColumnIndex = cursor.getColumn();
			if(selectedTableItem == null || selectedTableItem.isDisposed())
				return;
			IDatapoolCell[] rowData = (IDatapoolCell[])selectedTableItem.getData(TAG_DATA); 
			IDatapoolCell cell = rowData[cursor.getColumn() - 1];
			if(cell == null)
				return;
			else
			{
				// If the text has not changed, don't make any changes.
				if(text.equals(cell.getStringValue()))
				{
					if(textEditor != null && !textEditor.isDisposed())
						textEditor.dispose();
					return;	
				}
			}
			cell.setCellValue(text);
			selectedTableItem.setText(cursor.getColumn(), text);
			cursor.setSelection(table.getSelectionIndex(), cursor.getColumn());
			datapoolPart.markDirty();
			if(textEditor != null && !textEditor.isDisposed())
				textEditor.dispose();
		}					
	}

	/*
	 * Creates TableColumn objects from the DPLVariable objects in the 
	 * datapool.  The indexes are shifted by one because there is a 
	 * column for the row headers.
	 */
	private void makeColumns()
	{
		makeHeaderColumn();
		for(int i = 0; i < datapool.getVariableCount(); i++)
		{
			TableColumn tableColumn = new TableColumn(table, SWT.NONE, i + 1);
			IDatapoolVariable variable = (IDatapoolVariable)datapool.getVariable(i);
			tableColumn.setResizable(true);
			IDatapoolSuggestedType suggestedType = (IDatapoolSuggestedType)variable.getSuggestedType();
			String type = new String();
			if(suggestedType != null);
				type = suggestedType.getSuggestedClassName();			
			tableColumn.setText(variable.getName() + DatapoolPlugin.getResourceString("DATA_EDT_DIVIDER") + type); //$NON-NLS-1$
			tableColumn.setData(TAG_VARIABLE, variable); 
			tableColumn.setData(TAG_VARIABLE_INDEX, new Integer(i)); 
			tableColumn.setWidth(100);
			tableColumn.addSelectionListener(headerListener);	
			tableColumn.addControlListener(resizeColumnListener);		
		}		
    }

	/*
	 * Creates the column header.
	 */
	private void makeHeaderColumn() {
		TableColumn recordTableColumn = new TableColumn(table, SWT.NONE, 0);
		recordTableColumn.setResizable(true);
		recordTableColumn.setWidth(100);
		recordTableColumn.setData(TAG_VARIABLE, null);
		recordTableColumn.addControlListener(resizeColumnListener); 
	}

	/*
	 * Creates TableItem objects from the DPLEquivalenceClass and the IDatapoolRecord
	 * objects in the datapool.  Records do not need to have cells for every
	 * variable.
	 */
	private void makeRows()
	{
		for(int i = 0; i < datapool.getEquivalenceClassCount(); i++)
		{
			IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(i);
			for(int j = 0; j < equivalenceClass.getRecordCount(); j++)
			{
				IDatapoolRecord record = (IDatapoolRecord)equivalenceClass.getRecord(j);
				int cellCount = record.getCellCount();
								
				String rowContents[] = new String[datapool.getVariableCount() + 1];
				IDatapoolCell[] rowData = new IDatapoolCell[datapool.getVariableCount()];
				rowContents[0] = equivalenceClass.getName() + DatapoolPlugin.getResourceString("DATA_EDT_DIVIDER") + String.valueOf(j);  //$NON-NLS-1$

				for(int k = 0; k < cellCount; k++)
				{
					IDatapoolCell cell = (IDatapoolCell)record.getCell(k);
					IDatapoolVariable cellVariable = (IDatapoolVariable)cell.getCellVariable();
					int index = findColumnIndex(cellVariable.getName());
					String cellValue = new String();
					if(cell.getStringValue() != null)
						cellValue = cell.getStringValue();
					rowContents[index] = cellValue;
					rowData[index - 1] = cell;
				}				
							
				TableItem item = new TableItem(table, SWT.NULL);
				item.setText(rowContents);
				item.setData(TAG_DATA, rowData);
				item.setData(TAG_RECORD, record); 
				item.setData(TAG_EQUIVALENCE_CLASS, equivalenceClass); 
			}	
		}
	}

	/*
	 * Checks to see if the datapool is empty of equivalence classes causing
	 * the table to be without rows.
	 */
	private boolean isWithoutRows()
	{
		if(datapool.getEquivalenceClassCount() > 0)
		{
			if(datapool.getEquivalenceClassCount() == 1)
			{
				IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(0);
				if(equivalenceClass.getRecordCount() == 0)
					return true;
				else
					return false;
			}
			else
				return false;
			
		}
		else
		{
			return true;
		}
	}

	/*
	 * Checks to see if the datapool is empty of variables causing the table 
	 * to be without columns.
	 */
	private boolean isWithoutColumns()
	{
		if(datapool.getVariableCount() == 0)
			return true;
		else
			return false;
	}
	
	/*
	 * Finds the column index from the varible's ID.
	 */
	private int findColumnIndex(String name)
	{
		for(int i = 0; i < table.getColumnCount(); i++)
		{
			TableColumn tableColumn = table.getColumn(i);
			IDatapoolVariable variable = (IDatapoolVariable)tableColumn.getData(TAG_VARIABLE); 
			if(variable != null && name.equals(variable.getName()))
				return i;	
		}	
		return -1;
	}
	/*
	 * Sets the scroll bars location when the cursor or selection moves.
	 */
	private void setScrollBars() {
		Point cursorPoint = cursor.getLocation();		
		Composite greatGrandParent = parent.getParent().getParent();
		if(greatGrandParent != null && greatGrandParent instanceof ScrolledComposite) 
		{
			int scrollableComponentHeight = greatGrandParent.getBounds().height;
			int scrollableComponentWidth = greatGrandParent.getBounds().width;
			
			int xAdjustment = cursorPoint.x - scrollableComponentWidth + cursor.getBounds().width + 30;
			int yAdjustment = cursorPoint.y - scrollableComponentHeight + cursor.getBounds().height + 53;
			((ScrolledComposite)greatGrandParent).setOrigin(xAdjustment, yAdjustment);
		}
	}

	/*
	 * Resize the scrollbar by xChange and yChange.
	 */
	private void resizeScrollBars()
	{
		try
		{
			Composite greatGrandParent = parent.getParent().getParent();
						
			int currentHorizontalMaximum = greatGrandParent.getHorizontalBar().getMaximum();
			int currentVerticalMaximum = greatGrandParent.getVerticalBar().getMaximum();
	
			int scrollableComponentHeight = greatGrandParent.getBounds().height;
			int scrollableComponentWidth = greatGrandParent.getBounds().width;
	
			int tableWidth = table.getBounds().width;
			int tableHeight = table.getBounds().height;
			
			int horizontalAdjust = 0;
			int verticalAdjust = 68;
			for(int i = 0; i < table.getColumnCount(); i++)
				horizontalAdjust += table.getColumn(i).getWidth();
			for(int i = 0; i < table.getItemCount(); i++)
			{
				TableItem item = table.getItem(i);
				verticalAdjust += item.getBounds(0).height + 1;
			}
	
			if(horizontalAdjust < scrollableComponentWidth)
				greatGrandParent.getHorizontalBar().setVisible(false);
			else
			{
				greatGrandParent.getHorizontalBar().setVisible(true);
				
				if(currentHorizontalMaximum < 119)
					greatGrandParent.getHorizontalBar().setMaximum(horizontalAdjust - scrollableComponentWidth + 44);
				else
					greatGrandParent.getHorizontalBar().setMaximum(horizontalAdjust + 12);
			}

			if(verticalAdjust < scrollableComponentHeight)
				greatGrandParent.getVerticalBar().setVisible(false);
			else			
			{
				greatGrandParent.getVerticalBar().setVisible(true);

				if(currentVerticalMaximum < 119)
					greatGrandParent.getVerticalBar().setMaximum(verticalAdjust - scrollableComponentHeight + 58);
				else
					greatGrandParent.getVerticalBar().setMaximum(verticalAdjust - 12);
			}

		}
		catch(Exception e)
		{
		}
	}

	/*
	 * Resets the scroll bar upon changes in control size.  The boolean 
	 * indicates if the resize request is from the container of the table or 
	 * from an internal table element.
	 */
	private void resetScrollBarSize(boolean container)
	{
		try
		{
			Composite greatGrandParent = table.getParent().getParent().getParent();	
			int horizontalMax = greatGrandParent.getHorizontalBar().getMaximum();
			int verticalMax = greatGrandParent.getVerticalBar().getMaximum();
	
			int scrollableComponentHeight = greatGrandParent.getBounds().height;
			int scrollableComponentWidth = greatGrandParent.getBounds().width;
	
			int horizontalAdjust = 0;
			int verticalAdjust = 0;
			for(int i = 0; i < table.getColumnCount(); i++)
				horizontalAdjust += table.getColumn(i).getWidth();
			for(int i = 0; i < table.getItemCount(); i++)
			{
				TableItem item = table.getItem(i);
				verticalAdjust += item.getBounds(0).height + 1;
			}
	
			if(horizontalAdjust + 14 < scrollableComponentWidth)
				horizontalAdjust = 0;
			if(verticalAdjust < scrollableComponentHeight)
				verticalAdjust = 0;
	
			if(container)
			{
				greatGrandParent.getHorizontalBar().setMaximum(horizontalMax + horizontalAdjust - MAX_TABLE_WIDTH - 26);
				greatGrandParent.getVerticalBar().setMaximum(verticalMax + verticalAdjust - MAX_TABLE_HEIGHT - 4);
			}
			else
			{
				if(horizontalMax < 119)
					greatGrandParent.getHorizontalBar().setMaximum(horizontalAdjust - scrollableComponentWidth + 44);
				else
					greatGrandParent.getHorizontalBar().setMaximum(horizontalAdjust + 12);

				if(verticalMax < 119)
					greatGrandParent.getVerticalBar().setMaximum(verticalAdjust - scrollableComponentHeight + 58);
				else
					greatGrandParent.getVerticalBar().setMaximum(verticalAdjust - 12);

//				greatGrandParent.getHorizontalBar().setMaximum(horizontalAdjust + 14);
//				greatGrandParent.getVerticalBar().setMaximum(verticalAdjust);				
			}
		}
		catch(Exception e)
		{
		}
	}

	// SelectionListener
	
	/* 
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e) 
	{
		int selectedRowIndex = table.getSelectionIndex();
		if(cursor != null && !cursor.isDisposed())
		{
			int selectedColumnIndex = cursor.getColumn();
			if(selectedColumnIndex >= 0)
				cursor.setSelection(selectedRowIndex, selectedColumnIndex);
		}		
	}

	/* 
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e) 
	{
	}
	

	// Context Menu

	private InsertRowAction insertRowAction = null;	
	private DeleteRowAction deleteRowAction = null;
 	private EditRowAction editRowAction = null;
	private InsertColumnAction insertColumnAction = null;
	private DeleteColumnAction deleteColumnAction = null;
 	private EditColumnAction editColumnAction = null;
	private InsertRowGroupAction insertRowGroupAction = null;
	private DeleteRowGroupAction deleteRowGroupAction = null;
	private EditRowGroupAction editRowGroupAction = null;
	
	/*
	 * Creates all the actions
	 */
	private void makeActions()
	{
		ISelectionProvider provider = viewer;
		
		insertRowAction = new InsertRowAction(provider, this);
		deleteRowAction = new DeleteRowAction(provider, this);
		editRowAction = new EditRowAction(provider, this);
		insertColumnAction = new InsertColumnAction(provider, this);
		deleteColumnAction = new DeleteColumnAction(provider, this);
		editColumnAction = new EditColumnAction(provider, this);		
		insertRowGroupAction = new InsertRowGroupAction(provider, this);
		deleteRowGroupAction = new DeleteRowGroupAction(provider, this);
		editRowGroupAction = new EditRowGroupAction(provider, this);

	}
	
		
	// IMenuListener
		
	/**
 	* see menuAboutToShow(org.eclipse.jface.action.IMenuManager) in org.eclipse.jface.action.IMenuListener
	*/
	public void menuAboutToShow(IMenuManager imenuMgr)
	{
		MenuManager menuMgr = (MenuManager)imenuMgr;

		// Add actions to menu.
		menuMgr.add(insertRowGroupAction);				
		menuMgr.add(deleteRowGroupAction);				
		menuMgr.add(editRowGroupAction);				
		menuMgr.add(insertRowAction);				
		menuMgr.add(deleteRowAction);				
		menuMgr.add(editRowAction);				
		menuMgr.add(insertColumnAction);				
		menuMgr.add(deleteColumnAction);
		menuMgr.add(editColumnAction);

		// Set the mode.
		int menuMode = 0;
		if(cursor != null && !cursor.isDisposed())
			if(cursor.getColumn() == 0)
				menuMode = 0;
			else
				menuMode = 1;
		if(table.getSelectionIndex() == -1)
			menuMode = 2;
		if(datapool.getEquivalenceClassCount() == 0)
			menuMode = 3;		
		if(datapool.getEquivalenceClassCount() == 0 && datapool.getVariableCount() == 0)
			menuMode = 4;		
		if(datapoolPart.isReadOnly())
			menuMode = 5;

		// Based on the mode, enable certain menu items.
		switch(menuMode)
		{
			case 0: 
				insertRowAction.aboutToShow(true);
				deleteRowAction.aboutToShow(true);
				editRowAction.aboutToShow(true);
				insertRowGroupAction.aboutToShow(true);
				deleteRowGroupAction.aboutToShow(true);
				editRowGroupAction.aboutToShow(true);
				insertColumnAction.aboutToShow(true);
				deleteColumnAction.aboutToShow(false);
				editColumnAction.aboutToShow(false);
				break;
			case 1:
				insertRowAction.aboutToShow(true);
				deleteRowAction.aboutToShow(true);
				editRowAction.aboutToShow(true);
				insertRowGroupAction.aboutToShow(true);
				deleteRowGroupAction.aboutToShow(true);
				editRowGroupAction.aboutToShow(true);
				insertColumnAction.aboutToShow(true);
				deleteColumnAction.aboutToShow(true);
				editColumnAction.aboutToShow(true);
				break;
			case 2:
			case 3:
			case 4:
				insertRowAction.aboutToShow(false);
				deleteRowAction.aboutToShow(false);
				editRowAction.aboutToShow(false);
				insertRowGroupAction.aboutToShow(true);
				deleteRowGroupAction.aboutToShow(false);
				editRowGroupAction.aboutToShow(false);
				insertColumnAction.aboutToShow(true);
				deleteColumnAction.aboutToShow(false);
				editColumnAction.aboutToShow(false);
				break;
			case 5:
				insertRowAction.aboutToShow(false);
				deleteRowAction.aboutToShow(false);
				editRowAction.aboutToShow(false);
				insertRowGroupAction.aboutToShow(false);
				deleteRowGroupAction.aboutToShow(false);
				editRowGroupAction.aboutToShow(false);
				insertColumnAction.aboutToShow(false);
				deleteColumnAction.aboutToShow(false);
				editColumnAction.aboutToShow(false);
				break;
			default:
				insertRowAction.aboutToShow(true);
				deleteRowAction.aboutToShow(true);
				editRowAction.aboutToShow(true);
				insertRowGroupAction.aboutToShow(true);
				deleteRowGroupAction.aboutToShow(true);
				editRowGroupAction.aboutToShow(true);
				insertColumnAction.aboutToShow(true);
				deleteColumnAction.aboutToShow(true);
				editColumnAction.aboutToShow(true);
				break;
		}

	}
	
	// Action handlers
	
	/*
	 * Inserts a new equivalence class/row group to the table.  The newly
	 * created equivalence class will have one record.  Brings up a dialog to 
	 * gather information about the equivalence class/row group.  Adds the 
	 * equivalence class to the datapool object.  Inserts a row in the table.
	 */
	public void insertRowGroup()
	{
		TableItem previousTableItem = null;
		if(cursor != null && !cursor.isDisposed())
			previousTableItem = cursor.getRow();
		IDatapoolEquivalenceClass previousEquivalenceClass = null;
		if(previousTableItem != null)
			previousEquivalenceClass = (IDatapoolEquivalenceClass)previousTableItem.getData(TAG_EQUIVALENCE_CLASS); 

		DataPoolRowGroupDialog dialog = new DataPoolRowGroupDialog(Display.getCurrent().getActiveShell(), datapool, table, null, previousEquivalenceClass, DatapoolPlugin.getResourceString("DATA_ROW_GRP_DLG_TITLE_INS"));  //$NON-NLS-1$
		if (dialog.open() == IDialogConstants.OK_ID)
		{
			setWaitCursor();			
			String name = dialog.getName();
			if(name != null && name.length() != 0)
			{
				IDatapoolEquivalenceClass equivalenceClass = datapool.constructEquivalenceClass();
				int insertionIndex = dialog.getInsertionIndex();
				datapool.insertEquivalenceClass(equivalenceClass, insertionIndex + 1);
				equivalenceClass.setName(name);	
				IDatapoolRecord record = equivalenceClass.constructRecord(createCells(datapool));
				equivalenceClass.insertRecord(record, 0);
			}
			unsetWaitCursor();
		}
	}

	/*
	 * Deletes the currently selected equivalence class/row group.  Removes 
	 * the equivalence class and all assocaited records from the datapool 
	 * object.  Removes the rows from the table.
	 */
	public void deleteRowGroup()
	{
		TableItem tableItem = null;
		if(cursor != null && !cursor.isDisposed())
			tableItem = cursor.getRow();
		else
			return;
			
		setWaitCursor();
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)tableItem.getData(TAG_EQUIVALENCE_CLASS);
		int equivalenceClassIndex = datapool.getEquivalenceClassIndexById(equivalenceClass.getId());
		datapool.removeEquivalenceClass(equivalenceClassIndex);
		unsetWaitCursor();
	}

	/*
	 * Edits the currently selected equivalence class/row group.  Renames 
	 * and/or moves the equivalence class in the datapool object.  Updates the
	 * table to reflect the edit and/or move.
	 */
	public void editRowGroup()
	{
		TableItem tableItem = null;
		if(cursor != null && !cursor.isDisposed())
			tableItem = cursor.getRow();
		else
			return;
		editRowGroupAux(tableItem);
	}

	/*
	 * Function that actually does the editing of the equivalence class/row group.
	 */
	private void editRowGroupAux(TableItem tableItem)
	{
		IDatapoolEquivalenceClass selectedEquivalenceClass = null;
		if(tableItem != null)
			selectedEquivalenceClass = (IDatapoolEquivalenceClass)tableItem.getData(TAG_EQUIVALENCE_CLASS);
		int selectedEquivalenceClassIndex = -1;
		if(selectedEquivalenceClass != null)
			selectedEquivalenceClassIndex = datapool.getEquivalenceClassIndexById(selectedEquivalenceClass.getId());
		IDatapoolEquivalenceClass previousEquivalenceClass = null;
		if(selectedEquivalenceClassIndex != -1 && selectedEquivalenceClassIndex != 0)
			previousEquivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(selectedEquivalenceClassIndex - 1);

		DataPoolRowGroupDialog dialog = new DataPoolRowGroupDialog(Display.getCurrent().getActiveShell(), datapool, table,  selectedEquivalenceClass, previousEquivalenceClass, DatapoolPlugin.getResourceString("DATA_ROW_GRP_DLG_TITLE_EDIT")); //$NON-NLS-1$
		if (dialog.open() == IDialogConstants.OK_ID)
		{
			setWaitCursor();
			String name = dialog.getName();
			if(!selectedEquivalenceClass.getName().equals(name))
			{
				selectedEquivalenceClass.setName(name);	
			}
			int insertionIndex = dialog.getInsertionIndex();
			if(insertionIndex == selectedEquivalenceClassIndex - 1)
			{
				unsetWaitCursor();
				return;
			}			
			datapool.moveEquivalenceClass(selectedEquivalenceClassIndex, insertionIndex);
			unsetWaitCursor();				
		}
	}

	/*
	 * Inserts a new record/row to the table.  Brings up a dialog to gather 
	 * information about the where to place the row.  Adds the record to the 
	 * datapool object.  Inserts a row in the table.
	 */
	public void insertRow()
	{
		TableItem previousTableItem = null;
		if(cursor != null && !cursor.isDisposed())
			previousTableItem = cursor.getRow();
		else
			return;

		DataPoolRowDialog dialog = new DataPoolRowDialog(Display.getCurrent().getActiveShell(), datapool, table, null, previousTableItem, DatapoolPlugin.getResourceString("DATA_ROW_DLG_TITLE_INS"));  //$NON-NLS-1$
		if (dialog.open() == IDialogConstants.OK_ID)
		{
			setWaitCursor();
			int insertionEquivalenceClassIndex = dialog.getInsertionEquivalenceClassIndex();
			int insertionRecordIndex = dialog.getInsertionRecordIndex();
			if(insertionEquivalenceClassIndex != -1 && insertionRecordIndex != -1)
			{
				IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(insertionEquivalenceClassIndex);
				IDatapoolRecord record = equivalenceClass.constructRecord(createCells(datapool));
				equivalenceClass.insertRecord(record, insertionRecordIndex + 1);
			}
			else
			{
				IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(insertionEquivalenceClassIndex);
				IDatapoolRecord record = equivalenceClass.constructRecord(createCells(datapool));
				equivalenceClass.insertRecord(record, 0);
			}
			unsetWaitCursor();		
		}
	}

	/*
	 * Deletes the currently selected record/row.  Removes the record from the 
	 * datapool object.  Removes the row from the table.
	 */
	public void deleteRow()
	{
		TableItem tableItem = null;
		if(cursor != null && !cursor.isDisposed())
			tableItem = cursor.getRow();
		else
			return;

		setWaitCursor();
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)tableItem.getData(TAG_EQUIVALENCE_CLASS);
		IDatapoolRecord record = (IDatapoolRecord)tableItem.getData(TAG_RECORD);
		int equivalenceClassIndex = datapool.getEquivalenceClassIndexById(equivalenceClass.getId());
		int recordIndex = getRecordIndex(equivalenceClass, record);
		equivalenceClass.removeRecord(recordIndex);
		if(equivalenceClass.getRecordCount() == 0)
		{
			ignoreECNotification = true;
			datapool.removeEquivalenceClass(equivalenceClassIndex);			
		}
		unsetWaitCursor();
	}

	/*
	 * Edits the currently selected record/row.  Moves the record in the 
	 * datapool object.  Updates the table to reflect move.
	 */
	public void editRow()
	{
		TableItem tableItem = null;
		if(cursor != null && !cursor.isDisposed())
			tableItem = cursor.getRow();
		else
			return;
		editRowAux(tableItem);		
	}

	/*
	 * Function that actually does the editing of the record/row.
	 */
	private void editRowAux(TableItem tableItem)
	{
		int rowIndex = table.getSelectionIndex();
		TableItem previousTableItem = null;
		if(rowIndex != 0)
			previousTableItem = table.getItem(rowIndex - 1);

		DataPoolRowDialog dialog = new DataPoolRowDialog(Display.getCurrent().getActiveShell(), datapool, table,  tableItem, previousTableItem, DatapoolPlugin.getResourceString("DATA_ROW_DLG_TITLE_EDIT"));  //$NON-NLS-1$
		if (dialog.open() == IDialogConstants.OK_ID)
		{
			setWaitCursor();
			IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)tableItem.getData(TAG_EQUIVALENCE_CLASS);
			int equivalenceClassIndex = datapool.getEquivalenceClassIndexById(equivalenceClass.getId());
			IDatapoolRecord record = (IDatapoolRecord)tableItem.getData(TAG_RECORD);
			int recordIndex = getRecordIndex(equivalenceClass, record);

			int insertionEquivalenceClassIndex = dialog.getInsertionEquivalenceClassIndex();
			int insertionRecordIndex = dialog.getInsertionRecordIndex();
			if(equivalenceClassIndex == insertionEquivalenceClassIndex)
			{
				if(recordIndex != insertionRecordIndex + 1)
				{
					if(insertionRecordIndex > recordIndex)
						equivalenceClass.moveRecord(recordIndex, insertionRecordIndex);
					else
						equivalenceClass.moveRecord(recordIndex, insertionRecordIndex + 1);	
				}
			}
			else
			{
				if(insertionEquivalenceClassIndex != -1 && insertionRecordIndex != -1)
				{
					IDatapoolEquivalenceClass targetEquivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(insertionEquivalenceClassIndex);
					equivalenceClass.removeRecord(recordIndex);
					targetEquivalenceClass.insertRecord(record, insertionRecordIndex + 1);
					if(equivalenceClass.getRecordCount() == 0)
					{
						ignoreECNotification = true;
						datapool.removeEquivalenceClass(equivalenceClassIndex);
					}
				}
				else
				{
					IDatapoolEquivalenceClass targetEquivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(insertionEquivalenceClassIndex);
					equivalenceClass.removeRecord(recordIndex);
					targetEquivalenceClass.insertRecord(record, 0);
					if(equivalenceClass.getRecordCount() == 0)
					{
						ignoreECNotification = true;
						datapool.removeEquivalenceClass(equivalenceClassIndex);
					}
				}
			}
			unsetWaitCursor();		
		}
	}

	/*
	 * Inserts a new variable/column to the table.  Brings up a dialog to 
	 * gather information about the variable/column.  Adds the variable to the 
	 * datapool object.  Inserts a column in the table.
	 */
	public void insertColumn()
	{
		int selectedColumnIndex = -1;
		if(cursor != null && !cursor.isDisposed())
			selectedColumnIndex = cursor.getColumn();
		TableColumn previousTableColumn = null;
		if(selectedColumnIndex > 0)
			previousTableColumn = table.getColumn(selectedColumnIndex);
		IDatapoolVariable previousVariable = null;
		if(previousTableColumn != null)
			previousVariable = (IDatapoolVariable)previousTableColumn.getData(TAG_VARIABLE);

		DataPoolColumnDialog dialog = new DataPoolColumnDialog(Display.getCurrent().getActiveShell(), datapool, null, previousVariable, DatapoolPlugin.getResourceString("DATA_COL_DLG_TITLE_INS"));  //$NON-NLS-1$
		if ( dialog.open() == IDialogConstants.OK_ID)
		{
			setWaitCursor();
			IDatapoolVariable variable = datapool.constructVariable();
			variable.setName(dialog.getName());
			IDatapoolSuggestedType suggestedType = (IDatapoolSuggestedType)variable.getSuggestedType();
//			suggestedType.setSuggestedType(IDatapoolSuggestedType.TYPE_COMPLEX);			
//			suggestedType.setSuggestedClassName(dialog.getType());
			setVariableType(suggestedType, dialog.getType());
			variable.setSuggestedType(suggestedType);
			int insertionIndex = findColumnIndex(dialog.getInsertionVariableName());
			if(insertionIndex == -1)
				insertionIndex = 0;

			datapool.insertVariable(variable, insertionIndex);
			unsetWaitCursor();		
		}
	}

	/*
	 * Deletes the currently selected variable/column.  Removes the column and 
	 * all assocaited cells from the datapool object.  Removes the column from 
	 * the table.
	 */
	public void deleteColumn()
	{
		if(cursor == null || cursor.isDisposed())
			return;
		int selectedColumnIndex = cursor.getColumn();
		int selectedRowIndex = table.getSelectionIndex();
		// Don't delete the row name column.
		if (selectedColumnIndex < 1)	
			return;
		
		setWaitCursor();
		TableColumn tableColumn = table.getColumn(selectedColumnIndex);
		IDatapoolVariable variable = (IDatapoolVariable)tableColumn.getData(TAG_VARIABLE);
		int variableIndex = datapool.getVariableIndexById(variable.getId());
		datapool.removeVariable(variableIndex);
		unsetWaitCursor();
	}
	
	/*
	 * Edits the currently selected variable/column.  Renames and/or moves the 
	 * variable in the datapool object.  Updates the table to reflect the edit 
	 * and/or move.
	 */
	public void editColumn()
	{
		int selectedColumnIndex = cursor.getColumn();
		TableColumn tableColumn = (TableColumn) table.getColumn(selectedColumnIndex);
		editColumnAux(tableColumn);		
	}

	/*
	 * Function that actually does the editing of the variable/column.
	 */	
	private void editColumnAux(TableColumn tableColumn)
	{
		int columnIndex = table.indexOf(tableColumn);
		TableColumn previousTableColumn = null;
		if(columnIndex != 0)
			previousTableColumn = table.getColumn(columnIndex-1);
		IDatapoolVariable previousVariable = null;
		if(previousTableColumn != null)
			previousVariable = (IDatapoolVariable)previousTableColumn.getData(TAG_VARIABLE); 
		IDatapoolVariable variable = (IDatapoolVariable)tableColumn.getData(TAG_VARIABLE); 
		DataPoolColumnDialog dialog = new DataPoolColumnDialog(Display.getCurrent().getActiveShell(), datapool, variable, previousVariable, DatapoolPlugin.getResourceString("DATA_COL_DLG_TITLE_EDIT"));  //$NON-NLS-1$
		if ( dialog.open() == IDialogConstants.OK_ID)
		{
			setWaitCursor();
			String name = dialog.getName();
			String type = dialog.getType();
			String insertionVariableID = dialog.getInsertionVariableID();
			String previousVariableID = new String();
			if(previousVariable != null)
				previousVariableID = previousVariable.getId();

			IDatapoolSuggestedType suggestedType = (IDatapoolSuggestedType)variable.getSuggestedType();			
			if(name.equals(variable.getName()) && 
				type.equals(suggestedType.getSuggestedClassName()) &&
				insertionVariableID.equals(previousVariableID))
			{
				unsetWaitCursor();
				return;	
			}
			variable.setName(dialog.getName());
//			suggestedType.setSuggestedType(IDatapoolSuggestedType.TYPE_COMPLEX);						
//			suggestedType.setSuggestedClassName(dialog.getType());
			setVariableType(suggestedType, dialog.getType());
			variable.setSuggestedType(suggestedType);
			Integer variableIndex = (Integer)tableColumn.getData(TAG_VARIABLE_INDEX); 

			int insertionIndex = findColumnIndex(dialog.getInsertionVariableName());
			if(insertionIndex == columnIndex - 1)
			{
				unsetWaitCursor();
				return;
			}
			if(insertionIndex == -1)
				datapool.moveVariable(columnIndex - 1, 0);
			else
				if(insertionIndex > columnIndex)
					datapool.moveVariable(columnIndex - 1, insertionIndex - 1);
				else
					datapool.moveVariable(columnIndex - 1, insertionIndex);
			unsetWaitCursor();				
		}
	}
	
	/* 
	 * Clears the value in cell.
	 */
	public void clearCell()
	{
		applyEditingValue(new String());
	}
	
	// IDatapoolListener
	
	/*
	 * The following methods follow a spec that will eventually correspond to a
	 * listener interface that will make callbacks to the Datapool table.  
	 * Based on the callback, the table will update accordingly.  
	 */

	public void variableAdded(IDatapool datapool, int newVariableIndex) 
	{	
		TableColumn tableColumn = new TableColumn(table, SWT.NONE, newVariableIndex + 1);
		IDatapoolVariable variable = (IDatapoolVariable)datapool.getVariable(newVariableIndex);
		tableColumn.setResizable(true);
		IDatapoolSuggestedType suggestedType = (IDatapoolSuggestedType)variable.getSuggestedType();
		String type = new String();
		if(suggestedType != null);
			type = suggestedType.getSuggestedClassName();
		tableColumn.setText(variable.getName() + DatapoolPlugin.getResourceString("DATA_EDT_DIVIDER") + type);  //$NON-NLS-1$
		tableColumn.setData(TAG_VARIABLE, variable);
		tableColumn.setData(TAG_VARIABLE_INDEX, new Integer(newVariableIndex));
		tableColumn.addSelectionListener(headerListener);
		tableColumn.setWidth(100);
		tableColumn.addSelectionListener(headerListener);
		tableColumn.addControlListener(resizeColumnListener);		
		tableUtil.insertColumn(tableColumn, newVariableIndex);	

		if(cursor != null && !cursor.isDisposed())
		{
			int selectedRowIndex = table.getSelectionIndex();
			if(selectedRowIndex >= 0)
			{
				table.setSelection(selectedRowIndex);
				cursor.setSelection(selectedRowIndex, newVariableIndex + 1);
				cursor.setFocus();
				resizeScrollBars();
				setScrollBars();
			}

		}
		datapoolPart.markDirty();		
	}	
	
	public void variableRemoved(IDatapool datapool, int variableIndex)
	{
		tableUtil.deleteColumn(variableIndex + 1);

		if(cursor != null && !cursor.isDisposed())
		{
			if(variableIndex > 0)
			{
				int selectedRowIndex = table.getSelectionIndex();
				if(selectedRowIndex >= 0)
				{
					table.select(selectedRowIndex);
					cursor.setSelection(selectedRowIndex, variableIndex);
					resizeScrollBars();
					setScrollBars();
				}
			}
			else
				table.deselectAll();

		}
		datapoolPart.markDirty();		
	}
	
	public void variableMoved(IDatapool datapool, int sourceVariableIndex, int targetVariableIndex)
	{
		tableUtil.moveColumn(sourceVariableIndex + 1, targetVariableIndex + 1);
		if(cursor != null && !cursor.isDisposed())
			if(targetVariableIndex > 0)
			{
				int selectedIndex = table.getSelectionIndex();
				if(selectedIndex >= 0)
				{
					table.select(selectedIndex);
					cursor.setSelection(selectedIndex, targetVariableIndex);
					setScrollBars();
				}
			}
		datapoolPart.markDirty();		
	}
	
	public void variableChanged(IDatapool datapool, int variableIndex)
	{
		IDatapoolVariable variable = (IDatapoolVariable)datapool.getVariable(variableIndex);
		int columnIndex = findColumnIndex(variable.getName());
		TableColumn tableColumn = table.getColumn(columnIndex);
		IDatapoolSuggestedType suggestedType = (IDatapoolSuggestedType)variable.getSuggestedType();
		String type = new String();
		if(suggestedType != null);
			type = suggestedType.getSuggestedClassName();		
		tableColumn.setText(variable.getName() + DatapoolPlugin.getResourceString("DATA_EDT_DIVIDER") + type);  //$NON-NLS-1$
		tableColumn.setData(TAG_VARIABLE, variable);
		tableColumn.setData(TAG_VARIABLE_INDEX, new Integer(variableIndex));
		datapoolPart.markDirty();		
	}
	
	public void equivalenceClassChanged(IDatapool datapool, int equivalenceClassIndex)
	{
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(equivalenceClassIndex);
		
		for(int i = 0; i < equivalenceClass.getRecordCount(); i++)
		{
			// Assumes rows are synced with datapool
			int tableRowIndex = findRowIndex(equivalenceClassIndex, i);
			TableItem tableItem = table.getItem(tableRowIndex);
			tableItem.setText(0, equivalenceClass.getName() + DatapoolPlugin.getResourceString("DATA_EDT_DIVIDER") + String.valueOf(i)); //$NON-NLS-1$
			datapoolPart.markDirty();		
		}
	}
	
	public void equivalenceClassAdded(IDatapool datapool, int newEquivalenceClassIndex)
	{
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(newEquivalenceClassIndex);
		IDatapoolEquivalenceClass previousEquivalenceClass = null;
		int insertionIndex = 0;
		if(newEquivalenceClassIndex > 0)
		{
			previousEquivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(newEquivalenceClassIndex - 1);
			insertionIndex = findRowIndex(newEquivalenceClassIndex - 1, previousEquivalenceClass.getRecordCount() - 1) + 1;
		}

		for(int j = 0; j < equivalenceClass.getRecordCount(); j++)
		{
			IDatapoolRecord record = (IDatapoolRecord)equivalenceClass.getRecord(j);
			String rowContents[] = new String[table.getColumnCount()];
			IDatapoolCell[] rowData = new IDatapoolCell[table.getColumnCount() - 1];
			rowContents[0] = equivalenceClass.getName() + DatapoolPlugin.getResourceString("DATA_EDT_DIVIDER") + String.valueOf(j); //$NON-NLS-1$
			for(int k = 0; k < record.getCellCount(); k++)
			{
				IDatapoolCell cell = (IDatapoolCell)record.getCell(k);
				org.eclipse.hyades.execution.runtime.datapool.IDatapoolVariable cellVariable = cell.getCellVariable();
				int index = findColumnIndex(cellVariable.getName());
				rowContents[index] = cell.getStringValue();	
				rowData[index - 1] = cell;
			}
												
			TableItem item = new TableItem(table, SWT.NULL, insertionIndex + j);
			item.setText(rowContents);
			item.setData(TAG_DATA, rowData);
			item.setData(TAG_RECORD, record); 
			item.setData(TAG_EQUIVALENCE_CLASS, equivalenceClass);
		}
		datapoolPart.markDirty();	
	}

	public void equivalenceClassRemoved(IDatapool datapool, int equivalenceClassIndex)
	{
		if(ignoreECNotification)
		{
			ignoreECNotification = false;
			return;
		}
		// Assumes rows are synced with datapool
		int tableRowIndex = findRowIndex(equivalenceClassIndex, 0);
		TableItem tableItem = table.getItem(tableRowIndex);
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)tableItem.getData(TAG_EQUIVALENCE_CLASS);
		boolean removeRow = true;
		while(removeRow)
		{
			table.remove(tableRowIndex);
			if(tableRowIndex < table.getItemCount())
			{
				tableItem = table.getItem(tableRowIndex);
				IDatapoolEquivalenceClass nextEquivalenceClass = (IDatapoolEquivalenceClass)tableItem.getData(TAG_EQUIVALENCE_CLASS);
				if(!nextEquivalenceClass.equals(equivalenceClass))
					removeRow = false;
			}
			else
				removeRow = false;
		}
		datapoolPart.markDirty();		
		if(cursor != null && !cursor.isDisposed() && table != null)
		{
			int tableSize = table.getItemCount();
			if(tableSize == 0)
			{
				table.deselectAll();
				cursor.dispose();	
				return;								
			}
			if(tableSize > tableRowIndex)
			{
				table.setSelection(tableRowIndex);
				cursor.setSelection(tableRowIndex, 0);
				resizeScrollBars();				
				setScrollBars();
				return;
			}
			if(tableSize > tableRowIndex - 1)	
			{
				table.setSelection(tableRowIndex - 1);
				cursor.setSelection(tableRowIndex - 1, 0);
				resizeScrollBars();
				setScrollBars();
			}
		}		
	}
	
	public void equivalenceClassMoved(IDatapool datapool, int sourceEquivalenceClassIndex, int targetEquivalenceClassIndex)
	{
		if(sourceEquivalenceClassIndex == targetEquivalenceClassIndex)
			return;

		int targetIndex = 0;
		if(targetEquivalenceClassIndex != -1)
		{
			IDatapoolEquivalenceClass targetEquivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(targetEquivalenceClassIndex);
			int targetRecordCount = targetEquivalenceClass.getRecordCount();
			if(sourceEquivalenceClassIndex < targetEquivalenceClassIndex)
				targetRecordCount -= 1;
			targetIndex = findRowIndex(targetEquivalenceClassIndex, targetRecordCount);
		}
		
		IDatapoolEquivalenceClass sourceEquivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(sourceEquivalenceClassIndex);
		int sourceRecordCount = sourceEquivalenceClass.getRecordCount();
		int sourceIndex = findRowIndex(sourceEquivalenceClassIndex, 0);
		for(int i = 0; i < sourceEquivalenceClass.getRecordCount(); i++)
		{
			if(sourceEquivalenceClassIndex > targetEquivalenceClassIndex)
				tableUtil.moveRow(sourceIndex + sourceRecordCount - 1, targetIndex);
			else
				tableUtil.moveRow(sourceIndex, targetIndex);
		}
		if(cursor != null && !cursor.isDisposed() && table != null)
		{
			table.setSelection(targetIndex);
			cursor.setSelection(targetIndex, 0);
			setScrollBars();
		}		
		datapoolPart.markDirty();		
	}

	public void recordAdded(IDatapool datapool, int equivalenceClassIndex, int newRecordIndex)
	{
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(equivalenceClassIndex);
		IDatapoolRecord record = (IDatapoolRecord)equivalenceClass.getRecord(newRecordIndex);
		String rowContents[] = new String[table.getColumnCount()];
		IDatapoolCell[] rowData = new IDatapoolCell[table.getColumnCount() - 1];
		rowContents[0] = equivalenceClass.getName() + DatapoolPlugin.getResourceString("DATA_EDT_DIVIDER") + String.valueOf(newRecordIndex); //$NON-NLS-1$
		for(int i = 0; i < record.getCellCount(); i++)
		{
			IDatapoolCell cell = (IDatapoolCell)record.getCell(i);
			org.eclipse.hyades.execution.runtime.datapool.IDatapoolVariable cellVariable = cell.getCellVariable();
			int index = findColumnIndex(cellVariable.getName());
			rowContents[index] = cell.getStringValue();	
			rowData[index - 1] = cell;			
		}
		int insertionIndex = findRowIndex(equivalenceClassIndex, newRecordIndex);
		TableItem tableItem = new TableItem(table, SWT.NULL, insertionIndex);
		tableItem.setText(rowContents);
		tableItem.setData(TAG_DATA, rowData);
		tableItem.setData(TAG_RECORD, record);
		tableItem.setData(TAG_EQUIVALENCE_CLASS, equivalenceClass);
		repopulateRowLabels(equivalenceClassIndex);
		if(table.getItemCount() == 1)
			createCursor();		
		if(cursor != null && !cursor.isDisposed() && table != null)
		{
			table.setSelection(insertionIndex);
//			if(table.getColumnCount() > 1)
//				cursor.setSelection(insertionIndex, 1);
//			else
			cursor.setSelection(insertionIndex, 0);
			cursor.setFocus();
			resizeScrollBars();			
			setScrollBars();	
		}		
		datapoolPart.markDirty();		
	}
	
	public void recordRemoved(IDatapool datapool, int equivalenceClassIndex, int recordIndex)
	{
		int tableRowIndex = findRowIndex(equivalenceClassIndex, recordIndex);
		table.remove(tableRowIndex);
		repopulateRowLabels(equivalenceClassIndex);

		datapoolPart.markDirty();				
		if(cursor != null && !cursor.isDisposed() && table != null)
		{
			int tableSize = table.getItemCount();
			if(tableSize == 0)
			{
				table.deselectAll();
				cursor.dispose();	
				return;								
			}
			if(tableSize > tableRowIndex)
			{
				table.setSelection(tableRowIndex);
				cursor.setSelection(tableRowIndex, 0);
				resizeScrollBars();
				setScrollBars();
				return;
			}
			if(tableSize > tableRowIndex - 1)	
			{
				table.setSelection(tableRowIndex - 1);
				cursor.setSelection(tableRowIndex - 1, 0);
				resizeScrollBars();
				setScrollBars();
			}
		}		
	}
	
	public void recordMoved(IDatapool datapool, int equivalenceClassIndex, int sourceRecordIndex, int targetRecordIndex)
	{
		int sourceRowIndex = findRowIndex(equivalenceClassIndex, sourceRecordIndex);
		int targetRowIndex = findRowIndex(equivalenceClassIndex, targetRecordIndex);
		tableUtil.moveRow(sourceRowIndex, targetRowIndex);
		repopulateRowLabels(equivalenceClassIndex);
		datapoolPart.markDirty();		
	}
	
	public void cellChanged(IDatapool datapool, int equivalenceClassIndex, int recordIndex, int variableIndex)
	{
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(equivalenceClassIndex);
		IDatapoolRecord record = (IDatapoolRecord)equivalenceClass.getRecord(recordIndex);
		IDatapoolVariable variable = (IDatapoolVariable)datapool.getVariable(variableIndex);

		IDatapoolCell cell = null;		
		for(int i = 0; i < record.getCellCount(); i++)
		{
			cell = (IDatapoolCell)record.getCell(i);
			if(cell.getCellVariable().equals(variable))
				break;
			else
				cell = null;
		}

		if(cell != null)
		{
			String value = cell.getStringValue();
			int tableRowIndex = findRowIndex(equivalenceClassIndex, recordIndex);
			TableItem tableItem = table.getItem(tableRowIndex);
			tableItem.setText(equivalenceClassIndex + 1, value);
			cursor.setSelection(tableRowIndex, equivalenceClassIndex + 1);
			datapoolPart.markDirty();
		}
		datapoolPart.markDirty();		
	}

	public void equivalenceClassReordered(IDatapool datapool, int equivalenceClassIndex)
	{
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(equivalenceClassIndex);
		
		for(int i = 0; i < equivalenceClass.getRecordCount(); i++)
		{
			IDatapoolRecord record = (IDatapoolRecord)equivalenceClass.getRecord(i);
			int newTableRowIndex = findRowIndex(equivalenceClassIndex, i);
			int oldTableRowIndex = findRowIndex(equivalenceClassIndex, record);
			tableUtil.swapRow(oldTableRowIndex, newTableRowIndex);
		}
		datapoolPart.markDirty();		
	}
	
	public void recordChanged(IDatapool datapool, int EquivClassIndex, int recordIndex) 
	{
		return;
	}

	public void save(IDatapool datapool) 
	{
		datapoolFactory.save(datapool);
	}

	// End IDatapoolListener

	/* 
	 * Finds the index of the row in the table that corresponds to the record 
	 * given by the record object.  Returns -1 if the record is not found.
	 */
	private int findRowIndex(int equivalenceClassIndex, IDatapoolRecord record)
	{
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(equivalenceClassIndex);
		int startingRowIndex = findRowIndex(equivalenceClassIndex, 0);
		for(int i = 0; i < equivalenceClass.getRecordCount(); i++)
		{
			IDatapoolRecord currentRecord = (IDatapoolRecord)table.getItem(startingRowIndex + i).getData(TAG_RECORD);
			if(currentRecord.equals(record))
				return startingRowIndex + i;
		}
		return -1;
	}

	/*
	 * Finds the index of the row in the table that corresponds to the record
	 * at the given equivalence class and record index in the datapool object.
	 */
	private int findRowIndex(int equivalenceClassIndex, int recordIndex)
	{
		int count = recordIndex;
		for(int i = 0; i < equivalenceClassIndex; i++)
		{
			IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(i);
			count += equivalenceClass.getRecordCount();
		}
		return count;
	}

	/*
	 * Recreates the row labels for a row group when a memeber changes or
	 * moves.
	 */	
	private void repopulateRowLabels(int equivalenceClassIndex)
	{
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(equivalenceClassIndex);
		String name = equivalenceClass.getName();
		int tableRowIndex = findRowIndex(equivalenceClassIndex, 0);
		for(int i = 0; i < equivalenceClass.getRecordCount(); i++)
		{
			if(table.getItemCount() <= tableRowIndex + 1)
				continue;
			TableItem tableItem = table.getItem(tableRowIndex + i);
			tableItem.setText(0, name + DatapoolPlugin.getResourceString("DATA_EDT_DIVIDER") + String.valueOf(i)); //$NON-NLS-1$
		}
	}
	
	private int getRecordIndex(IDatapoolEquivalenceClass equivalenceClass, IDatapoolRecord record)
	{
		for(int i = 0; i < equivalenceClass.getRecordCount(); i++)
		{
			org.eclipse.hyades.execution.runtime.datapool.IDatapoolRecord compRecord = equivalenceClass.getRecord(i);
			if(compRecord.equals(record))
				return i;
		}
		return -1;
	}
	
	private Object[] createCells(IDatapool datapool)
	{
	 	Object[] cells = new Object[datapool.getVariableCount()];
	 	for(int i = 0; i < datapool.getVariableCount(); i++)
	 		cells[i] = new String();
	 	return cells;
	}
	
	public TableViewer getViewer()
	{
		return viewer;
	}
	
	private void setVariableType(IDatapoolSuggestedType suggestedType, String type)
	{
		if(suggestedType == null)
			return;
		if(TypeChecker.getInstance().isBoolean(type))
			suggestedType.setSuggestedType(IDatapoolSuggestedType.TYPE_BOOLEAN);
		else if(TypeChecker.getInstance().isNumber(type))
			suggestedType.setSuggestedType(IDatapoolSuggestedType.TYPE_NUMBER);
		else if(TypeChecker.getInstance().isString(type))
			suggestedType.setSuggestedType(IDatapoolSuggestedType.TYPE_STRING);
		else if(TypeChecker.getInstance().isEnumeration(type))
			suggestedType.setSuggestedType(IDatapoolSuggestedType.TYPE_ENUMERATION);
		else
			suggestedType.setSuggestedType(IDatapoolSuggestedType.TYPE_COMPLEX);
		
		suggestedType.setSuggestedClassName(type);	
	}
}
