/*******************************************************************************
 * Copyright (c) 2003, 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.datapool.internal.dialog;

import java.util.Vector;

import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.edit.datapool.IDatapoolVariable;
import org.eclipse.hyades.execution.runtime.datapool.IDatapoolSuggestedType;
import org.eclipse.hyades.test.ui.datapool.DatapoolPlugin;
import org.eclipse.hyades.test.ui.datapool.internal.util.GridDataUtil;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

/**
 * Dialog to collect information about a variable/column that is being
 * inserted or edited by the datapool table.  
 * 
 * @author psun
 *
 */
public class DataPoolColumnDialog extends Dialog implements KeyListener, SelectionListener{

	private static final String TAG_VARIABLES = "variables"; //$NON-NLS-1$

	
	private static final int HEIGHT = 100;
	private static final int WIDTH = 400;
	private Shell shell = null;
	private IDatapool datapool = null;
	private String title = null;
	private IDatapoolVariable selectedVariable = null;
	private IDatapoolVariable previousVariable = null;

	private Text nameField = null;
	private Text typeField = null;
	private Color nameFieldForeground = null;
	private Color typeFieldForeground = null;
	private Combo insertionVariables = null;
	private Label errorLabel = null;

	private String name = null;
	private String type = null;
	private String insertionVariableID = null;
	private String insertionVariableName = null;
	
	public DataPoolColumnDialog(Shell parentShell, 
									     IDatapool datapool,
									     IDatapoolVariable selectedVariable,
									     IDatapoolVariable previousVariable,
									     String title)
	{
		super(parentShell);
		this.shell = parentShell;
		this.datapool = datapool;
		this.title = title; 
		this.selectedVariable = selectedVariable;
		this.previousVariable = previousVariable;
	}

	/**
	 * @see org.eclipse.jface.window.Window#createContents(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createContents(Composite parent)
	{
		Control control = super.createContents(parent);
		enableOK();
		return control;
	}

	/**
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createDialogArea(Composite parent)
	{
		getShell().setText(title);
		Composite superComposite = (Composite)super.createDialogArea(parent);
		GridData gridData = GridDataUtil.createFill();
		gridData.heightHint = HEIGHT;
		gridData.widthHint = WIDTH;
		superComposite.setLayoutData(gridData);		

		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		superComposite.setLayout(gridLayout);

		Label nameLabel = new Label(superComposite, SWT.NONE);
		nameLabel.setText(DatapoolPlugin.getResourceString("DATA_COL_DLG_NAME")); //$NON-NLS-1$

		nameField = new Text(superComposite, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		gridData.horizontalSpan = 1;
		gridData.widthHint = WIDTH - 100;
		nameField.setLayoutData(gridData);
		nameField.addKeyListener(this);
		nameFieldForeground = nameField.getForeground();

		Label typeLabel = new Label(superComposite, SWT.NONE);
		typeLabel.setText(DatapoolPlugin.getResourceString("DATA_COL_DLG_TYPE")); //$NON-NLS-1$

		typeField = new Text(superComposite, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		gridData.horizontalSpan = 1;
		typeField.setLayoutData(gridData);
		typeField.addKeyListener(this);
		typeFieldForeground = typeField.getForeground();
	
		IDatapoolVariable[] variables = getVariables(null);
		if(selectedVariable != null)
		{
			name = selectedVariable.getName();
			nameField.setText(name);
			nameField.setSelection(name.length() + 1);
			IDatapoolSuggestedType suggestedType = selectedVariable.getSuggestedType();
			if(suggestedType != null)
			{
				type = suggestedType.getSuggestedClassName();
				if(type != null)
				{
					typeField.setText(type);
					typeField.setSelection(type.length() + 1);
				}
			}
		}
		String[] variableLabels = getVariableLabels(variables);

		Label insertionLabel = new Label(superComposite, SWT.NONE);
		if(title.equals(DatapoolPlugin.getResourceString("DATA_COL_DLG_TITLE_INS"))) //$NON-NLS-1$
			insertionLabel.setText(DatapoolPlugin.getResourceString("DATA_COL_DLG_INS")); //$NON-NLS-1$
		else
			insertionLabel.setText(DatapoolPlugin.getResourceString("DATA_COL_DLG_MOVE")); //$NON-NLS-1$

		insertionVariables = new Combo(superComposite, SWT.DROP_DOWN | SWT.READ_ONLY);
		insertionVariables.setItems(variableLabels);
		insertionVariables.setData(TAG_VARIABLES, variables);
	    insertionVariables.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL));
		insertionVariables.addSelectionListener(this);
		
		if(previousVariable != null)
		{
			int selectIndex = findVariableIndexInCombo(variables, previousVariable);
			insertionVariables.select(selectIndex);
			insertionVariableName = variables[selectIndex].getName();
			insertionVariableID = variables[selectIndex].getId();
		}
		else
		{
			insertionVariables.select(insertionVariables.getItemCount() - 1);
			insertionVariableName = new String();
			insertionVariableID = new String();
		}

		gridData = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		gridData.horizontalSpan = 2;
		errorLabel = new Label(superComposite, SWT.NONE);
		errorLabel.setLayoutData(gridData);

		return superComposite;
	}

	private boolean enableOK()
	{
		boolean enableButton = true;
		String typeErrorText = new String();
		String nameErrorText = new String();
		Display display= Display.getDefault();
		Color color= display.getSystemColor(SWT.COLOR_RED);

		String type = typeField.getText();
		if(type != null && type.length() >= 1)
		{
			boolean isTypeValid = isTypeValid(type);
			if(!isTypeValid)
			{
				typeField.setForeground(color);
				getButton(IDialogConstants.OK_ID).setEnabled(false);
				typeErrorText = DatapoolPlugin.getResourceString("DATA_COL_DLG_ERROR_TYPE_NOT_VALID");						
				enableButton = false;									
			}
			else
			{				
				typeField.setForeground(typeFieldForeground);
			}			
		}
		String name = nameField.getText();
		if(name != null && name.length() > 0)
		{
			boolean isNameUnique = isNameUnique(name);
			boolean isNameValid = isNameValid(name);
			if(isNameUnique && isNameValid)
			{		
				nameField.setForeground(nameFieldForeground);
			}
			else
			{
				nameField.setForeground(color);
				if(!isNameUnique)
				{
					nameErrorText = DatapoolPlugin.getResourceString("DATA_COL_DLG_ERROR_NAME_NOT_UNIQUE");
				}
				else if(!isNameValid)
				{
					nameErrorText = DatapoolPlugin.getResourceString("DATA_COL_DLG_ERROR_NAME_NOT_VALID");
				}
				if(nameErrorText.length() > 0)
				{
					nameErrorText += "  ";
				}
				enableButton = false;
			}
		}

		if(enableButton)
		{		
			getButton(IDialogConstants.OK_ID).setEnabled(true);
			errorLabel.setText(new String());
			return true;
		}
		else
		{
			String errorText = DatapoolPlugin.getResourceString("DATA_DLG_ERROR_INDICATOR") + nameErrorText + typeErrorText;
			errorLabel.setText(errorText);
			errorLabel.setForeground(color);
			getButton(IDialogConstants.OK_ID).setEnabled(false);		
			return false;
		}
	}
	
	private IDatapoolVariable[] getVariables(String exclusionName)
	{
		int variableCount = datapool.getVariableCount();
		Vector tempList = new Vector();
		for(int i = 0; i < variableCount; i++)
		{
			IDatapoolVariable variable = (IDatapoolVariable)datapool.getVariable(i);
			String variableName = variable.getName();
			if(variableName.equals(exclusionName))
				continue;
			else
				tempList.add(variable);
		}

		int listSize = tempList.size();
		IDatapoolVariable[] returnList = new IDatapoolVariable[listSize];
		tempList.toArray(returnList);
		return returnList;
	}
	
	private String[] getVariableLabels(IDatapoolVariable[] variables)
	{
		if(variables == null)
			return null;
		int listSize = variables.length;
		String[] returnList = new String[listSize + 1];
		for(int i = 0; i < listSize; i++)
		{
			IDatapoolVariable variable = variables[i];
			String type = new String();
			returnList[i] = variable.getName(); 
		}	
		returnList[listSize] = DatapoolPlugin.getResourceString("DATA_COL_DLG_HEAD_VAR"); //$NON-NLS-1$
		return returnList;
	}

	public String getName()
	{
		return name;
	}

	public String getType()
	{
		return type;
	}
	
	public String getInsertionVariableName()
	{
		return insertionVariableName;
	}

	public String getInsertionVariableID()
	{
		return insertionVariableID;
	}

	private int findVariableIndexInCombo(IDatapoolVariable[] variables, IDatapoolVariable variable)
	{
		if(variables == null || variable == null)
			return -1;
		for(int i = 0; i < variables.length; i++)
		{
			if(variables[i].equals(variable))
				return i;	
		}
		return -1;
	}

	private boolean isNameUnique(String text)
	{
		if(datapool == null)
			return true;
		for(int i = 0; i < datapool.getVariableCount(); i++)
		{
			IDatapoolVariable variable = (IDatapoolVariable)datapool.getVariable(i);
			if(!variable.equals(selectedVariable) && text.equals(variable.getName()))
				return false;
		}
		return true;			
	}

	// ISelectionListener

	/* 
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e) {
		Combo insertVariableCombo = (Combo)e.widget;
		int selectionIndex = insertVariableCombo.getSelectionIndex();
		IDatapoolVariable[] variables = (IDatapoolVariable[])insertVariableCombo.getData(TAG_VARIABLES);
		if(selectionIndex < variables.length)
		{
			IDatapoolVariable variable = variables[selectionIndex];
			insertionVariableName = variable.getName();
			insertionVariableID = variable.getId();
		}
		else
		{
			insertionVariableName = new String();
			insertionVariableID = new String();
		}
	}

	/* 
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e) 
	{
		return;	
	}

	// IKeyListener

	/* 
	 * @see org.eclipse.swt.events.KeyListener#keyPressed(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyPressed(KeyEvent e) 
	{		
		return;
	}

	/* 
	 * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyReleased(KeyEvent e) 
	{
		if(nameField != null)
			name = nameField.getText();
		if(typeField != null)
			type = typeField.getText();
		enableOK();
	}

	private boolean isNameValid(String name)
	{
		String invalidCharacters = DatapoolPlugin.getResourceString("DATA_DLG_INVALID_NAME_CHARACTERS");
		return !hasInvalidCharacters(name, invalidCharacters);
	}
	
	private boolean isTypeValid(String type)
	{
		String invalidCharacters = DatapoolPlugin.getResourceString("DATA_DLG_INVALID_TYPE_CHARACTERS");
		String invalidFirstCharacter = DatapoolPlugin.getResourceString("DATA_DLG_INVALID_TYPE_FIRST_CHARACTER");
		if(hasInvalidCharacters(type.substring(0,1), invalidFirstCharacter))
		{
			return false;
		}
		return !hasInvalidCharacters(type, invalidCharacters);
	}
	
	private boolean hasInvalidCharacters(String value, String invalidCharacters)
	{
		for(int i = 0; i < invalidCharacters.length(); i++)
		{
			String character = invalidCharacters.substring(i, i+1);
			if(value.lastIndexOf(character) != -1)
				return true;	
		}
		return false;		
	}
}
