/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.internal.execution.recorder.local.appadapters;

import java.util.HashMap;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;

/**
 * Factory which manages Recorder Application Adapters which are comprised of multiple objects which work together 
 * to communicate and function using the RAC
 * @author Ernest Jessee
 */
public class RecorderAppAdapterFactory
{
	/**
	 * ID of the extension point associated with this factory.  At first call, the factory will query the system metadata and 
	 * load all extension point objects with this ID.
	 */
	private String extensionPointID = "org.eclipse.hyades.execution.recorder.RecorderApplication"; //$NON-NLS-1$
	
	/**
	 * the default intstance of this factory	
	 */
	private static RecorderAppAdapterFactory instance = null;
	
	/**
	 * Hashmap for referencing adapters by id.
	 */
	private HashMap elementsByID = new HashMap();
	
	/**
	 * Hashmap for referencing adapter clients (IRecorderApplicationAdapterClient) by ID
	 */
	private HashMap clients = new HashMap();
		
		
	/**
	 * private constructor (singleton)
	 */
	private RecorderAppAdapterFactory(){}
	
	/**
	 * returns default instance of the RecorderAppAdapterFactory
	 * @return RecorderAppAdapterFactory
	 * 
	 */
	public static RecorderAppAdapterFactory getInstance()
	{
		if(instance == null)
		{
			instance = new RecorderAppAdapterFactory();
			instance.loadMetdata();
			
		}
		
		return instance;
	}
	
	
	/**
	 * called at initial instantiation to load the xml data from various plugin.xml files.
	 */
	private void loadMetdata()
	{
		IConfigurationElement[] adapterElements = Platform.getPluginRegistry().getConfigurationElementsFor(extensionPointID);
		for(int i=0; i<adapterElements.length; i++)
		{
			IConfigurationElement element = adapterElements[i];
			elementsByID.put(element.getAttribute("id"),element); //$NON-NLS-1$
		}
		
	}
	
	/**
	 * returns the name of the registered adapter with the specified ID
	 * @param String adapterID
	 * @return String
	 */
	public String getAdapterName(String adapterID)
	{
		return ((IConfigurationElement)elementsByID.get(adapterID)).getAttribute("name"); //$NON-NLS-1$
	}
	
	/**
	 * returns the full path of the agent object (IRecorderApplicationAdapter) for the Recorder Application Adapter with the specified ID
	 */
	public String getAgentClasspath(String adapterID)
	{
		if (adapterID.equalsIgnoreCase("-1"))
			return adapterID;
		else
			return ((IConfigurationElement)elementsByID.get(adapterID)).getAttribute("agentclass"); //$NON-NLS-1$
	}
	
	/**
	 * returns the client object for the adapter with the specified ID
	 * @param String adapterID
	 * @return IRecorderApplicationAdapter
	 */
	public IRecorderApplicationAdapterClient getClient(String adapterID) throws InstantiationException, IllegalAccessException, ClassNotFoundException
	{
		Object obj = clients.get(adapterID);
		if(obj!=null)
			return (IRecorderApplicationAdapterClient)obj;
		else
		{		
			if (adapterID.equalsIgnoreCase("-1"))
				return null;
			String clientClasspath = ((IConfigurationElement)elementsByID.get(adapterID)).getAttribute("clientclass"); //$NON-NLS-1$
			IRecorderApplicationAdapterClient client = (IRecorderApplicationAdapterClient)Class.forName(clientClasspath).newInstance();
			clients.put(adapterID,client);
			return client;
		}
	}
	
	
	/**
	 * returns a String Array of id's corresponding to all registered application adapters
	 * @return String[]
	 */
	public String[] getAdapterIDs()
	{
		Object[] array = elementsByID.keySet().toArray();
		String retArray[] = new String[array.length];
		for(int i=0; i<array.length; i++)
		{
			retArray[i]=(String)array[i];
		}
		return retArray;
	}
	
	
	/**
	 * returns an application id by index
	 * @param int index
	 * @return String 
	 */
	public String getAdapterIDByIndex(int index)
	{
		Object[] values = elementsByID.values().toArray();
		
		if(index>values.length-1)
			return null;
		
		IConfigurationElement element = (IConfigurationElement)values[index];
			
		return element.getAttribute("id"); //$NON-NLS-1$
	}
	
	/**
	 * returns the IConfigurationElement for an appliation adapter with the specified ID
	 * @param String ID
	 * @return IConfigurationElement
	 */
	public IConfigurationElement getAdapterConfigElement(String ID)
	{
		return (IConfigurationElement)elementsByID.get(ID);
	}
	
	
	

}
