package org.eclipse.hyades.tests.logging.java;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.hyades.logging.events.ICommonBaseEvent;
import org.eclipse.hyades.logging.events.IComponentIdentification;
import org.eclipse.hyades.logging.events.ISimpleEventFactory;
import org.eclipse.hyades.logging.events.SimpleEventFactoryImpl;
import org.eclipse.hyades.logging.java.CommonBaseEventFilter;
import org.eclipse.hyades.logging.java.CommonBaseEventLogRecord;
import org.eclipse.hyades.logging.java.LoggingAgentHandler;
import org.eclipse.hyades.logging.java.XmlFormatter;

/** 
 * CLASS: JavaLoggingTest.java
 * 
 * 
 * DESCRIPTION:  Sample class to demonstrate the usage of the Hyades support for logging Common Base Event log records in JSR-047 logging APIs implemented in JDK 1.4.0 or above.
 * 
 * 
 * ASSUMPTIONS:  1) This class must be executed using JDK 1.4.0 or above .
 *               2) The required JAR files must be specified in the JVM's classpath system variable.
 *               3) JDK 1.4.0 or above must be specified in the PATH environment variable.
 *               4) When launching this application from the Profiling Perspective of the Workbench, logging is enabled in the Profiling and Logging preferences (Window--> Preferences --> Profiling and Logging --> Enable logging) 
 *                  and the 'org.eclipse.hyades.tests.logging.java.JavaLoggingTest' logging agent is added to the Profiling and Logging preferences (Window --> Preferences --> Profiling and Logging --> Logging Agents --> New agent --> Add agent).
 *               5) When either launching this application from the Profiling Perspective of the Workbench or attaching to a logging agent using the Workbench, the Agent Controller (RAC) is installed and running.
 *               6) When launching this application from the Profiling Perspective of the Workbench, the Agent Controller (RAC) is configured during installation to use JDK 1.4.0 or above.
 *
 *                 
 * DEPENDENCIES: The following JAR files are required to compile and execute JavaLoggingTest.java:
 * 
 * 		         hexr.jar                       -		org.eclipse.hyades.execution.remote
 * 		         hl14.jar                       -		org.eclipse.hyades.logging.java14
 * 		         hlcore.jar                     -		org.eclipse.hyades.logging.core
 * 		         hlevents.jar                   -		org.eclipse.hyades.logging.core
 *
 * 
 * COMPILE COMMAND:   javac -classpath hexr.jar;hlevents.jar;hlcore.jar;hl14.jar;. JavaLoggingTest.java
 * 
 * EXECUTE COMMAND:   java  -classpath hexr.jar;hlevents.jar;hlcore.jar;hl14.jar;. JavaLoggingTest
 *  
 * 
 * @author      Paul Slauenwhite
 * @version	May 29, 2003
 * @since       JDK 1.4.0
 * @see		java.util.logging.Handler
 * @see		java.util.logging.Level
 * @see		java.util.logging.Logger
 * @see		org.eclipse.hyades.logging.events.ICommonBaseEvent
 * @see		org.eclipse.hyades.logging.events.SimpleEventFactoryImpl
 * @see		org.eclipse.hyades.logging.java.CommonBaseEventLogRecord
 * @see		org.eclipse.hyades.logging.java.LoggingAgentHandler
 */

public class JavaLoggingTest {

    //Amount of time (seconds) to wait to allow time to attach to the logging agent:
    private static final int WAITTIME = 30;

    //The name of the Java logger:
    private static final String LOGGERS_NAME = "org.eclipse.hyades.tests.logging.java.JavaLoggingTest";

    public static void main(String[] args) {

        try {

            //Create a logger named 'org.eclipse.hyades.tests.logging.java.JavaLoggingTest':
            Logger logger = Logger.getLogger(LOGGERS_NAME);

            //Set the logger to log all messages:
            logger.setLevel(Level.ALL);

            //Create a new instance of a Logging Agent Handler:
            Handler loggingAgentHandler = new LoggingAgentHandler();

            //Set the Logging Agent Handler's log record formatter:
            //NOTE: Alternatively, set the Logging Agent Handler's log record formatter in the <JRE 1.4.0 or above>/lib/logging.properties:
            //      org.eclipse.hyades.logging.java.LoggingAgentHandler.formatter = org.eclipse.hyades.logging.java.XmlFormatter
            loggingAgentHandler.setFormatter(new XmlFormatter());

            //Set the Logging Agent Handler's log record filter:
            //NOTE: Alternatively, set the Logging Agent Handler's log record filter in the <JRE 1.4.0 or above>/lib/logging.properties:
            //      org.eclipse.hyades.logging.java.LoggingAgentHandler.filter = org.eclipse.hyades.logging.java.CommonBaseEventFilter
            loggingAgentHandler.setFilter(new CommonBaseEventFilter());

            //Set the Logging Agent Handler's logging level:
            //NOTE: Alternatively, set the Logging Agent Handler's logging level in the <JRE 1.4.0 or above>/lib/logging.properties:
            //      org.eclipse.hyades.logging.java.LoggingAgentHandler.level = SEVERE
            loggingAgentHandler.setLevel(Level.SEVERE);

            //Add the LoggingAgentHandler to the logger:
            //NOTE: Alternatively, set the Logging Agent Handler to the logger in the <JRE 1.4.0 or above>/lib/logging.properties:
            //      handlers= <current handlers>, org.eclipse.hyades.logging.java.LoggingAgentHandler
            logger.addHandler(loggingAgentHandler);

            //Retrieve the instance of the Simple Event Factory:
            ISimpleEventFactory simpleEventFactory = SimpleEventFactoryImpl.getInstance();

            //Create a new instance of a Source Component:
            IComponentIdentification sourceComponentId = simpleEventFactory.createComponentIdentification();
            sourceComponentId.setComponent("Test Source Component");
            sourceComponentId.setSubComponent("Test Source Component Subcomponent");
            sourceComponentId.setComponentIdType("Test Source Component ID Type");
            sourceComponentId.setLocation("Test Source Component Location");
            sourceComponentId.setLocationType("Test Source Component Location Type");

            //Create a new instance of a Common Base Event:
            ICommonBaseEvent commonBaseEvent = simpleEventFactory.createCommonBaseEvent();
            commonBaseEvent.setSourceComponentId(sourceComponentId);
            commonBaseEvent.setCreationTime(System.currentTimeMillis());
            commonBaseEvent.setSeverity(((short) (10)));
            commonBaseEvent.setMsg("Java Common Base Event log message");

            //Create a new instance of a Common Base Event log record:
            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(Level.SEVERE);

            //Set the Common Base Event log record's Common Base Event:
            commonBaseEventLogRecord.setCommonBaseEvent(commonBaseEvent);

            //Set the Common Base Event log record's logger to the 'org.eclipse.hyades.tests.logging.java.JavaLoggingTest' logger:
            commonBaseEventLogRecord.setLoggerName(LOGGERS_NAME);

            //Log the Common Base Event log record:
            logger.log(commonBaseEventLogRecord);

            //Log the an invalid log message (i.e. filtered by the LoggingAgentHandler):
            logger.log(Level.SEVERE, "Java invalid LoggingAgentHandler log message.");

            //Wait for 2 seconds until all log messages are written to the console handler (i.e. standard error):
            try {
                Thread.sleep(2000);
            }
            catch (InterruptedException i) {
            }

            System.out.println("Please attach to the 'org.eclipse.hyades.tests.logging.java.JavaLoggingTest' logging agent in the next " + WAITTIME + " second" + (WAITTIME == 1 ? "" : "s") + ":");
            System.out.println("NOTE: When attaching to a logging agent using the Workbench, please remember to enable logging in the Profiling and Logging preferences (Window --> Preferences --> Profiling and Logging --> Enable logging) and the Agent Controller (RAC) is running.");

            //Create the timer scale:
            System.out.print("I");

            for (int counter = 0; counter < WAITTIME; counter++)
                System.out.print("-");

            System.out.print("I\nI");

            //Increment the timer:
            try {

                for (int counter = 0; counter < WAITTIME; counter++) {

                    System.out.print("-");

                    Thread.sleep(1000);
                }
            }
            catch (InterruptedException i) {
            }

            System.out.println("I");

            System.out.println("JavaLoggingTest successfully ended!");
        }
        catch (Throwable t) {
            System.out.println("ERROR - JavaLoggingTest unsuccessfully ended!");
            System.out.println("REASON: " + t.getMessage());
        }
    }
}
