/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.extractors;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.hyades.logging.adapter.MessageString;
import org.eclipse.hyades.logging.adapter.impl.Extractor;
import org.eclipse.hyades.logging.adapter.AdapterInvalidConfig;
import org.eclipse.hyades.logging.adapter.internal.util.AdapterSensor;
import org.eclipse.hyades.logging.adapter.util.Messages;

/**
 * @author rduggan
 *
 */
public class SimpleExtractor extends Extractor {
	
	private static final int MESSAGE_STRING_ARRAY_SIZE = 8;
	
	/* The storage for strings between invocations */
	protected StringArray inputs;
	
	private String localLineSeparator = System.getProperty("line.separator");
	
	public void update() throws AdapterInvalidConfig {
		super.update();		
		
		/* create our store for keeping data between invocations */
		inputs=new StringArray();
	}
	
	/**
	 * @see org.eclipse.hyades.logging.adaptor.IExtractor#processStrings(java.lang.String[])
	 */
	public MessageString[] processStrings(String[] strings)	{
		
		/* Where we will stick our results */
		MessageString[] finalResult=null;
		
		
		/* add new input to left over input if any
		 * null is a valid input, particularly during a flush operation
		 */
		if (strings != null) {
			for (int i = 0; i < strings.length; i++) {
				if (strings[i] != null) {
					inputs.enqueueString(strings[i]);
				}
				strings[i] = null;
			}
		}
		
		/* If there is nothing to do then return. */ 
		if (inputs == null || inputs.lastAddedString == inputs.lastRemovedString) {
			return null;
		}
		
		/* If there are multiple lines then handle them differently then single lines */
		if (!getContainsLineBreaks()) {
			finalResult=processSeparateLines();
		}
		else {
			finalResult=processMultipleLines();
		}
	
		/* Log our results if we have any */
		if (finalResult != null) {
			for (int i = 0; i < finalResult.length; i++) {
				if (finalResult[i] != null)	{
					log(Messages.getString("HyadesGA_CBE_Simple_Extractor_Message_Info",finalResult[i].getValue()), AdapterSensor.FINEST_LEVEL);
				}
			}
		}
		return finalResult;
	}

	private MessageString[] processSeparateLines() {
		return processMultipleLines();
	}
	
	private MessageString[] processMultipleLines() {
		
		List results=new ArrayList();
		int endOffset;
		
		/* Build a new string that consists of the next string in our buffer
		 * and continues until the end of all our strings.
		 */
		String searchString =
			buildString(
				inputs,
				inputs.lastRemovedString + 1,
				0,
				inputs.lastAddedString,
				inputs.contents[inputs.lastAddedString].length() - 1);
				
				
		/* Store the next character beyond our search */
		endOffset = searchString.length() - 1;
		
		/* The results of our search */
		StringPosition start=new StringPosition();
		StringPosition end=new StringPosition();
		
		/* process the search string we have created */
		processSearchString(results, searchString, start, end);
		
		/* If the start is not valid we need more data */
		String temp = null;
		if(start.isValid()) {
			if(!end.isValid()) {
				if(!flushingMode) {
					temp = searchString.substring(start.getStartPosition(), endOffset + 1);
				}
			}
		}
		else {
			temp=searchString;
		}
		
		/* This flushes the entire inputs and put in the current string for the next time around */
		inputs.flush();
		if(temp!=null) {
			inputs.enqueueString(temp);	
		}
		
		
		/* Format our results and return */
		Object[] entries=results.toArray();
		if(entries.length>0) {
			MessageString[] messages=new MessageString[entries.length];
			for(int i=0; i<entries.length; i++) {
				messages[i]=(MessageString)entries[i];
			}
			return messages;
		}

		return null;
	}
	
	
	/**
	 * Use the specified StringArray to build one single string that will be used 
	 * for our searching.
	 * 
	 * @param array
	 * @param arrayStartIndex
	 * @param startPosition
	 * @param arrayStopIndex
	 * @param stopPosition
	 * @return
	 */
	private String buildString(StringArray array, int arrayStartIndex, int startPosition, int arrayStopIndex, int stopPosition) {
		StringBuffer result = new StringBuffer();
		int copyStart, copyEnd;
		
		/* Walk through each of the specified entries in the array */
		for (int i = arrayStartIndex; i <= arrayStopIndex; i++)	{
			
			/* If this is the first entry store specified offset.  Otherwise the offset
			 * is the beginning
			 */
			if(i==arrayStartIndex) {
				copyStart = startPosition;
			}
			else {
				copyStart = 0;
			}
			
			/* If this is the last entry then store the next character beyond our scope.
			 * Otherwise the end is the length of the entry 
			 */
			if (i == arrayStopIndex) {
				copyEnd = stopPosition + 1;
			}
			else {
				copyEnd = array.contents[i].length();
			}
			
			/* Add the appropriate linebreak symbol for all of the input strings
			 * except the first one.
			 */
			if (i!=arrayStartIndex) {
			    if(getLineBreakSymbol()!=null && !getLineBreakSymbol().equals("")) {
					result.append(getLineBreakSymbol());
			    }
			    else {
			    	result.append(localLineSeparator);
			    }
			}
			
			/* Append the substring */
			result.append(array.contents[i].substring(copyStart, copyEnd));
		}
		return result.toString();
	}
	
	
	private void processSearchString(List results, String searchString, StringPosition start, StringPosition end) {
		
		int startOffset=0;
		int endOffset=searchString.length()-1;
		
		/* Loop through the string finding as many messages as possible */
		while (true) {
			
			/* If we have a specified start pattern then find where it starts 
			 * or ends, depending upon whether we are including our start pattern or not.
			 * If the start pattern is not provided then use the beginning of the current
			 * string.
			 */
			if (getStartPattern()!=null && !getStartPattern().equals("")) {
				search(start, getStartPattern(), searchString, startOffset);
				
				/* If we have not located the start pattern then we are done */
				if(!start.isValid()) {
					return;	
				}
			}
			else {
				start.setStartPosition(startOffset);
				start.setEndPosition(startOffset);
				start.setValid(true);
			}
			
			/* If we have located the start of the message then we will now
			 * locate the end of the sting.  If no end has been specified then we
			 * look for the next start.  
			 */
			if (start.isValid() && start.getEndPosition()<endOffset) {
				if(getEndPattern()==null || getEndPattern().equals("")) {
					search(end, getStartPattern(), searchString, start.getEndPosition()+1);
				}
				else  {
					search(end, getEndPattern(), searchString, start.getEndPosition()+1);
				}
			}
			
			/* If we have the start and no end we need more data.  The only exception to this is when we are flushing */
			if(!end.isValid()) {
				if(flushingMode) {
					end.setStartPosition(endOffset);
					end.setEndPosition(endOffset);
					results.add(adjustMessage(searchString, start, end));
				}
				return;
			}
			
			/* If we have a start and an end we are good to go */
			if(start.isValid() && end.isValid()) {
				results.add(adjustMessage(searchString, start, end));
				/* If there is an end position specified we use the end of it.  Otherwise we use the begining */
				if(getEndPattern()==null || getEndPattern().equals("")) {
					startOffset=end.getStartPosition();
				}
				else {
					startOffset=end.getEndPosition();
				}
				// This string is just for testing in the debugger 
				//String newString=searchString.substring(startOffset, searchString.length()-1);
				start.setValid(false);
				end.setValid(false);
			}
		}
	}
	
	/**
	 * Extract our final message based upon whether the end and start patterns are 
	 * included.
	 * @param searchString
	 * @param startPosition
	 * @param endPosition
	 * @return
	 */
	protected MessageString adjustMessage(String searchString, StringPosition startPosition, StringPosition endPosition) {
		int adjustedStart;
		int adjustedEnd;
		int endOffset = searchString.length()-1;
		if(getIncludeStartPattern()) {
			adjustedStart = startPosition.getStartPosition();
		}
		else {
			adjustedStart = startPosition.getEndPosition()+1;
		}
		if(getIncludeEndPattern()) {
			adjustedEnd = endPosition.getEndPosition()+1;
		}
		else {
			/* check if we are at the end of the string (ie we are flushing) */
			if (endPosition.getStartPosition() == endOffset) {
				/* To include the last character we must specify one past the end
				 * of the string for substring below to work.
				 */
				adjustedEnd = endPosition.getStartPosition()+1;
			}
			else {
				adjustedEnd = endPosition.getStartPosition();
			}
		}
		MessageString result=new MessageString();
		result.setValue(searchString.substring(adjustedStart, adjustedEnd));
		
		return result;		
	}
	
	
	/**
	 * Search for the pattern in the specified search string.  If the pattern is located in the 
	 * search string then the position variable is loaded with the start of the pattern and the 
	 * end of the pattern.
	 * 
	 * @param pos
	 * @param pattern
	 * @param searchString
	 * @param startOffset
	 */
	protected void search(StringPosition position, String pattern, String searchString, int startOffset) {
		int foundPosition = -1;
		if (pattern != null) {
			foundPosition = searchString.indexOf(pattern, startOffset);
		}
		if (foundPosition > -1 && foundPosition<searchString.length()) {
			position.setStartPosition(foundPosition);
			if(pattern.length()>1) {
				position.setEndPosition(foundPosition + pattern.length()-1);
			}
			else {
				position.setEndPosition(foundPosition + pattern.length());
			}
			position.setValid(true);
		}
	}
	
	/**
	 * A wrapper for String[] that keeps track of the current offset in the array
	 */ 
	class StringArray {
		public String[] contents;
		public int offset;
		public int lastAddedString=-1;
		public int lastRemovedString=-1;
		
		public StringArray() {
			contents=new String[MESSAGE_STRING_ARRAY_SIZE];
			offset=0;
			lastAddedString=lastRemovedString=-1;
		}
		
		private void increaseStringArraySize() {
			String[] tempArray = new String[contents.length * 2];
			System.arraycopy(contents, 0, tempArray, 0, contents.length);
			contents = tempArray;
		}
		
		public void enqueueString(String newString) {
			if (offset + 1 >= contents.length) {
				increaseStringArraySize();
			}
			contents[offset++] = newString;
			lastAddedString=offset-1;
		}	
		
		public void flush() {
			for(int i=0; i<contents.length; i++) {
				contents[i]=null;
			}
			offset=0;
			lastAddedString=lastRemovedString=-1;
		}
	}
	
	class StringPosition implements Cloneable {
		private int startPosition;
		private int endPosition;
		private boolean valid=false;
		
		public int getStartPosition() {
			return startPosition;
		}
		
		public void setStartPosition(int pos) {
			startPosition=pos;
		}
		
		int getEndPosition() {
			return endPosition;
		}
		
		public void setEndPosition(int pos) {
			endPosition=pos;
		}
		
		public boolean isValid() {
			return valid;
		}
		
		public void setValid(boolean valid) {
			this.valid=valid;
		}
		
		protected Object clone() throws CloneNotSupportedException {
			StringPosition rhs=new StringPosition();
			rhs.startPosition=this.startPosition;
			rhs.endPosition=this.endPosition;
			rhs.valid=this.valid;
			return rhs;
		}
	}

}
