/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.logging.adapter.formatters;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.logging.adapter.AdapterInvalidConfig;
import org.eclipse.hyades.logging.adapter.ICBEPropertyConstants;
import org.eclipse.hyades.logging.adapter.IDirectedGraph;
import org.eclipse.hyades.logging.adapter.IFormatter;
import org.eclipse.hyades.logging.adapter.impl.ProcessUnit;
import org.eclipse.hyades.logging.adapter.internal.util.AdapterSensor;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.core.Guid;
import org.eclipse.hyades.logging.events.IAssociatedEvent;
import org.eclipse.hyades.logging.events.IAssociationEngine;
import org.eclipse.hyades.logging.events.IAvailableSituation;
import org.eclipse.hyades.logging.events.ICommonBaseEvent;
import org.eclipse.hyades.logging.events.IComponentIdentification;
import org.eclipse.hyades.logging.events.IConfigureSituation;
import org.eclipse.hyades.logging.events.IConnectSituation;
import org.eclipse.hyades.logging.events.IContextDataElement;
import org.eclipse.hyades.logging.events.ICreateSituation;
import org.eclipse.hyades.logging.events.IDependencySituation;
import org.eclipse.hyades.logging.events.IDestroySituation;
import org.eclipse.hyades.logging.events.IExtendedDataElement;
import org.eclipse.hyades.logging.events.IFeatureSituation;
import org.eclipse.hyades.logging.events.IMsgCatalogToken;
import org.eclipse.hyades.logging.events.IMsgDataElement;
import org.eclipse.hyades.logging.events.IOtherSituation;
import org.eclipse.hyades.logging.events.IReportSituation;
import org.eclipse.hyades.logging.events.IRequestSituation;
import org.eclipse.hyades.logging.events.ISimpleEventFactory;
import org.eclipse.hyades.logging.events.ISituation;
import org.eclipse.hyades.logging.events.ISituationType;
import org.eclipse.hyades.logging.events.IStartSituation;
import org.eclipse.hyades.logging.events.IStopSituation;
import org.eclipse.hyades.logging.events.SimpleEventFactoryImpl;
import org.eclipse.hyades.logging.events.exceptions.UnsupportedVersionException;

/**
 * @author smith
 *
 * This Formatter class processes a HashMap to generate a CBE.
 * The HashMap key is the CBE property name string and the value
 * is the CBE property value. 
 * 
 * The HashMap key is a sequence of period delimited property name
 * string tokens representing the path to a specific CBE property.
 * We need a path to represent the CBE property because
 * CBE's have a class hierarchy.  For example, the CBE has a Reporter Component
 * ID and a Source Component ID property.  Both of these are ComponentIdentification
 * objects.  So, for example, to represent the location property of the Source
 * Component ID the HashMap key would be:
 * COMMONBASEEVENT.COMMONBASEEVENT_SOURCECOMPONENTID.COMPONENTID_LOCATION
 * Some CBE properties are arrays so we can represent an element of an array
 * with an array index.  For example contextDataElements is an array of
 * ContextDataElements.  To represent the name property of the second ContextDataElement
 * in the array the HashMap key would be:
 * COMMONBASEEVENT.COMMONBASEEVENT_CONTEXTDATAELEMENT[2].CONTEXTDATAELEMENT_NAME
 * A special case is the children attribute of ExtendedDataElement. It is a
 * collection of ExtendedDataElements which aren't ordered.  Therefore, the
 * name of the ExtendedDataElement will be used to represent the child ExtendedDataElement.
 * For example, to represent the type property of the child ExtendedDataDlement with name 
 * "ErrorCode" in the third ExtendedDataElement in the CBE extendedDataElements 
 * array the HashMap key would be:
 * COMMONBASEEVENT.COMMONBASEEVENT_EXTENDEDDATAELEMENT[3].EXTENDEDDATAELEMENT_CHILD.ErrorCode.EXTENDEDDATAELEMENT_TYPE
 *  
 */
public class CBEFormatter extends ProcessUnit implements IFormatter, ICBEPropertyConstants {

	/**
	 * Constant to indicate to use a built-in value
	 */
	private final static String BUILTIN = "##BUILTIN";
	/**
	 * Factory for creating events
	 */
	private static ISimpleEventFactory eventFactory = SimpleEventFactoryImpl.getInstance();

	/**
	 * IP address of the machine parsing the log file. 
	 */
	private String localHostId = null;

	/**
	 * Format of the IP address of the machine parsing the log file.  This
	 * field will be set either to the String "IPV4" or the String "IPV6".
	 */
	private String localHostIdFormat = null;

	// Local host constants
	private final static String HOST_ID_FORMAT_IPV4 = "IPV4";
	private final static String HOST_ID_FORMAT_IPV6 = "IPV6";
	private final static String NONE = "None";
	private final static String UNKNOWN_HOST = "Unknown Host";


	/**
	 * @see org.eclipse.hyades.logging.adapter.IProcessUnit#processEventItems(java.lang.Object[])
	 */
	public Object[] processEventItems(Object[] msgs) {
		/* If we have no messages return */
		if(msgs==null || msgs.length==0) {
			return null;
		}
		return processDirectedGraphs(msgs);
	}

	/**
	 * Process an array of messages
	 * @param msgs An array of HashMap objects representing the data in a group of messages
	 * @return ICommonBaseEvent[] The CBE's generated from the HashMaps passed in
	 */
	private ICommonBaseEvent[] processDirectedGraphs(Object[] msgs) {
		
		
		/* Create our CBE's that will be ouputs of this procesing.  There is one CBE per message */
		ICommonBaseEvent[] cbes = new ICommonBaseEvent[msgs.length];
		
		// For each message, process the list of rule matches
		for (int k = 0; k < msgs.length; k++) {
			
			/* Keep a cache of all elements that can have multiple entries. This is required
			 * as some types will have multiple rules that apply to them.  We need to find the element
			 * if previous rules have already been matched.  The scope of the cache is strictly the
			 * duration of the processing step for this CBE instance.
			 */ 
			HashMap cache=new HashMap(); 
			
			
			// Get the nodes of the graph
			Iterator ni= ((List)msgs[k]).iterator();
		
			ICommonBaseEvent cbe = eventFactory.createCommonBaseEvent();
			
			// Iterate over the nodes of the graph
			while (ni.hasNext()) {
				IDirectedGraph  graphEntry = (IDirectedGraph)ni.next();
	
				// Get the path
				List path=graphEntry.getPath();
			
				try {		
						
					if (((String)path.get(0)).equals(COMMONBASEEVENT)) {
						
						if (((String)path.get(2)).equals(COMMONBASEEVENT_EXTENSIONNAME)) {
							cbe.setExtensionName((String)graphEntry.getValue());
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_LOCALINSTANCEID)) {
							String value = (String)graphEntry.getValue();
							
							if (value.equals(BUILTIN)) {
								value = getLocalHostId() + System.currentTimeMillis() + Integer.toString(msgs[k].hashCode());
							}
							cbe.setLocalInstanceId(value);
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_GLOBALINSTANCEID)) {
							String value = (String)graphEntry.getValue();
							
							if (value.equals(BUILTIN)) {
								value = (new Guid().toString());
							}
							cbe.setGlobalInstanceId(value);
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_CREATIONTIME)) {
							String strValue;
							Long lValue;
							try {
								strValue = (String)graphEntry.getValue();
								cbe.setCreationTime(strValue);
							}
							catch (ClassCastException e) {
								/* time returned by parser may be in milliseconds */
								lValue = (Long)graphEntry.getValue();
								cbe.setCreationTime(lValue.longValue());
							}
							catch (Exception e) {
								throw e;
							}
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_SEVERITY)) {
							cbe.setSeverity(Short.parseShort((String)graphEntry.getValue()));
						}						
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_PRIORITY)) {
							cbe.setPriority(Short.parseShort((String)graphEntry.getValue()));
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_MSG)) {
							cbe.setMsg((String)graphEntry.getValue());
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_REPEATCOUNT)) {
							cbe.setRepeatCount(Short.parseShort((String)graphEntry.getValue()));
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_ELAPSEDTIME)) {
							cbe.setElapsedTime(Long.parseLong((String)graphEntry.getValue()));
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_SEQUENCENUMBER)) {
							cbe.setSequenceNumber(Long.parseLong((String)graphEntry.getValue()));
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_VERSION)) {
							try {
								cbe.setPreferredVersion((String)graphEntry.getValue());
							}
							catch (UnsupportedVersionException e) {
								// Do nothing in this case because the version is set automatically when the
								// CBE is created.
							}
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_OTHERDATA)) {
							cbe.setOtherData((String[])graphEntry.getValue());
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_MSGDATAELEMENT)) {
						
							// Get the message data element from the CBE if it exists,
							// otherwise create it.
							IMsgDataElement mde = cbe.getMsgDataElement();
							if (mde == null) {
								mde = eventFactory.createMsgDataElement();
								cbe.setMsgDataElement(mde);
							}
							// Set the message data element property
							
							if (((String)path.get(4)).equals(MSGDATAELEMENT_MSGID)) {
								mde.setMsgId((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(MSGDATAELEMENT_MSGIDTYPE)) {
								mde.setMsgIdType((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(MSGDATAELEMENT_MSGCATALOGID)) {
								mde.setMsgCatalogId((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(MSGDATAELEMENT_MSGCATALOGTOKENS)) {
								
								String[] result={(String)graphEntry.getValue()};
								
								IMsgCatalogToken msgCatalogToken = null;
								IMsgCatalogToken[] msgCatalogTokens = new IMsgCatalogToken[result.length];
								
								for (int counter = 0; counter < result.length; counter++) {
									msgCatalogToken = eventFactory.createMsgCatalogToken();
									msgCatalogToken.setValue(result[counter]);
									msgCatalogTokens[counter] = msgCatalogToken;
                                }
                                
								mde.setMsgCatalogTokens(msgCatalogTokens);
							}
							else if (((String)path.get(4)).equals(MSGDATAELEMENT_MSGCATALOGTYPE)) {
								mde.setMsgCatalogType((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(MSGDATAELEMENT_MSGCATALOG)) {
								mde.setMsgCatalog((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(MSGDATAELEMENT_MSGLOCALE)) {
								mde.setMsgLocale((String)graphEntry.getValue());
							}
							else {
								// error - incorrect name
								log(Messages.getString("HyadesGA_CBE_Formatter_MsgDataElement_Warning",getUniqueID(),(String)path.get(4)),AdapterSensor.WARN_LEVEL);
							}
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_CONTEXTDATAELEMENTS)) {
							
							/* Extract our key that identifies this rule.  We will use COMMONBASEEVENT_ASSOCIATEDEVENTS + index as the key */
							String key=((String)path.get(2))+(String)path.get(3);


							/* Get our IExtendedDataElement or create a new one */
							IContextDataElement cde=(IContextDataElement)cache.get(key);
							if(cde==null) {
								cde = eventFactory.createContextDataElement();
								cbe.addContextDataElement(cde);	
								cache.put(key, cde);
							}

							if (((String)path.get(4)).equals(CONTEXTDATAELEMENT_CONTEXTID)) {
								// contextId and contextValue properties are mutually exclusive so
								// check for existence of contextValues before setting contextId
								if (cde.getContextValue() != null) {									
									log(Messages.getString("HyadesGA_CBE_Formatter_Duplicate_ContextDataElement_Warning",getUniqueID(),(String)path.get(4),CONTEXTDATAELEMENT_CONTEXTVALUE),AdapterSensor.WARN_LEVEL);
								}
								else {
									cde.setContextId((String)graphEntry.getValue());
								}
							}
							else if (((String)path.get(4)).equals(CONTEXTDATAELEMENT_TYPE)) {
								cde.setType((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(CONTEXTDATAELEMENT_NAME)) {
								cde.setName((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(CONTEXTDATAELEMENT_CONTEXTVALUE)) {
								// contextId and contextValue properties are mutually exclusive so
								// check for existence of contextId before setting contextValue
								if (cde.getContextId() != null) {
									log(Messages.getString("HyadesGA_CBE_Formatter_Duplicate_ContextDataElement_Warning",getUniqueID(),(String)path.get(4),CONTEXTDATAELEMENT_CONTEXTID),AdapterSensor.WARN_LEVEL);
								}
								else {									
									cde.setContextValue((String)graphEntry.getValue());
								}
							}
							else {
								// error - incorrect name
								log(Messages.getString("HyadesGA_CBE_Formatter_ContextDataElement_Warning",getUniqueID(),(String)path.get(4)),AdapterSensor.WARN_LEVEL);
							}
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_REPORTERCOMPONENTID) || ((String)path.get(2)).equals(COMMONBASEEVENT_SOURCECOMPONENTID)) {
							// Get the component identification from the CBE if it exists,
							// otherwise create it
							IComponentIdentification cid;
							if (((String)path.get(2)).equals(COMMONBASEEVENT_REPORTERCOMPONENTID)) {
								cid = cbe.getReporterComponentId();
								if (cid == null) {
									cid = eventFactory.createComponentIdentification();
									cbe.setReporterComponentId(cid);
								}
							}
							else {
								cid = cbe.getSourceComponentId();
								if (cid == null) {
									cid = eventFactory.createComponentIdentification();
									cbe.setSourceComponentId(cid);
								}
							}
							
							// Set the component identification property
							
							if (((String)path.get(4)).equals(COMPONENTID_LOCATION)) {
								String value = (String)graphEntry.getValue();
								
								if (value.equals(BUILTIN)) {
									value = getLocalHostId();
								}
								cid.setLocation(value);
							}
							else if (((String)path.get(4)).equals(COMPONENTID_LOCATIONTYPE)) {
								String value = (String)graphEntry.getValue();
								
								if (value.equals(BUILTIN)) {
									value = getLocalHostIdFormat();
								}
								cid.setLocationType(value);
							}
							else if (((String)path.get(4)).equals(COMPONENTID_APPLICATION)) {
								cid.setApplication((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(COMPONENTID_EXECUTIONENVIRONMENT)) {
								cid.setExecutionEnvironment((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(COMPONENTID_COMPONENT)) {
								cid.setComponent((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(COMPONENTID_SUBCOMPONENT)) {
								cid.setSubComponent((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(COMPONENTID_COMPONENTIDTYPE)) {
								cid.setComponentIdType((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(COMPONENTID_COMPONENTTYPE)) {
								cid.setComponentType((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(COMPONENTID_INSTANCEID)) {
								cid.setInstanceId((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(COMPONENTID_PROCESSID)) {
								cid.setProcessId((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(COMPONENTID_THREADID)) {
								cid.setThreadId((String)graphEntry.getValue());
							}
							else {
								// error - incorrect name
								log(Messages.getString("HyadesGA_CBE_Formatter_ComponentIdentification_Warning",getUniqueID(),(String)path.get(4)),AdapterSensor.WARN_LEVEL);
							}	
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_SITUATION)) {
	
							// Get the situation from the CBE if it exists,
							// otherwise create it.
							ISituation sit = cbe.getSituation();
							if (sit == null) {
								sit = eventFactory.createSituation();
								cbe.setSituation(sit);
							}
							
							// Set the appropiate situation property
							
							if (((String)path.get(4)).equals(SITUATION_CATEGORYNAME)) {
								sit.setCategoryName((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(SITUATION_SITUATIONTYPE)) {					
								// Set the situation type properites	
								if (((String)path.get(6)).equals(AVAILABLESITUATION)) {
									IAvailableSituation asit = null;
									ISituationType sittype = sit.getSituationType();
									if (sittype == null) {
										asit = eventFactory.createAvailableSituation();
										sit.setSituationType(asit);
									}
									else if (sittype instanceof IAvailableSituation){
										asit = (IAvailableSituation)sittype;
									}
									else {
										// error - incorrect situtation type
										// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
									}
									if (asit != null) {
										if (((String)path.get(8)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
											asit.setReasoningScope((String)graphEntry.getValue());
										}
										else if (((String)path.get(8)).equals(AVAILABLESITUATION_OPERATIONDISPOSITION)) {
											asit.setOperationDisposition((String)graphEntry.getValue());
										}
										else if (((String)path.get(8)).equals(AVAILABLESITUATION_PROCESSINGDISPOSITION)) {
											asit.setProcessingDisposition((String)graphEntry.getValue());
										}
										else if (((String)path.get(8)).equals(AVAILABLESITUATION_AVAILABILITYDISPOSITION)) {
											asit.setAvailabilityDisposition((String)graphEntry.getValue());
										}
										else {
											// error - incorrect situtation type property
											log(Messages.getString("HyadesGA_CBE_Formatter_AvailableSituation_Warning",getUniqueID(),(String)path.get(8)),AdapterSensor.WARN_LEVEL);										
										}														
									}
								}	
								else if (((String)path.get(6)).equals(REQUESTSITUATION)) {
									IRequestSituation asit = null;
									ISituationType sittype = sit.getSituationType();
									if (sittype == null) {
										asit = eventFactory.createRequestSituation();
										sit.setSituationType(asit);
									}
									else if (sittype instanceof IRequestSituation){
										asit = (IRequestSituation)sittype;
									}
									else {
										// error - incorrect situtation type
										// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
									}
									if (asit != null) {
										if (((String)path.get(8)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
											asit.setReasoningScope((String)graphEntry.getValue());
										}
										else if (((String)path.get(8)).equals(SITUATIONTYPE_SUCCESSDISPOSITION)) {
											asit.setSuccessDisposition((String)graphEntry.getValue());
										}
										else if (((String)path.get(8)).equals(SITUATIONTYPE_SITUATIONQUALIFIER)) {
											asit.setSituationQualifier((String)graphEntry.getValue());
										}
										else {
											// error - incorrect situtation type property
											log(Messages.getString("HyadesGA_CBE_Formatter_RequestSituation_Warning",getUniqueID(),(String)path.get(8)),AdapterSensor.WARN_LEVEL);										
										}														
									}												
								}
								else if (((String)path.get(6)).equals(STARTSITUATION)) {
										
									IStartSituation asit = null;
									ISituationType sittype = sit.getSituationType();
									if (sittype == null) {
										asit = eventFactory.createStartSituation();
										sit.setSituationType(asit);
									}
									else if (sittype instanceof IStartSituation){
										asit = (IStartSituation)sittype;
									}
									else {
										// error - incorrect situtation type
										// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
									}
									if (asit != null) {
										if (((String)path.get(8)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
											asit.setReasoningScope((String)graphEntry.getValue());
										}
										else if (((String)path.get(8)).equals(SITUATIONTYPE_SUCCESSDISPOSITION)) {
											asit.setSuccessDisposition((String)graphEntry.getValue());
										}
										else if (((String)path.get(8)).equals(SITUATIONTYPE_SITUATIONQUALIFIER)) {
											asit.setSituationQualifier((String)graphEntry.getValue());
										}
										else {
											// error - incorrect situtation type property
											log(Messages.getString("HyadesGA_CBE_Formatter_StartSituation_Warning",getUniqueID(),(String)path.get(8)),AdapterSensor.WARN_LEVEL);										
										}														
									}											
								}											
								else if (((String)path.get(6)).equals(STOPSITUATION)) {
									IStopSituation asit = null;
									ISituationType sittype = sit.getSituationType();
									if (sittype == null) {
										asit = eventFactory.createStopSituation();
										sit.setSituationType(asit);
									}
									else if (sittype instanceof IStopSituation){
										asit = (IStopSituation)sittype;
									}
									else {
										// error - incorrect situtation type
										// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
									}
									if (asit != null) {
										if (((String)path.get(8)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
											asit.setReasoningScope((String)graphEntry.getValue());
										}
										else if (((String)path.get(8)).equals(SITUATIONTYPE_SUCCESSDISPOSITION)) {
											asit.setSuccessDisposition((String)graphEntry.getValue());
										}
										else if (((String)path.get(8)).equals(SITUATIONTYPE_SITUATIONQUALIFIER)) {
											asit.setSituationQualifier((String)graphEntry.getValue());
										}
										else {
											// error - incorrect situtation type property
											log(Messages.getString("HyadesGA_CBE_Formatter_StopSituation_Warning",getUniqueID(),(String)path.get(8)),AdapterSensor.WARN_LEVEL);										
										}														
									}													
								}
								else if (((String)path.get(6)).equals(CONNECTSITUATION)) {
									IConnectSituation asit = null;
									ISituationType sittype = sit.getSituationType();
									if (sittype == null) {
										asit = eventFactory.createConnectSituation();
										sit.setSituationType(asit);
									}
									else if (sittype instanceof IConnectSituation){
										asit = (IConnectSituation)sittype;
									}
									else {
										// error - incorrect situtation type
										// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
									}
									if (asit != null) {
										if (((String)path.get(8)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
											asit.setReasoningScope((String)graphEntry.getValue());
										}
										else if (((String)path.get(8)).equals(SITUATIONTYPE_SUCCESSDISPOSITION)) {
											asit.setSuccessDisposition((String)graphEntry.getValue());
										}
										else if (((String)path.get(8)).equals(CONNECTSITUATION_SITUATIONDISPOSITION)) {
											asit.setSituationDisposition((String)graphEntry.getValue());
										}
										else {
											// error - incorrect situtation type property
											log(Messages.getString("HyadesGA_CBE_Formatter_ConnectSituation_Warning",getUniqueID(),(String)path.get(8)),AdapterSensor.WARN_LEVEL);										
										}														
									}												
								}													
								else if (((String)path.get(6)).equals(CONFIGURESITUATION)) {
									IConfigureSituation asit = null;
									ISituationType sittype = sit.getSituationType();
									if (sittype == null) {
										asit = eventFactory.createConfigureSituation();
										sit.setSituationType(asit);
									}
									else if (sittype instanceof IConfigureSituation){
										asit = (IConfigureSituation)sittype;
									}
									else {
										// error - incorrect situtation type
										// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
									}
									if (asit != null) {
										if (((String)path.get(8)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
											asit.setReasoningScope((String)graphEntry.getValue());
										}
										else if (((String)path.get(8)).equals(SITUATIONTYPE_SUCCESSDISPOSITION)) {
											asit.setSuccessDisposition((String)graphEntry.getValue());
										}
										else {
											// error - incorrect situtation type property
											log(Messages.getString("HyadesGA_CBE_Formatter_ConfigureSituation_Warning",getUniqueID(),(String)path.get(8)),AdapterSensor.WARN_LEVEL);										
										}
									}												
								}
								else if (((String)path.get(6)).equals(CREATESITUATION)) {
									ICreateSituation asit = null;
									ISituationType sittype = sit.getSituationType();
									if (sittype == null) {
										asit = eventFactory.createCreateSituation();
										sit.setSituationType(asit);
									}
									else if (sittype instanceof ICreateSituation){
										asit = (ICreateSituation)sittype;
									}
									else {
										// error - incorrect situtation type
										// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
									}
									if (asit != null) {
										if (((String)path.get(8)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
											asit.setReasoningScope((String)graphEntry.getValue());
										}
										else if (((String)path.get(8)).equals(SITUATIONTYPE_SUCCESSDISPOSITION)) {
											asit.setSuccessDisposition((String)graphEntry.getValue());
										}
										else {
											// error - incorrect situtation type property
											log(Messages.getString("HyadesGA_CBE_Formatter_CreateSituation_Warning",getUniqueID(),(String)path.get(8)),AdapterSensor.WARN_LEVEL);										
										}
									}												
								}
								else if (((String)path.get(6)).equals(DESTROYSITUATION)) {
							
									IDestroySituation asit = null;
									ISituationType sittype = sit.getSituationType();
									if (sittype == null) {
										asit = eventFactory.createDestroySituation();
										sit.setSituationType(asit);
									}
									else if (sittype instanceof IDestroySituation){
										asit = (IDestroySituation)sittype;
									}
									else {
										// error - incorrect situtation type
										// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
									}
									if (asit != null) {
										if (((String)path.get(8)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
											asit.setReasoningScope((String)graphEntry.getValue());
										}
										else if (((String)path.get(8)).equals(SITUATIONTYPE_SUCCESSDISPOSITION)) {
											asit.setSuccessDisposition((String)graphEntry.getValue());
										}
										else {
											// error - incorrect situtation type property
											log(Messages.getString("HyadesGA_CBE_Formatter_CreateSituation_Warning",getUniqueID(),(String)path.get(8)),AdapterSensor.WARN_LEVEL);										
										}
									}												
								}
								else if (((String)path.get(6)).equals(REPORTSITUATION)) {
									IReportSituation asit = null;
									ISituationType sittype = sit.getSituationType();
									if (sittype == null) {
										asit = eventFactory.createReportSituation();
										sit.setSituationType(asit);
									}
									else if (sittype instanceof IReportSituation){
										asit = (IReportSituation)sittype;
									}
									else {
										// error - incorrect situtation type
										// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
									}
									if (asit != null) {
										if (((String)path.get(8)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
											asit.setReasoningScope((String)graphEntry.getValue());
										}
										else if (((String)path.get(8)).equals(REPORTSITUATION_REPORTCATEGORY)) {
											asit.setReportCategory((String)graphEntry.getValue());
										}
										else {
											// error - incorrect situtation type property
											log(Messages.getString("HyadesGA_CBE_Formatter_ReportSituation_Warning",getUniqueID(),(String)path.get(8)),AdapterSensor.WARN_LEVEL);										
										}
									}												
								}
								else if (((String)path.get(6)).equals(FEATURESITUATION)) {
									IFeatureSituation asit = null;
									ISituationType sittype = sit.getSituationType();
									if (sittype == null) {
										asit = eventFactory.createFeatureSituation();
										sit.setSituationType(asit);
									}
									else if (sittype instanceof IFeatureSituation){
										asit = (IFeatureSituation)sittype;
									}
									else {
										// error - incorrect situtation type
										// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
									}
									if (asit != null) {
										if (((String)path.get(8)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
											asit.setReasoningScope((String)graphEntry.getValue());
										}
										else if (((String)path.get(8)).equals(FEATURESITUATION_FEATUREDISPOSITION)) {
											asit.setFeatureDisposition((String)graphEntry.getValue());
										}
										else {
											// error - incorrect situtation type property
											log(Messages.getString("HyadesGA_CBE_Formatter_FeatureSituation_Warning",getUniqueID(),(String)path.get(8)),AdapterSensor.WARN_LEVEL);										
										}
									}												
								}
								else if (((String)path.get(6)).equals(DEPENDENCYSITUATION)) {
									IDependencySituation asit = null;
									ISituationType sittype = sit.getSituationType();
									if (sittype == null) {
										asit = eventFactory.createDependencySituation();
										sit.setSituationType(asit);
									}
									else if (sittype instanceof IDependencySituation){
										asit = (IDependencySituation)sittype;
									}
									else {
										// error - incorrect situtation type
										// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
									}
									if (asit != null) {
										if (((String)path.get(8)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
											asit.setReasoningScope((String)graphEntry.getValue());
										}
										else if (((String)path.get(8)).equals(DEPENDENCYSITUATION_DEPENDENCYDISPOSITION)) {
											asit.setDependencyDisposition((String)graphEntry.getValue());
										}
										else {
											// error - incorrect situtation type property
											log(Messages.getString("HyadesGA_CBE_Formatter_DependencySituation_Warning",getUniqueID(),(String)path.get(8)),AdapterSensor.WARN_LEVEL);										
										}
									}												
								}
								else if (((String)path.get(6)).equals(OTHERSITUATION)) {
									IOtherSituation asit = null;
									ISituationType sittype = sit.getSituationType();
									if (sittype == null) {
										asit = eventFactory.createOtherSituation();
										sit.setSituationType(asit);
									}
									else if (sittype instanceof IOtherSituation){
										asit = (IOtherSituation)sittype;
									}
									else {
										// error - incorrect situtation type
										// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
									}
									if (asit != null) {
										if (((String)path.get(8)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
											asit.setReasoningScope((String)graphEntry.getValue());
										}
										else if (((String)path.get(8)).equals(OTHERSITUATION_ANYDATA)) {
											asit.setAnyData((String[])graphEntry.getValue());
										}
										else {
											// error - incorrect situtation type property
											log(Messages.getString("HyadesGA_CBE_Formatter_DependencySituation_Warning",getUniqueID(),(String)path.get(8)),AdapterSensor.WARN_LEVEL);										
										}
									}												
								}
								else {
									// error - incorrect name
									log(Messages.getString("HyadesGA_CBE_Formatter_Invalid_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.WARN_LEVEL);										
								}
							}
							else {
								
								// error - incorrect name
								log(Messages.getString("HyadesGA_CBE_Formatter_Invalid_Situation_Warning",getUniqueID(),(String)path.get(4)),AdapterSensor.WARN_LEVEL);										
							}
							
						}
						else if (((String)path.get(2)).startsWith(COMMONBASEEVENT_ASSOCIATEDEVENTS)) {
							
							
							/* Extract our key that identifies this rule.  We will use COMMONBASEEVENT_ASSOCIATEDEVENTS + index as the key */
							String key=((String)path.get(2))+(String)path.get(3);


							/* Get our IExtendedDataElement or create a new one */
							IAssociatedEvent ae=(IAssociatedEvent)cache.get(key);
							if(ae==null) {
								ae = eventFactory.createAssociatedEvent();
								cbe.addAssociatedEvent(ae);	
								cache.put(key, ae);
							}
								
	
							if (((String)path.get(4)).equals(ASSOCIATEDEVENT_ASSOCIATIONENGINEREFERENCE)) {
								// As of CBE 1.0.1 this property is a reference to an association engine so 
								// we shouldn't be getting this property from the parser
								// Log an warning message.
								log(Messages.getString("HyadesGA_CBE_Formatter_Invalid_AssociatedEvent_Warning",getUniqueID(),(String)path.get(4)),AdapterSensor.WARN_LEVEL);										
							}
							else if (((String)path.get(4)).equals(ASSOCIATEDEVENT_ASSOCIATIONENGINEINFO)) {
		
								// Get the AssociationEngineInfo from the AssociatedEvent if it exists,
								// otherwise create it.
								IAssociationEngine aei = ae.getAssociationEngineInfo();
								if (aei == null) {
									aei = eventFactory.createAssociationEngine();
									ae.setAssociationEngineInfo(aei);
								}
								
								// Set the AssociationEngine property
								if (((String)path.get(6)).equals(ASSOCIATIONENGINE_ID)) {
									aei.setId((String)graphEntry.getValue());
								}
								else if (((String)path.get(6)).equals(ASSOCIATIONENGINE_NAME)) {
									aei.setName((String)graphEntry.getValue());
								}
								else if (((String)path.get(6)).equals(ASSOCIATIONENGINE_TYPE)) {
									aei.setType((String)graphEntry.getValue());
								}
								else {
									// error - incorrect name
									log(Messages.getString("HyadesGA_CBE_Formatter_Invalid_AssociationEngine_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.WARN_LEVEL);										
								}
							}
							else if (((String)path.get(4)).equals(ASSOCIATEDEVENT_RESOLVEDEVENT)) {
								ae.addResolvedEvent((String)graphEntry.getValue());
							}
							else {
								// error - incorrect name
								log(Messages.getString("HyadesGA_CBE_Formatter_Invalid_AssociatedEvent_Warning",getUniqueID(),(String)path.get(4)),AdapterSensor.WARN_LEVEL);										
							}
						}
						else if (((String)path.get(2)).startsWith(COMMONBASEEVENT_EXTENDEDDATAELEMENTS)) {
	
							/* Extract our key that identifies this rule.  We will use COMMONBASEEVENT_EXTENDEDDATAELEMENTS + index as the key */
							String key=((String)path.get(2))+(String)path.get(3);
							
							
							/* Get our IExtendedDataElement or create a new one */
							IExtendedDataElement ede=(IExtendedDataElement)cache.get(key);
							if(ede==null) {
								ede = eventFactory.createExtendedDataElement();	
								cbe.addExtendedDataElement(ede);
								cache.put(key, ede);
							}
							
									
									
							// If we have a valid extended data element then set its property with the data
							String property=((String)path.get(4));
							int depth=4;
							while (property != null) {											
						
								if (property.equals(EXTENDEDDATAELEMENT_NAME)) {
									ede.setName((String)graphEntry.getValue());
									property = null;
								}
								else if (property.equals(EXTENDEDDATAELEMENT_TYPE)) {
									ede.setType((String)graphEntry.getValue());
									property = null;
								}
								else if (property.equals(EXTENDEDDATAELEMENT_VALUES)) {
									String[] result={(String)graphEntry.getValue()};
									ede.setValues(result);
					
									property = null;
								}
								else if (property.equals(EXTENDEDDATAELEMENT_HEXVALUE)) {
									// values and hexvalues properties are mutually exclusive so
									// add check for existence of values before setting hexvalues
									if (ede.getValues() != null) {										
										log(Messages.getString("HyadesGA_CBE_Formatter_Duplicate_ExtendedDataElement_Warning",getUniqueID(),property,EXTENDEDDATAELEMENT_VALUES),AdapterSensor.WARN_LEVEL);										
									}
									else {
										ede.setHexValues((char[])graphEntry.getValue());
									}
									property = null;
								}
								else if (property.equals(EXTENDEDDATAELEMENT_CHILDREN)) {
									// Get the name of the child Extended Data Element
									String edeName = ((String)path.get(++depth));
									
									// Get the child ExtendedDataElement from the ExtendedDataElement if it exists,
									// otherwise create it.
									List edeArr = ede.getChildDataElements();
									if (edeArr == null) {
										IExtendedDataElement edeTmp = eventFactory.createExtendedDataElement();
										edeTmp.setName(edeName);
										ede.addChildDataElement(edeTmp);
										ede = edeTmp;
									}
									else {
										IExtendedDataElement edeTmp = null;
										// Search for the ExendedDataElement in the arrray
										for (int j=0; j < edeArr.size(); j++) {
											if (((IExtendedDataElement)(edeArr.get(j))).getName().equals(edeName)) {
												edeTmp = ((IExtendedDataElement)(edeArr.get(j)));
											}
										}
										
										// If it was not found in the array, create it
										if (edeTmp == null) {
											edeTmp = eventFactory.createExtendedDataElement();
											edeTmp.setName(edeName);
											ede.addChildDataElement(edeTmp);
										}
										
										ede = edeTmp;
									}
									
									// Get the next ID for the next Extended Data Element property
									depth += 2;
									property = ((String)path.get(depth));

								}
								else {
									// error - incorrect name
									log(Messages.getString("HyadesGA_CBE_Formatter_Invalid_ExtendedDataElement_Warning",getUniqueID(),property),AdapterSensor.WARN_LEVEL);										
									property = null;
								}
							} /* end while */
						}
						else {
							// error - incorrect name
							log(Messages.getString("HyadesGA_CBE_Formatter_Invalid_CommmonBaseEvent_Warning",getUniqueID(),((String)path.get(2))),AdapterSensor.WARN_LEVEL);										
						}
					}
					else {
						// error - invalid CBE property name string
						log(Messages.getString("HyadesGA_CBE_Formatter_Invalid_Property_Warning",getUniqueID(),((String)path.get(0))),AdapterSensor.WARN_LEVEL);										
					}
				}
				catch(NumberFormatException e) {
					log(Messages.getString("HyadesGA_CBE_Formatter_Non_Numeric_Property_Warning",getUniqueID(),((String)path.get(2))),AdapterSensor.WARN_LEVEL);										
				}
				catch(Exception e) {					
					log(Messages.getString("HyadesGA_CBE_Formatter_Exception_Warning",getUniqueID(),((String)path.get(2)),e.getMessage()),AdapterSensor.WARN_LEVEL);										
				}
			} /* end while */
			cbes[k] = cbe;
		} /* end for */
		
		return cbes;
	}

	/**)
	 * @see org.eclipse.hyades.logging.adapter.IProcessUnit#testProcessEventItems(java.lang.Object[])
	 */
	public Object[] testProcessEventItems(Object[] msgs)
		throws AdapterInvalidConfig {

		return testProcessDirectedGraphs(msgs);
	}

	/** 
	 * Test processing an array of HashMap objects
	 */
	private ICommonBaseEvent[] testProcessDirectedGraphs(Object[] elements) throws AdapterInvalidConfig  {
		if(!(elements[0] instanceof List)) {
			throw new AdapterInvalidConfig("This formatter will only accept arrays of Lists");
		}
		ICommonBaseEvent[] eventArray = new ICommonBaseEvent[1];
		eventArray[0] = eventFactory.createCommonBaseEvent();
		return eventArray;
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.IComponent#update()
	 */
	public void update() throws AdapterInvalidConfig {
		// first get the basic configuration set
		super.update();

	}

	/**
	 * Function to determine IP address and fully-qualified host name
	 * of the machine parsing the log file. 
	 */
	private void getLocalHostInfo() {
		try {
			localHostId = java.net.InetAddress.getLocalHost().getHostAddress();
			if (localHostId.indexOf(':') != -1)
				localHostIdFormat = HOST_ID_FORMAT_IPV6;
			else
				localHostIdFormat = HOST_ID_FORMAT_IPV4;
		}
		catch (Exception e) {
			localHostId = UNKNOWN_HOST;
			localHostIdFormat = NONE;
		}

	}

	/**
	 * @return
	 */
	private String getLocalHostId() {
		if (localHostId == null)
			getLocalHostInfo();
		return localHostId;
	}

	/**
	 * @return
	 */
	private String getLocalHostIdFormat() {
		if (localHostIdFormat == null)
			getLocalHostInfo();
		return localHostIdFormat;
	}

}
