/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.internal.util;
import org.eclipse.hyades.logging.adapter.ICBEGraphInfo;
import org.eclipse.hyades.logging.adapter.MessageGraph;
import org.eclipse.hyades.logging.adapter.impl.DirectedGraphImpl;
import org.eclipse.hyades.logging.adapter.impl.Sensor;
import org.eclipse.hyades.logging.events.ICommonBaseEvent;
import org.eclipse.hyades.logging.events.IComponentIdentification;
/**
 * @author smith
 * 
 * A modified version of AdapterLogger to log MessageGraph objects instead
 * of CBEs. It converts the CBEs passed in to MessageGraph objects.
 * This class is used to test the CBEFormatter.  To use this class, modify
 * org.eclipse.hyades.logging.adapter.impl.Component to use it instead of
 * AdapterLogger as the logger.
 */
public class AdapterGraphLogger extends Sensor implements ICBEGraphInfo
{
	/**
	 * OFF is not a level ever set in an event, however it is a logging level filter that can be
	 * set in order to indicate no logging should take place.
	 */
	public static final short OFF_LEVEL = -1;
	/**
	 * Unknown logging level value. For example random logging in a development environment
	 * or uncategorized logging that should be observed whenever log data is collected.
	 */
	public static final short UNKNOWN_LEVEL = 0;
	/**
	 * Informational logging level value. For example successful activity reporting.
	 */
	public static final short INFO_LEVEL = 10;
	/**
	 * Harmless or detailed info logging level value.
	 */
	public static final short HARMLESS_LEVEL = 20;
	/**
	 * Warning logging level value. For example informational data that may 
	 * require remedial action.
	 */
	public static final short WARN_LEVEL = 30;
	/**
	 * Minor Error logging level value. For example a recoverable error that requires no
	 * immediate remedial action.
	 */
	public static final short MINOR_LEVEL = 40;
	/**
	 * Critical Error logging level value. For example a recoverable but serious error that
	 * likely requires remedial action.
	 */
	public static final short CRITICAL_LEVEL = 50;
	/**
	 * Fatal Error logging level value. For example an unrecoverable error.
	 */
	public static final short FATAL_LEVEL = 60;
	/**
	 * Trace logging level value. For example stack tracing.
	 */
	public static final short TRACE_LEVEL = 70;
	/**
	 * Fine debugging logging level value. For example stack tracing.
	 */
	public static final short FINE_LEVEL = 72;
	/**
	 * Finer debugging logging level value. For example procedural flows and values.
	 */
	public static final short FINER_LEVEL = 74;
	/**
	 * Finest debugging logging level value. For example detailed dumps.
	 */
	public static final short FINEST_LEVEL = 76;
	/**
	 * All logging is enabled.
	 */
	public static final short ALL_LEVEL = 99;
	//
	private int arraySize = 256;
	private MessageGraph[] events = new MessageGraph[arraySize];
	private int lastAddedEvent = -1;
	private int lastRemovedEvent = -1;
	static private AdapterGraphLogger instance = null;
	private boolean availableToRead = true;
	/**
	 * 
	 */
	public AdapterGraphLogger()
	{
		super();
		instance = this;
	}
	/**
	 * 
	 * @return AdapterLogger
	 */
	static public synchronized AdapterGraphLogger getInstance()
	{
		if (instance == null)
			instance = new AdapterGraphLogger();
		return instance;
	}
	/**
	 * 	simulates a getNext
	 * @see com.ibm.acad.general.sensor.ISensor#testGetNext()
	 */
	public synchronized Object[] testGetNext()
	{
		return testGetNextEvent();
	}
	/**
	 * implements the testGetNext behaviour by returning a test MessageGraph
	 * @return org.eclipse.hyades.logging.adapter.MessageGraph[]
	 */
	public synchronized MessageGraph[] testGetNextEvent()
	{
		MessageGraph[] eventArray = new MessageGraph[1];
		eventArray[0] = new MessageGraph();
		return eventArray;
	}
	//TODO: HS increase the granularity of the locks so the impact is smaller
	public Object[] getNext()
	{
//		while (!availableToRead)
//		{
//		}
		return getNextEvent();
	}
	public synchronized MessageGraph[] getNextEvent()
	{
		synchronized (this)
		{
			if (lastAddedEvent == -1 || lastRemovedEvent == lastAddedEvent)
				return null;
			MessageGraph[] outputArray = new MessageGraph[lastAddedEvent - lastRemovedEvent];
			System.arraycopy(events, lastRemovedEvent + 1, outputArray, 0, lastAddedEvent - lastRemovedEvent);
			lastRemovedEvent = lastAddedEvent;
			return outputArray;
		}
	}
	public synchronized void addEvent(ICommonBaseEvent newEvent)
	{
//		availableToRead = false;
		synchronized (this)
		{
			// Create a MessageGraph from the CBE
			MessageGraph msg = new MessageGraph();
			
			DirectedGraphImpl node;
			int[] intarr;
			
			intarr = new int[1];
			node = new DirectedGraphImpl();
			intarr[0] = COMMONBASEEVENT_CREATIONTIME;
			node.setPath(intarr);
			node.setValue(newEvent.getCreationTime());
			msg.addNode(node);
			
			node = new DirectedGraphImpl();
			intarr[0] = COMMONBASEEVENT_SEVERITY;
			node.setPath(intarr);
			node.setValue(String.valueOf((int)newEvent.getSeverity()));
			msg.addNode(node);

			node = new DirectedGraphImpl();
			intarr[0] = COMMONBASEEVENT_MSG;
			node.setPath(intarr);
			node.setValue(newEvent.getMsg());
			msg.addNode(node);

			intarr = new int[2];
			IComponentIdentification compId = newEvent.getSourceComponentId();

			node = new DirectedGraphImpl();
			intarr[0] = COMMONBASEEVENT_SOURCECOMPONENTID;
			intarr[1] = COMPONENTID_LOCATION;
			node.setPath(intarr);
			node.setValue(compId.getLocation());
			msg.addNode(node);			
			
			node = new DirectedGraphImpl();
			intarr[1] = COMPONENTID_LOCATIONTYPE;
			node.setPath(intarr);
			node.setValue(compId.getLocationType());
			msg.addNode(node);
			
			node = new DirectedGraphImpl();
			intarr[1] = COMPONENTID_COMPONENT;
			node.setPath(intarr);
			node.setValue(compId.getComponent());
			msg.addNode(node);	
			
			node = new DirectedGraphImpl();
			intarr[1] = COMPONENTID_COMPONENTIDTYPE;
			node.setPath(intarr);
			node.setValue(compId.getComponentIdType());
			msg.addNode(node);	
			
			node = new DirectedGraphImpl();
			intarr[1] = COMPONENTID_SUBCOMPONENT;
			node.setPath(intarr);
			node.setValue(compId.getSubComponent());
			msg.addNode(node);	
										
			if (lastAddedEvent + 1 == events.length)
				increaseArraySize();
			events[++lastAddedEvent] = msg;
		}
		//TODO: HS remove this System out
//		availableToRead = true;
	}
	
	private synchronized void increaseArraySize()
	{
		//		synchronized (this)
		{
			MessageGraph[] tempArray = new MessageGraph[arraySize * 2];
			arraySize = arraySize * 2;
			compressArrayInto(tempArray);
			events = tempArray;
		}
	}
	private synchronized void compressArrayInto(MessageGraph[] targetArray)
	{
		System.arraycopy(events, lastRemovedEvent + 1, targetArray, 0, lastAddedEvent - lastRemovedEvent);
		lastAddedEvent = lastAddedEvent - lastRemovedEvent;
		lastRemovedEvent = -1;
	}
}
