/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

//==============================================================================
// ProbeKit.cpp
//------------------------------------------------------------------------------
// External representation for the probe kit.
//==============================================================================
#include "CommonDef.h"

#define _BCIENGINTERFACE_DLL_EXPORT // Make the "C" interface exported
#pragma warning(disable:4786)
#include "BCIEngProbe.h"
#include "BCIEngProbeInterface.h"


//------------------------------------------------------------------------------
inline static unsigned ExceptionToError(CBCIEngProbeInterfaceException& ex)
{
	unsigned uErr = ex.GetReason();
	uErr <<= 16;
	uErr |= ex.GetSource();
	return uErr;
}

//==============================================================================
// Implementation of BCI Engine interface for the Probe Kit
//


//------------------------------------------------------------------------------
void 
CBCIEngProbeInterface::GetEngVersion(unsigned& o_uVersion)
{
	o_uVersion = BCI_ENG_PROBE_1_0;
}
	
//------------------------------------------------------------------------------
void
CBCIEngProbeInterface::GetEngDescription(CSTR& o_szDescription)
{
	o_szDescription = "BCI Engine for Probe Kit. Version 1.0";
}

//------------------------------------------------------------------------------

// Helper func for CBCIEngProbeInterface::Initialize
//
// Armed with a buffer full of text, we want to split it along the
// newline boundaries by stomping on the newlines. Sounds simple, except 
// the text may have been generated on one OS and parsed on another, so
// you can readily tell where the line terminators are. For example, on
// Windows, it is CR/LF, on most Unix systems it is CR, on Mac it is LF.
//
// Since you don't know the source, you can't reliably handle the problem.
// This function is a hack that will work for Windows, most Unix varieties
// and Macintosh. It will definitely NOT work with non-ASCII char sets or
// for platforms that don't use some combination of ASCII CR and LF chars
// for line terminators. Hackety-hack.
//
// When we find either a CR or LF (\n or \r, on ASCII systems), we replace
// them with a null byte. This is OK because we control the input stream
// enough to know there won't be random occurences of these chars. If they're
// there at all, they're the line terminators. Also, the calling function,
// CBCIEngProbeInterface::Initialize, treats extra null bytes as if they're 
// blank lines in the input file.

#if defined(_WINDOWS_) || defined(SOLARIS) || defined(HPUX) || \
    (defined(LINUX) && defined(IA32_ARCH))
static void 
break_into_lines(char *buf, size_t len)
{
	size_t i;

	if ( buf == NULL || *buf == '\0' ) {
		return;
	}

        for (i = 0; i < len; i++) {
		if ( buf[i] == '\n' || buf[i] == '\r' ) {
			buf[i] = '\0';
		}
	}
}
#else
#  error "Platform-specific configuration required"
#endif

//------------------------------------------------------------------------------
void 
CBCIEngProbeInterface::Initialize(CSTR i_pchOptions, size_t i_cbOptions)
{
	CProbe* pprobe = NULL;
	char* szRecType;
	char* szProbePackage;
	char* szProbeClass;
	char* szProbeMethod;
	char* szProbeSignature;
	char* szProbeAction;
	char* szRefType;
	char* szRefClass;
	char* szRefMethod;
	char* szRefSignature;
	char* szRefArgs;
	char* szDummy;
	CProbeFragment::fragmentType_t fragType;
	size_t  cbScan = 0;
	char*	szScan = (char*)i_pchOptions;
	int n = 0;

    break_into_lines(szScan, i_cbOptions);
	for(;cbScan < i_cbOptions; szScan += n + 1, cbScan += n + 1)
	{
		n = strlen(szScan);
		if (n == 0) {
			// blank line? continue
			continue;
		}
		szRecType = strtok(szScan, " \t");
		if (szRecType == NULL) {
			// blank line, continue
			continue;
		}

		if(stricmp(szRecType, "REM") == 0)
		{
			// Comment. Continue the line-reading loop
			continue;
		}
		else if(stricmp(szRecType, "PROBE") == 0)
		{
			// Start of new probe.
			// Flush out the old one being read, if any
			if(NULL != pprobe)
			{
				m_peng.AddProbe(pprobe);
				pprobe = NULL;
			}

			szDummy = strtok(NULL, " \t");
			if (szDummy != NULL) {
				CBCIEngProbeInterfaceException ex(CBCIEngProbeInterfaceException::EXSRC_INTERFACE,
					                              CBCIEngProbeInterfaceException::EX_BAD_PROBE_ARGS);
				throw ex; //"Bad PROBE line, too many parameters";
			}

			pprobe = m_peng.CreateProbe();
		}
		else if (stricmp(szRecType, "RULE") == 0)
		{
			szProbePackage = strtok(NULL, " \t");
			szProbeClass = strtok(NULL, " \t");
			szProbeMethod = strtok(NULL, " \t");
			szProbeSignature = strtok(NULL, " \t");
			szProbeAction = strtok(NULL, " \t");
			szDummy = strtok(NULL, " \t");

			if (szProbeAction == NULL) {
				// string ended before reading the action token
				CBCIEngProbeInterfaceException ex(CBCIEngProbeInterfaceException::EXSRC_INTERFACE,
					                              CBCIEngProbeInterfaceException::EX_BAD_PROBE_ARGS);
				throw ex; //"Bad RULE line, not enough parameters";
			}
			if (szDummy != NULL) {
				throw "Bad RULE line, too many parameters";
			}

			CFilterRule::action_t action;
			if (stricmp(szProbeAction, "include") == 0) {
				action = CFilterRule::ACTION_INCLUDE;
			}
			else if (stricmp(szProbeAction, "exclude") == 0) {
				action = CFilterRule::ACTION_EXCLUDE;
			}
			else {
				CBCIEngProbeInterfaceException ex(CBCIEngProbeInterfaceException::EXSRC_INTERFACE,
					                              CBCIEngProbeInterfaceException::EX_INVALID_FILTER_TYPE);
				throw ex; //"Bad action string in RULE line, must be include or exclude";
			}
			pprobe->AddFilterRule(szProbePackage, szProbeClass, szProbeMethod, szProbeSignature, action);
		}
		else if(NULL != pprobe && stricmp(szRecType, "REF") == 0)
		{
			// Pick off the type (entry, exit, catch) and the class, method, and signature,
			// plus the argument list - which will be NULL if it's missing
			szRefType = strtok(NULL, " \t");
			szRefClass = strtok(NULL, " \t");
			szRefMethod = strtok(NULL, " \t");
			szRefSignature = strtok(NULL, " \t");
			szRefArgs = strtok(NULL, " \t");		// might be NULL if this ref takes no args
			szDummy = strtok(NULL, " \t");

			if (szRefArgs == NULL) szRefArgs = "";
			if (szRefSignature == NULL) {
				CBCIEngProbeInterfaceException ex(CBCIEngProbeInterfaceException::EXSRC_INTERFACE,
					                              CBCIEngProbeInterfaceException::EX_BAD_PROBE_ARGS);
				throw ex; //"Bad REF line, not enough parameters";
			}
			if (szDummy != NULL) {
				CBCIEngProbeInterfaceException ex(CBCIEngProbeInterfaceException::EXSRC_INTERFACE,
					                              CBCIEngProbeInterfaceException::EX_BAD_PROBE_ARGS);
				throw ex; //"Bad REF line, too many parameters";
			}

			if(stricmp(szRefType, "ONENTRY") == 0)
				fragType = CProbeFragment::PROBE_ONENTRY;
			else if(stricmp(szRefType, "ONEXIT") == 0)
				fragType = CProbeFragment::PROBE_ONEXIT;
			else if(stricmp(szRefType, "ONCATCH") == 0)
				fragType = CProbeFragment::PROBE_ONCATCH;
			else if(stricmp(szRefType, "BEFORECALL") == 0)
				fragType = CProbeFragment::PROBE_BEFORECALL;
			else if(stricmp(szRefType, "AFTERCALL") == 0)
				fragType = CProbeFragment::PROBE_AFTERCALL;
			else {
				CBCIEngProbeInterfaceException ex(CBCIEngProbeInterfaceException::EXSRC_INTERFACE,
													CBCIEngProbeInterfaceException::EX_BAD_PROBE_ARGS);
				throw ex;
			}
			pprobe->AddFragment(fragType, szRefClass, szRefMethod, szRefSignature, "static", szRefArgs);
		}
		// else some unknown opcode or something, just ignore
	}

	// End of file - write out the last probe we were scanning, if any
	if (pprobe != NULL) {
		m_peng.AddProbe(pprobe);
	}

}


//------------------------------------------------------------------------------
void 
CBCIEngProbeInterface::Instrument(void* i_pInClass, size_t i_cbInClass, 
		                          void** o_ppOutClass, size_t* o_pcbOutClass)
{
	try
	{
		m_peng.Instrument(i_pInClass, i_cbInClass, o_ppOutClass, o_pcbOutClass);
	}
	catch (CJClassFileException& i_ex)
	{
		CBCIEngProbeInterfaceException ex(CBCIEngProbeInterfaceException::EXSRC_JCLASSFILE,
					                      i_ex.GetReason());
		throw ex;
	}
	catch (CBCIEngProbeException& i_ex) 
	{
		CBCIEngProbeInterfaceException ex(CBCIEngProbeInterfaceException::EXSRC_BCIENG,
					                      i_ex.GetReason(),
										  i_ex.GetStringReason());
		throw ex;
	}
	catch (CBCIEngException& i_ex)
	{
		CBCIEngProbeInterfaceException ex(CBCIEngProbeInterfaceException::EXSRC_BCIENG,
					                      i_ex.GetReason());
		throw ex;
	}
	catch (CModuleException& i_ex)
	{
		CBCIEngProbeInterfaceException ex(CBCIEngProbeInterfaceException::EXSRC_MODULE,
					                      i_ex.GetReason());
		throw ex;
	}
}

//------------------------------------------------------------------------------


//==============================================================================
// Plain vanilla C implementation of the probe kit interface
//

//------------------------------------------------------------------------------
unsigned GetEngVersion(pbcieng_t i_pbcieng, unsigned* o_puVersion)
{
	unsigned uErr = 0;
	try
	{
		CBCIEngProbeInterface* peng = (CBCIEngProbeInterface*)i_pbcieng;
		peng->GetEngVersion(*o_puVersion);
	}
	catch(...)
	{
	}
	return uErr;
}

//------------------------------------------------------------------------------
unsigned GetEngDescription(pbcieng_t i_pbcieng, char** o_szDescription)
{
	unsigned uErr = 0;
	try
	{
		CBCIEngProbeInterface* peng = (CBCIEngProbeInterface*)i_pbcieng;
		peng->GetEngDescription(*(const char**)o_szDescription);
	}
	catch(...)
	{
	}
	return uErr;
}

//------------------------------------------------------------------------------
unsigned Initialize(pbcieng_t i_pbcieng, CSTR i_pchOptions, size_t i_cbOptions)
{
	unsigned uErr = 0;
	try
	{
		CBCIEngProbeInterface* peng = (CBCIEngProbeInterface*)i_pbcieng;
		peng->Initialize(i_pchOptions, i_cbOptions);
	}
	catch (CBCIEngProbeInterfaceException& ex)
	{
		uErr = ExceptionToError(ex);
	}
	return uErr;
}

//------------------------------------------------------------------------------
unsigned Instrument(pbcieng_t i_pbcieng, void* i_pInClass, size_t i_cbInClass, 
		            void** o_ppOutClass, size_t* o_pcbOutClass)
{
	unsigned uErr = 0;
	try
	{
		CBCIEngProbeInterface* peng = (CBCIEngProbeInterface*)i_pbcieng;
		peng->Instrument(i_pInClass, i_cbInClass, o_ppOutClass, o_pcbOutClass);
	}
	catch (CBCIEngProbeInterfaceException& ex)
	{
		uErr = ExceptionToError(ex);
	}
	return uErr;
}

//------------------------------------------------------------------------------
unsigned SetAllocator(pbcieng_t i_pbcieng, pfnMalloc_t i_pfnMalloc)
{
	unsigned uErr = 0;
	CBCIEngProbeInterface* peng = (CBCIEngProbeInterface*)i_pbcieng;
	peng->SetAllocator(i_pfnMalloc);
	return uErr;
}

//------------------------------------------------------------------------------
unsigned SetCallback(pbcieng_t i_pbcieng, pfnCallback_t i_pfnCallback, unsigned i_uFlags)
{
	unsigned uErr = 0;
	CBCIEngProbeInterface* peng = (CBCIEngProbeInterface*)i_pbcieng;
	peng->SetCallback(i_pfnCallback, i_uFlags);
	return uErr;
}

//------------------------------------------------------------------------------
unsigned CreateBCIEngine(pbcieng_t* o_eng)
{
	unsigned uErr = 0;
	*o_eng = (pbcieng_t)new CBCIEngProbeInterface;
	return uErr;
}

//------------------------------------------------------------------------------
unsigned DestroyBCIEngine(pbcieng_t i_eng)
{
	delete (CBCIEngProbeInterface*)i_eng;
	return 0;
}
//= End of ProbeKit.cpp ========================================================
