/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.test.common.internal.wizard;

import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.common.configuration.CFGArtifact;
import org.eclipse.hyades.models.common.configuration.CFGArtifactLocationPair;
import org.eclipse.hyades.models.common.configuration.CFGComparableProperty;
import org.eclipse.hyades.models.common.configuration.CFGInstance;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.CFGMachineConstraint;
import org.eclipse.hyades.models.common.configuration.CFGPropertyGroup;
import org.eclipse.hyades.models.common.configuration.Common_ConfigurationFactory;
import org.eclipse.hyades.models.common.interactions.BVRProperty;
import org.eclipse.hyades.models.common.interactions.Common_Behavior_InteractionsFactory;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.test.common.TestCommonPlugin;
import org.eclipse.hyades.test.common.internal.util.ContextIds;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.ui.internal.util.CoreUtil;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.hyades.ui.internal.util.JavaUtil;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnPixelData;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.help.WorkbenchHelp;

public class EnvironmentVariableViewer implements SelectionListener, IDisposable
{
	private final static String CLASSPATH = "CLASSPATH";
	
	private static class PreferenceContentProvider implements IStructuredContentProvider
	{
		/**
		 * @see IStructuredContentProvider#getElements(Object)
		 */
		public Object[] getElements(Object element)
		{
			if(element instanceof CFGArtifact)
			{
				CFGPropertyGroup propertyGroup = (CFGPropertyGroup)((CFGArtifact)element).getPropertyGroups().get(0);
				return propertyGroup.getProperties().toArray();
			}

			return new Object[0];
		}
		/**
		 * @see IContentProvider#dispose()
		 */
		public void dispose()
		{
		}
		/**
		 * @see IContentProvider#inputChanged(Viewer, Object, Object)
		 */
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
		{
		}
	}
	private static class PreferenceLabelProvider extends LabelProvider implements ITableLabelProvider
	{
		/**
		 * @see ITableLabelProvider#getColumnImage(Object, int)
		 */
		public Image getColumnImage(Object element, int columnIndex)
		{

			return null;
		}
		/**
		 * @see ITableLabelProvider#getColumnText(Object, int)
		 */
		public String getColumnText(Object element, int columnIndex)
		{

			if (element instanceof BVRProperty)
			{
				BVRProperty myTableElement = (BVRProperty)element;

				if (columnIndex == 0)
					return myTableElement.getName();

				if (columnIndex == 1)
					return myTableElement.getValue();

			}
			return "";
		}
	}
	public static class EnvironmentDialog extends Dialog implements Listener
	{
		private BVRProperty element;
		private EnvironmentUI environmentUI;

		public EnvironmentDialog(Shell shell, BVRProperty element)
		{
			super(shell);
			this.element = element;
		}

		protected Control createDialogArea(Composite parent)
		{

			if (element == null)
				getShell().setText(TestUIPlugin.getString("ADD_ENV_DESC_DLG"));
			else
				getShell().setText(TestUIPlugin.getString("EDIT_ENV_DESC_DLG"));

			Composite result = new Composite(parent, SWT.NULL);
			result.setLayoutData(GridDataUtil.createFill());
			result.setLayout(new GridLayout());

			environmentUI = new EnvironmentUI(getShell());
			Composite content = environmentUI.createControl(result);

			if (element != null)
			{
				environmentUI.setName(element.getName());
				environmentUI.setValue(element.getValue());
				environmentUI.setDescription(element.getDescription());
			}

			WorkbenchHelp.setHelp(content, TestCommonPlugin.getID() + ContextIds.RUN_ENV_VIEWER_DLG);								

			return content;
		}

		protected Control createButtonBar(Composite parent)
		{
			Control control = super.createButtonBar(parent);
			environmentUI.registerListener(this);

			if (element == null)
				getButton(IDialogConstants.OK_ID).setEnabled(false);

			return control;

		}

		protected void okPressed()
		{

			if (element == null)
				element = Common_Behavior_InteractionsFactory.eINSTANCE.createBVRProperty();

			element.setName(environmentUI.getName());
			element.setValue(environmentUI.getValue());
			element.setDescription(environmentUI.getDescription());
			super.okPressed();
		}

		public BVRProperty getEnvironmentElement()
		{
			return element;

		}

		public void handleEvent(Event e)
		{
			getButton(IDialogConstants.OK_ID).setEnabled(!environmentUI.getName().equals("") && !environmentUI.getValue().equals(""));
		}
	}
	private static class EnvironmentUI implements SelectionListener, ModifyListener
	{
		private Text _name;
		private Text _classpath;
		private Text _description;
		private Button _browseClass;
		private Button _browseDirPath;
		private Button _browseJarPath;
		private Shell shell;
		private Listener _listener;

		public EnvironmentUI(Shell shell)
		{
			super();
			this.shell = shell;
		}
		/**
		 * Insert the method's description here.
		 * Creation date: (8/10/2001 12:39:10 PM)
		 * @return org.eclipse.swt.widgets.Composite
		 * @param parent org.eclipse.swt.widgets.Composite
		 */
		public Composite createArgumentsControl(Composite parent)
		{
			GridData gd;

			Label label = new Label(parent, SWT.NULL);
			gd = new GridData();
			gd.verticalAlignment = GridData.BEGINNING;
			label.setText(TestUIPlugin.getString("ENV_VALUE"));
			label.setLayoutData(gd);

			_classpath = new Text(parent, SWT.BORDER | SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.WRAP);
			gd = GridDataUtil.createFill();
			gd.heightHint = 100;
			gd.widthHint = 250;
			_classpath.setLayoutData(gd);

			Composite btnGroup = new Composite(parent, SWT.NULL);
			gd = GridDataUtil.createVerticalFill();
			GridLayout layout = new GridLayout();
			layout.numColumns = 1;
			btnGroup.setLayoutData(gd);
			btnGroup.setLayout(layout);

			_browseDirPath = new Button(btnGroup, SWT.CENTER | SWT.PUSH);

			gd = new GridData();
			gd.horizontalAlignment = GridData.FILL;
			gd.verticalAlignment = GridData.FILL;
			_browseDirPath.setLayoutData(gd);

			_browseDirPath.setText(TestUIPlugin.getString("ADD_DIRECTORY"));
			_browseJarPath = new Button(btnGroup, SWT.CENTER | SWT.PUSH);

			gd = new GridData();
			gd.horizontalAlignment = GridData.FILL;
			gd.verticalAlignment = GridData.VERTICAL_ALIGN_BEGINNING;

			_browseJarPath.setLayoutData(gd);

			_browseJarPath.setText(TestUIPlugin.getString("ADD_JAR"));

			//add listeners
			_browseDirPath.addSelectionListener(this);
			_browseJarPath.addSelectionListener(this);

			WorkbenchHelp.setHelp(_classpath, TestCommonPlugin.getID() + ContextIds.RUN_ENV_VIEWER_DLG_C);								
			//WorkbenchHelp.setHelp(_browseClass, CommonPlugin.getPluginId() + ContextIds.RUN_ENV_VIEWER_DLG_BC);								
			WorkbenchHelp.setHelp(_browseDirPath, TestCommonPlugin.getID() + ContextIds.RUN_ENV_VIEWER_DLG_BD);								
			WorkbenchHelp.setHelp(_browseJarPath, TestCommonPlugin.getID() + ContextIds.RUN_ENV_VIEWER_DLG_BJ);								

			return parent;
		}
		public Composite createControl(Composite parent)
		{
			Label label;

			Composite content = new Composite(parent, SWT.NULL);
			GridLayout layout = new GridLayout();
			layout.numColumns = 3;
			layout.verticalSpacing = 10;
			layout.horizontalSpacing = 5;
			content.setLayout(layout);
			GridData gd = GridDataUtil.createFill();
			gd.horizontalIndent = 5;
			content.setLayoutData(gd);

			label = new Label(content, SWT.NULL);
			label.setText(TestUIPlugin.getString("ENV_NAME"));
			_name = new Text(content, SWT.SINGLE | SWT.BORDER);
			gd = GridDataUtil.createHorizontalFill();
			gd.horizontalSpan = 2;
			_name.setLayoutData(gd);

			_name.setFocus();
			_name.selectAll();
			_name.addModifyListener(this);

			createArgumentsControl(content);
			_classpath.addModifyListener(this);

			label = new Label(content, SWT.NULL);
			label.setText(TestUIPlugin.getString("ENV_DESCRIPTION"));
			gd = new GridData();
			gd.verticalAlignment = GridData.BEGINNING;
			label.setLayoutData(gd);
			_description = new Text(content, SWT.BORDER | SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.WRAP);
			gd = GridDataUtil.createHorizontalFill();
			gd.heightHint = 100;
			_description.setLayoutData(gd);

			WorkbenchHelp.setHelp(_description, TestCommonPlugin.getID() + ContextIds.RUN_ENV_VIEWER_DLG_D);								
			WorkbenchHelp.setHelp(_name, TestCommonPlugin.getID() + ContextIds.RUN_ENV_VIEWER_DLG_N);								

			return content;
		}
		/**
		 * Insert the method's description here.
		 * Creation date: (5/8/2001 12:25:04 PM)
		 * @return java.lang.String
		 */
		public String getValue()
		{
			return _classpath.getText().trim();
		}
		public Control getBrowseJarButton()
		{
			return _browseJarPath;
		}

		public Control getBrowseDirButton()
		{
			return _browseDirPath;
		}

		public String getName()
		{
			return _name.getText().trim();
		}

		public String getDescription()
		{
			return _description.getText().trim();
		}

		public void widgetDefaultSelected(SelectionEvent event)
		{
		}
		/**
		 * Insert the method's description here.
		 * Creation date: (08/21/2000 7:49:33 PM)
		 * @param e com.ibm.swt.events.SelectionEvent
		 */
		public void widgetSelected(SelectionEvent e)
		{
			String osName = System.getProperty("os.name");

			if (e.widget == _browseClass)
			{
				FileDialog dlg = new FileDialog(shell);
				dlg.setFilterExtensions(new String[] { "*.class" });
				dlg.open();

				String fileName = dlg.getFileName();
				if (fileName != null && !fileName.equals(""))
				{
					if (fileName.endsWith(".class"))
						fileName = fileName.substring(0, fileName.length() - 6);

					String classpath = dlg.getFilterPath();
					if ((osName != null) && (osName.startsWith("Windows")) && (classpath.endsWith(":")))
					{
						classpath += "\\";
					}

					String oldClassPath = _classpath.getText().trim();
					oldClassPath = oldClassPath.equals("") || oldClassPath.endsWith(File.pathSeparator) ? oldClassPath : oldClassPath + File.pathSeparator;
					_classpath.setText(oldClassPath + classpath);

				}

			}
			else if (e.widget == _browseDirPath)
			{
				DirectoryDialog dlg = new DirectoryDialog(shell);
				String classpath = dlg.open();

				if (classpath != null && !classpath.equals(""))
				{

					if ((osName != null) && (osName.startsWith("Windows")) && (classpath.endsWith(":")))
					{
						classpath += "\\";
					}

					String oldClassPath = _classpath.getText().trim();
					oldClassPath = oldClassPath.equals("") || oldClassPath.endsWith(File.pathSeparator) ? oldClassPath : oldClassPath + File.pathSeparator;
					classpath = oldClassPath + classpath;
					_classpath.setText(classpath);
					_classpath.setFocus();
				}

			}
			else if (e.widget == _browseJarPath)
			{
				FileDialog dlg = new FileDialog(shell);
				dlg.setFilterExtensions(new String[] { "*.jar;*.zip" });
				dlg.open();

				String classpath = dlg.getFileName();

				if (classpath != null && !classpath.equals(""))
				{
					classpath = dlg.getFilterPath() + File.separator + classpath;

					String oldClassPath = _classpath.getText().trim();
					oldClassPath = oldClassPath.equals("") || oldClassPath.endsWith(File.pathSeparator) ? oldClassPath : oldClassPath + File.pathSeparator;
					classpath = oldClassPath + classpath;
					_classpath.setText(classpath);
				}
			}
		}

		/**
		 * Sent when the text is modified.
		 *
		 * @param e an event containing information about the modify
		 */
		public void modifyText(ModifyEvent e)
		{
			notifyListener();
		}

		/**
		 * @param text
		 */
		public void setValue(String text)
		{
			if(text != null)
				_classpath.setText(text);
			else
				_classpath.setText("");
		}

		/**
		 * @param text
		 */
		public void setName(String text)
		{
			if(text != null)
				_name.setText(text);
			else
				_name.setText("");
		}

		public void setDescription(String text)
		{
			if(text != null)
				_description.setText(text);
			else
				_description.setText("");
		}

		public void registerListener(Listener listener)
		{
			_listener = listener;
		}

		public void notifyListener()
		{

			if (_listener != null)
				_listener.handleEvent(null);
		}

	}

	private final static String[] COLUMNS = { "STRING", "STRING" };

	private TableViewer tableViewer;
	private Button btnAdd;
	private Button btnRename;
	private Button btnRemove;

	private Button btnUp;
	private Button btnDown;

	private TPFTestSuite testSuite;
	private CFGArtifact testSuiteArtifact;
	
	private boolean calculatedClasspath = false;

	public EnvironmentVariableViewer(TPFTest test)
	{
		if (test instanceof TPFTestSuite)
			testSuite = (TPFTestSuite)test;
		else if (test instanceof TPFTestCase)
			testSuite = ((TPFTestCase)test).getTestSuite();		
	}

	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		testSuite = null;
		testSuiteArtifact = null;
	}

	public void createControl(Composite parent)
	{
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayoutData(GridDataUtil.createFill());
		GridLayout gridLayout = new GridLayout();
		gridLayout.marginHeight = 0;
		gridLayout.marginWidth = 0;
		composite.setLayout(gridLayout);

		createTable(composite);
		
		WorkbenchHelp.setHelp(composite, TestCommonPlugin.getID() + ContextIds.RUN_ENV_VIEWER);
		WorkbenchHelp.setHelp(btnAdd, TestCommonPlugin.getID() + ContextIds.RUN_ENV_VIEWER_ADD);
		WorkbenchHelp.setHelp(btnDown, TestCommonPlugin.getID() + ContextIds.RUN_ENV_VIEWER_DOWN);
		WorkbenchHelp.setHelp(btnRemove, TestCommonPlugin.getID() + ContextIds.RUN_ENV_VIEWER_DEL);
		WorkbenchHelp.setHelp(btnRename, TestCommonPlugin.getID() + ContextIds.RUN_ENV_VIEWER_EDIT);
		WorkbenchHelp.setHelp(btnUp, TestCommonPlugin.getID() + ContextIds.RUN_ENV_VIEWER_UP);
		WorkbenchHelp.setHelp(tableViewer.getControl(), TestCommonPlugin.getID() + ContextIds.RUN_ENV_VIEWER_TABLE);

	}

	private void createButtons(Composite parent)
	{
		btnAdd = new Button(parent, SWT.PUSH);
		btnAdd.setText(TestUIPlugin.getString("label.Add"));
		GridData data = GridDataUtil.createHorizontalFill();
		btnAdd.setLayoutData(data);
		btnAdd.setEnabled(false);

		btnRename = new Button(parent, SWT.PUSH);
		btnRename.setText(TestUIPlugin.getString("label.Update"));
		data = GridDataUtil.createHorizontalFill();
		btnRename.setLayoutData(data);
		btnRename.setEnabled(false);

		btnRemove = new Button(parent, SWT.PUSH);
		btnRemove.setText(TestUIPlugin.getString("label.Remove"));
		data = GridDataUtil.createHorizontalFill();
		btnRemove.setLayoutData(data);
		btnRemove.setEnabled(false);

		btnUp = new Button(parent, SWT.PUSH);
		btnUp.setText(TestUIPlugin.getString("LBL_UP"));
		data = GridDataUtil.createHorizontalFill();
		data.verticalAlignment = GridData.VERTICAL_ALIGN_END;
		btnUp.setLayoutData(data);
		btnUp.setEnabled(false);

		btnDown = new Button(parent, SWT.PUSH);
		btnDown.setText(TestUIPlugin.getString("LBL_DOWN"));
		data = GridDataUtil.createHorizontalFill();
		data.verticalAlignment = GridData.VERTICAL_ALIGN_END;
		btnDown.setLayoutData(data);
		btnDown.setEnabled(false);
	}

	private Control createTable(Composite parent)
	{
		Composite composite = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		layout.verticalSpacing = 10;
		layout.numColumns = 2;
		composite.setLayout(layout);
		composite.setLayoutData(GridDataUtil.createFill());

		createTableViewer(composite);

		Composite buttons = new Composite(composite, SWT.NONE);
		GridData data = new GridData();
		data.horizontalSpan = 2;
		data = new GridData(GridData.FILL_VERTICAL);
		buttons.setLayoutData(data);
		layout = new GridLayout();
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		layout.numColumns = 1;
		buttons.setLayout(layout);

		createButtons(buttons);

		btnAdd.addSelectionListener(this);
		btnRename.addSelectionListener(this);
		btnRemove.addSelectionListener(this);
		btnUp.addSelectionListener(this);
		btnDown.addSelectionListener(this);

		return composite;
	}

	private void createTableViewer(Composite parent)
	{
		// Create the table viewer.
		tableViewer = new TableViewer(parent, SWT.BORDER | SWT.SINGLE | SWT.FULL_SELECTION);

		// Create the table control.
		Table table = tableViewer.getTable();
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		GridData data = GridDataUtil.createFill();
		table.setLayoutData(data);
		table.computeSize(SWT.DEFAULT, SWT.DEFAULT);

		TableLayout tableLayout = new TableLayout();
		CellEditor[] cellEditors = new CellEditor[COLUMNS.length];

		TableColumn nameColumn = new TableColumn(table, SWT.LEFT);
		nameColumn.setText(TestUIPlugin.getString("ENV_NAME"));
		tableLayout.addColumnData(new ColumnPixelData(100));
		cellEditors[0] = new TextCellEditor(table);

		TableColumn valueColumn = new TableColumn(table, SWT.LEFT);
		valueColumn.setText(TestUIPlugin.getString("ENV_VALUE"));
		tableLayout.addColumnData(new ColumnPixelData(230));
		cellEditors[1] = new TextCellEditor(table);

		table.setLayout(tableLayout);

		// Adjust the table viewer.
		tableViewer.setColumnProperties(COLUMNS);
		tableViewer.setContentProvider(new PreferenceContentProvider());
		tableViewer.setLabelProvider(new PreferenceLabelProvider());
		tableViewer.getTable().addSelectionListener(this);
	}
	
	public boolean isCalculatedClasspath()
	{
		return calculatedClasspath;
	}

	public void widgetDefaultSelected(SelectionEvent e)
	{
		widgetSelected(e);
	}
	
	public void widgetSelected(SelectionEvent event)
	{
		CFGPropertyGroup propertyGroup = (CFGPropertyGroup)testSuiteArtifact.getPropertyGroups().get(0);
		EList properties = propertyGroup.getProperties();
		
		if (event.widget == btnAdd)
		{
			EnvironmentDialog dlg = new EnvironmentDialog(btnAdd.getShell(), null);
			if(dlg.open() == Window.OK)
			{
				BVRProperty property = dlg.getEnvironmentElement();
				CFGComparableProperty comparableProperty = Common_ConfigurationFactory.eINSTANCE.createCFGComparableProperty();
				comparableProperty.setName(property.getName());
				comparableProperty.setValue(property.getValue());
				comparableProperty.setDescription(property.getDescription());
				properties.add(comparableProperty);
				tableViewer.add(property);
				tableViewer.setSelection(new StructuredSelection(property));
				calculatedClasspath = false;
			}
		}

		else if (event.widget == btnRemove)
		{
			int index = tableViewer.getTable().getSelectionIndex();
			if(index >= 0)
			{
				if(properties.remove(index) != null)
				{
					tableViewer.getTable().remove(index);
					index = (index<properties.size())?index:(properties.size()-1);
					if(index >= 0)
						tableViewer.getTable().select(index);
					calculatedClasspath = false;
				}
			}
		}

		else if (event.widget == btnRename)
		{
			IStructuredSelection structuredSelection = (IStructuredSelection)tableViewer.getSelection();
			if (structuredSelection.size() > 0)
			{
				Object obj = structuredSelection.getFirstElement();
				if (obj instanceof BVRProperty)
				{
					EnvironmentDialog dlg = new EnvironmentDialog(btnRename.getShell(), (BVRProperty)obj);
					dlg.open();
					if (dlg.getReturnCode() == Window.OK)
						tableViewer.update(obj, null);
					calculatedClasspath = false;
				}
			}
		}

		else if (event.widget == btnUp)
		{
			int index = tableViewer.getTable().getSelectionIndex();
			if(index > 0)
			{
				CoreUtil.swap(properties, index, index-1);
				tableViewer.refresh();
				tableViewer.getTable().select(index-1);
				calculatedClasspath = false;
			}
		}

		else if (event.widget == btnDown)
		{
			int index = tableViewer.getTable().getSelectionIndex();
			if(index < (properties.size()-1))
			{
				CoreUtil.swap(properties, index, index+1);
				tableViewer.refresh();
				tableViewer.getTable().select(index+1);
				calculatedClasspath = false;
			}
		}

		btnAdd.setEnabled(testSuiteArtifact != null);
		
		boolean enable = 	(testSuiteArtifact != null) 
							&& (!tableViewer.getSelection().isEmpty());
		btnRemove.setEnabled(enable);
		btnRename.setEnabled(enable);
		btnUp.setEnabled(enable && (tableViewer.getTable().getSelectionIndex() > 0));
		btnDown.setEnabled(enable && (tableViewer.getTable().getSelectionIndex() < (properties.size()-1)));
	}

	public void setInput(TPFDeployment deployment)
	{
		if(calculatedClasspath && (testSuiteArtifact != null))
		{
			CFGPropertyGroup propertyGroup = (CFGPropertyGroup)testSuiteArtifact.getPropertyGroups().get(0);
			for(int i = 0; i < propertyGroup.getProperties().size(); i++)
			{
				propertyGroup.getProperties().remove(0);	
			}	
		}
			
		testSuiteArtifact = calculateTestSuiteArtifact(deployment);
		
		
		calculateClasspath(CALC_CP_ABORT);
		tableViewer.setInput(testSuiteArtifact);
		
		btnAdd.setEnabled(testSuiteArtifact != null);

		boolean enable = 	(testSuiteArtifact != null) 
							&& (!tableViewer.getSelection().isEmpty());
		btnDown.setEnabled(enable);
		btnRemove.setEnabled(enable);
		btnRename.setEnabled(enable);
		btnUp.setEnabled(enable);
	}

	public CFGArtifact calculateTestSuiteArtifact(TPFDeployment deployment)
	{
		if (deployment == null)
			return null;

		if (testSuite == null)
			return null;

		if (testSuite.eResource().getResourceSet() != deployment.eResource().getResourceSet())
			return null;

		CFGArtifact artifact = testSuiteArtifact;
		// TODO: Hack...Using first pair
		CFGArtifactLocationPair pair = (CFGArtifactLocationPair)deployment.getArtifactLocations().get(0); 
		if ((artifact != null) && pair != null && (pair.getArtifact().equals(artifact)))
			return artifact;

		artifact = null;
		if(pair == null)
			pair = Common_ConfigurationFactory.eINSTANCE.createCFGArtifactLocationPair();
		
		if(pair.getArtifact() == null)
		{
			artifact = (CFGArtifact)deployment.getArtifacts().get(0);
			if(artifact == null)
			{
				artifact = Common_ConfigurationFactory.eINSTANCE.createCFGArtifact();
				CFGPropertyGroup propertyGroup = Common_ConfigurationFactory.eINSTANCE.createCFGPropertyGroup();
				artifact.getPropertyGroups().add(propertyGroup);
				deployment.getArtifacts().add(artifact);
			}
			pair.setArtifact(artifact);		
		}
		else
		{
			artifact = pair.getArtifact();
		}

		if(pair.getLocation() == null)
		{
			// TODO: Hack...Using first location
			CFGLocation location = (CFGLocation)deployment.getRefLocations().get(0);
			if(location == null)
			{
				location = Common_ConfigurationFactory.eINSTANCE.createCFGMachineConstraint();
				((CFGMachineConstraint)location).setHostname("localhost");
				deployment.getRefLocations().add(location);
			}
			pair.setLocation(location);
		}
		CFGInstance testSuiteInstance = null;
		if(testSuite.getInstances().size() == 0)
		{
			testSuiteInstance = Common_ConfigurationFactory.eINSTANCE.createCFGInstance();
			testSuite.getInstances().add(testSuiteInstance);
		}
		else
		{
			boolean found = false;
			for(int i = 0; i < testSuite.getInstances().size(); i++)
			{
				testSuiteInstance = (CFGInstance) testSuite.getInstances().get(i);
				if(artifact.getInstances().contains(testSuiteInstance))
					found = true;
			}
			if(!found)
			{			
				testSuiteInstance = Common_ConfigurationFactory.eINSTANCE.createCFGInstance();
				testSuite.getInstances().add(testSuiteInstance);
			}
		}

		artifact.getInstances().add(testSuiteInstance);
		testSuite.getArtifacts().add(artifact);
		return artifact;
	}


	/*
	 * Flags to define how to handle the existing classpath entries.
	 */
	private static final int CALC_CP_ABORT = 0;	//if there is a CP entry, don't do anything
	private static final int CALC_CP_OVERWRITE = 1;
	
	protected void calculateClasspath(int flag)
	{
		if((testSuite == null) || (testSuiteArtifact == null))
			return;
			
		calculatedClasspath = true;		
		CFGPropertyGroup propertyGroup = (CFGPropertyGroup)testSuiteArtifact.getPropertyGroups().get(0);
		BVRProperty[] properties = findPropertiesByName(propertyGroup.getProperties(), CLASSPATH, false);
 
		if(properties.length > 0)
		{
			switch(flag)
			{
				case CALC_CP_ABORT:
					calculatedClasspath = false;
					return;
					
				case CALC_CP_OVERWRITE:
					calculatedClasspath = true;
					for (int i = 0, maxi = properties.length; i < maxi; i++)
						propertyGroup.getProperties().remove(properties[i]);

			}
		}
			
		if((testSuite.getBehavior() == null) || (testSuite.getBehavior().getLocation() == null))
			return;
			
// TODO: Fix Bug		
//		String foo = testSuite.eResource().getURI().segment(1);
		IResource location = ResourcesPlugin.getWorkspace().getRoot().findMember(testSuite.getBehavior().getLocation());
		if((location == null) || (!location.exists()))
			return; 
		
		String[] classpathEntries = JavaUtil.getProjectClasspath(location.getProject(), false);
		for(int i = 0, maxi = classpathEntries.length; i < maxi; i++)
		{
			if(addClasspathEntry(classpathEntries[i]))
			{
				CFGComparableProperty property = Common_ConfigurationFactory.eINSTANCE.createCFGComparableProperty();
				property.setName(CLASSPATH);
				property.setValue(classpathEntries[i]);
				propertyGroup.getProperties().add(property);
			}
		}
	}
	
	protected boolean addClasspathEntry(String cp)
	{
		if(cp.indexOf("junit.jar") > 0)
			return false;

		if(cp.indexOf("xercesImpl.jar") > 0)
			return false;

		if(cp.indexOf("xmlParserAPIs.jar") > 0)
			return false;

		if(cp.indexOf("hexr.jar") > 0)
			return false;

		if(cp.indexOf("hexcore.jar") > 0)
			return false;

		if(cp.indexOf("common.runner.jar") > 0)
			return false;

		if(cp.indexOf("java.runner.jar") > 0)
			return false;

		if(cp.indexOf("http.runner.jar") > 0)
			return false;

		if(cp.indexOf("manual.runner.jar") > 0)
			return false;
					
		return true;
	}
	
	protected BVRProperty[] findPropertiesByName(List properties, String name, boolean caseSensitive)
	{
		if((properties != null) && (!properties.isEmpty()) && (name != null))
		{
			List ret = new ArrayList();
			for (Iterator i = properties.iterator(); i.hasNext();)
			{
				BVRProperty property = (BVRProperty)i.next();
				if(caseSensitive)
				{
					if(name.equals(property.getName()))
						ret.add(property);
				}
				else
				{
					if(name.equalsIgnoreCase(property.getName()))
						ret.add(property);						
				}
			}
			return (BVRProperty[])ret.toArray(new BVRProperty[ret.size()]);
		}
		
		return new BVRProperty[0];
	}
}