/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.test.common.internal.wizard;

import java.text.DateFormat;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.common.configuration.CFGArtifact;
import org.eclipse.hyades.models.common.configuration.CFGArtifactLocationPair;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.CFGPropertyGroup;
import org.eclipse.hyades.models.common.facades.behavioral.IDeployableComponent;
import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.interactions.BVRProperty;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.util.SaveManager;
import org.eclipse.hyades.test.common.TestCommonPlugin;
import org.eclipse.hyades.test.ui.TestUI;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchWizard;

public abstract class RunWizard 
extends Wizard implements IWorkbenchWizard
{
	protected static final String SET_DESTINATION = "SET_DESTINATION";
	protected static final String SET_LOCATION = "SET_LOCATION";
	
	protected RunDestinationPage destinationPage;
	protected RunLocationPage locationPage;
	protected IStructuredSelection selection;
	protected ResourceSet resourceSet;
	protected TPFTest     test;
	
	/**
	 * Constructor for RunWizard
	 */
	public RunWizard()
	{
		super();
				
		setDefaultPageImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_WIZBAN_RUN));

		IDialogSettings dialogSettings = TestUIPlugin.getInstance().getDialogSettings();
		IDialogSettings settings = dialogSettings.getSection("RunWizard");
		if(settings == null)
			settings = dialogSettings.addNewSection("RunWizard");
		setDialogSettings(settings);
	}

	/**
	 * @see org.eclipse.jface.wizard.IWizard#dispose()
	 */
	public void dispose()
	{
		test = null;

		if(resourceSet != null)
		{
			resourceSet.getResources().clear();
			resourceSet = null;
		}
		
		super.dispose();
	}

	/**
	 * @see org.eclipse.jface.wizard.IWizard#createPageControls(org.eclipse.swt.widgets.Composite)
	 */
	public void createPageControls(Composite pageContainer)
	{
		GridData gridData = GridDataUtil.createFill();
		gridData.heightHint = 400;
		gridData.widthHint = 450;
		pageContainer.setLayoutData(gridData);
		
		super.createPageControls(pageContainer);
	}

	/**
	 * @see Wizard#addPages()
	 */
	public void addPages()
	{
		setWindowTitle(TestUIPlugin.getString("RUN_WTITLE"));
		
		destinationPage = createDestionationPage();
		if(destinationPage != null)
		{
			addPage(destinationPage);
			String settings = getDialogSettings().get(SET_DESTINATION + getTest().getId());
			if(settings == null)
			{
				IFile file = EMFUtil.getWorkspaceFile(getTest());
				if(file != null)
					destinationPage.setInitialDestination(file.getParent());
			}
			else
			{
				IResource resource = ResourcesPlugin.getWorkspace().getRoot().findMember(settings);
				if(resource != null)
				{
					if(resource instanceof IContainer)
						destinationPage.setInitialDestination((IContainer)resource);
					else
						destinationPage.setInitialDestination(resource.getParent());
				}
			}
			
		}
			
		locationPage = createLocationPage();
		if(locationPage != null)
		{
			addPage(locationPage);
		
			String settings = getDialogSettings().get(SET_LOCATION);
			if(settings != null)
				locationPage.setInitialLocationURI(URI.createURI(settings));
		}
	}
	
	protected String getExecutionResultInitialName()
	{
		StringBuffer name = new StringBuffer(getTest().getName()).append("_");
		name.append(DateFormat.getDateTimeInstance(DateFormat.DEFAULT, DateFormat.MEDIUM).format(new Date()));
		
		return name.toString();
	}
	
	protected RunDestinationPage createDestionationPage()
	{
		return new RunDestinationPage();
	}

	protected RunLocationPage createLocationPage()
	{
		return new RunLocationPage(true);
	}
	
	/**
	 * @return
	 */
	public ResourceSet getResourceSet() {
		return resourceSet;
	}

	/**
	 * @return
	 */
	public TPFTest getTest() {
		return test;
	}

	/**
	 * @see org.eclipse.ui.IWorkbenchWizard#init(org.eclipse.ui.IWorkbench, org.eclipse.jface.viewers.IStructuredSelection)
	 */
	public void init(IWorkbench workbench, IStructuredSelection selection)
	{
		this.selection = selection;
		
		if(selection != null && !selection.isEmpty())
		{
			Object object = ((IStructuredSelection)selection).getFirstElement();
			if(object instanceof TPFTest)
			{
				resourceSet = new ResourceSetImpl();
				test = (TPFTest)EMFUtil.getEObject(resourceSet, (TPFTest)object, true);
				getTest().eResource().setTrackingModification(true);
			}
		}
	}

	/**
	 * 
	 * @return test instance name
	 */
	public String getExecutionResultName()
	{
		if(destinationPage == null)
			return null;
		return destinationPage.getExecutionResultName();
	}
	
	/**
	 * 
	 * @return container path
	 */
	public IPath getContainerPath()
	{
		if(destinationPage == null)
			return null;
		return 	destinationPage.getContainerPath();
	}

	public TPFDeployment getDeployment()
	{
		if(locationPage == null)
			return null;
		return locationPage.getDeployment();
	}

	public BVRProperty[] getEnvironmentVariables()
	{
		if(locationPage != null)
		{
			CFGArtifact artifact = locationPage.getTestSuiteArtifact();
			if(artifact != null)
			{
				CFGPropertyGroup propertyGroup = (CFGPropertyGroup)artifact.getPropertyGroups().get(0);
				Collection environmentVariables = propertyGroup.getProperties();
				environmentVariables = EcoreUtil.copyAll(environmentVariables);	
				return (BVRProperty[])environmentVariables.toArray(new BVRProperty[environmentVariables.size()]);
			}
		}
		return new BVRProperty[0];
	}

	protected void saveSettings()
	{
		IPath destinationPath = getContainerPath();
		if(destinationPath != null)
			getDialogSettings().put(SET_DESTINATION + getTest().getId(), getContainerPath().toString());
		
		TPFDeployment deployment = getDeployment();
		CFGArtifactLocationPair pair = (CFGArtifactLocationPair)deployment.getArtifactLocations().get(0);
		CFGLocation location = null;
		if(pair != null)
			location = pair.getLocation();

		if(location != null)
		{
			URI uri = EcoreUtil.getURI(location);
			getDialogSettings().put(SET_LOCATION, uri.toString());
		}		
	}
	
	/**
	 * @see org.eclipse.jface.wizard.IWizard#performFinish()
	 */
	public final boolean performFinish()
	{
		ITestSuite testSuite = null;
		if(getTest() instanceof ITestSuite)
			testSuite = (ITestSuite)getTest();
		else if(getTest() instanceof ITestCase)
			testSuite = ((ITestCase)getTest()).getOwner();

		if((locationPage != null) && (locationPage.getTestSuiteArtifact() != null))
		{
			List environmentVariableProperties = ((IDeployableComponent)testSuite).getEnvironmentVariables().getProperties();
			environmentVariableProperties.clear();
			environmentVariableProperties.addAll(Arrays.asList(getEnvironmentVariables()));
			
			if(locationPage.isCalculatedClasspath())
				getTest().eResource().setModified(false);
		}

		if(!doPerformFinish())
			return false;

		saveSettings();
		
		if(getTest().eResource().isModified())
		{
			if((locationPage != null) && (locationPage.isCalculatedClasspath()))
			{
				CFGArtifact artifact = locationPage.getTestSuiteArtifact();
	
				if(artifact != null)
				{
					CFGPropertyGroup propertyGroup = (CFGPropertyGroup)artifact.getPropertyGroups().get(0);
					for(int i = 0; i < propertyGroup.getProperties().size(); i++)
					{
						propertyGroup.getProperties().remove(0);
					}
				}
			}
			
			// JPT -- if the test suite has any artifacts associated with it
			// (which should be a unidirectional association the other direction, anyway)
			// then this call will throw an exception because the artifact is not in a 
			// resource (yet)
			// TODO: address this issue at the model level
			TPFTest theTest = getTest();
			if ( theTest instanceof TPFTestSuite )
			{
				((TPFTestSuite)theTest).getArtifacts().clear();
			}
			
			try
			{
				EMFUtil.save(getTest().eResource());
			}
			catch (Exception e)
			{
				TestCommonPlugin.logError(e);
			}
		}
		
		SaveManager.getInstance().start(TestUIPlugin.getInstance().getPreferenceStore().getLong(TestUI.SAVE_INTERVAL));
		return true;
	}
	
	abstract protected boolean doPerformFinish();
}