/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.adapter.internal;

import org.eclipse.hyades.trace.ui.*;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.*;
import org.eclipse.emf.common.util.EList;
import org.eclipse.ui.part.IPage;
import org.eclipse.hyades.trace.views.util.internal.OpenSource;
import java.util.Enumeration;

import org.eclipse.jface.action.*;

import org.eclipse.hyades.trace.views.internal.*;

/**
 * @author hind.hasnaoui@fr.ibm.com  --
 * create a Coverage Page each time a profiling instance
 *  is selected in the "Profiling monitor view"
 */
public class CoverageViewer extends TraceViewer {

	protected static final String _title =
		TraceUIPlugin.getString("STR_COVERAGE_ST_TITLE");

	protected Action _packageLevel;
	protected Action _classLevel;
	protected Action _methodLevel;
	protected Action _openSource;   

	/**
	 * 
	 */
	public CoverageViewer() {
		super();
	}

	/**
	 * return the selected agent in the Profiling monitor view
	 * called by addViewPage() -handelProfileEvent() - refreshVisiblePage()and
	 * removePage() of the parent class (TraceViewer)
	 */
	public EObject getObjectToView(EObject selObject) {
		if (selObject == null)
			return selObject;

		if (selObject instanceof TRCProcessProxy) {
			int nbOfAgents = 0;
			TRCAgentProxy a = null;
			EList agents = ((TRCProcessProxy) selObject).getAgentProxies();
			for (int idx = 0; idx < agents.size(); idx++) {
				TRCAgentProxy agent = (TRCAgentProxy) agents.get(idx);
				if (agent == null || agent.eIsProxy())
					continue;
				if (agent
					.getType()
					.equals(HyadesConstants.PROFILE_AGENT_TYPE)) {
					nbOfAgents++;
					a = agent;
				}
			}
			if (nbOfAgents == 1) {
				return a;
			}
		}
		return selObject;
	}

	/**
	 * create a coverage page
	 */
	public TraceViewerPage createPage(EObject mofObject) {
		return new CoveragePage(mofObject, this);
	}

	/**
	 * return the view title
	 */
	public String getViewTitle() {
		return _title;
	}

	/**
	 * @see IWorkbenchPart#dispose
	 */
	public void dispose() {
		Enumeration pages = _pages.elements();
		while (pages.hasMoreElements()) {
			IPage page = (IPage) pages.nextElement();

			if (page instanceof CoveragePage)
				 ((CoveragePage) page).dispose();
		}

		_classLevel = null;
		_packageLevel = null;
		_methodLevel = null;
		_openSource = null;

		if (fContextMenu != null)
			fContextMenu.dispose();

		fContextMenu = null;

		super.dispose();
	}

	/**
	 * set the focus on the current page
	 */
	public void setFocus() {
		CoveragePage page = (CoveragePage) getCurrentPage();
		if (page != null)
			page.setFocus();
	}

	/**
	 * Three action buttons are defined:
	 * -packageLevel: display all the tree (package/class/method)
	 * -classlevel: display only the class tree
	 * -method level: display only the methods
	 */
	public void makeActions() {
		if (isInitializedMenu())
			return; //create the toolbar only once

		initializedMenu(true);

		String strPackageLevel =
			TraceUIPlugin.getString("STR_ST_PACKAGE_LEVEL");
		_packageLevel = new Action(strPackageLevel) {
			public void run() {

				CoveragePage page = (CoveragePage) getCurrentPage();

				if (page == null)
					return;

				if (page.getView().getLevel() == 1)
					_packageLevel.setChecked(true);
				else {
					_classLevel.setChecked(false);
					_methodLevel.setChecked(false);

					page.getView().setLevel(1);
					page.getView().redrawTable();
				}

			}
			public int getStyle() {
				return IAction.AS_CHECK_BOX;
			}

		};

		_packageLevel.setText(strPackageLevel);
		TracePluginImages.setImageDescriptors(
			_packageLevel,
			TracePluginImages.T_LCL,
			TracePluginImages.IMG_PACKAGE);
		_packageLevel.setDescription(strPackageLevel);
		_packageLevel.setToolTipText(strPackageLevel);
		_packageLevel.setEnabled(true);
		_packageLevel.setChecked(true);

		String strClassLevel = TraceUIPlugin.getString("STR_ST_CLASS_LEVEL");

		_classLevel = new Action(strClassLevel) {
			public void run() {
				CoveragePage page = (CoveragePage) getCurrentPage();

				if (page == null)
					return;

				if (page.getView().getLevel() == 2)
					_classLevel.setChecked(true);
				else {
					_packageLevel.setChecked(false);
					_methodLevel.setChecked(false);

					page.getView().setLevel(2);
					page.getView().redrawTable();
				}
			}

			public int getStyle() {
				return IAction.AS_CHECK_BOX;
			}

		};

		_classLevel.setText(strClassLevel);
		TracePluginImages.setImageDescriptors(
			_classLevel,
			TracePluginImages.T_LCL,
			TracePluginImages.IMG_CLASS);
		_classLevel.setDescription(strClassLevel);
		_classLevel.setToolTipText(strClassLevel);
		_classLevel.setEnabled(true);

		String strMethodLevel = TraceUIPlugin.getString("STR_ST_METHOD_LEVEL");

		_methodLevel = new Action(strMethodLevel) {
			public void run() {
				CoveragePage page = (CoveragePage) getCurrentPage();

				if (page == null)
					return;

				if (page.getView().getLevel() == 3)
					_methodLevel.setChecked(true);
				else {
					_packageLevel.setChecked(false);
					_classLevel.setChecked(false);

					page.getView().setLevel(3);
					page.getView().redrawTable();
				}

			}
			public int getStyle() {
				return IAction.AS_CHECK_BOX;
			}
		};
		_methodLevel.setText(strMethodLevel);
		TracePluginImages.setImageDescriptors(
			_methodLevel,
			TracePluginImages.T_LCL,
			TracePluginImages.IMG_METHOD);
		_methodLevel.setDescription(strMethodLevel);
		_methodLevel.setToolTipText(strMethodLevel);
		_methodLevel.setEnabled(true);

		String openSourceStr = TraceUIPlugin.getString("STR_SOURCE");
		_openSource = new Action(openSourceStr)
		{
			public void run()
			{
				CoveragePage page = (CoveragePage)getCurrentPage();

				if(page == null)
				  return;

				OpenSource.openSource(UIPlugin.getDefault().getSelectionModel(page.getMOFObject()).getFirstElement());
			}
		};

		_openSource.setText(openSourceStr);
		TracePluginImages.setImageDescriptors(_openSource,TracePluginImages.T_TOOL,TracePluginImages.IMG_SOURCE);	
		_openSource.setDescription(openSourceStr);   
		_openSource.setToolTipText(openSourceStr);
		_openSource.setEnabled(false);
	
		IToolBarManager tbm = getViewSite().getActionBars().getToolBarManager();

		tbm.removeAll();
		tbm.add(new Separator());
		tbm.add(_packageLevel);
		tbm.add(_classLevel);
		tbm.add(_methodLevel);
		tbm.add(new Separator());
		tbm.add(_openSource);

		// require update because toolbar control has been created by this point,
		// but manager does not update it automatically once it has been created
		getViewSite().getActionBars().updateActionBars();
	}

	/**
	 * @return the class level action
	 */
	public Action getClassLevel() {
		return _classLevel;
	}

	/**
	 * @return the method level action
	 */
	public Action getMethodLevel() {
		return _methodLevel;
	}

	/**
	 * @return the package level action
	 */
	public Action getPackageLevel() {
		return _packageLevel;
	}

	/**
	 * Insert the method's description here.
	 * Creation date: (8/21/2001 4:20:37 PM)
	 * @return org.eclipse.jface.action.Action
	 */
	public Action openSource() {
		return _openSource;
	}
}