/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import org.eclipse.hyades.models.trace.*;
import org.eclipse.hyades.models.trace.impl.TRCClassImpl;
import org.eclipse.hyades.models.trace.impl.TRCMethodImpl;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.hyades.trace.views.adapter.internal.ClassStatisticPage;
import org.eclipse.hyades.trace.views.util.internal.*;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.custom.TableTreeItem;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.*;

public class ClassStatistic extends StatisticView {
	
	private final TRCClassImpl.TRCClassSnapshot cdelta1 = new TRCClassImpl.TRCClassSnapshot();
	private final TRCClassImpl.TRCClassSnapshot cdelta2 = new TRCClassImpl.TRCClassSnapshot();

	private final TRCMethodImpl.TRCMethodSnapshot mdelta1 = new TRCMethodImpl.TRCMethodSnapshot();
	private final TRCMethodImpl.TRCMethodSnapshot mdelta2 = new TRCMethodImpl.TRCMethodSnapshot();
				 
	
	/**
	 * 
	 *
	 */
	public class ClassStatisticFilter extends StatisticFilter {

		public ClassStatisticFilter() {
			super();
		}


		public boolean select(Viewer viewer, Object parent, Object element) {
			
			boolean flag = true;
			if(!(element instanceof TRCClass))
			  return true;
						
			String compareText = "";
			
			if (_noPattern)
				return true;
			
			if(element instanceof TRCClass)   
			   compareText = ((TRCClass)element).getName();
			   
			if (!_caseSensitive) {
				compareText = compareText.toLowerCase();
			}
			if (_exactMatch) {
				return compareText.compareTo(_prefix) == 0;
			}
			if (_prefix != "") {
				flag = compareText.startsWith(_prefix);
			}
			if (flag && _suffix != "") {
				flag = compareText.endsWith(_suffix);
			}
			if (flag) {
				for (int k = 0; k < _textList.size(); k++) {
					String str1 = (String) _textList.get(k);

					int index = compareText.lastIndexOf(str1);
					if (index == -1) {
						flag = false;
						break;
					}

					compareText = compareText.substring(index + str1.length());
				}
			}
			return flag;
		}
		
	}
	
	public class ClassStatisticSorter extends StatisticSorter {

		public ClassStatisticSorter() {
			super();
		}

		public int compare(Viewer viewer, Object e1, Object e2) {

            double d =0;
            
			if (e1 instanceof TRCClass) {
					
				TRCClass c1 = (TRCClass) e1;
				TRCClass c2 = (TRCClass)e2;				
				TRCClassImpl.TRCClassSnapshot cs1 = (TRCClassImpl.TRCClassSnapshot)c1.retrieveSnapshot();
				TRCClassImpl.TRCClassSnapshot cs2 = (TRCClassImpl.TRCClassSnapshot)c2.retrieveSnapshot();
				
				switch (_pos) {
					case 0 : //class name
						return (int) _sortSequence
							* (int) c1.getName().compareToIgnoreCase(c2.getName());

					case 1 :	//package name
						return (int) _sortSequence
							* (int) c1.getPackage().getName()
								.compareToIgnoreCase(c2.getPackage().getName());

					case 2 : //total inst
					  if(!_info.isDeltaColumn())
					  {																									
						return (int) _sortSequence
							* (cs1.getTotalInstances() - cs2.getTotalInstances());
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						return (int) _sortSequence * (cdelta1.getTotalInstances() - cdelta2.getTotalInstances());					  	
					  }

					case 3 : //live inst
					  if(!_info.isDeltaColumn())
					  {																									
						return (int) _sortSequence
							* (cs1.getTotalInstances()
								- cs1.getCollectedInstances()
								- cs2.getTotalInstances()
								+ cs2.getCollectedInstances());
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						return (int) _sortSequence * ((cdelta1.getTotalInstances() - cdelta1.getCollectedInstances())
							   - (cdelta2.getTotalInstances() - cdelta2.getCollectedInstances()));					  	
					  }

					case 4 : //collected
					  if(!_info.isDeltaColumn())
					  {																									
						  if(isShowPercent())
						  {
						  	  int inst = cs1.getTotalInstances();
						  	  if(inst == 0) inst = 1;					  	  
						  	  double size = (double)cs1.getCollectedInstances()/inst;
						  	  
						  	  inst = cs2.getTotalInstances();
						  	  if(inst == 0) inst = 1;					  	  
						  	  size -= (double)cs2.getCollectedInstances()/inst;
						  	  
							  return (int) _sortSequence * (int)(size*10000);					  	  					  	  
						  }
						  else
						  {										
							return (int) _sortSequence
								* (cs1.getCollectedInstances() - cs2.getCollectedInstances());
						  }
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						return (int) _sortSequence * (cdelta1.getCollectedInstances() - cdelta2.getCollectedInstances());					  	
					  }

					case 5 : //total size
					  if(!_info.isDeltaColumn())
					  {																									
						return (int) _sortSequence
							* (cs1.getTotalSize() - cs2.getTotalSize());
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_SIZE);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__TOTAL_SIZE);
						return (int) _sortSequence * (cdelta1.getTotalSize() - cdelta2.getTotalSize());					  	
					  }

					case 6 : //active size
					  if(!_info.isDeltaColumn())
					  {																									
						return (int) _sortSequence
							* (cs1.getTotalSize()
								- cs1.getCollectedSize()
								- cs2.getTotalSize()
								+ cs2.getCollectedSize());
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_SIZE);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__TOTAL_SIZE);
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_SIZE);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__COLLECTED_SIZE);
						return (int) _sortSequence * ((cdelta1.getTotalSize() - cdelta1.getCollectedSize())
							   - (cdelta2.getTotalSize() - cdelta2.getCollectedSize()));					  	
					  }

					case 7 : //base time
					  if(!_info.isDeltaColumn())
					  {																									
						  d = cs1.getBaseTime() - cs2.getBaseTime();
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__BASE_TIME);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__BASE_TIME);
						d = cdelta1.getBaseTime() - cdelta2.getBaseTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					
					case 8 : //inherited base time
					  if(!_info.isDeltaColumn())
					  {																									
						  d = cs1.getInheritedBaseTime() - cs2.getInheritedBaseTime();
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_BASE_TIME);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__INHERITED_BASE_TIME);
						d = cdelta1.getInheritedBaseTime() - cdelta2.getInheritedBaseTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					
					case 9 : //cumulative time
					  if(!_info.isDeltaColumn())
					  {																									
						  d = cs1.getCumulativeTime() - cs2.getCumulativeTime();
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__CUMULATIVE_TIME);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__CUMULATIVE_TIME);
						d = cdelta1.getCumulativeTime() - cdelta2.getCumulativeTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					
					case 10 : //inherited cumulative time
					  if(!_info.isDeltaColumn())
					  {																									
						  d = cs1.getInheritedCumulativeTime() - cs2.getInheritedCumulativeTime();
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_CUMULATIVE_TIME);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__INHERITED_CUMULATIVE_TIME);
						d = cdelta1.getInheritedCumulativeTime() - cdelta2.getInheritedCumulativeTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					
					case 11 : //calls
					  if(!_info.isDeltaColumn())
					  {																									
						return (int) _sortSequence * (cs1.getCalls() - cs2.getCalls());
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__CALLS);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__CALLS);
						return (int) _sortSequence * (cdelta1.getCalls() - cdelta2.getCalls());					  	
					  }

					case 12 : //inherited calls
					  if(!_info.isDeltaColumn())
					  {																									
						return (int) _sortSequence 	* (cs1.getInheritedCalls() - cs2.getInheritedCalls());
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_CALLS);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__INHERITED_CALLS);
						return (int) _sortSequence * (cdelta1.getInheritedCalls() - cdelta2.getInheritedCalls());					  	
					  }

				}
			} else {
				
				TRCMethod m1 = (TRCMethod) e1;
				TRCMethod m2 = (TRCMethod) e2;
				TRCMethodImpl.TRCMethodSnapshot ms1 = (TRCMethodImpl.TRCMethodSnapshot)m1.retrieveSnapshot();
				TRCMethodImpl.TRCMethodSnapshot ms2 = (TRCMethodImpl.TRCMethodSnapshot)m2.retrieveSnapshot();
				
				 
				switch (_pos) {
					case 0 : //method name
					  StringBuffer buf1 = new StringBuffer();
					  StringBuffer buf2 = new StringBuffer();					  
					  return (int) _sortSequence * (int)buf1.append(m1.getName()).append(m1.getSignature()).toString()
					        .compareToIgnoreCase(buf2.append(m2.getName()).append(m2.getSignature()).toString());
					
					case 7 : //base time
					  if(!_info.isDeltaColumn())
					  {																									
						  d = ms1.getBaseTime() - ms2.getBaseTime();
					  }
					  else
					  {
						m1.computeDelta(mdelta1,TracePackage.TRC_METHOD__BASE_TIME);
						m2.computeDelta(mdelta2,TracePackage.TRC_METHOD__BASE_TIME);
						d = mdelta1.getBaseTime() - mdelta2.getBaseTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					
					case 9 : //cumulative time
					  if(!_info.isDeltaColumn())
					  {																									
						  d = ms1.getCumulativeTime() - ms2.getCumulativeTime();
					  }
					  else
					  {
						m1.computeDelta(mdelta1,TracePackage.TRC_METHOD__CUMULATIVE_TIME);
						m2.computeDelta(mdelta2,TracePackage.TRC_METHOD__CUMULATIVE_TIME);
						d = mdelta1.getCumulativeTime() - mdelta2.getCumulativeTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					
					case 11 : //calls
					  if(!_info.isDeltaColumn())
					  {																									
						return (int) _sortSequence * (ms1.getCalls() - ms2.getCalls());

					  }
					  else
					  {
						m1.computeDelta(mdelta1,TracePackage.TRC_METHOD__CALLS);
						m2.computeDelta(mdelta2,TracePackage.TRC_METHOD__CALLS);
						return (int) _sortSequence * (mdelta1.getCalls() - mdelta2.getCalls());					  	
					  }					 
				}
			}

			return 0;
		}

	}

	/**
	 * 
	 */
	public class ClassStatisticContentProvider
		implements ITreeContentProvider {

		public void dispose() {
		}

		public Object getParent(Object element) {
			
			if(element instanceof TRCMethod)
			   return ((TRCMethod)element).getDefiningClass();
			   
			return null;
		}

		public void inputChanged(
			Viewer viewer,
			Object oldInput,
			Object newInput) {
		}

		public Object[] getElements(Object inputElement) {
			return PerftraceUtil.getAllClasses(_page.getMOFObject());
		}

		public Object[] getChildren(Object element) {
			tmpList.clear();

			if (element instanceof TRCClass)
				return ((TRCClass) element).getMethods().toArray();

			return tmpList.toArray();
		}

		public boolean hasChildren(Object element) {
			if (element instanceof TRCClass)
				return ((TRCClass) element).getMethods().size() > 0;

			return false;
		}
	}

	/**
	 * 
	 */
	public class ClassStatisticLabelProvider
		extends LabelProvider
		implements ITableLabelProvider {
		protected StatisticView _viewer;

		public ClassStatisticLabelProvider(StatisticView viewer) {
			super();
			_viewer = viewer;
		}

		public Image getColumnImage(Object obj, int col) {
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(info.isDeltaColumn())
			   return null;
			
			int state = 0;
			double d = 0;
			if(obj instanceof TRCClass)	
			{
				TRCClass clas = (TRCClass)obj;

				if(((TRCClass)obj).computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES)<0)
				{
					if(col == 1)
						return TracePluginImages.getImage(TracePluginImages.IMG_DELTANEW);
					return null;	
				}
				
				switch(pos)
				{
					case 2: //total inst
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES);					
					  state = cdelta1.getTotalInstances();
					  break;
					  					 
					case 3: //live inst
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
					  state = cdelta1.getTotalInstances() - cdelta1.getCollectedInstances();
					  break;
					  
					case 4://collected
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
					  state = cdelta1.getCollectedInstances();
					  break;
					  
					case 5: //total size
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_SIZE);
					  state = cdelta1.getTotalSize();
					  break;
					  
					case 6: //active size
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_SIZE);
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_SIZE);
					  state = cdelta1.getTotalSize() - cdelta1.getCollectedSize();
					  break;
									  					  
					case 7: //base time
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__BASE_TIME);					
					  d = cdelta1.getBaseTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;

					case 8: //inherited base time
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_BASE_TIME);
					  d = cdelta1.getInheritedBaseTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  break;
					  					  
					case 9: //cumulative time
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__CUMULATIVE_TIME);
					  d = cdelta1.getCumulativeTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;
					  
					case 10: //inherited cumulative time
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_CUMULATIVE_TIME);
					  d = cdelta1.getInheritedCumulativeTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;
					  
					case 11: //calls
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__CALLS);
					  state = cdelta1.getCalls();
					  break;
					  
					case 12: //inherited calls
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_CALLS);
					  state = cdelta1.getInheritedCalls();
					  break;
					
				}				  				  				 				   				  
			}
			else
			{
	            //TRMethod				
				TRCMethod m = (TRCMethod)obj;

				if(m.computeDelta(mdelta1,TracePackage.TRC_METHOD__BASE_TIME)<0)
				{
					if(col == 1)
						return TracePluginImages.getImage(TracePluginImages.IMG_DELTANEW);
					return null;	
				}
				
				switch(pos)
				{
					case 7: //base time
					  m.computeDelta(mdelta1,TracePackage.TRC_METHOD__BASE_TIME);
					  d = mdelta1.getBaseTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;
					  					  					  
					case 9: //cumulative time
					  m.computeDelta(mdelta1,TracePackage.TRC_METHOD__CUMULATIVE_TIME);
					  d = mdelta1.getCumulativeTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;
						  
					case 11: //calls
					  m.computeDelta(mdelta1,TracePackage.TRC_METHOD__CALLS);
					  state = mdelta1.getCalls();
					  break;
					  
				}
			}
                  
			if (state < 0) 
				return TracePluginImages.getImage(TracePluginImages.IMG_DELTADOWN);
			if(state > 0)
				return TracePluginImages.getImage(TracePluginImages.IMG_DELTAUP);
      							
			return null;		
		}

		public String getColumnText(Object obj, int col) {

			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			

			if (obj instanceof TRCClass) {
				
				TRCClass clas = (TRCClass) obj;
				TRCClassImpl.TRCClassSnapshot cs = (TRCClassImpl.TRCClassSnapshot)clas.retrieveSnapshot();
				
				switch (pos) {
					case 0 : //class name
						return ((TRCClass) obj).getName();

					case 1 :
   						return PerftraceUtil.getPackageName(clas.getPackage(), _page.getMOFObject()).toString();

					case 2 : //total inst
					  if(!info.isDeltaColumn())
					  {					
						  if(isShowPercent())
						  {
						  	  return PerftraceUtil.formatAsPercentage(((double)cs.getTotalInstances())/_totalInst);
						  }
						  else
						  {													  					
							return String.valueOf(cs.getTotalInstances());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						return String.valueOf(cdelta1.getTotalInstances());					  	
					  }

					case 3 : //live inst
					  if(!info.isDeltaColumn())
					  {										
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(((double)(cs.getTotalInstances()
									- cs.getCollectedInstances()))/_totalInst);
						  }
						  else
						  {													  										
							return String.valueOf(cs.getTotalInstances()- cs.getCollectedInstances());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						return String.valueOf(cdelta1.getTotalInstances() - cdelta1.getCollectedInstances());					  	
					  }

					case 4 : //collected
					  if(!info.isDeltaColumn())
					  {															
						  if(isShowPercent())
						  {
						  	  int size = cs.getTotalInstances();
						  	  if(size == 0) size = 1;
						  	   
						  	  return TString.formatAsPercentage(((double)cs.getCollectedInstances())/size);
						  }
						  else
						  {																						
							return String.valueOf(cs.getCollectedInstances());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						return String.valueOf(cdelta1.getCollectedInstances());					  	
					  }

					case 5 : //total size
					  if(!info.isDeltaColumn())
					  {																				
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(((double)cs.getTotalSize())/_totalSize);
						  }
						  else
						  {													  										
							return String.valueOf(cs.getTotalSize());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_SIZE);
						return String.valueOf(cdelta1.getTotalSize());					  	
					  }

					case 6 : //active size
					
					  if(!info.isDeltaColumn())
					  {																									
					  	  long size = cs.getTotalSize() - cs.getCollectedSize();					
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(((double) size)/_activeSize);
						  }
						  else
						  {													  															
							return String.valueOf(size);
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_SIZE);
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_SIZE);
						return String.valueOf(cdelta1.getTotalSize() - cdelta1.getCollectedSize());					  	
					  }

					case 7 : //base time
					  if(!info.isDeltaColumn())
					  {																														
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(cs.getBaseTime()/_maxTime);
						  }
						  else
						  {													  										
							return TString.formatTimeValue(cs.getBaseTime());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__BASE_TIME);
						return TString.formatTimeValue(cdelta1.getBaseTime());					  	
					  }

					case 8 : //inherited base time
					  if(!info.isDeltaColumn())
					  {																																			
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(cs.getInheritedBaseTime()/_maxTime);
						  }
						  else
						  {													  										
							return TString.formatTimeValue(cs.getInheritedBaseTime());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_BASE_TIME);
						return TString.formatTimeValue(cdelta1.getInheritedBaseTime());					  	
					  }

					case 9 : //cumulative time
					  if(!info.isDeltaColumn())
					  {																																								
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(cs.getCumulativeTime() / _maxTime);
						  }
						  else
						  {													  					
					
							return TString.formatTimeValue(cs.getCumulativeTime());
						  }
					  }
					  else
				  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__CUMULATIVE_TIME);
						return TString.formatTimeValue(cdelta1.getCumulativeTime());					  	
					  }
						  
					case 10 : //inherited cumulative time
					  if(!info.isDeltaColumn())
					  {																																													
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(cs.getInheritedCumulativeTime() / _maxTime);
						  }
						  else
						  {													  										
							return TString.formatTimeValue(cs.getInheritedCumulativeTime());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_CUMULATIVE_TIME);
						return TString.formatTimeValue(cdelta1.getInheritedCumulativeTime());					  	
					  }
						  
					case 11 : //calls
					  if(!info.isDeltaColumn())
					  {																																																		
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(((double)cs.getCalls()) / _totalCalls);
						  }
						  else
						  {													  										
							  return String.valueOf(cs.getCalls());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__CALLS);
						return String.valueOf(cdelta1.getCalls());					  	
					  }
						  
					case 12 : //inherited calls
					  if(!info.isDeltaColumn())
					  {																																																							
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(((double)cs.getInheritedCalls()) / _totalCalls);
						  }
						  else
						  {													  										
							  return String.valueOf(cs.getInheritedCalls());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_CALLS);
						return String.valueOf(cdelta1.getInheritedCalls());					  	
					  }	
					  					  
				}
			}
			else
			{
                TRCMethod m = (TRCMethod)obj;
				TRCMethodImpl.TRCMethodSnapshot ms = (TRCMethodImpl.TRCMethodSnapshot)m.retrieveSnapshot();
                
				//TRCMethod
				switch (pos)
				{										
					case 0 : //method name
					    StringBuffer buf = new StringBuffer();
					    return buf.append(m.getName()).append(m.getSignature()).toString();
					    
					case 7 : //base time
					  if(!info.isDeltaColumn())
					  {																																																												
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(ms.getBaseTime() / _maxTime);
						  }
						  else
						  {													  									
							return TString.formatTimeValue(ms.getBaseTime());
						  }
					  }
					  else
					  {
						m.computeDelta(mdelta1,TracePackage.TRC_METHOD__BASE_TIME);
						return TString.formatTimeValue(mdelta1.getBaseTime());					  	
					  }
	
					case 9 : //cumulative time
					  if(!info.isDeltaColumn())
					  {																																																																	
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(ms.getCumulativeTime() / _maxTime);
						  }
						  else
						  {													  									
							return TString.formatTimeValue(ms.getCumulativeTime());
						  }
					  }
					  else
					  {
						 m.computeDelta(mdelta1,TracePackage.TRC_METHOD__CUMULATIVE_TIME);					  	
						 return TString.formatTimeValue(mdelta1.getCumulativeTime());					  	
					  }
	
					case 11 : //calls
					  if(!info.isDeltaColumn())
					  {																																																																						
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(((double)ms.getCalls()) / _totalCalls);
						  }
						  else
						  {													  									
							return String.valueOf(ms.getCalls());
						  }
					  }
					  else
					  {
						m.computeDelta(mdelta1,TracePackage.TRC_METHOD__CALLS);
						return String.valueOf(mdelta1.getCalls());					  	
					  }					
				}					  
					
			}

			return "";
		}
	}

	class ClassTreeViewer extends TableTreeViewer {
		public ClassTreeViewer(Composite parent) {
			super(parent);
		}

		public ClassTreeViewer(TableTree table) {
			super(table);
		}

		public void expandItem(TableTreeItem item) {
			item.setExpanded(true);
			createChildren(item);

		}

	}
	protected Composite createTable(Composite parent, int options) {
		return new TableTree(parent, options);
	}

	public ClassStatistic(Composite parent, TraceViewerPage page) {
		super(parent, page);	
		_viewerFilter = new ClassStatisticFilter();				
	}

	/**
	 * getClassColumnsPerferencesKey returns the string key used to find the
	 * column preferences information in the preferences store.
	 * @return java.lang.String key name
	 */
	public String getColumnsPreferencesKey() {
		// Note this string is not to be translated and must be changed whenever the 
		// default columns template is changed by adding, removing or renaming columns.
		// Changing the version will result in the default preferences being used.
		return "Class60";

	}

	public String getDefaultColumnsTemplate() {
		//Class Columns Data
		String classColumn =
			TraceUIPlugin.getString("STR_ST_CLASS_NAME")
				+ ":0:"
				+ String.valueOf(
					ColumnData.NONDELETABLE
						| ColumnData.IS_VISIBLE
						| ColumnData.NONMOVABLE)
				+ ":200,"
				+ TraceUIPlugin.getString("STR_ST_PACKAGE")
				+ ":1:"
				+ String.valueOf(ColumnData.IS_VISIBLE)
				+ ":left:100,"
				+ TraceUIPlugin.getString("STR_ST_INSTANCES")
				+ ":2:"
				+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)
				+ ":right:100,"
				+ TraceUIPlugin.getString("STR_ST_LIVE_INSTANCES")
				+ ":3:"
				+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)
				+ ":right:100,"
				+ TraceUIPlugin.getString("STR_ST_COLLECTED")
				+ ":4:"
				+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)
				+ ":right:100,"
				+ TraceUIPlugin.getString("STR_ST_SIZE")
				+ ":5:"
				+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)
				+ ":right:100,"
				+ TraceUIPlugin.getString("STR_ST_LIVE_STORAGE")
				+ ":6:"
				+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)
				+ ":right:100,"
				+ TraceUIPlugin.getString("STR_ST_CPU_TIME")
				+ ":7:"
				+ String.valueOf(ColumnData.SUPPORTS_DELTA)				
				+ ":right:100,"
				+ TraceUIPlugin.getString("STR_ST_CPU_INHTIME")
				+ ":8:"
				+ String.valueOf(ColumnData.SUPPORTS_DELTA)								
				+ ":right:100,"
				+ TraceUIPlugin.getString("STR_ST_STACK_TIME")
				+ ":9:"
				+ String.valueOf(ColumnData.SUPPORTS_DELTA)								
				+ ":right:100,"
				+ TraceUIPlugin.getString("STR_ST_STACK_INHTIME")
				+ ":10:"
				+ String.valueOf(ColumnData.SUPPORTS_DELTA)				
				+ ":right:100,"				
				+ TraceUIPlugin.getString("STR_ST_CALLS")
				+ ":11:"
				+ String.valueOf(ColumnData.SUPPORTS_DELTA)								
				+ ":right:50,"
				+ TraceUIPlugin.getString("STR_ST_INHCALLS")
				+ ":12:"
				+ String.valueOf(ColumnData.SUPPORTS_DELTA)								
				+ ":right:100";
				
		return classColumn;
	}

	public Table getTable() {
		return ((TableTree) getTableViewer().getControl()).getTable();
	}

	protected StructuredViewer createTableViewer(Composite table) {
		ClassTreeViewer tv = new ClassTreeViewer((TableTree) table);
		return (StructuredViewer) tv;
	}

	protected void handleSelectionEvent() {
		((ClassStatisticPage) getTraceViewerPage()).openSource().setEnabled(
			false);

		ITraceSelection model =
		UIPlugin.getDefault().getSelectionModel(_page.getMOFObject());
		if (model.size() > 0) {
			((ClassStatisticPage) getTraceViewerPage())
				.openSource()
				.setEnabled(
				true);

			Object sel = model.getFirstElement();
			// If the selection is an object then turn the selection into the class of the object
			if (sel instanceof TRCObject) {

				sel = PerftraceUtil.getClass((TRCObject) sel);
			}
			else if (sel instanceof TRCClass) {
				select((TRCClass) sel);
			}
			else if(sel instanceof TRCMethod)
			{
				select((TRCMethod)sel);
			}
			else if(sel instanceof TRCMethodInvocation)
			{
				select(((TRCMethodInvocation)sel).getMethod());
			}
			
		}
	}

	/**
	 * Called when the context menu is about to open.
	 * @see IFillMenuTarget#fillContextMenu
	 */
	public void menuAboutToShow(IMenuManager menu) {
		menu.add(fSeparator);

		menu.add(getUpdateAction());
		menu.add(fSeparator);
		menu.add(
			getChooseColumnsAction(
				getColumnDataList(),
				getColumnsPreferencesKey()));

	}

	private void select(TRCMethod obj) {
		
		TRCClass cls = obj.getDefiningClass();		
		TableTree table = ((TableTreeViewer)getTableViewer()).getTableTree();
		
		TableTreeItem[] items = table.getItems();
		for (int idx = 0; idx < items.length; idx++) {
			TableTreeItem item = items[idx];
			if(item.getData() == cls)
			{
				((ClassTreeViewer) getTableViewer()).expandItem(item);
				TableTreeItem[] childItems = item.getItems();
	
				for (int i = 0; i < childItems.length; i++) {
					TableTreeItem childItem = childItems[i];
					if (obj == childItem.getData()) {
							((ClassTreeViewer) getTableViewer()).expandItem(item);
							table.setSelection(new TableTreeItem[] { childItem });
							return;
					}
				}
			}
		}
		
		table.deselectAll();		
		
	}

	private void select(TRCClass obj) {
		
		TableTree table = ((TableTreeViewer)getTableViewer()).getTableTree();
        
        TableTreeItem[] items = table.getItems();
		for (int idx = 0; idx < items.length; idx++) {
			Object data = items[idx].getData();
			if (data != null && data == obj) {
				table.setSelection(new TableTreeItem[] {items[idx]});
				
				return;
			}
		}
		
		table.deselectAll();		
	}

	public void updateButtons() {

		ITraceSelection _selectionmodel =
		UIPlugin.getDefault().getSelectionModel(_page.getMOFObject());

		((ClassStatisticPage) getTraceViewerPage()).openSource().setEnabled(
			_selectionmodel.size() > 0);

		((ClassStatisticPage) getTraceViewerPage()).percentMode().setChecked(isShowPercent());
		((ClassStatisticPage) getTraceViewerPage()).deltaColumns().setChecked(showingDeltaColumns());

	}

	protected void updateDetailsPane() {
		int selIndex = getTable().getSelectionIndex();
		if (selIndex < 0) {
			return;
		}

		Item item = getTable().getItem(selIndex);
		Object itemData = item.getData();

		if (itemData == null) {
			return;
		}
		if (itemData != null && itemData instanceof TableTreeItem) {
			itemData = ((TableTreeItem) itemData).getData();
		}
	}

	public void updateModelSelection() {
		
		ISelection selection = getTableViewer().getSelection();
		if(selection != null && !selection.isEmpty())
		{
			Object sel = ((IStructuredSelection)selection).getFirstElement();
			
			UIPlugin.getDefault().getSelectionModel(
				_page.getMOFObject()).add(
				sel);
	
			ViewSelectionChangedEvent event = UIPlugin.getDefault().getViewSelectionChangedEvent();
			event.setSource(_page.getMOFObject());
			UIPlugin.getDefault().notifyViewSelectionChangedListener(event);			
		}
	}

	public IContentProvider getContentProvider() {
		return new ClassStatisticContentProvider();
	}

	public LabelProvider getTableLabelProvider() {
		return new ClassStatisticLabelProvider(this);
	}

	public void update() {
		if (_firstTime) {
			getTableViewer().addFilter(getViewerFilter());
			_firstTime = false;
			Table table = getTable();
			TableColumn firstColumn = table.getColumn(0);

			_viewerSorter = new ClassStatisticSorter();

			getViewerSorter().setSortedColumn(firstColumn);
			getTableViewer().setSorter(getViewerSorter());

		}
		// set the input of the viewer
		if(isShowPercent())
		{
			_totalCalls = PerftraceUtil.getTotalCalls(_page.getMOFObject());	
			_totalInst = PerftraceUtil.getTotalInstances(_page.getMOFObject());
			_activeInst = PerftraceUtil.getActiveInstances(_page.getMOFObject());	
			_totalSize = PerftraceUtil.getTotalSize(_page.getMOFObject());	
			_activeSize = PerftraceUtil.getActiveSize(_page.getMOFObject());						
			_maxTime = PerftraceUtil.getMaximumTime(_page.getMOFObject());
		}

		getTableViewer().setInput(_page.getMOFObject());

		getTable().setRedraw(false);
		getTableViewer().refresh();
		getTable().setRedraw(true);

		handleSelectionEvent();

	}

	public void handleViewSelectionChangedEvent(ViewSelectionChangedEvent event)
	{
		if(!getTraceViewerPage().getTraceViewer().isProcessRefreshEvents()		
		|| getTable().isFocusControl())
		  return;
		  
		handleSelectionEvent();  
	}
	
	public boolean isEmpty()
	{
		if (PerftraceUtil.getAllClasses(_page.getMOFObject()).length > 0)
			return false;
		return true;
		
	}

}