/**********************************************************************
 * Copyright (c) 2003-2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.util.internal;
import org.eclipse.hyades.models.trace.*;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import java.util.ArrayList;
import java.util.HashMap;

import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.internal.ui.*;
import org.eclipse.hyades.trace.ui.internal.core.*;
import org.eclipse.hyades.trace.ui.internal.util.*;

/**
 * 
 * @author hind.hasnaoui@fr.ibm.com /
 *
 * This class compute the method coverage 
 * 
 */
public class CoverageAnalysis {

	private CoverageFolder root = null;
	private HashMap coveragesMap = null;
		
	/**
	 * @param object
	 */
	public CoverageAnalysis(EObject object) {
		//Filter the list of packages
		Object packs[] = PerftraceUtil.getAllPackages(object);
		EList listPack = new BasicEList();
		for (int i = 0; i < packs.length; i++) {
			listPack.add(i, packs[i]);
		}
		EList filteredListPack = filterPackages(listPack);

		//Compute method coverage
		if (filteredListPack != null) {
			root = compute(filteredListPack, object);
		}
	}

	/**
	 * @param parent
	 * @param obj
	 * @return  an existing or new CoverageLeaf to store coverage data for this EObject
	 */
	public CoverageLeaf FindCoverageLeaf(CoverageFolder parent, EObject obj) {
		if (coveragesMap == null) {
			coveragesMap = new HashMap();
		}
		TRCMethod m = (TRCMethod)obj;
		String fullSig = getFullSig(m);
		CoverageLeaf cl = (CoverageLeaf)coveragesMap.get(fullSig);
		if (cl == null) {
			cl = new CoverageLeaf(this, parent);
			coveragesMap.put(fullSig, cl);
			parent.getChilds().add(cl);
		}
		cl.addTraceObjects(obj);
		return cl;
	}
	
	/**
	 * @param parent
	 * @param obj
	 * @return an existing or new CoverageFolder to store coverage data for this EObject
	 */
	public CoverageFolder FindCoverageFolder(CoverageFolder parent, EObject obj) {
		if (coveragesMap == null) {
			coveragesMap = new HashMap();
		}
		String fullSig = null;
		if (obj instanceof TRCClass) {
			TRCClass c = (TRCClass)obj;
			fullSig = getFullSig(c);
		} else if (obj instanceof TRCPackage) {
			TRCPackage p = (TRCPackage)obj;
			fullSig = getFullSig(p);
		} else {
			fullSig = obj.toString();
		}
		CoverageFolder cf = (CoverageFolder)coveragesMap.get(fullSig);
		if (cf == null) {
			cf = new CoverageFolder(this, parent);
			coveragesMap.put(fullSig, cf);
			parent.getChilds().add(cf);
		}
		cf.addTraceObjects(obj);
		return cf;
	}
	
	/**
	 * @param e
	 * @return source location signature if any
	 */
	private static String getSourceSig(TRCSourceInfo e) {
		return (e != null ? e.getLocation()+":" : "");
	}
	
	/**
	 * @param e
	 * @return qualified name of the class
	 */
	private static String getSig(TRCClass e) {
		return (e.getPackage().getName() == null ? "" : (e.getPackage().getName()+"."))+
				e.getName();
	}
	
	/**
	 * @param e
	 * @return
	 */
	public static String getFullSig(TRCMethod e) {
		return  getSourceSig(e.getSourceInfo())+
				e.getLineNo()+":"+
				getSig(e.getDefiningClass())+"."+
				e.getName()+
				e.getSignature();
	}

	/**
	 * @param e
	 * @return
	 */
	public static String getFullSig(TRCClass e) {
		return  getSourceSig(e.getSourceInfo())+
				e.getLineNo()+":"+
				getSig(e);
	}

	/**
	 * @param e
	 * @return
	 */
	public static String getFullSig(TRCPackage e) {
		return e.getName();
	}
	
	/**
	 * This method filter the packages list 
	 * based on the active filter 
	 * the empty packages are also removed
	 */
	protected EList filterPackages(EList list) {
		if (list == null)
			return null;

		ArrayList listFilter = filters();
		EList newList = new BasicEList(); 
		
		for (int j = 0; j < list.size(); ++j) {
			TRCPackage pac = (TRCPackage) list.get(j);

			if (!isFiltered(pac, listFilter)) {
				newList.add(pac);
			}
		}

		return newList;

	}

	/**
	 * This method compute the method coverage
	 * @param listPack: filtered packages list
	 * @return the coverage object
	 */
	protected CoverageFolder compute(EList listPack, EObject proc) {
		CoverageFolder cov = new CoverageFolder(this, null);
		cov.addTraceObjects(proc);
		cov.parse(listPack);
		cov.compute();
		return cov;
	}

	/**
	 * Compute the class level tree
	 * @return
	 */
	public Coverage getClassLevel() {
		// Root
		CoverageFolder classLevel = duplicate(root);

		ArrayList classList = new ArrayList();
		classList.clear();

		ArrayList packList = root.getChilds();

		for (int i = 0; i < packList.size(); i++) {
			Coverage covPack = (Coverage) packList.get(i);
			ArrayList list = covPack.getChilds();
			for (int j = 0; j < list.size(); j++) {
				classList.add(list.get(j));
			}

		}

		classLevel.setChilds(classList);

		return classLevel;
	}

	/**
	 * Compute the method level tree
	 * @return
	 */
	public Coverage getMethodLevel() {
		int i, j;
		ArrayList list = new ArrayList();
		;
		ArrayList classList = new ArrayList();
		ArrayList methodList = new ArrayList();
		ArrayList methodListF = new ArrayList();

		//	Root
		CoverageFolder methodLevel = duplicate(root);

		ArrayList packList = root.getChilds();

		for (i = 0; i < packList.size(); i++) {
			Coverage covPack = (Coverage) packList.get(i);
			list = covPack.getChilds();
			for (j = 0; j < list.size(); j++) {
				classList.add(list.get(j));
			}

		}

		for (i = 0; i < classList.size(); i++) {
			Coverage covClass = (Coverage) classList.get(i);
			list = covClass.getChilds();
			for (j = 0; j < list.size(); j++) {
				methodList.add(list.get(j));
			}

		}

		// filter sub classes
		for (i = 0; i < methodList.size(); i++) {
			Coverage covElement = (Coverage) methodList.get(i);
			// if sub class
			if ((covElement.getTraceObj()) instanceof TRCClass) {
				list = covElement.getChilds();
				for (j = 0; j < list.size(); j++) {
					methodListF.add(list.get(j));
				}

			} else
				methodListF.add(covElement);
		}

		methodLevel.setChilds(methodListF);

		return methodLevel;
	}

	/**
	 * Duplicate the coverage root object
	 * @param covArg
	 * @return
	 */
	protected CoverageFolder duplicate(CoverageFolder covArg) {
		CoverageFolder cov = new CoverageFolder(this, null, covArg);
		return cov;

	}

	/**
	 * @return the list of the active filter
	 */
	protected ArrayList filters() {
		ArrayList listFilters = new ArrayList();

		String filters =
			UIPlugin.getDefault().getPreferenceStore().getString(
				TraceConstants.FILTERS_SET_KEY);

		if (filters != null) {
			ArrayList listFilterSetElement =
				TraceProfileFiltersUI.getFiltersSet(filters);
			FilterSetElement filterElemt = null;
			boolean active = false;

			// retrieve the active filter
			for (int i = 0; i < listFilterSetElement.size(); i++) {
				filterElemt = (FilterSetElement) listFilterSetElement.get(i);

				if (filterElemt.getEnabled()) {
					active = true;
					break;
				}
			}
			if ((filterElemt != null) && (active)) {
				ArrayList children = filterElemt.getChildren();

				if (children.size() != 0) {
					for (int j = 0; j < children.size(); j++) {
						FilterTableElement filterTableEl =
							(FilterTableElement) children.get(j);
						if (filterTableEl
							.getVisibility()
							.equals(
								UIPlugin.getResourceString("STR_EXCLUDE"))) {
							String text = filterTableEl.getText();
							if (text.indexOf("*") != -1)
								text = text.substring(0, text.indexOf("*"));

							listFilters.add(text);
						}
					}

				}
			}

		}
		return listFilters;
	}

	/**
	 * The empty packages are removed and
	 * those listed in the active filter
	 * @param pac
	 * @return
	 */
	protected boolean isFiltered(TRCPackage pac, ArrayList list) {
		if (pac.getClasses() == null ||
			pac.getClasses().size() == 0) {
			//if empty package
			return true;
		}

		else if (list.size() != 0) {
			String pacName = pac.getName();

			for (int k = 0; k < list.size(); ++k) {
				if (pacName.startsWith((String) list.get(k))) {
					return true;
				}
			}
			return false;
		}
		return false;
	}

	/**
	 * The empty classes are removed too
	 * @param c
	 * @return
	 */
	protected boolean isFiltered(TRCClass c) {
		if (c.getMethods() == null ||
			c.getMethods().size() == 0) {
			//if empty class
			return true;
		}
		return false;
	}

	/**
	 * @return the root folder (summary) that has been computed during construction
	 */
	public CoverageFolder getRoot() {
		return root;
	}

}