/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.extension;

import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;

import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.editor.form.EventForm;
import org.eclipse.hyades.test.ui.internal.editor.form.ExecutionForm;
import org.eclipse.hyades.test.ui.editor.extension.BaseEditorExtension;
import org.eclipse.hyades.test.ui.editor.form.util.WidgetFactory;
import org.eclipse.hyades.ui.editor.IHyadesEditorPart;

/**
 * Hyades execution editor extension that is suitable for all the
 * deployment objects.
 * 
 * <p>This class is used whenever it is not possible neither to identify nor
 * to locate the appropriate extension editor.
 * 
 * @author marcelop
 * @since 0.3.0
 */
public class ExecutionEditorExtension
extends BaseEditorExtension
{
	private static final int PAGE_OVERVIEW = 0;
	private static final int PAGE_EVENTS = 1;

	public static final String EXTENSION_ID = "org.eclipse.hyades.test.ui.editor.extension.Execution";
	
	private ExecutionForm executionForm;
	private EventForm eventsForm;
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		executionForm.getWidgetFactory().dispose();
		executionForm.dispose();
		eventsForm.dispose();
		
		super.dispose();
	}

	/**
	 * Returns the execution that is manipulated by this editor
	 * extension.
	 * @return TPFExecutionResult 
	 */
	public TPFExecutionResult getExecution()
	{
		return (TPFExecutionResult)getHyadesEditorPart().getEditorObject();
	}
	
	/**
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#createPages()
	 */
	public void createPages()
	{
		IHyadesEditorPart hyadesEditorPart = getHyadesEditorPart();
		WidgetFactory widgetFactory = new WidgetFactory();
		
		executionForm = new ExecutionForm(this, widgetFactory);
		hyadesEditorPart.addPage(executionForm.createControl());
		hyadesEditorPart.setPageText(0, TestUIPlugin.getString("W_OVERVIEW"));

		eventsForm = new EventForm(this, widgetFactory);
		hyadesEditorPart.addPage(eventsForm.createControl());
		hyadesEditorPart.setPageText(PAGE_EVENTS, TestUIPlugin.getString("W_EVENTS"));
		
		executionForm.updateTitle();
	}
	
	/**
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#editorFileChanged()
	 */
	public boolean editorFileChanged()
	{
		reload();
		if(getHyadesEditorPart().getActivePage() == PAGE_EVENTS)
		{
			if(getSelection().isEmpty())
			{
				try
				{
					setSelection(new StructuredSelection(getExecution().getExecutionHistory().getExecutionEvents().get(0)));
				}
				catch (RuntimeException e)
				{
				}
			}
		}
		
		return true;
	}

	/**
	 * @see org.eclipse.hyades.ui.util.IRefreshable#refreshContent(java.lang.Object)
	 */
	public void refreshContent(Object data)
	{
		executionForm.load();
		executionForm.updateTitle();
		
		eventsForm.load();
	}

	/**
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#getSelection()
	 */
	public IStructuredSelection getSelection()
	{
		switch(getHyadesEditorPart().getActivePage())
		{
			case PAGE_OVERVIEW:
				return new StructuredSelection(getExecution());
				
			case PAGE_EVENTS:
				return (IStructuredSelection)eventsForm.getSelection();
		}
		
		return StructuredSelection.EMPTY;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#pageActivated(int)
	 */
	public boolean pageActivated(int index)
	{
		switch(index)
		{
			case PAGE_OVERVIEW:
				return executionForm.activated();
				
			case PAGE_EVENTS:
				return eventsForm.activated();
		}
		
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#setSelection(org.eclipse.jface.viewers.IStructuredSelection)
	 */
	public void setSelection(IStructuredSelection structuredSelection)
	{
		if(structuredSelection.size() == 1)
		{
			Object object = structuredSelection.getFirstElement();
			if(object == getExecution())
			{
				// don't do anything.
			}
			else if(object instanceof TPFExecutionEvent)
			{
				getHyadesEditorPart().setActivePage(PAGE_EVENTS);
				eventsForm.selectReveal(structuredSelection);
			}
		}
	}	
}
