/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.model.ui;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.test.ui.IHyadesTestNavigatorConverter;
import org.eclipse.hyades.test.ui.IHyadesTestNavigatorFilter;
import org.eclipse.hyades.test.ui.IHyadesTestNavigatorProvider;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProviderChangedEvent;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.model.IWorkbenchAdapter;

/**
 * @author jgout
 */
public class TestNavigatorProvider implements ITreeContentProvider, ILabelProvider, ILabelProviderListener {

	private Map visibleElementFromFile;
	private Map visibleElementFromObject;
	private List filters;
	private List listeners;

	class fileExtensionElement {
		private IHyadesTestNavigatorProvider provider;
		private IHyadesTestNavigatorConverter converter;

		public fileExtensionElement(IHyadesTestNavigatorProvider provider, IHyadesTestNavigatorConverter converter) {
			this.provider = provider;
			this.converter = converter;
		}

		public IHyadesTestNavigatorConverter getConverter() {
			return converter;
		}

		public IHyadesTestNavigatorProvider getProvider() {
			return provider;
		}
	}

	public TestNavigatorProvider() {
		visibleElementFromFile = new HashMap();
		visibleElementFromObject = new HashMap();
		listeners = new LinkedList();
		filters = new LinkedList();
	}

	/**
	 * @param extName
	 * @param provider
	 */
	public void addVisibleFile(String extName, IHyadesTestNavigatorProvider provider, IHyadesTestNavigatorConverter converter) {
		//- remove '.' (dot) character if found
		if (extName.startsWith("."))
			extName = extName.substring(1);
		if (visibleElementFromFile.containsKey(extName)) {
			TestUIPlugin.logInfo(
				"Extension file: '." + extName + "' can't be registered as a visible element in the Test Navigator due to a previous registration by another plug-in");
			return;
		}
		//- register core provider as delegated provider change listener
		if(provider != null) provider.addListener(this);
		//- save data in the global map
		visibleElementFromFile.put(extName, new fileExtensionElement(provider, converter));
	}

	/**
	 * @param objectName
	 * @param provider
	 */
	public void addVisibleObject(String objectName, IHyadesTestNavigatorProvider provider) {
		if (visibleElementFromObject.containsKey(objectName)) {
			TestUIPlugin.logInfo(
				"Extension file: '." + objectName + "' can't be registered as a visible element in the Test Navigator due to a previous registration by another plug-in");
			return;
		}
		visibleElementFromObject.put(objectName, provider);
		//- register core provider as delegated provider change listener
		provider.addListener(this);
	}

	/**
	 * @param filter
	 */
	public void addFilter(IHyadesTestNavigatorFilter filter) {
		filters.add(filter);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
	 */
	public Object getParent(Object element) {
		if (element instanceof IResource)
			return ((IResource) element).getParent();
		return null;
	}

	private IHyadesTestNavigatorProvider getProviderFromFile(IFile file) {
		String fileExtension = file.getFileExtension();
		if (fileExtension != null) {
			fileExtensionElement fileExt = (fileExtensionElement) visibleElementFromFile.get(fileExtension);
			if(fileExt != null) return fileExt.getProvider();
		}
		return null;
	}

	private IHyadesTestNavigatorConverter getConverterFromFile(IFile file) {
		String fileExtension = file.getFileExtension();
		if (fileExtension != null) {
			fileExtensionElement fileExt = (fileExtensionElement) visibleElementFromFile.get(fileExtension);
			if(fileExt != null) return fileExt.getConverter();
		}
		return null;
	}

	private IHyadesTestNavigatorProvider getProviderFromObject(Object object) {
		return (IHyadesTestNavigatorProvider) visibleElementFromObject.get(object);
	}

	private boolean filter(Object element) {
		for (Iterator it = filters.iterator(); it.hasNext();) {
			IHyadesTestNavigatorFilter filter = (IHyadesTestNavigatorFilter) it.next();
			if (filter.isFiltered(element))
				return true;
		}
		return false;
	}

	private Object[] filterChildren(Object[] children) {
		if (children == null)
			return new Object[0];

		LinkedList res = new LinkedList();
		for (int i = 0; i < children.length; i++) {
			if (!filter(children[i]))
				res.add(children[i]);
		}
		return res.toArray();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
	 */
	public boolean hasChildren(Object parentElement) {
		if (parentElement == null)
			return false;
		if (parentElement instanceof IContainer) {
			IContainer container = (IContainer) parentElement;
			if (!container.exists())
				return false;

			IResource[] resources = null;
			try {
				resources = container.members();
			} catch (CoreException e) {
				TestUIPlugin.logError(e);
				return false;
			}
			for (int i = 0; i < resources.length; i++) {
				if (resources[i].getType() != IResource.FILE) {
					if (!filter(resources[i]))
						return true;
				} else {
					String fileExtension = resources[i].getFileExtension();
					if (fileExtension != null) {
						if (isVisibleFile((IFile)resources[i]) && !filter(resources[i])) return true;
					}
				}
			}
			return false;
		} else if (parentElement instanceof IFile) {
			//- delagating for declared files
			IFile res = (IFile) parentElement;
			IHyadesTestNavigatorProvider provider = getProviderFromFile(res);
			if (provider != null)
				return filterChildren(provider.getChildren(res)).length > 0;
		} else {
			//- delegating for declared objects
			IHyadesTestNavigatorProvider provider = (IHyadesTestNavigatorProvider) visibleElementFromObject.get(parentElement.getClass().getName());
			if (provider != null)
				return filterChildren(provider.getChildren(parentElement)).length > 0;
		}
		return false;
	}

	/**
	 * @param resource
	 * @return
	 */
	private boolean isVisibleFile(IFile file) {
		String fileExtension = file.getFileExtension();
		if (fileExtension != null) {
			return visibleElementFromFile.containsKey(fileExtension);
		}
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
	 */
	public Object[] getChildren(Object parentElement) {
		if (parentElement instanceof IContainer) {
			List children = new UniqueEList();
			IResource[] resources = null;
			try {
				resources = ((IContainer) parentElement).members();
			} catch (CoreException e) {
				TestUIPlugin.logError(e);
				return new Object[0];
			}

			for (int i = 0, maxi = resources.length; i < maxi; i++) {
				if (resources[i].getType() != IResource.FILE) {
					if (!filter(resources[i]))
						children.add(resources[i]);
				} else {
					if (!filter(resources[i])) {
						Object node = null;
						//- lookup first provider for this file
						IHyadesTestNavigatorProvider provider = getProviderFromFile((IFile) resources[i]);
						if (provider != null) {
							node = (IFile) resources[i];
						} else {
							//- if no provider provided it should have a converter
							IHyadesTestNavigatorConverter converter = getConverterFromFile((IFile) resources[i]);
							if (converter != null) {
								node = converter.getObjectFromResource(resources[i]);
								//- this object would be filtered ... try it
								if(filter(node)) continue;
							}
						}
						if (node != null && !filter(node))
							children.add(node);
					}
				}
			}
			return children.toArray();
		} else if (parentElement instanceof IFile) {
			IFile res = (IFile) parentElement;
			IHyadesTestNavigatorProvider provider = getProviderFromFile(res);
			if (provider != null)
				return filterChildren(provider.getChildren(res));
		} else {
			IHyadesTestNavigatorProvider provider = (IHyadesTestNavigatorProvider) visibleElementFromObject.get(parentElement.getClass().getName());
			if (provider != null)
				return filterChildren(provider.getChildren(parentElement));
		}
		return new Object[0];
	}

	private IWorkbenchAdapter getWorkbenchAdapter(Object object) {
		IWorkbenchAdapter workbenchAdapter = (IWorkbenchAdapter) Platform.getAdapterManager().getAdapter(object, IWorkbenchAdapter.class);
		if (workbenchAdapter == this)
			return null;

		return workbenchAdapter;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.ILabelProvider#getImage(java.lang.Object)
	 */
	public Image getImage(Object element) {
		if (element == null)
			return null;
		if (element instanceof IContainer) {


			ImageDescriptor descriptor;
			IWorkbenchAdapter workbenchAdapter = getWorkbenchAdapter(element);
			if (workbenchAdapter != null) {

				descriptor = workbenchAdapter.getImageDescriptor(element);
				if (descriptor != null)
					return descriptor.createImage();
			}
		} else if (element instanceof IFile) {

			return getProviderFromFile((IFile) element).getImage(element);
		} else
		{
			//- if not a typed element or a not overload typed element
			IHyadesTestNavigatorProvider provider = (IHyadesTestNavigatorProvider) visibleElementFromObject.get(element.getClass().getName());
			if (provider != null)
				return provider.getImage(element);
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.ILabelProvider#getText(java.lang.Object)
	 */
	public String getText(Object element) {
		if (element == null)
			return "";
		if (element instanceof IContainer) {
			IWorkbenchAdapter workbenchAdapter = getWorkbenchAdapter(element);
			if (workbenchAdapter != null) {

				return workbenchAdapter.getLabel(element);
			}
		} else if (element instanceof IFile) {
			return getProviderFromFile((IFile) element).getText(element);
		} else  
		{
			//- if not a typed element or a not overload typed element
			IHyadesTestNavigatorProvider provider = (IHyadesTestNavigatorProvider) visibleElementFromObject.get(element.getClass().getName());
			if (provider != null)
				return provider.getText(element);
		}
		return "";
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
	 */
	public Object[] getElements(Object inputElement) {
		if (inputElement instanceof IWorkspaceRoot)
			return ((IWorkspaceRoot) inputElement).getProjects();

		return getChildren(inputElement);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.IBaseLabelProvider#dispose()
	 */
	public void dispose() {
//		visibleElementFromFile.clear();
//		visibleElementFromObject.clear();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
	 */
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.IBaseLabelProvider#addListener(org.eclipse.jface.viewers.ILabelProviderListener)
	 */
	public void addListener(ILabelProviderListener listener) {
		listeners.add(listener);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.IBaseLabelProvider#isLabelProperty(java.lang.Object, java.lang.String)
	 */
	public boolean isLabelProperty(Object element, String property) {
		// TODO Auto-generated method stub
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.IBaseLabelProvider#removeListener(org.eclipse.jface.viewers.ILabelProviderListener)
	 */
	public void removeListener(ILabelProviderListener listener) {
		if (!listeners.remove(listener)) {
			TestUIPlugin.logInfo("Unable to remove listener '" + listener.toString());
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.ILabelProviderListener#labelProviderChanged(org.eclipse.jface.viewers.LabelProviderChangedEvent)
	 */
	public void labelProviderChanged(LabelProviderChangedEvent event) {
		for (Iterator it = listeners.iterator(); it.hasNext();) {
			ILabelProviderListener listener = (ILabelProviderListener) it.next();
			listener.labelProviderChanged(event);
		}
	}

	/** Returns the array of file extensions found in test navigator extender
	 * @return file extensions of visible files
	 */
	public String[] getVisibleFiles() {
		Set s = visibleElementFromFile.keySet();
		String[] res = new String[s.size()];
		for (Iterator it = s.iterator(); it.hasNext();) {
			res[0] = (String) it.next();
		}
		return res;
	}

}
