/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.action;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IWorkingSet;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.ActionContext;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.actions.AddBookmarkAction;
import org.eclipse.ui.actions.AddTaskAction;
import org.eclipse.ui.actions.DeleteResourceAction;
import org.eclipse.ui.actions.WorkingSetFilterActionGroup;
import org.eclipse.ui.dialogs.PropertyDialogAction;
import org.eclipse.ui.ide.IDEActionFactory;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.eclipse.ui.views.framelist.TreeFrame;
import org.eclipse.ui.views.navigator.ResourceNavigatorMessages;

import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.model.ui.RemoveChildrenAction;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;
import org.eclipse.hyades.ui.internal.action.CollapseAllAction;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.hyades.ui.internal.navigator.ITreeNavigator;
import org.eclipse.hyades.ui.internal.navigator.action.FrameListActionGroup;
import org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup;
import org.eclipse.hyades.ui.internal.navigator.action.NavigatorSorterActionGroup;
import org.eclipse.hyades.ui.internal.navigator.action.ToggleLinkingAction;

/**
 * Group action for the Test Navigator.
 * 
 * <p>This implementation is based on the 
 * {@link org.eclipse.ui.views.navigator.MainActionGroup} class.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class TestNavigatorActionGroup 
extends NavigatorActionGroup
{
	protected OpenEditorAction openAction;
	protected AddBookmarkAction addBookmarkAction;
	protected AddTaskAction addTaskAction;	
	protected PropertyDialogAction propertyDialogAction;
	protected CollapseAllAction collapseAllAction;
	protected ToggleLinkingAction toggleLinkingAction;
	protected ToggleFolderAction toggleFolderAction;
	protected ToggleEObjectChildrenAction toggleEObjectChildren;
	protected OpenReportAction openReportAction;
	protected GenerateAction generateAction;	
	protected RunAction runAction;	
		
	protected DeleteResourceAction deleteResourceAction;
	protected RemoveChildrenAction deleteChildEObject;
	
	protected NavigatorSorterActionGroup sorterActionGroup;
	protected NewActionGroup newActionGroup;
	protected WorkingSetFilterActionGroup workingSetFilterActionGroup;
	protected FrameListActionGroup frameListActionGroup;
	
	/**
	 * Constructor for NavigatorActionGroup
	 * @param testNavigator
	 */
	public TestNavigatorActionGroup(TestNavigator testNavigator)
	{
		super(testNavigator);
	}
	
	/**
	 * @see org.eclipse.ui.actions.ActionGroup#dispose()
	 */
	public void dispose()
	{
		newActionGroup.dispose();
		workingSetFilterActionGroup.dispose();
		sorterActionGroup.dispose();
		frameListActionGroup.dispose();
		generateAction.dispose();
		runAction.dispose();
		
		super.dispose();
	}

	/**
	 * @see org.eclipse.ui.actions.ActionGroup#setContext(org.eclipse.ui.actions.ActionContext)
	 */
	public void setContext(ActionContext context)
	{
		super.setContext(context);
		
		sorterActionGroup.setContext(context);
		newActionGroup.setContext(context);
		workingSetFilterActionGroup.setContext(context);
		frameListActionGroup.setContext(context); 
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup#createActions()
	 */
	protected void createActions()
	{
		super.createActions();
		Shell shell = getNavigator().getSite().getShell();
		
		openAction = new OpenEditorAction((TestNavigator)getNavigator(), TestUIPlugin.getString("ACT_OPEN"));
		addBookmarkAction = new AddBookmarkAction(shell);
		addTaskAction = new AddTaskAction(shell);
		propertyDialogAction = new PropertyDialogAction(shell, getNavigator().getViewer());	

		collapseAllAction = new CollapseAllAction(((ITreeNavigator)getNavigator()).getTreeViewer());
		collapseAllAction.useDefaultDecorators();

		toggleLinkingAction = new ToggleLinkingAction(getNavigator(), ResourceNavigatorMessages.getString("ToggleLinkingAction.text")); 
		toggleLinkingAction.setToolTipText(ResourceNavigatorMessages.getString("ToggleLinkingAction.toolTip")); 
		toggleLinkingAction.setImageDescriptor(getImageDescriptor("elcl16/synced.gif"));
		toggleLinkingAction.setHoverImageDescriptor(getImageDescriptor("elcl16/synced.gif"));

		toggleFolderAction = new ToggleFolderAction((TestNavigator)getNavigator(), TestUIPlugin.getString("ACT_SHOW_FLD_TXT"));
		toggleFolderAction.setToolTipText(TestUIPlugin.getString("ACT_SHOW_FLD_DSC"));
		TestUIImages.INSTANCE.setImageDescriptors(toggleFolderAction, TestUIImages.ACT_SHOW_FOLDER);

		toggleEObjectChildren = new ToggleEObjectChildrenAction((TestNavigator)getNavigator(), TestUIPlugin.getString("ACT_SHOW_EOBJ_CH_TXT"));
		toggleEObjectChildren.setToolTipText(TestUIPlugin.getString("ACT_SHOW_EOBJ_CH_DSC"));
		TestUIImages.INSTANCE.setImageDescriptors(toggleEObjectChildren, TestUIImages.ACT_SHOW_EOBJECT_CHILDREN);

		ISharedImages images = PlatformUI.getWorkbench().getSharedImages();
		deleteResourceAction = new DeleteResourceAction(shell);
		deleteResourceAction.setDisabledImageDescriptor(images.getImageDescriptor(ISharedImages.IMG_TOOL_DELETE_DISABLED));
		deleteResourceAction.setImageDescriptor(images.getImageDescriptor(ISharedImages.IMG_TOOL_DELETE));		
		deleteResourceAction.setHoverImageDescriptor(images.getImageDescriptor(ISharedImages.IMG_TOOL_DELETE_HOVER));
		
		deleteChildEObject = new RemoveChildrenAction(true)
		{
			protected List saveResources(Set changedResources)
			{
				TestNavigator testNavigator = (TestNavigator)getNavigator();
				testNavigator.getTreeViewer().remove(getStructuredSelection().toArray());
				
				testNavigator.setMonitoringResourceChange(false);
				try
				{
					return super.saveResources(changedResources);
				}
				finally
				{
					testNavigator.setMonitoringResourceChange(true);
				}
			}
		};
		deleteChildEObject.setStructuredViewer(((TestNavigator)getNavigator()).getViewer());
		deleteChildEObject.setShowErrorDialog(true);
		
		openReportAction = new OpenReportAction();	
		generateAction = new GenerateAction();	
		runAction = new RunAction();			
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup#createActionGroups()
	 */
	protected void createActionGroups()
	{
		super.createActionGroups();
		sorterActionGroup = new NavigatorSorterActionGroup(getNavigator());
		newActionGroup = new NewActionGroup((TestNavigator)getNavigator());
		
		IPropertyChangeListener workingSetUpdater = new IPropertyChangeListener()
		{
			public void propertyChange(PropertyChangeEvent event)
			{
				String property = event.getProperty();
				
				if (WorkingSetFilterActionGroup.CHANGE_WORKING_SET.equals(property))
				{
					INavigator navigator = getNavigator();
					Object newValue = event.getNewValue();
					
					if (newValue instanceof IWorkingSet)
					{	
						((TestNavigator)navigator).setWorkingSet((IWorkingSet) newValue);
					}
					else if (newValue == null)
					{
						((TestNavigator)navigator).setWorkingSet(null);
					}
				}
			}
		};
		workingSetFilterActionGroup = new WorkingSetFilterActionGroup(getNavigator().getViewer().getControl().getShell(), workingSetUpdater);
		workingSetFilterActionGroup.setWorkingSet(((TestNavigator)getNavigator()).getWorkingSet());
		frameListActionGroup = new FrameListActionGroup((TestNavigator)getNavigator());
	}

	/**
	 * @see org.eclipse.ui.actions.ActionGroup#fillContextMenu(org.eclipse.jface.action.IMenuManager)
	 */
	public void fillContextMenu(IMenuManager menu)
	{
		super.fillContextMenu(menu);
		IStructuredSelection structuredSelection = (IStructuredSelection)getContext().getSelection();
		IStructuredSelection resourceStructuredSelection = getResourceSelection(structuredSelection);
		
		newActionGroup.fillContextMenu(menu);
		frameListActionGroup.fillContextMenu(menu);
		
		openAction.selectionChanged(structuredSelection);
		if(openAction.isApplicableForSelection())
		{
			menu.add(openAction);
			menu.add(new Separator());			
		}
		
		generateAction.selectionChanged(structuredSelection);
		if(generateAction.isApplicableForSelection())
		{
			menu.add(generateAction);
		}
	
		runAction.selectionChanged(structuredSelection);
		if(runAction.isApplicableForSelection())
		{
			menu.add(runAction);
		}

		deleteResourceAction.setEnabled(false);		
		if(resourceStructuredSelection.size() == structuredSelection.size())
		{
			menu.add(new Separator());
			deleteResourceAction.selectionChanged(resourceStructuredSelection);
			menu.add(deleteResourceAction);
			menu.add(new Separator());
			
			addBookmarkAction.selectionChanged(resourceStructuredSelection);
			if(addBookmarkAction.isEnabled())
				menu.add(addBookmarkAction);
		}
		if(!deleteResourceAction.isEnabled())
		{
			menu.add(new Separator());
			deleteChildEObject.selectionChanged(structuredSelection);
			if(deleteChildEObject.isEnabled())
				menu.add(deleteChildEObject);
		}

		openReportAction.selectionChanged(structuredSelection);
		if(openReportAction.isApplicableForSelection())
		{
			menu.add(openReportAction);
		}
		
		if(structuredSelection.size() == 1)
		{
			menu.add(new Separator());
			propertyDialogAction.selectionChanged(structuredSelection);
			if(propertyDialogAction.isApplicableForSelection())
				menu.add(propertyDialogAction);
		}
	}
		
	/**
	 * @see org.eclipse.ui.actions.ActionGroup#fillActionBars(org.eclipse.ui.IActionBars)
	 */
	public void fillActionBars(IActionBars actionBars)
	{
		actionBars.setGlobalActionHandler(ActionFactory.PROPERTIES.getId(),	propertyDialogAction);
		actionBars.setGlobalActionHandler(IDEActionFactory.BOOKMARK.getId(), addBookmarkAction);
		actionBars.setGlobalActionHandler(IDEActionFactory.ADD_TASK.getId(), addTaskAction);
		actionBars.setGlobalActionHandler(ActionFactory.DELETE.getId(), deleteResourceAction);

		frameListActionGroup.fillActionBars(actionBars);
		newActionGroup.fillActionBars(actionBars);
		workingSetFilterActionGroup.fillActionBars(actionBars);
		sorterActionGroup.fillActionBars(actionBars);
		
		super.fillActionBars(actionBars);

		IMenuManager menu = actionBars.getMenuManager();
		menu.add(new Separator());
		menu.add(toggleLinkingAction);
		//menu.add(toggleFolderAction);
		menu.add(toggleEObjectChildren);
		
		IToolBarManager toolBar = actionBars.getToolBarManager();
		toolBar.add(new Separator());
		toolBar.add(collapseAllAction);		
		toolBar.add(toggleLinkingAction);
		//toolBar.add(toggleFolderAction);
	}
	
	/**
	 * @see org.eclipse.ui.actions.ActionGroup#updateActionBars()
	 */
	public void updateActionBars()
	{
		TestNavigator testNavigator = (TestNavigator)getNavigator();
		Object currentFrameInput = ((TreeFrame)testNavigator.getFrameList().getCurrentFrame()).getInput();
		toggleFolderAction.setEnabled((currentFrameInput == null) || (testNavigator.getInitialViewerInput() == currentFrameInput));
		toggleEObjectChildren.setEnabled((currentFrameInput == null) || (!(currentFrameInput instanceof EObject)));
		
		IStructuredSelection structuredSelection = (IStructuredSelection) getContext().getSelection();
		IStructuredSelection resourceStructuredSelection = getResourceSelection(structuredSelection);
		
		propertyDialogAction.setEnabled(structuredSelection.size() == 1);
		addBookmarkAction.selectionChanged(resourceStructuredSelection);
		deleteResourceAction.selectionChanged(resourceStructuredSelection);
		addTaskAction.selectionChanged(structuredSelection);	
	
		frameListActionGroup.updateActionBars();
		sorterActionGroup.updateActionBars();
		newActionGroup.updateActionBars();
		workingSetFilterActionGroup.updateActionBars();
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup#runDefaultAction(org.eclipse.jface.viewers.IStructuredSelection)
	 */
	public void runDefaultAction(IStructuredSelection selection)
	{
		openAction.selectionChanged(selection);
		openAction.run();
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup#handleKeyPressed(org.eclipse.swt.events.KeyEvent)
	 */
	public void handleKeyPressed(KeyEvent event)
	{
		if(event.character == SWT.DEL && (event.stateMask == 0))
		{
			if(deleteResourceAction.isEnabled())
				deleteResourceAction.run();
		}

		super.handleKeyPressed(event);
	}

	/**
	 * Returns an structured selection with resource elements based on the given
	 * structured selection.
	 * @param structuredSelection
	 * @return IStructuredSelection
	 * @see TestNavigator#getFile(Object)
	 */
	protected IStructuredSelection getResourceSelection(IStructuredSelection structuredSelection)
	{
		TestNavigator testNavigator = (TestNavigator)getNavigator();
		List resources = new ArrayList();
		for(Iterator i = structuredSelection.iterator(); i.hasNext();)
		{
			Object object = i.next();
			if((object instanceof IResource) && (((IResource)object).getType() != IResource.FILE))
			{
				resources.add(object);
			}
			else
			{
				IFile file = testNavigator.getFile(object);
				if(file != null)
					resources.add(file);				
			}			
		}
		return new StructuredSelection(resources);
	}

	/**
	 * Returns the image descriptor from the eclipse UI plugin with the given relative 
	 * path.
	 * @return ImageDescriptor;
	 */
	protected ImageDescriptor getImageDescriptor(String relativePath)
	{
		String iconPath = "icons/full/";
		try
		{
			AbstractUIPlugin plugin = (AbstractUIPlugin) Platform.getPlugin(PlatformUI.PLUGIN_ID);
			URL installURL = plugin.getDescriptor().getInstallURL();
			URL url = new URL(installURL, iconPath + relativePath);
			return ImageDescriptor.createFromURL(url);
		}
		catch(MalformedURLException e)
		{
			return ImageDescriptor.getMissingImageDescriptor();
		}
	}	
}
