/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.logging.adapter.config;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.util.Hashtable;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.logging.Log;
import org.apache.xml.serialize.LineSeparator;
import org.apache.xml.serialize.OutputFormat;
import org.apache.xml.serialize.XMLSerializer;
import org.eclipse.hyades.logging.adapter.Adapter;
import org.eclipse.hyades.logging.adapter.AdapterException;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.events.ICommonBaseEvent;
import org.eclipse.hyades.logging.parsers.LogParserException;
import org.eclipse.hyades.logging.parsers.Parser;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * This class is used as in interface between the UI and the generic adapter.
 * The UI side is expecting an implementor of IParser and will call setUserInput() and parse(log).
 * This class will implement the plumbing to map the two methods above to those in the generic adapter.
 */
public class StaticParserWrapper extends Parser {
	private Adapter adapter;
	private String parserName;
	private String agentName;
	private String key;
//	protected Plugin thisPlugin;
//	protected ResourceBundle resourceBundle;
	protected String newDirectory = null;
	protected String newFileName = null;
	protected String currentPlugin;

	// Hash table to store all active loggers
	private static Hashtable activeWrappers = new Hashtable();

	public StaticParserWrapper() {
		// Generate a key for this instance
		key = String.valueOf(System.currentTimeMillis());
		adapter = new Adapter();
		parserName = new String(Messages.getString("HyadesGAUnknownNameTag"));
		agentName = new String(Messages.getString("HyadesGAUnknownNameTag"));

		currentPlugin = "org.eclipse.hyades.logging.adapter.config";
	}

	public static Log getLogInstance(String logKey) {
		return (Log)activeWrappers.remove(logKey);
	}
	
	/**
	 * @see org.eclipse.hyades.logging.parsers.IParser.parse(org.apache.commons.logging.Log)
	 */
	public void parse(Log argLogger) throws LogParserException {
		try {
			activeWrappers.put(key, argLogger);
			adapter.start(false,false);
		} catch (AdapterException e) {
		}
	}

	public void setUserInput(Hashtable table) throws LogParserException {
		// Pass the instance key to the parser
		table.put("loggerKey", key);

		String newConfigFile = getNewConfigFile(table);
		adapter.setContextConfigPath(newConfigFile);
		adapter.setComponentConfigPath(newConfigFile);
	}
	
	/**
	 * Since this is not a real parser we don't need to implement the getName() method
	 * 
	 * @see org.eclipse.hyades.logging.parsers.IParser.getName()
	 */
	public String getName() {

		return agentName;
	}

	// 
	/**
	 * Since this is not a real parser we don't need to implement the parseNext() method
	 * 
	 * @see org.eclipse.hyades.logging.parsers.IParser.parseNext()
	 */
	public ICommonBaseEvent[] parseNext() throws LogParserException {
		return null;
	}

	// Since this is not a real parser we don't need to implement the getVersion() method
	public String getVersion() {
		return null;
	}

	/**
	 * Method to get an input stream to the config file
	 * @param fileName - config file name
	 * @param table - table of user input parameters
	 * @return InputStream
	 * @throws LogParserException 
	 */	
	private InputStream getConfigInputStream(String fileName, Hashtable table) throws LogParserException
	{
		String fileSeparator = System.getProperty("file.separator");
		// Get input stream for config file relative to the current directory
		try {
			// If we have the wrong file separator in the config file path then
			if (fileName.indexOf('/') != -1 && fileSeparator != "/") {
				// Replace file separators
				fileName = fileName.replace('/', fileSeparator.charAt(0));
			}
			// Remove ./
			if (fileName.startsWith("." + fileSeparator)) {
				fileName = fileName.substring(2);
			}
			return new FileInputStream(fileName);
		} catch (FileNotFoundException exc) {
			String fileName2 = (String) table.get(Messages.getString("HyadesGAWebClientConfigFileRootAttributeName")) + fileSeparator + fileName;
			try {
				return new FileInputStream(fileName2);
			}  catch (FileNotFoundException exc2) {
				throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_No_Config_File_Error", fileName));
			} catch (SecurityException exc2) {
				throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Open_Error", fileName));
			}
		} catch (SecurityException exc) {
			throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Open_Error", fileName));
		}		
	}

	/**
	 * Method to create a new config file based on an existing config file and 
	 * a hash table of configuration parameter values
	 * @param table - hash table of configuration parameters
	 * @return String - name of new config file
	 * @throws LogParserException 
	 */		
	private String getNewConfigFile(Hashtable table) throws LogParserException
	{
		DocumentBuilderFactory docFactory = DocumentBuilderFactory.newInstance();;
		Document doc = null;

		// Resolve the version number of the log in order to load the correct config file
		String version = (String) (table.get(Messages.getString("HyadesGALogVersionTagName")));
		String config_file = null;
		
		// If no version attribute given, use default
		if(version == null)
		{
			version = Messages.getString("HyadesGADefaultLogVersionTagName");
		}

		// Get the config file defined for this version
		config_file =  (String) (table.get(version));
		
		if(config_file == null)
		{
			// Cannot find the corresponding config file, use default
			config_file = (String) (table.get(Messages.getString("HyadesGADefaultLogVersionTagName")));
			
			// If the default (which should be) is not defined, throw an exception
			if(config_file == null) {
				throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_No_Config_File_Error"));
			}
		}

		// Get the input stream for the config file

		InputStream inStream = null;
		String configFileDirectory = null;
		
		try {
			// Try to get the input stream using the plugin

			Boolean append = Boolean.TRUE;

			Class pathClass = Class.forName("org.eclipse.core.runtime.Path");
			Constructor pathConstructor = pathClass.getConstructor(new Class[] {config_file.getClass()});
			Object path = pathConstructor.newInstance(new Object[] {config_file});
			
			Class platformClass = Class.forName("org.eclipse.core.runtime.Platform");			
			Class pluginClass = Class.forName("org.eclipse.core.runtime.Plugin");
			
			Object plugin = platformClass.getMethod("getPlugin", new Class[] { currentPlugin.getClass()}).invoke(null, new Object[] { currentPlugin });

			inStream = (InputStream)(pluginClass.getMethod("openStream", new Class[] {Class.forName("org.eclipse.core.runtime.IPath"), boolean.class}).invoke(plugin, new Object[] {path, append}));

			configFileDirectory = new File(((URL)(pluginClass.getMethod("find", new Class[] {Class.forName("org.eclipse.core.runtime.IPath")}).invoke(plugin, new Object[] {path}))).getFile()).getParentFile().getAbsolutePath();
		} catch	(ClassNotFoundException e) {
			// We must be running outside of eclipse so try to find the config
			// file relative to the current directory

			inStream = getConfigInputStream(config_file, table);
			configFileDirectory = new File(config_file).getParentFile().getAbsolutePath();
		} catch	(InvocationTargetException e) {
			// We may be running outside of eclipse but the eclipse classes may be in
			// the path so try to find the config file relative to the current directory
			// To Do:  The exception should be logged, however, in case we are running inside
			// eclipse and there was an error from the invoked method calls.

			inStream = getConfigInputStream(config_file, table);
			configFileDirectory = new File(config_file).getParentFile().getAbsolutePath();
		} catch (Exception e) {
			// Some other exception occurred - throw a LogParserException
			throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Open_Error", config_file), e);
		}

		String logFilePath = ((String)(table.get(Messages.getString("HyadesGAInputLogFilenameTag"))));
	    File logFile = null;

	    if((logFilePath != null) && (logFilePath.trim().length() > 0)){
		    logFile = new File(logFilePath.trim());
	    }

		try {
			doc = docFactory.newDocumentBuilder().parse(inStream);
		} catch (SAXException e) {
			return null;
		} catch (IOException e) {
			return null;
		} catch (ParserConfigurationException e) {
			return null;
		} catch (Exception e) {
			return null;
		}
		/**
		 * Get the sensor config - there should only be one sensor - and fill in
		 * the properties and attributes with the values from the input hash table
		 */
		NodeList sensorList = doc.getElementsByTagName(Messages.getString("HyadesGASensorTagName"));
		int sensorCount = sensorList.getLength();
		for (int i = 0; i < sensorCount; ++i)
		{
			Element sensorElement = (Element) sensorList.item(i);
			
			// We should always have a sensor instance but check in case an invalid
			// config file is used.
			if (sensorElement != null) {
				Element sensorNode;
				Element sensorPropertyElement;
				Element sensorTypeInstance = null;
				String propValue;
				String propName;
				boolean directoryProperty = false;
				boolean fileNameProperty = false;
				Element directoryPropertyElement = null;
				Element fileNamePropertyElement = null;
				
				// Get the sensor children (properties or sensor type instances)
				NodeList sensorNodes = sensorElement.getChildNodes();
				for (int k = 0; k < sensorNodes.getLength(); k++) {
					if (sensorNodes.item(k).getNodeType() == Node.ELEMENT_NODE) {
						sensorNode = (Element) sensorNodes.item(k);
						// Process the sensor property
						if (sensorNode.getTagName().equals(Messages.getString("HyadesGAPropertyTagName"))) {
							sensorPropertyElement = sensorNode;

							propName = sensorPropertyElement.getAttribute(Messages.getString("HyadesGAPropertyNameTag"));
							propValue = sensorPropertyElement.getAttribute(Messages.getString("HyadesGAPropertyValueTag"));
							
							if (propName.equals(Messages.getString("HyadesGAdirectoryAttributeName"))) {
								sensorPropertyElement.setAttribute(Messages.getString("HyadesGAPropertyValueTag"), logFile.getParent());
								directoryPropertyElement = sensorPropertyElement;
							}
							else if (propName.equals(Messages.getString("HyadesGAfileNameAttributeName"))) {
								sensorPropertyElement.setAttribute(Messages.getString("HyadesGAPropertyValueTag"), logFile.getName());
								fileNamePropertyElement = sensorPropertyElement;
							}
							else if (propName.equals(Messages.getString("HyadesGAconverterCmdAttributeName"))) {
								if (propValue != null && propValue.length() > 0) {
									String converter = modifyConverter(propValue, configFileDirectory, table, logFile);
									sensorPropertyElement.setAttribute(Messages.getString("HyadesGAPropertyValueTag"), converter);
								}
							}
							else if (propName.equals(Messages.getString("HyadesGAstaticParserClassAttributeName"))) {
								parserName = propValue;							
							}
							else {
								propValue = (String)table.get(propName);
								if(propValue != null)
								{
									sensorPropertyElement.setAttribute(Messages.getString("HyadesGAPropertyValueTag"), propValue.trim());
								}
							}
						}
						// Process the Generic sensor type instance
						else if (sensorNode.getTagName().equals(Messages.getString("HyadesGAGenericSensorTagName"))) {
							sensorTypeInstance = sensorNode;
							// Get the sensor properties for this sensor type instance
							NodeList sensorPropertyList = sensorTypeInstance.getElementsByTagName(Messages.getString("HyadesGASensorPropertyTagName"));
							int sensorPropertyCount = sensorPropertyList.getLength();
							for(int j = 0; j < sensorPropertyCount; ++j)
							{
								sensorPropertyElement = (Element) sensorPropertyList.item(j);
								propName = sensorPropertyElement.getAttribute(Messages.getString("HyadesGASensorPropertyNameTag"));
								propValue = sensorPropertyElement.getAttribute(Messages.getString("HyadesGASensorPropertyValueTag"));
							
								if (propName.equals(Messages.getString("HyadesGAdirectoryAttributeName"))) {
									sensorPropertyElement.setAttribute(Messages.getString("HyadesGASensorPropertyValueTag"), logFile.getParent());
									directoryPropertyElement = sensorPropertyElement;
								}
								else if (propName.equals(Messages.getString("HyadesGAfileNameAttributeName"))) {
									sensorPropertyElement.setAttribute(Messages.getString("HyadesGASensorPropertyValueTag"), logFile.getName());
									fileNamePropertyElement = sensorPropertyElement;
								}
								else if (propName.equals(Messages.getString("HyadesGAconverterCmdAttributeName"))) {
									if (propValue != null && propValue.length() > 0) {
										String converter = modifyConverter(propValue, configFileDirectory, table, logFile);
										sensorPropertyElement.setAttribute(Messages.getString("HyadesGASensorPropertyValueTag"), converter);
									}
								}
								else if (propName.equals(Messages.getString("HyadesGAstaticParserClassAttributeName"))) {
									parserName = propValue;							
								}
								else {
									propValue = (String)table.get(propName);
									if(propValue != null)
									{
										sensorPropertyElement.setAttribute(Messages.getString("HyadesGASensorPropertyValueTag"), propValue.trim());
									}
								}
							}
						}
						// Process the sensor type instance
						else if (sensorNode.getTagName().equals(Messages.getString("HyadesGAStaticParserSensorTagName")) ||
								 sensorNode.getTagName().equals(Messages.getString("HyadesGASingleFileSensorTagName"))) {
							sensorTypeInstance = sensorNode;
							// Get the sensor properties for this sensor type instance
							NodeList sensorPropertyList = sensorTypeInstance.getElementsByTagName(Messages.getString("HyadesGASensorPropertyTagName"));
							int sensorPropertyCount = sensorPropertyList.getLength();
							for(int j = 0; j < sensorPropertyCount; ++j)
							{
								sensorPropertyElement = (Element) sensorPropertyList.item(j);
								propName = sensorPropertyElement.getAttribute(Messages.getString("HyadesGASensorPropertyNameTag"));
								propValue = (String)table.get(propName);
								if(propValue != null)
								{
									sensorPropertyElement.setAttribute(Messages.getString("HyadesGASensorPropertyValueTag"), propValue.trim());
								}
							}

							// Get the sensor type attributes
							NamedNodeMap sensorTypeAttributeList = sensorTypeInstance.getAttributes();
							int sensorTypeAttributeCount = sensorTypeAttributeList.getLength();
							String converter;
				
							for(int j = 0; j < sensorTypeAttributeCount; ++j)
							{
								Node sensorTypeAttribute = sensorTypeAttributeList.item(j);
								String attrName = sensorTypeAttribute.getNodeName();
								// Modify the attribute based on user input
								if(attrName.equals(Messages.getString("HyadesGAdirectoryAttributeName")))
								{
									sensorTypeAttribute.setNodeValue(logFile.getParent());
								}
								else if(attrName.equals(Messages.getString("HyadesGAfileNameAttributeName")))
								{
									sensorTypeAttribute.setNodeValue(logFile.getName());
								}
								else if(attrName.equals(Messages.getString("HyadesGAstaticParserClassAttributeName")))
								{
									parserName = sensorTypeAttribute.getNodeValue();
								}
								else if(attrName.equals(Messages.getString("HyadesGAconverterCmdAttributeName")))
								{
									converter = modifyConverter(sensorTypeAttribute.getNodeValue(), configFileDirectory, table, logFile);
									sensorTypeAttribute.setNodeValue(converter);
								}
								else
								{
									propValue = (String)table.get(attrName);
									if(propValue != null)
									{
										sensorTypeAttribute.setNodeValue(propValue.trim());
									}
								}
							}
						}
					}
				}

				// Get the sensor attributes
				NamedNodeMap sensorAttributeList = sensorElement.getAttributes();
				int sensorTypeAttributeCount = sensorAttributeList.getLength();
				
				for(int k = 0; k < sensorTypeAttributeCount; ++k)
				{
					Node sensorTypeAttribute = sensorAttributeList.item(k);
					String attrName = sensorTypeAttribute.getNodeName();
					
					// Modify the attribute based on user input
					propValue = (String)table.get(attrName);
					if(propValue != null && propValue.length() > 0)
					{
						sensorTypeAttribute.setNodeValue(propValue.trim());
					}
				}

				// Check if the directory of the log file to be parsed was updated by modifyConverter 				
				if (newDirectory != null) {
					// If so update it in the config file
					if (directoryPropertyElement != null) {
						directoryPropertyElement.setAttribute(Messages.getString("HyadesGASensorPropertyValueTag"), newDirectory);
					}
					else if (sensorTypeInstance != null) {
						// Set the attribute
						Node sensorAttribute = sensorTypeInstance.getAttributeNode(Messages.getString("HyadesGAdirectoryAttributeName"));
						sensorAttribute.setNodeValue(newDirectory);					
					}
				}
				
				// Check if the name of the log file to be parsed was updated by modifyConverter 				
				if (newFileName != null) {
					// If so update it in the config file
					if (fileNameProperty) {
						fileNamePropertyElement.setAttribute(Messages.getString("HyadesGASensorPropertyValueTag"), newFileName);
					}
					else if (sensorTypeInstance != null) {
						// Set the attribute
						Node sensorAttribute = sensorTypeInstance.getAttributeNode(Messages.getString("HyadesGAfileNameAttributeName"));
						sensorAttribute.setNodeValue(newFileName);					
					}	
				}
			}
		}

		/**
		 * Add the logger key to the outputter so the the outputter can write the the UI
		 * logger directly - there should only be one outputter.
		 */
		NodeList outputterList = doc.getElementsByTagName(Messages.getString("HyadesGAOutputterTagName"));
		int outputterCount = outputterList.getLength();
		for (int i = 0; i < outputterCount; ++i) {
			Element outputterElement = (Element) outputterList.item(i);
			Element outputterProperty = doc.createElement(Messages.getString("HyadesGAPropertyTagName"));
			outputterProperty.setAttribute(Messages.getString("HyadesGAPropertyValueTag"), key);
			outputterProperty.setAttribute(Messages.getString("HyadesGAPropertyNameTag"),Messages.getString("HyadesGAStaticParserLoggerKeyName"));
			outputterElement.appendChild(outputterProperty);
		}


		// Save the resulting xml

		// Create a temporary file for the new adapter configuration
		File newFile;
		try {		
			newFile = File.createTempFile("GLA", ".adapter");
		}
		catch (Exception e){
			// If a temporary file cannot be generated in the OS tmp directory
			// try creating it in the same directory as the current config file.
			File oldFile = new File(config_file);
			String tmpFileName = new String("GLA" + System.currentTimeMillis() + ".adapter");
			newFile = new File(oldFile.getParent(), tmpFileName);
		}
		// Ensure the temporary file gets deleted
		newFile.deleteOnExit();
		
		// Save the configuration in the temporary file
		saveConfigFile(doc, newFile);

		return newFile.getAbsolutePath();
	}

	/**
	 * Method to perform any modifications to the converter string based on user
	 * input from the UI
	 * @param converter  - converter string to be updated
	 * @param configFileDirectory  - the absolute path to the directory of the config file 
	 * @param table  - of user input values
	 * @param logFile - log file to be parsed
	 * @return String - new converter string
	 */
	protected String modifyConverter(String converter, String configFileDirectory, Hashtable table, File logfile) {
		return converter;
	}

	/**
	 * Save the new adapter configuration to the specified file
	 * @param document - adapter configuration
	 * @param configFile - new adapter file
	 */	
	private void saveConfigFile(Document document, File configFile)
	{
		String platform = System.getProperty("os.name", null);
		
		OutputFormat format = new OutputFormat(document);
		
		// Set platform specific attributes
		if (platform.startsWith("Windows")) {
			format.setLineSeparator(LineSeparator.Windows);
		}
		else if(platform.equals("z/OS") || platform.equals("OS/390")) {
			format.setEncoding("IBM-1047");
		}
		
		format.setIndenting(true);
		format.setLineWidth(0);             
		format.setPreserveSpace(true);
		XMLSerializer serializer;
		try {
			serializer = new XMLSerializer(new FileWriter(configFile), format);
			serializer.asDOMSerializer();
			serializer.serialize(document);
		} catch (IOException e) {
		}
	}

}
