/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/*
 * Created on Jul 29, 2003
 *
 */
package org.eclipse.hyades.log.ui.internal.wizards;

import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IPluginRegistry;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.internal.execution.local.common.Constants;
import org.eclipse.hyades.log.ui.internal.LogUIPlugin;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.security.util.GridUtil;
import org.eclipse.hyades.security.util.TString;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.ui.internal.util.XMLUtil;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.viewers.ColumnLayoutData;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.FontMetrics;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * @author apnan
 *
 */
public class ImportLogWizardPage extends WizardPage implements SelectionListener{

	public static final int NEW_LOGSET = 0;
	public static final int OPEN_LOGSET = 1;
	public static final int EDIT_LOGSET = 2;
	public static final int RENAME_LOGSET = 3;
	private final String LOGSETS_KEY = "logsets";
	private final String LOGSET_TAG = "logset";
	private final String LOGELEMENT_TAG = "LogFileElement";
	private final String DEFAULT_LOGSET = "default";
	public static final int ADD = 4;
	public static final int EDIT = 5;
	public static final int BUTTON_WIDTH = 90;

	private LogParserUI _parserUI;
	//private LogFileDetailsUI _detailsUI;
	private Table logsTable;
	private TableViewer fTableViewer;
	private List logElements = new ArrayList();
	private Button addBtn, removeBtn, editBtn, newLogSetBtn, openLogSetBtn, editLogSetBtn;
	private LogFileElement selection;
	private LogFileDetailsUI dialog;
	private LogSetDialog logSetDialog;
	private Label logset_desc;
	private Label type_label;
	private LogSetManager lsMgr;
	private int dialog_type;
	
	public ImportLogWizardPage() {
		
		super("ImportLogWizardPage");		
		setTitle(LogUIPlugin.getResourceString("LOG_PARSER_PAGE_TITLE"));
		setDescription(LogUIPlugin.getResourceString("LOG_PARSER_PAGE_DESC"));
	}
	
	public void createControl(Composite parent) {
	
		Composite result = new Composite(parent, SWT.NONE);
		result.setLayoutData(GridUtil.createFill());
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		result.setLayout(layout);
			
		logset_desc = new Label(result, SWT.NONE);
		GridData data = GridUtil.createHorizontalFill();
		data.horizontalSpan = 2;		
		logset_desc.setLayoutData(data);		
		logset_desc.setFont(JFaceResources.getBannerFont());
		
		Label logs_desc = new Label(result, SWT.NONE);
		data = GridUtil.createHorizontalFill();
		data.horizontalSpan = 2;		
		logs_desc.setLayoutData(data);
		logs_desc.setText(LogUIPlugin.getResourceString("STR_LOG_PDESC"));
		
		createLogTable(result);

		Composite buttonGrp = new Composite(result, SWT.NONE);	
		GridData btnData = GridUtil.createFill();
		GridLayout btnLayout = new GridLayout();
		btnLayout.numColumns = 1;		
		btnData.verticalAlignment = GridData.BEGINNING;
		buttonGrp.setLayout(btnLayout);
		buttonGrp.setLayoutData(btnData);		
		int len = computeButtonsMinWidth(result);
		createLogButtons(buttonGrp, len);
		createLogSetButtons(buttonGrp, len);					

		type_label = new Label(result, SWT.WRAP);
		data = GridUtil.createHorizontalFill();
		data.horizontalSpan = 2;
		data.heightHint = 30;		
		type_label.setLayoutData(data);

		loadDefaultLogSet();
			
		setControl(result);
				
		setPageComplete(false);
		
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
		parent,
			LogUIPlugin.getPluginId() + ".pp0000");
			
		parent.setFocus();
		
		
	}
		
	/**
	 * 
	 */
	protected void loadDefaultLogSet() {

		lsMgr = new LogSetManager();
		lsMgr.initDefault();		
		lsMgr.loadLogSets();
		
	}

	private void createLogTable(Composite parent){
		
		logsTable = new Table(parent, SWT.BORDER|SWT.H_SCROLL|SWT.V_SCROLL|SWT.FULL_SELECTION|SWT.MULTI);
		GridData tblData = GridUtil.createFill();
		tblData.heightHint = 75;
		tblData.widthHint = 330;	
		logsTable.setLayoutData(tblData);

		TableColumn column0= new TableColumn(logsTable, SWT.NONE); 

		TableColumn column1= new TableColumn(logsTable, SWT.NONE);		
		column1.setText(LogUIPlugin.getResourceString("STR_IMPORT_LOG_HOST")); 

		TableColumn column2= new TableColumn(logsTable, SWT.NONE);
		column2.setText(LogUIPlugin.getResourceString("STR_IMPORT_LOG_PATH"));
		
		ColumnLayoutData c0Layout = new ColumnWeightData(10, false);		
		// path column
		ColumnLayoutData c1Layout = new ColumnWeightData(50, true);

		// host column
		ColumnLayoutData c2Layout = new ColumnWeightData(125, true);

		// set columns in Table layout
		TableLayout tblLayout = new TableLayout();
		tblLayout.addColumnData(c0Layout);
		tblLayout.addColumnData(c1Layout);
		tblLayout.addColumnData(c2Layout);
		logsTable.setLayout(tblLayout);
		logsTable.setHeaderVisible(true);
		logsTable.setLinesVisible(true);
		fTableViewer= new TableViewer(logsTable);		

		fTableViewer.setLabelProvider(new TableLabelProvider());
		
		fTableViewer.setContentProvider(new TableContentProvider());
		fTableViewer.setInput(logElements);
		
		logsTable.addSelectionListener(this);
		
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
		logsTable,
			LogUIPlugin.getPluginId() + ".pp0000");
	
	}
	
	private void createLogButtons(Composite parent, int len){
						
		addBtn = new Button(parent, SWT.PUSH);
		addBtn.setText(LogUIPlugin.getResourceString("STR_IMPORT_LOG_ADD"));
		GridData btnData = GridUtil.createHorizontalFill();
		btnData.widthHint = len;
		btnData.horizontalAlignment = GridData.CENTER;
		addBtn.setLayoutData(btnData);		
		
		editBtn = new Button(parent, SWT.PUSH);
		editBtn.setText(LogUIPlugin.getResourceString("STR_IMPORT_LOG_EDIT"));
		btnData = GridUtil.createHorizontalFill();
		btnData.widthHint = len;
		btnData.horizontalAlignment = GridData.CENTER;
		editBtn.setLayoutData(btnData);
			
		removeBtn = new Button(parent, SWT.PUSH);
		removeBtn.setText(LogUIPlugin.getResourceString("STR_IMPORT_LOG_REMOVE"));
		btnData = GridUtil.createHorizontalFill();
		btnData.widthHint = len;
		btnData.horizontalAlignment = GridData.CENTER;
		removeBtn.setLayoutData(btnData);
		
		addBtn.addSelectionListener(this);
		editBtn.addSelectionListener(this);
		removeBtn.addSelectionListener(this);
		
		updateButtons();
		
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
		addBtn,
			LogUIPlugin.getPluginId() + ".pp0010");
			
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
		editBtn,
			LogUIPlugin.getPluginId() + ".pp0020");
			
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
		removeBtn,
			LogUIPlugin.getPluginId() + ".pp0030");

	}
	
	private void createLogSetButtons(Composite parent, int len){
				
		Label space = new Label(parent, SWT.NONE);
		GridData data = GridUtil.createVerticalFill();
		data.heightHint = 120;
		space.setLayoutData(data);
		
		newLogSetBtn = new Button(parent, SWT.PUSH);
		newLogSetBtn.setText(LogUIPlugin.getResourceString("STR_IMPORT_LOG_NEW_SET"));
		GridData btnData = GridUtil.createHorizontalFill();
		btnData.widthHint = len;
		btnData.horizontalAlignment = GridData.CENTER;
		newLogSetBtn.setLayoutData(btnData);		
		
		openLogSetBtn = new Button(parent, SWT.PUSH);
		openLogSetBtn.setText(LogUIPlugin.getResourceString("STR_IMPORT_LOG_OPEN_SET"));
		btnData = GridUtil.createHorizontalFill();
		btnData.widthHint = len;
		btnData.horizontalAlignment = GridData.CENTER;
		openLogSetBtn.setLayoutData(btnData);
			
		editLogSetBtn = new Button(parent, SWT.PUSH);
		editLogSetBtn.setText(LogUIPlugin.getResourceString("STR_IMPORT_LOG_EDIT_SET"));
		btnData = GridUtil.createHorizontalFill();
		btnData.widthHint = len;
		btnData.horizontalAlignment = GridData.CENTER;
		editLogSetBtn.setLayoutData(btnData);
		
		newLogSetBtn.addSelectionListener(this);
		openLogSetBtn.addSelectionListener(this);
		editLogSetBtn.addSelectionListener(this);
		
		updateButtons();
		
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
		newLogSetBtn,
			LogUIPlugin.getPluginId() + ".ls0000");
			
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
		openLogSetBtn,
			LogUIPlugin.getPluginId() + ".ls0010");
			
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
		editLogSetBtn,
			LogUIPlugin.getPluginId() + ".ls0020");
		

	}
	
	private int computeButtonsMinWidth(Control control){
		
		String sAdd = LogUIPlugin.getResourceString("STR_IMPORT_LOG_ADD");
		String sEdit = LogUIPlugin.getResourceString("STR_IMPORT_LOG_EDIT");
		String sRemove = LogUIPlugin.getResourceString("STR_IMPORT_LOG_REMOVE");
		int maxLen1 = Math.max(Math.max(sAdd.length(), sEdit.length()) , sRemove.length());
		
		String sNewLogSet = LogUIPlugin.getResourceString("STR_IMPORT_LOG_NEW_SET");
		String sOpenLogSet = LogUIPlugin.getResourceString("STR_IMPORT_LOG_OPEN_SET");
		String sEditLogSet = LogUIPlugin.getResourceString("STR_IMPORT_LOG_EDIT_SET");
		int maxLen2 = Math.max(Math.max(sNewLogSet.length(), sOpenLogSet.length()), sEditLogSet.length()); 
		
		GC gc = new GC(control);
		gc.setFont(control.getFont());
		FontMetrics fm = gc.getFontMetrics();
		
		return Math.max(Dialog.convertWidthInCharsToPixels(fm, Math.max(maxLen1,maxLen2)+4),BUTTON_WIDTH);
		
	}
	private void updateButtons(){
		if(logElements.size()==0){
			editBtn.setEnabled(false);
			removeBtn.setEnabled(false);
			return;
		}

		editBtn.setEnabled(logsTable.getSelectionCount()>0);
		removeBtn.setEnabled(logsTable.getSelectionCount()>0);			
	}
	
	public void refresh(){
		fTableViewer.refresh();
	}
	
	public boolean finish() {
		lsMgr.saveLogSets();
		lsMgr.getLogSets().clear();
		lsMgr.getLogSetsClone().clear();
		return true;
	}

	public void dispose()
	{
		super.dispose();
		//_parserUI.dispose();
	}
	
	public void addElement(LogFileElement element){
		logElements.add(element);
		List logList = (List)lsMgr.getLogSets().get(lsMgr.getDefaultLogSet());
		if(logList==null){
			logList = new ArrayList();
			logList.add(element);
			lsMgr.getLogSets().put(lsMgr.getDefaultLogSet(), logList);
		}else{
			logList.add(element);
		}
		
	}
	
	public void removeElement(LogFileElement element){
		logElements.remove(element);
		List logList = (List)lsMgr.getLogSets().get(lsMgr.getDefaultLogSet());
		if (logList!=null)
			logList.remove(element);
		
	}
	
	public LogFileElement getSelectedElement(){
		return selection;
	}
	
	public void setSelectedElement(LogFileElement selection) {
		this.selection = selection;
	}
	
	public List getLogFileElements(){
		return logElements;
	}
		
	public boolean isPageComplete(){
		
		List elem = getLogFileElements();
		boolean complete = elem.size()>0 ? true:false;
		for (Iterator iter = elem.iterator(); iter.hasNext();) {
			LogFileElement element = (LogFileElement) iter.next();
			// need to update the valid flag for all the elements in order to display the corresponding icon in the elements table
			element.setValid(isLogElementValid(element));
			complete = complete && element.isValid();
		}
		return complete;			
	}
	
				
	public LogFileDetailsUI getDialog(int type){
		
		if(dialog==null || type!=dialog_type){
			dialog_type = type;
			dialog = new LogFileDetailsUI(getShell(), this, type);
		}
		return dialog;

	}
	
	public LogFileDetailsUI getDialog(){

		return dialog;

	}
	
	class TableContentProvider implements IStructuredContentProvider{
	
			/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
		 */
		public Object[] getElements(Object inputElement) {			
			return logElements.toArray();
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
		 */
		public void dispose() {

		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public void inputChanged(
			Viewer viewer,
			Object oldInput,
			Object newInput) {

		}

	}
	
	class TableLabelProvider extends LabelProvider implements ITableLabelProvider{

		private Hashtable images =  new Hashtable();
			/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(java.lang.Object, int)
		 */
		public Image getColumnImage(Object element, int columnIndex) {
			
			if(columnIndex==0){

				return getImageHelper(((LogFileElement)element).getParser());

			}
			return null; 
		}

		private Image getImageHelper(LogParserItem parser){
			Image img = (Image)images.get(parser.getDescription());
			if(img==null){
				if (parser.getImageDescriptor() == null) return null;
				img = parser.getImageDescriptor().createImage();
				images.put(parser.getDescription(), img);
			}
			return img;
			
		}
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object, int)
		 */
		public String getColumnText(Object element, int columnIndex) {

			if(columnIndex==1){
				return ((LogFileElement)element).getHost();				
			}
			if(columnIndex==2){
				List fields = ((LogFileElement)element).getParser().getFields();
				int s = fields.size();
				ParserField field = null;
				int j=0;				
				for (;j<s;j++) {
					field = (ParserField) fields.get(j);
					if(field!=null && field.getId().equals("file_path")){
						break;
					}
				}
				
				if(j>=0 && j<((LogFileElement)element).getValues().size())
					return (String)((LogFileElement)element).getValues().get(j);				
			}
			
			return "";
		}
		
		
		/*
		 * @see IBaseLabelProvider#dispose()
		 */
		public void dispose() {
			for (Enumeration enum = images.elements(); enum.hasMoreElements();) {
				Image image = (Image) enum.nextElement();
				if (image.isDisposed() == false)
					image.dispose();
			}
			images.clear();
			
			super.dispose();
		}
		
		

	}

	public class LogSetManager{		
		
		private IPreferenceStore preferenceStore = LogUIPlugin.getDefault().getPreferenceStore();
		private String defaultLogSet;
		private Map logSetsList = new HashMap();;
		private Map parsers = new HashMap();
		private Map logSetsClone;
		
		LogSetManager(){
		}
						
		public void initDefault(){

			defaultLogSet = DEFAULT_LOGSET;

			//loadLogSets();						
			loadParserRegistry();
			
		}
		
		public void loadLogSets(){
			
			String prefXML = preferenceStore.getString(LOGSETS_KEY);
			
			if(prefXML.length()==0){
				logSetsList.put(DEFAULT_LOGSET, null);
				updateLogSetDescription(defaultLogSet);
				return;
			}
			Element prefElement = XMLUtil.loadDom(prefXML, LOGSETS_KEY);
			if(prefElement == null){
				logSetsList.put(DEFAULT_LOGSET, null);
				updateLogSetDescription(defaultLogSet);
				return;
			}
						
			NodeList logSetsNodeList = XMLUtil.getChildrenByName(prefElement, LOGSET_TAG);

			for(int i=0, maxi=logSetsNodeList.getLength(); i<maxi; i++)
			{
				if(logSetsNodeList.item(i) instanceof Element)
					loadLogSet((Element)logSetsNodeList.item(i));
			}

			List logList = (List)logSetsList.get(defaultLogSet);
			if(logList!=null){
				logElements.addAll(logList);
			}
			
			updateLogSetDescription(defaultLogSet);
			refresh();
			
		}
		
		private void loadLogSet(Element logSetElement){
			
			if(XMLUtil.getValue(logSetElement, "default").equals("1")){
				defaultLogSet = XMLUtil.getValue(logSetElement, "name");
			}
			List logElements = new ArrayList();
						
			NodeList logNodeList = XMLUtil.getChildrenByName(logSetElement, LOGELEMENT_TAG);
			Element element = null;
			for(int i=0, maxi=logNodeList.getLength(); i<maxi; i++)
			{
				if(logNodeList.item(i) instanceof Element)
					element = (Element)logNodeList.item(i);
					logElements.add(loadElement(element));
			}
			
			logSetsList.put(XMLUtil.getValue(logSetElement, "name"), logElements);
		}
		
		private  LogFileElement loadElement(Element prefElement){
			
			LogFileElement logElement = new LogFileElement();
			logElement.setHost(XMLUtil.getValue(prefElement, "host"));
			String port = XMLUtil.getValue(prefElement, "port");
			logElement.setPort(port==null?Constants.CTL_PORT_NUM_SERVER:Integer.parseInt(port));
			logElement.setMonitor(XMLUtil.getValue(prefElement, "monitor"));
			logElement.setProject(XMLUtil.getValue(prefElement, "project"));
			logElement.setValid(XMLUtil.getValue(prefElement, "isValid").equals("true")?true:false);			
			String uri = XMLUtil.getValue(prefElement, "mergedAgent");
			logElement.setParser((LogParserItem)parsers.get(XMLUtil.getValue(prefElement, "parserId")));			
			logElement.setUseLargeLogSupport("true".equals(XMLUtil.getValue(prefElement, "useLargeLogSupport"))?true:false);
			
			if(uri!=null && uri.trim().length() > 0){
				TRCAgentProxy agent = null;
				try{	
					agent = (TRCAgentProxy)UIPlugin.getDefault().getResourceSet().getEObject(URI.createURI(uri), true);
				}catch(Exception e){
					//fail silently, no merged agent will be set
				}
					
				logElement.setMergedAgent(agent);
			}
			XMLUtil.getChildrenByName(prefElement, "value");
			NodeList nodeList = XMLUtil.getChildrenByName(prefElement, "value");
			for(int i=0, maxi=nodeList.getLength(); i<maxi; i++)
			{
				if(nodeList.item(i) instanceof Element)
					logElement.getValues().add(XMLUtil.getValue((Element)nodeList.item(i), "val"));
			}
			return logElement;
					
		}
		
		public void openLogSet(String name){
			List logsList = (List)logSetsList.get(name);
			logElements.clear();
			logElements.addAll(logsList);
			refresh();
		}
		
		public Map getLogSets(){
			return logSetsList;
		}
		
		public Map getLogSetsClone(){
			if(logSetsClone==null){
				logSetsClone = new HashMap();
			}
			logSetsClone.clear();
			logSetsClone.putAll(logSetsList);
			return logSetsClone;
		}
		
		private void loadParserRegistry()
		{
			IPluginRegistry reg = Platform.getPluginRegistry();
	
			int max = 1;
			IConfigurationElement[] config =
				reg.getConfigurationElementsFor("org.eclipse.hyades.logging.parsers", "logParser");
	
			for (int idx = 0; idx < config.length; idx++) {
				IConfigurationElement elem = config[idx];
						
				LogParserItem parser = new LogParserItem(elem.getAttribute("id")
												 , elem.getAttribute("name")
												 , elem.getAttribute("ui_name")
												 , elem.getAttribute("description"));
				parser.setConfigurationElement(elem);
				parser.setClass(elem.getAttribute("class"));								 
				parser.setUIType(elem.getAttribute("ui_type"));								 
					
				String iconPath = elem.getAttribute("icon");
				if(iconPath != null && !iconPath.equals(""))
				{
					URL pluginURL = elem.getDeclaringExtension().getDeclaringPluginDescriptor().getInstallURL();
					try {
						URL url = new URL(pluginURL, iconPath);
						ImageDescriptor image = ImageDescriptor.createFromURL(url);
						parser.setImageDescriptor(image);
					}
					catch(Exception exc)
					{
						// if the URL does not have a valid format, just log and ignore the exception
						IStatus status = new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 2,"exception.malformedURL", exc); //$NON-NLS-1$
						LogUIPlugin.getDefault().getLog().log(status);
						continue;
					
					}
				
				}
														 
				IConfigurationElement[] felems = elem.getChildren("field");
				for(int i=0; i<felems.length; i++)
				{
				   IConfigurationElement felem = felems[i];
				   ParserField pfield = new ParserField(felem.getAttribute("id")
														, felem.getAttribute("name"));
				   pfield.setConfigurationElement(felem);
				   pfield.setDefaultValue(felem.getAttribute("defaultValue"));	
				   pfield.setContextId(felem.getAttribute("helpContextId"));	
				   pfield.setTooltip(felem.getAttribute("tooltip"));	
				   pfield.setFilters(felem.getAttribute("browseType"));
				   pfield.useBrowse(felem.getAttribute("useBrowse").equals("true"));
				   pfield.setUiType(felem.getAttribute("ui_type"));
				   pfield.setListener(felem.getAttribute("listener"));
			   
				   parser.addEntry(pfield);			   
				}
				
				IConfigurationElement[] fparameters = elem.getChildren("parserParameter");
				for(int i=0; i<fparameters.length; i++)
				{
				   IConfigurationElement fparameter = fparameters[i];
				   ParserParameter para = new ParserParameter(fparameter.getAttribute("name"),fparameter.getAttribute("value"));
				   parser.addParameter(para);			   
				}
			
				parsers.put(parser.getId(), parser);
			}
					
		}

		
		public LogParserItem getLogParser(String key){
			return (LogParserItem)parsers.get(key);
		}
		
		public Map getParserRegistry(){
			return parsers;
		}
		
		public void saveLogSets(){
			
			StringBuffer xmlString = new StringBuffer();
			
			xmlString.append("<?xml");
			xmlString.append(XMLUtil.createXMLAttribute("version","1.0", false)).append(XMLUtil.createXMLAttribute("encoding","UTF-8", false));
			xmlString.append("?>");
			
			xmlString.append(" <");
			xmlString.append(LOGSETS_KEY);
			xmlString.append("> ");			
			
			Set keySet = logSetsList.keySet();
			String key = "";
			for (Iterator iter = keySet.iterator(); iter.hasNext();) {
				 key = (String) iter.next();
				 xmlString.append(generateLogSetPrefEntry(key));				
			}			
			
			xmlString.append(" </");				
			xmlString.append(LOGSETS_KEY);
			xmlString.append(">");
			
			preferenceStore.setValue(LOGSETS_KEY, xmlString.toString());			

		}
				
		
		private String generateLogSetPrefEntry(String name){
			
			StringBuffer xmlString = new StringBuffer();
			xmlString.append(" <");
			xmlString.append(LOGSET_TAG);
			xmlString.append(XMLUtil.createXMLAttribute("name", name,false));
			xmlString.append(XMLUtil.createXMLAttribute("default", name.equals(defaultLogSet) ? "1":"0", false));
			xmlString.append("> ");
			
			List logSet = (List)logSetsList.get(name);
			if (logSet!=null) 
			{
				LogFileElement element = null;
				for (Iterator iter = logSet.iterator(); iter.hasNext();) {
					 element = (LogFileElement) iter.next();
					 xmlString.append(generateLogElementPrefEntry(element));
				}
			}			

			xmlString.append(" </");				
			xmlString.append(LOGSET_TAG);
			xmlString.append("> ");

			return xmlString.toString();
		}
		
		private String generateLogElementPrefEntry(LogFileElement element){
			
			StringBuffer xml = new StringBuffer();
					
			//Body
			xml.append("<LogFileElement ");
			xml.append(XMLUtil.createXMLAttribute("host", element.getHost(),false));
			xml.append(XMLUtil.createXMLAttribute("port", String.valueOf(element.getPort()),false));
			xml.append(XMLUtil.createXMLAttribute("monitor", element.getMonitor(),false));
			xml.append(XMLUtil.createXMLAttribute("project", element.getProject(),false));
			xml.append(XMLUtil.createXMLAttribute("isValid", element.isValid()?"true":"false",false));
			xml.append(XMLUtil.createXMLAttribute("useLargeLogSupport", element.isUseLargeLogSupport()?"true":"false",false));
			xml.append(XMLUtil.createXMLAttribute("parserId", element.getParser().getId(),false));
			xml.append(XMLUtil.createXMLAttribute("mergedAgent", (element.getMergedAgent()!=null?EcoreUtil.getURI(element.getMergedAgent()).toString():""),false));
			xml.append(">");
			for(int i=0;i<element.getValues().size();i++){	
				xml.append(" <value ");
				xml.append(XMLUtil.createXMLAttribute("val",(String)element.getValues().get(i),false));
				xml.append("/> ");	
			}
					
			xml.append("</LogFileElement>");
		
			return xml.toString();		
		}
		
		
		public void setDefaultLogSet(String newValue){
			
			defaultLogSet = newValue;
					
		}
		
		public String getDefaultLogSet(){
			if(defaultLogSet==null){
				initDefault();
			}
			return defaultLogSet;
		}
	}
	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e) {

	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e) {
		
		if(e.widget==addBtn){
			selection = new LogFileElement();
			addElement(selection);
			fTableViewer.setSelection(new StructuredSelection(selection));
			
			getDialog(ADD).setWindowTitle(LogUIPlugin.getResourceString("STR_ADD_LOG_FILE"));
			
			if(getDialog(ADD).open()==Window.OK){
							
				refresh();				
			}else{
				removeElement(selection);
			}
			dialog = null;			
		}
		else if (e.widget==editBtn){
			
			getDialog(EDIT).setWindowTitle(LogUIPlugin.getResourceString("STR_EDIT_LOG_FILE"));		
			if(getDialog(EDIT).open()==Window.OK){
				
				refresh();
			}
			dialog = null;
		}
		else if (e.widget==removeBtn){
			IStructuredSelection sel = (IStructuredSelection)fTableViewer.getSelection();
			if(!sel.isEmpty()){
				for (Iterator iter = sel.iterator(); iter.hasNext();) {
					LogFileElement element = (LogFileElement) iter.next();
					removeElement(element);					
				}
			}
			type_label.setText("");
			refresh();
			
		}else if(e.widget==logsTable){
			int index = logsTable.getSelectionIndex();
			selection = (LogFileElement)logElements.get(index);
			type_label.setText(selection.getParser().getDescription());
		}else if(e.widget==newLogSetBtn){
			onNewLogSetAction();
		}		
		else if(e.widget==openLogSetBtn){
			onOpenLogSetAction();
		}else if(e.widget==editLogSetBtn){
			onEditLogSetsAction();
		}
		setPageComplete(isPageComplete());
		updateButtons();
	}

	
	private void onNewLogSetAction(){

		logSetDialog = new LogSetDialog(getShell(), NEW_LOGSET);
		logSetDialog.setLogSets(lsMgr.getLogSets());
		type_label.setText("");

		if(logSetDialog.open()==Window.OK){
			lsMgr.setDefaultLogSet(logSetDialog.getDefaultLogSet());
			lsMgr.getLogSets().put(lsMgr.getDefaultLogSet(), null);
			updateLogSetDescription(lsMgr.getDefaultLogSet());
			getLogFileElements().clear();
			refresh();
			setPageComplete(isPageComplete());
		}
	
	}
	
	private void onOpenLogSetAction(){

		logSetDialog = new LogSetDialog(getShell(), OPEN_LOGSET, this);
		logSetDialog.setLogSets(lsMgr.getLogSets());
		logSetDialog.setDefaultLogSet(lsMgr.getDefaultLogSet());	
		if(logSetDialog.open()==Window.OK && logSetDialog.hasLogSetSelectionChanged()){				
			lsMgr.setDefaultLogSet(logSetDialog.getDefaultLogSet());			
			updateLogSetDescription(logSetDialog.getDefaultLogSet());
			lsMgr.openLogSet(logSetDialog.getDefaultLogSet());
			setPageComplete(isPageComplete());				
		}			

	}
	
	private void onEditLogSetsAction(){

		logSetDialog = new LogSetDialog(getShell(), EDIT_LOGSET, this);
		logSetDialog.setLogSets(lsMgr.getLogSetsClone());
		logSetDialog.setDefaultLogSet(lsMgr.getDefaultLogSet());
		if(logSetDialog.open()==Window.OK){

			lsMgr.getLogSets().clear();
			lsMgr.getLogSets().putAll(logSetDialog.getLogSets());
			lsMgr.getLogSetsClone().clear();
			
			if(logSetDialog.getDefaultLogSet()==null){
				lsMgr.setDefaultLogSet(DEFAULT_LOGSET);
				getLogFileElements().clear();
				refresh();
			}
			else if(logSetDialog.hasCurrentLogSetChanged()){
				lsMgr.setDefaultLogSet(logSetDialog.getDefaultLogSet());
				lsMgr.openLogSet(logSetDialog.getDefaultLogSet());					
			}
			
			setPageComplete(isPageComplete());
			updateLogSetDescription(lsMgr.getDefaultLogSet());										
		}

	}
	
	public boolean isLogElementValid(LogFileElement elem){
		
		boolean valDest = (elem.getHost()!=null && elem.getHost().length()>0)&&
		(elem.getMonitor()!=null && elem.getMonitor().length()>0)&&
		(elem.getProject()!=null && elem.getProject().length()>0);
		boolean valValues = true;
		List values = elem.getValues();
		for (Iterator iter = values.iterator(); iter.hasNext();) {
			String element = (String) iter.next();
			if(element.length()==0){
				valValues = false;
				break;
			}
		}
		return valValues && valDest;
		
	}
	
	public void initLocation(){
		dialog.initializeLocation();
	}
	
	public void updateLogSetDescription(String desc){
		String text = LogUIPlugin.getResourceString("STR_IMPORT_LOG_SET_DESC");
		text = TString.change(text, "%1", desc);		
		logset_desc.setText(text);
	}
	
	public LogSetManager getLogSetManager(){
		return lsMgr;
	}

}