/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.models.internal.sdb.loader;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.xmi.XMIResource;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.internal.sdb.SDBDirective;
import org.eclipse.hyades.models.internal.sdb.SDBFactory;
import org.eclipse.hyades.models.internal.sdb.SDBMatchPattern;
import org.eclipse.hyades.models.internal.sdb.SDBRuntime;
import org.eclipse.hyades.models.internal.sdb.SDBSolution;
import org.eclipse.hyades.models.internal.sdb.SDBSymptom;
import org.eclipse.hyades.models.internal.sdb.impl.SDBFactoryImpl;
import org.eclipse.hyades.models.internal.sdb.impl.SDBPackageImpl;
import org.eclipse.hyades.models.internal.sdb.impl.SDBResourceFactoryImpl;
import org.eclipse.hyades.models.internal.sdb.impl.SDBResourceSetImpl;
import org.xml.sax.Attributes;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;


/**
 * Parser for xml file
 */
public class SDBLoader extends DefaultHandler implements IConstants {
    //~ Instance fields ----------------------------------------------------------------------------

    private HashMap _directives = new HashMap();
    private HashMap _references = new HashMap();
    private HashMap _solutions = new HashMap();
    private IProgressMonitor _monitor;
    private SDBFactory _factory;
    private SDBResourceSetImpl _resourceSet;
    private SDBRuntime _symptomDB;
    private SDBSymptom _symptom;
    private String errMsg;
    private StringBuffer buffer = new StringBuffer();
    private int counter;

    //~ Constructors -------------------------------------------------------------------------------

    public SDBLoader() {
        super();

        SDBPackageImpl.init();
        _factory = new SDBFactoryImpl();

        // init resourceSet
        _resourceSet = new SDBResourceSetImpl();
        Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("trcdbxmi", new SDBResourceFactoryImpl());
    }

    //~ Methods ------------------------------------------------------------------------------------

    public String getErrorMessage() {
        return errMsg;
    }

    /**
     * Receive notification of character data.
     */
    public void characters(char[] ch, int start, int length) throws SAXException {
        buffer.append(ch, start, length);
    }

    /**
     * Receive notification of the end of an element.
     */
    public void endElement(String namespaceURI, String localName, String qName) throws SAXException {
        if (qName.equals(MATCH_PATTERN_CLASS)) {
            createMatchPattern();
        } else if (qName.equals(SOLUTION_CLASS)) {
            createSolution();
        }

        /* else if (qName.equals(SYMPTOM_CLASS)) {
           _symptom = null;
           }*/
        buffer.setLength(0);
    }

    /**
     * @see ErrorHandler#error(SAXParseException)
     */
    public void error(SAXParseException ex) {
        errMsg = ex.getMessage();

        //errMsg = ex.toString();
		LoadersUtils.log(ex);
    }

    /**
     * @see ErrorHandler#fatalError(SAXParseException)
     */
    public void fatalError(SAXParseException ex) throws SAXException {
        errMsg = ex.getMessage();
		LoadersUtils.log(ex);

        //errMsg = ex.toString();
    }

    public void loadDatabase(String xmlFileName, String destination, IProgressMonitor mon) throws Exception {
        try {
            _monitor = mon;

            File xmlFile = new File(xmlFileName);
            int length = (int) xmlFile.length();

            _monitor.beginTask(xmlFileName, length / 1000);
			
			createRuntime(destination);				

            if (_symptomDB != null) {
                _symptomDB.setLocalExternalFileLocation(xmlFileName);
            }

            parse(xmlFileName);
        } finally {
            try {
                Map options = new HashMap();

                options.put(XMIResource.OPTION_DECLARE_XML, Boolean.TRUE);
                _symptomDB.eResource().save(options);

            } catch (Exception exc) {
                errMsg = exc.toString();
                _monitor.done();

				LoadersUtils.log(exc);
            }

            _monitor.done();
        }
    }

	public void loadDatabase(InputStream is, String urlLocation, String destination, IProgressMonitor mon) throws Exception {
        try {
            _monitor = mon;

            _monitor.beginTask(urlLocation, 100);

			createRuntime(destination);				

            if (_symptomDB != null) {
                _symptomDB.setSymptomUrl(urlLocation);
                _symptomDB.setName(destination);
            }

            parse(is);
        } finally {
            try {
                Map options = new HashMap();

                options.put(XMIResource.OPTION_DECLARE_XML, Boolean.TRUE);
                _symptomDB.eResource().save(options);

            } catch (Exception exc) {
                errMsg = exc.toString();
                _monitor.done();
				LoadersUtils.log(exc);
            }

            _monitor.done();
        }
    }

    public void parse(String symptomDBFile) throws Exception {
        errMsg = null;

        InputStream is = new FileInputStream(symptomDBFile);

        if (is == null) {
            return;
        }

        parse(is);
    }

	protected SAXParser makeParser() throws ParserConfigurationException, SAXException {
		  SAXParserFactory f = SAXParserFactory.newInstance();
//		SAXParserFactoryImpl f = new SAXParserFactoryImpl();
		
		f.setValidating(false);

		//        f.setNamespaceAware(true);
		return f.newSAXParser();
	}

    public void parse(InputStream is) throws Exception {
        errMsg = null;

        if (is == null) {
            return;
        }

        InputSource inputSource = new InputSource(is);

        try {
            SAXParser parser = makeParser();

////            parser.setFeature("http://apache.org/xml/features/nonvalidating/load-external-dtd", false);
////            parser.setErrorHandler(this);
////            parser.setContentHandler(this);
//			parser.parse(inputSource);
			parser.parse(inputSource, this);
            is.close();
        } catch (SAXException se) {
            errMsg = se.getMessage();
            LoadersUtils.log(se);

            //errMsg = se.toString();
        } catch (IOException ioe) {
            errMsg = ioe.getMessage();
            LoadersUtils.log(ioe);

            //errMsg = ioe.toString();
        }
    }

    /**
     * Receive notification of the beginning of an element.
     */
    public void startElement(String namespaceURI, String localName, String elementName, Attributes atts) throws SAXException {
        if (elementName.equals(V5_RUNTIME_CLASS)) {
            createRuntime(atts);
        } else if (elementName.equals(V5_MATCH_PATTERN_CLASS)) {
            createMatchPattern(atts);
        } else if (elementName.equals(V5_SOLUTION_CLASS)) {
            createSolution(atts);
        } else if (elementName.equals(V5_SYMPTOM_CLASS)) {
            createSymptom(atts);
        } else if (elementName.equals(V5_DIRECTIVE_CLASS)) {
            createDirective(atts);
        } else if (elementName.equals(SYMPTOM_CLASS)) {
            createV4Symptom(atts);
        }
    }

    /*
     *
     */
    public void warning(SAXParseException ex) {
        errMsg = ex.getMessage();
		LoadersUtils.log(ex);
    }

    /*
     *
     */
    private void addReference(String id, Object reference) {
        Object refs = _references.get(id);

        if (refs != null) {
            ((ArrayList) refs).add(reference);
        } else {
            ArrayList list = new ArrayList();

            list.add(reference);
            _references.put(id, list);
        }
    }

    private void createDirective(Attributes atts) {
        SDBDirective directive = _factory.createSDBDirective();

        _symptomDB.getDirectives().add(directive);

        directive.setId(atts.getValue("id"));
        directive.setDescription(atts.getValue("description"));
        directive.setDirectiveString(atts.getValue("directiveString"));

        //set solution references
        Object references = _references.get(directive.getId());

        if (references != null) {
            ArrayList list = (ArrayList) references;

            for (int idx = 0; idx < list.size(); idx++) {
                SDBSolution solution = (SDBSolution) list.get(idx);

                solution.getDirectives().add(directive);
                directive.getSolutions().add(solution);
            }

            _references.remove(directive.getId());
        }

        _directives.put(directive.getId(), directive);
    }

    /*
     *
     */
    private void createMatchPattern(Attributes atts) {
        SDBMatchPattern pattern = _factory.createSDBMatchPattern();

        _symptom.getPatterns().add(pattern);

        pattern.setName(atts.getValue("name"));
        pattern.setValue(atts.getValue("value"));

        String desc = _symptom.getDescription();

        if ((desc == null) || desc.equals("")) {
            _symptom.setDescription(pattern.getValue());
        }

        _monitor.worked(1);

        if (_monitor.isCanceled()) {
            throw new OperationCanceledException();
        }
    }

    private void createMatchPattern() {
        //createSymptom();
        String data = buffer.toString();

        while (data.startsWith("\n") || data.startsWith(" ")) {
            data = data.substring(1);
        }

        int idx = data.indexOf("\n");

        if (idx != -1) {
            String desc = data.substring(0, idx);

            if (desc.length() > 0) {
                _symptom.setDescription(desc);
            }
        } else {
            if (data.length() > 0) {
                _symptom.setDescription(data);
            }
        }

        /*data = data.substring(idx + 1);
           idx = data.indexOf("\n");*/
        int counter = 0;

        while (idx != -1) {
            if (data.substring(0, idx).length() > 0) {
                SDBMatchPattern pattern = _factory.createSDBMatchPattern();

                pattern.setName(String.valueOf(counter));
                pattern.setValue(data.substring(0, idx));

                _symptom.getPatterns().add(pattern);
                counter++;
            }

            data = data.substring(idx + 1);
            idx = data.indexOf("\n");
        }

        _monitor.worked(1);

        if (_monitor.isCanceled()) {
            throw new OperationCanceledException();
        }
    }

    /*
     *
     */
    private void createRuntime(String destination) {
		Resource monDoc = null;
		URI uri = null;
		
		if(destination.startsWith("platform:/resource/")){
			uri = URI.createURI(destination);
		}else{
			uri = URI.createFileURI(destination);
		}
		monDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
        EList ext = monDoc.getContents();

        _resourceSet.getResources().add(monDoc); // prevents reloading later				
        _symptomDB = _factory.createSDBRuntime();
        ext.add(_symptomDB);
        _symptomDB.setId("");
        _symptomDB.setName(destination);
        _symptomDB.setSymptomUrl("");
        _symptomDB.setLocalExternalFileLocation("");
        _symptomDB.setDescription("");
    }

    private void createRuntime(Attributes atts) {
        if (_symptomDB == null) {
            return;
        }

        _symptomDB.setId(atts.getValue("id"));
        _symptomDB.setName(atts.getValue("name"));
        _symptomDB.setSymptomUrl(atts.getValue("symptomUrl"));
        _symptomDB.setLocalExternalFileLocation(atts.getValue("localExternalFileLocation"));
    }

    /*
     *
     */
    private void createSolution(Attributes atts) {
        SDBSolution solution = _factory.createSDBSolution();

        _symptomDB.getSolutions().add(solution);

        solution.setId(atts.getValue("id"));
        solution.setDescription(atts.getValue("description"));

        String directives = atts.getValue("directives");

        if (directives != null) {
            int idx = directives.indexOf(" ");

            while (idx != -1) {
                String sol = directives.substring(0, idx);

                resolveRefDirective(sol, solution);

                directives = directives.substring(idx + 1);
                idx = directives.indexOf(" ");
            }

            if (!directives.equals("")) {
                resolveRefDirective(directives, solution);
            }
        }

        //set symptom references
        Object references = _references.get(solution.getId());

        if (references != null) {
            ArrayList list = (ArrayList) references;

            for (int idx = 0; idx < list.size(); idx++) {
                SDBSymptom symptom = (SDBSymptom) list.get(idx);

                symptom.getSolutions().add(solution);
                solution.getSymptoms().add(symptom);
            }

            _references.remove(solution.getId());
        }

        _solutions.put(solution.getId(), solution);

        _monitor.worked(1);

        if (_monitor.isCanceled()) {
            throw new OperationCanceledException();
        }
    }

    private void createSolution() {
        //createSymptom();
        SDBSolution solution = _factory.createSDBSolution();

        _symptomDB.getSolutions().add(solution);

        solution.getSymptoms().add(_symptom);
        _symptom.getSolutions().add(solution);

        String descr = buffer.toString();

        while (descr.startsWith("\n") || descr.startsWith(" ")) {
            descr = descr.substring(1);
        }

        int idx = descr.indexOf("\n");

        if (idx != -1) {
            solution.setDescription(descr.substring(0, idx));

            descr = descr.substring(idx);
            descr = descr.trim();

            if (descr.length() > 0) {
                SDBDirective directive = _factory.createSDBDirective();

                _symptomDB.getDirectives().add(directive);

                directive.setDescription(descr);
                directive.setId("Directive_" + counter++);

                directive.setDirectiveString("");

                solution.getDirectives().add(directive);
                directive.getSolutions().add(solution);
            }
        } else {
            solution.setDescription(descr);
        }

        solution.setId("Solution_" + counter);

        _monitor.worked(1);

        if (_monitor.isCanceled()) {
            throw new OperationCanceledException();
        }
    }

    /*
     *
     */
    private void createSymptom(Attributes atts) {
        _symptom = _factory.createSDBSymptom();
        _symptomDB.getSymptoms().add(_symptom);

        _symptom.setId(atts.getValue("id"));
        _symptom.setDescription(atts.getValue("description"));

        String solutions = atts.getValue("solutions");

        if (solutions != null) {
            int idx = solutions.indexOf(" ");

            while (idx != -1) {
                String sol = solutions.substring(0, idx);

                resolveRefSolution(sol, _symptom);

                solutions = solutions.substring(idx + 1);
                idx = solutions.indexOf(" ");
            }

            if (!solutions.equals("")) {
                resolveRefSolution(solutions, _symptom);
            }
        }

        _monitor.worked(1);

        if (_monitor.isCanceled()) {
            throw new OperationCanceledException();
        }
    }

    private void createV4Symptom(Attributes attr) {
        _symptom = _factory.createSDBSymptom();
        _symptom.setId(attr.getValue("recordid"));
        _symptom.setDescription(attr.getValue("recordid"));
        _symptomDB.getSymptoms().add(_symptom);

        _monitor.worked(1);

        if (_monitor.isCanceled()) {
            throw new OperationCanceledException();
        }
    }

    private void resolveRefDirective(String id, SDBSolution solution) {
        Object directive = _directives.get(id);

        if (directive != null) {
            SDBDirective dir = (SDBDirective) directive;

            dir.getSolutions().add(solution);
            solution.getDirectives().add(dir);
        } else {
            //the solution was not generated yet
            addReference(id, solution);
        }
    }

    private void resolveRefSolution(String id, SDBSymptom symptom) {
        Object solution = _solutions.get(id);

        if (solution != null) {
            SDBSolution sol = (SDBSolution) solution;

            sol.getSymptoms().add(symptom);
            symptom.getSolutions().add(sol);
        } else {
            //the solution was not generated yet
            addReference(id, symptom);
        }
    }
}
