/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.perfmon;

import org.eclipse.debug.core.model.IProcess;
import org.eclipse.hyades.perfmon.xml.*;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.ui.internal.util.ProcessAdapter;
import org.eclipse.hyades.internal.execution.local.common.*;
import org.eclipse.hyades.internal.execution.local.control.*;
import org.eclipse.hyades.internal.execution.local.control.Process;

import org.eclipse.hyades.model.statistical.*;

import org.eclipse.hyades.models.hierarchy.*;

import java.util.*;
import java.io.*;

public class PerfmonCustomLoader extends PerfmonLoader implements DataProcessor {

ArrayList listeners = new ArrayList();

long started;
String tracename;
	
Node node;
Process process;
Agent agent;

TRCAgent trcagent;

boolean running = false;

XMLParser parser = new XMLParser();
StatisticalXMLTranslator translator;

//OutputStream logfile;

	public PerfmonCustomLoader(long started, TRCAgent tagent, String modelname, String host, int port, String remote_host, int freq) throws Exception {
		this.started = started;
		init(modelname,tagent,host,port,remote_host,freq);
	}
	public PerfmonCustomLoader(long started, TRCAgent tagent, String modelname, String host, String remote_host, int freq) throws Exception {
		this.started = started;
		init(modelname,tagent,host,10002,remote_host,freq);
	}

	public void killProcess() throws Exception {
		node.killProcess(process);
	}
	
	public void setFrequency(int freq) throws IOException {
		setNVPair("var_int", "DC_WAITTIME", ""+freq);
	}

	public void setRemoteHost(String host) throws IOException {
		if (host.length() == 0) host = "NULL";
		PerfmonPlugin.DBG.info("Setting remote host to "+host);
		setNVPair("var_string", "PERFMON_REG_HOST", host);
	}

	public void getUpdatedTree() throws IOException {
		setNVPair("updatetree", "", "true");
	}

	public void getUpdatedTree(SDDescriptor update) throws IOException {
		if (update.getId() == null) return;
		PerfmonPlugin.DBG.info("Requesting tree update for descriptor "+update.getName()+" ("+update.getId()+")");
		setNVPair("updatetree", update.getId(), "true");
	}
	
	public void counterON(SDDescriptor counter) throws IOException, NullPointerException {
		String id = translator.getID(counter);
		if (id == null) throw new NullPointerException("descriptor ID not found");
		setNVPair("filter", id, "true");
	}
	
	public void counterOFF(SDDescriptor counter) throws IOException {
		String id = translator.getID(counter);
		if (id == null) throw new IOException("descriptor ID not found");
		setNVPair("filter", id, "false");
	}

	public SDDescriptor getModel() {
		return translator.getModel();	
	}
	
	public void addLoaderListener(StatisticalLoaderListener listener) {
		translator.addLoaderListener(listener);
	}

	public void removeLoaderListener(StatisticalLoaderListener listener) {
		translator.addLoaderListener(listener);
	}


	private Agent findAgent(Node node) throws NotConnectedException {
		Agent agent = null;
		Enumeration processes = node.listProcesses();
		
		//
		// Search for an existing agent
		//
		while (processes.hasMoreElements()) {
			Process p = (Process)processes.nextElement();
			PerfmonPlugin.DBG.info("checking RAC process "+p.getName());
			agent = findAgent(p);

			if (agent != null) {
				break;
			}
		}
		
		return agent;
	}

	private Agent findAgent(Process p) {
		if (p == null) return null;

		Enumeration agents = p.listAgents();

		Agent the_agent = null;

		if (!agents.hasMoreElements()) {
			PerfmonPlugin.DBG.info("found no agents in this process");
		}

		while (agents.hasMoreElements()) {
			Agent a = (Agent)agents.nextElement();

			PerfmonPlugin.DBG.info("found RAC agent "+a.getName()+"/"+a.getType());

			if (a.getName().startsWith("PerfmonAgent")
			||	a.getType().startsWith("PerfmonAgent")) {
				
				try {
//					String agentname = a.getName();
//					agentname = agentname.substring(agentname.indexOf("#")+1);

					if (!a.isAttached()) {
						process = p;
						the_agent = a;
						PerfmonPlugin.DBG.info("agent not attached - accepting");
//						long t = Long.parseLong(agentname); 
//						process = p;
//						agent = a;
						break;
					}
				} catch (Exception e) {
					PerfmonPlugin.DBG.warning("error checking agent "+e);
				}
				
			}
		}
		return the_agent;
	}

	private void init(String modelname, TRCAgent trcagent, String host, int port, String remote_host, int freq) throws Exception {

		this.trcagent = trcagent;

//logfile = new FileOutputStream("C:\\XMLLOG.txt");
				
		tracename = modelname;

		translator = new StatisticalXMLTranslator(trcagent,tracename);
		if (trcagent != null) {
			translator.getModel().setAgent(trcagent);
		}

		PerfmonPlugin.DBG.warning("asked to connect to "+host+":"+port);

		try {
			
			node = NodeFactory.createNode(host);
			
			PerfmonPlugin.DBG.info("connecting to RAC");
			
			node.connect(port);

			PerfmonPlugin.DBG.info("connected to RAC");

			//
			// Find an agent
			//			
			agent = findAgent(node); 
			
			//
			// No agent found, try to launch one
			//
			if (agent == null) {
				PerfmonPlugin.DBG.info("Launching RAC PerfmonAgent");
				process = ProcessFactory.createProcess(node,"PerfmonAgent");
				
				if (process != null) {
					
					 final IProcess adapter = new ProcessAdapter(process, null);
			            process.addProcessListener(new ProcessListener() {
			             
			             public void processLaunched(Process p)
			    {
			              UIPlugin.getDefault().registerLaunchProcess(adapter);
			             }
			             
			             public void processExited(Process p)
			    {
			              UIPlugin.getDefault().deregisterLaunchProcess(adapter);
			             }
			            });
			            
					process.launch();

					long t = System.currentTimeMillis() + 8000;	//search for a maximum of 25 seconds
					
					//wait for the process to become active
					while (!process.isActive() && System.currentTimeMillis() < t) {
						try {
							Thread.sleep(200);	
						} catch (Exception e) {
						}
					}

					t = System.currentTimeMillis() + 25000;	//search for a maximum of 25 seconds
					agent = findAgent(node); 
					while (agent == null && System.currentTimeMillis() < t) {
						Thread.sleep(1250);
						agent = findAgent(node);
					} 
				}
			}

			//
			// If we now have an agent, 
			//
			if (agent != null) {
				PerfmonPlugin.DBG.info("attaching to perfmon agent "+agent.getName());

				running = true;

				agent.attach();
				agent.startMonitoring(this);
				trcagent.setRuntimeId(agent.getUUID());

				PerfmonPlugin.DBG.info("attached To & Monitoring Perfmon Agent");
	
				setRemoteHost(remote_host);
				setFrequency(freq);
				getUpdatedTree();
				
	
			} else {
				PerfmonPlugin.DBG.warning("no perfmon agent found");
				throw new Exception("Could not create PerfmonAgent on RAC");
			}
			
		} catch (Exception e) {
			PerfmonPlugin.DBG.warning("problem initialising loader");

//////////////// FOR SAMPLE EVENTS ONLY
//temporarily disabled for sample events
			throw e;
////////////////////////////////////////

		}
	}

	public void setNVPair(String type, String name, String value) throws IOException {

		ControlMessage message=new ControlMessage();
		SetNVPairCommand command=new SetNVPairCommand();
		try {
			command.setProcessId(Long.parseLong(process.getProcessId()));
		}
		catch(InactiveProcessException e) {
			throw new IOException("Inactive process - try again soon?");
		}
		command.setAgentName(agent.getName());
		command.setType(type);
		command.setName(name);
		command.setValue(value);
		message.appendCommand(command);
		try {
			node.getConnection().sendMessage(message, null);
		} catch(IOException e) {
			//log errors?
			throw e;
		}
	}

	public boolean isRunning() {
		return running;
	}

	public void shutdown() {
		running = false;
		PerfmonPlugin.DBG.info("shutdown");
		if (agent != null) {
			try {
				PerfmonPlugin.DBG.info("detaching from agent");
				agent.stopMonitoring();
				agent.detach();	
				
			} catch (Exception e) {
				PerfmonPlugin.DBG.warning("failed to detach from agent",e);
			}
		}

	}

	protected void finalize() throws Throwable {
		super.finalize();
		shutdown();
	}

	/////////////////////////////////////////////
	//  DATA PROCESSOR IMPL
	/////////////////////////////////////////////

	public void incoming(String msg) {
		try {
//			logfile.write(msg.getBytes());
			parser.parse(translator,new ByteArrayInputStream(msg.getBytes()));
		} catch (Throwable t) {
t.printStackTrace();			
			PerfmonPlugin.DBG.warning("unable to parse message "+msg);
		}
	}

	private byte[] stringToBytes(String s) {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		for (int i = 0; i < s.length(); i++) {
			bout.write((byte)s.charAt(i));	
		}
		return bout.toByteArray();
	}

	public void incommingData(byte[] buffer, int length, java.net.InetAddress peer) {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < length; i++) {
			sb.append((char)buffer[i]);
		}
		incoming(sb.toString().trim());
	}

	public void incommingData(char[] buffer, int length, java.net.InetAddress peer) {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < length; i++) {
			sb.append((char)buffer[i]);
		}
		incoming(sb.toString().trim());
	}

	public void invalidDataType(byte[] data, int length, java.net.InetAddress peer) {
		PerfmonPlugin.DBG.warning("invalid datatype");
	}

	public void waitingForData() {
	}
}
