package org.eclipse.hyades.ui.sample.svg.generator;
/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

import java.io.File;
import java.io.IOException;
import java.io.Serializable;

import javax.xml.transform.TransformerException;

import org.eclipse.hyades.logging.core.XmlUtility;
import org.w3c.dom.Document;

/**
 * <code>GraphicDocumentGenerator</code> is an abstract base class for classes which take
 * content data input as Document Object Model (DOM) objects and then generate DOM objects 
 * which represent graphic visualizations of that content data.
 * 
 * @version 1.44.1.10
 */
public abstract class GraphicDocumentGenerator implements IGraphicDocumentGenerator, Serializable {
	/**
	 * Graphic type identifier.
	 */
	protected String graphicType;
	
	/**
	 * @see IGraphicDocumentGenerator#getGraphicType()
	 */
	public String getGraphicType() {
		return graphicType;
	}
	
	/**
     * @see IGraphicDocumentGenerator#generateGraphicDocument(Document)
     */       
	public Document generateGraphicDocument(Document data) throws DataInputProcessingException {	
		GraphicDocumentProperties properties = new GraphicDocumentProperties();
		properties.setDataDocument(data);		
		return generateGraphicDocument(properties);	
	}	
		
	/**
	 * @see IGraphicDocumentGenerator#generateAndSerializeToString(GraphicDocumentProperties)
	 */
	public String generateAndSerializeToString(GraphicDocumentProperties properties) throws DataInputProcessingException, IOException {
		Document generatedDocument = generateGraphicDocument(properties);
		return serializeGeneratedDocumentToString(generatedDocument);
	}
			
	/**
	 * @see IGraphicDocumentGenerator#generateAndSerializeToFile(GraphicDocumentProperties, String)
	 */
	public boolean generateAndSerializeToFile(GraphicDocumentProperties properties, String uri) throws DataInputProcessingException, IOException {
		Document generatedDocument = generateGraphicDocument(properties);
		if (generatedDocument != null) {
			return serializeGeneratedDocumentToFile(generatedDocument, uri);
		}
		else {
			return false;
		}	
	}
			
	/**
	 * @see IGraphicDocumentGenerator#serializeGeneratedDocumentToFile(Document, String)
	 */
	public boolean serializeGeneratedDocumentToFile(Document generatedDocument, String uri) throws IOException {

	    if (generatedDocument != null) {
		
	        try {
            
	            XmlUtility.serialize(generatedDocument,new File(uri));
	            
	            return true;		
	        } 
	        catch (TransformerException t) {
	            //Ignore since returning false.
	        }
		}

        return false;
	}		
		
	/**
	 * @see IGraphicDocumentGenerator#serializeGeneratedDocumentToString(Document)
	 */
	public String serializeGeneratedDocumentToString(Document generatedDocument) throws IOException {
	    
	    if (generatedDocument != null) {
			
		        try {
		            return (XmlUtility.serialize(generatedDocument));		
		        } 
		        catch (TransformerException t) {
		            //Ignore since returning null.
		        }
			}

	        return null;
	}
	
	/**
	 * XML-serializes and returns the generated <code>Document</code> object as a byte array.
	 * 
	 * @param generatedDocument the generated graphic
	 * @return byte[] The XML-serialized form of the <code>Document</code>
	 */	
	public static byte[] serializeGeneratedDocumentToStream(Document generatedDocument) {
        
	    if (generatedDocument != null) {
			
		        try {
		            return (XmlUtility.serializeAsByteArray(generatedDocument));		
		        } 
		        catch (TransformerException t) {
		            //Ignore since returning null.
		        }
			}

	    return null;
	}		
}
