/*******************************************************************************
 * Copyright (c) 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui;

import java.net.URL;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPluginDescriptor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.ui.plugin.AbstractUIPlugin;

import org.eclipse.emf.ecore.resource.Resource;

import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.models.common.facades.behavioral.impl.FacadeResourceFactoryImpl;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.ui.adapter.IResourceSetFileAdaptable;
import org.eclipse.hyades.test.ui.adapter.TestAdapterFactory;
import org.eclipse.hyades.test.ui.internal.model.TestUIResourceFactoryImpl;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.internal.logicalfolder.LogicalFolder;
import org.eclipse.hyades.ui.internal.util.JavaUtil;
import org.eclipse.hyades.ui.internal.util.ResourceBundleManager;

/**
 * Hyades Test UI Plugin class.  This class should not be used outside the context of
 * Eclipse.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class TestUIPlugin extends AbstractUIPlugin

{
	private static TestUIPlugin instance;
	private ResourceBundleManager resourceBundleManager;
	
	/**
	 * Extension point id used to register the test case creation wizards.
	 */
	public static final String EP_TEST_CASE_WIZARD = "newTestCaseWizards"; //$NON-NLS-1$
	
	/**
	 * Extension point id used to register the generate wizards.
	 */
	public static final String EP_GENERATE_WIZARDS = "generateWizards"; //$NON-NLS-1$

	/**
	 * Extension point id used to register the run wizards.
	 */
	public static final String EP_RUN_WIZARDS = "runWizards"; //$NON-NLS-1$

	/**
	 * Constructor for HyadesUIPlugin
	 * @param descriptor
	 */
	public TestUIPlugin(IPluginDescriptor descriptor)
	{
		super(descriptor);		
		instance = this;
	}
	
	/**
	 * Returns the instance of this class created by the eclipse framework.
	 * @return HyadesUIPlugin
	 */
	public static TestUIPlugin getInstance()
	{
		return instance;
	}
	
	/**
	 * @see org.eclipse.core.runtime.Plugin#startup()
	 */
	public void startup()
	throws CoreException
	{
		resourceBundleManager = new ResourceBundleManager(); 
		resourceBundleManager.add(getDescriptor().getResourceBundle());
		resourceBundleManager.add(HyadesUIPlugin.getInstance().getResourceBundle());

		Platform.getAdapterManager().registerAdapters(TestAdapterFactory.INSTANCE, IResourceSetFileAdaptable.class);
		Platform.getAdapterManager().registerAdapters(TestAdapterFactory.INSTANCE, IResource.class);
		Platform.getAdapterManager().registerAdapters(TestAdapterFactory.INSTANCE, LogicalFolder.class);
		Platform.getAdapterManager().registerAdapters(TestAdapterFactory.INSTANCE, CMNNamedElement.class);
		Platform.getAdapterManager().registerAdapters(TestAdapterFactory.INSTANCE, TPFExecutionEvent.class);
		Platform.getAdapterManager().registerAdapters(TestAdapterFactory.INSTANCE, Resource.class);

		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.TEST_SUITE_FILE_EXTENSION, new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.DEPLOYMENT_FILE_EXTENSION, new TestUIResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.EXECUTION_FILE_EXTENSION, new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.LOCATION_FILE_EXTENSION, new TestUIResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.DATAPOOL_FILE_EXTENSION, new TestUIResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.ARTIFACT_FILE_EXTENSION, new TestUIResourceFactoryImpl());


		JavaUtil.addResourcePatternToNotCopyList("*." + ICommonConstants.TEST_SUITE_FILE_EXTENSION); //$NON-NLS-1$
		JavaUtil.addResourcePatternToNotCopyList("*." + ICommonConstants.DEPLOYMENT_FILE_EXTENSION); //$NON-NLS-1$
		JavaUtil.addResourcePatternToNotCopyList("*." + ICommonConstants.LOCATION_FILE_EXTENSION); //$NON-NLS-1$
		JavaUtil.addResourcePatternToNotCopyList("*." + ICommonConstants.EXECUTION_FILE_EXTENSION); //$NON-NLS-1$
		JavaUtil.addResourcePatternToNotCopyList("*." + ICommonConstants.DATAPOOL_FILE_EXTENSION); //$NON-NLS-1$
		JavaUtil.addResourcePatternToNotCopyList("*." + ICommonConstants.ARTIFACT_FILE_EXTENSION); //$NON-NLS-1$

		super.startup();		

		try
		{
			TestUIImages.INSTANCE.initialize(new URL(getDescriptor().getInstallURL(), "icons/full/"), getImageRegistry()); //$NON-NLS-1$
		}
		catch(Exception e)
		{
			logError(e);
		}
	}

	/**
	 * @see org.eclipse.core.runtime.Plugin#shutdown()
	 */
	public void shutdown()
	throws CoreException
	{
		resourceBundleManager.dispose();
		TestUIExtension.dispose();
		TestUI.dispose();

		savePluginPreferences();
		Platform.getAdapterManager().unregisterAdapters(TestAdapterFactory.INSTANCE);
		TestAdapterFactory.INSTANCE.dispose();
		
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().remove(ICommonConstants.TEST_SUITE_FILE_EXTENSION);
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().remove(ICommonConstants.DEPLOYMENT_FILE_EXTENSION);
		/**
		 * @author psun
		 */
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().remove(ICommonConstants.DATAPOOL_FILE_EXTENSION);
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().remove(ICommonConstants.LOCATION_FILE_EXTENSION);
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().remove(ICommonConstants.ARTIFACT_FILE_EXTENSION);
		
		instance = null;		
		super.shutdown();
	}
	
	/**
	 * Returns the resource bundle used by this plugin.
	 * 
	 * <p>IMPORTANT: Don't use this method to retrieve values from the
	 * resource bundle.  For this purpose use the static <code>getString()</code> 
	 * defined in this class.
	 * 
	 * <p>This method is provided so this resource bundle can
	 * be used as the parent of another resource bundle.
	 * 
	 * @return ResourceBundle
	 */
	public ResourceBundle getResourceBundle()
	{
		return getDescriptor().getResourceBundle();
	}

	/**
	 * Returns this plugin's id.
	 * @return String
	 */
	public static String getID()
	{
		return instance.getDescriptor().getUniqueIdentifier();
	}
	
	/**
	 * Logs an error described by a throwable.
	 * 
	 * <p>This method should be used whenever a class in this plugin
	 * has to log an error since it adheres to the global logging
	 * strategy.
	 * 
	 * @param throwable
	 */
	public static void logError(Throwable throwable)
	{
		Status status = new Status(1, getID(), 0, throwable.toString(), throwable);
		getInstance().getLog().log(status);
	}
	
	/**
	 * Logs an error described by a text.
	 * 
	 * <p>This method should be whenever a class in this plugin
	 * has to log an error since it adheres to the global logging
	 * strategy.
	 * 
	 * @param text
	 */
	public static void logError(String text)
	{
		logError(new Throwable(text));
	}
	
	/**
	 * Logs an information described by a text.
	 * 
	 * <p>This method should be whenever a class in this plugin
	 * has to log an information since it adheres to the global logging
	 * strategy.
	 * 
	 * @param text
	 */
	public static void logInfo(String text)
	{
		Status status = new Status(3, getID(), 0, text, new Throwable(text));
		getInstance().getLog().log(status);
	}
	
	/**
	 * Returns the "plugin.properties" file's value associate to a given key.
	 * @param key
	 * @return String
	 * @throws java.util.MissingResourceException if the key is not in the file
	 * @throws NullPointerException if key is null
	 */
	public static String getString(String key)
	throws NullPointerException, MissingResourceException
	{
		return instance.resourceBundleManager.getString(key);
	}
	
	/**
	 * Returns the string value associate to a given key.  The key is passed to
	 * each resource bundle in the order they are retrieve by the 
	 * {@link #iterator()} method.
	 * 
	 * <p>The <code>arg</code> string defined replaces the %1 
	 * variable defined in the file's values.
	 * 
	 * <p>Example: If the value associated to the key <code>"a"</code> is 
	 * <code>"%0 %1 %2 %3 %4"</code> and arg is <code>"v1"</code>,
	 * the return of this method is <code>"%0 v1  %2 %3 %4"</code>.
	 * 
	 * @param key
	 * @param arg
	 * @return String
	 * @throws java.util.MissingResourceException if the key is not in the file
	 * @throws NullPointerException if key is null
	 */
	public static String getString(String key, String arg)
	throws NullPointerException, MissingResourceException
	{
		return instance.resourceBundleManager.getString(key, arg);
	}	

	/**
	 * Returns the "plugin.properties" file's value associate to a given key.
	 * 
	 * <p>The strings defined in <code>args</code> replaces the %n (where n>=1) 
	 * variables defined in the file's values.
	 * 
	 * <p>Example: If the value associated to the key <code>"a"</code> is 
	 * <code>"%0 %1 %2 %3 %4"</code> and args is <code>{"v1", null, "v3"}</code>,
	 * the return of this method is <code>"%0 v1  v3 %4"</code>.
	 * 
	 * @param key
	 * @param args
	 * @return String
	 * @throws java.util.MissingResourceException if the key is not in the file
	 * @throws NullPointerException if key is null
	 */
	public static String getString(String key, String[]args)
	throws NullPointerException, MissingResourceException
	{
		return instance.resourceBundleManager.getString(key, args);
	}
	
	/**
	 * Insert the method's description here.
	 * Creation date: (8/8/2001 5:11:57 PM)
	 * @param store org.eclipse.jface.preference.IPreferenceStore
	 */
	protected void initializeDefaultPreferences(IPreferenceStore store) {
		store.setDefault(TestUI.REPORT_OPEN_EDITOR, false);
		store.setDefault(TestUI.OPEN_EDITOR, true);
		store.setDefault(TestUI.SAVE_INTERVAL, 5000);			
		store.setDefault(TestUI.EXECUTION_FILTER, 3);	
		store.setDefault(TestUI.LOCALHOST_PORT, "10002");
		store.setDefault(TestUI.DEFAULT_HOST_NAME, "localhost");
	}
	
}
