/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.launch.extensions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.internal.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestComponent;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.launch.extensions.IDeploymentsProvider;
import org.eclipse.hyades.test.ui.launch.extensions.IExecutionHistoryDefaultsProvider;
import org.eclipse.hyades.test.ui.launch.extensions.IRunHandler;

/**
 * This class is the central point for registering and getting extensions, based
 * on the type and the kind of the Launchable Element.
 * @author jcanches
 */
public class LaunchConfigurationExtensionsManager {

	private static LaunchConfigurationExtensionsManager instance;
	
	public static LaunchConfigurationExtensionsManager getInstance() {
		if (instance == null) {
			instance = new LaunchConfigurationExtensionsManager();
		}
		return instance;
	}
	
	private static final String KIND_TEST = "test"; //$NON-NLS-1$
	private static final String KIND_COMPONENT = "testComponent"; //$NON-NLS-1$
	
	private LaunchConfigurationExtensionsManager() {
		registerRunHandlers();
		registerDeploymentsProviders();
		registerExecutionHistoryDefaultsProviders();
		registerSupportedTestTypes();
	}
	
	private Map testExtensionMap = new HashMap();
	private Map componentExtensionMap = new HashMap();
	
	private ExtensionDescriptor getExtensionDescriptor(String type, String kind) {
		Assert.isNotNull(kind);
		Assert.isNotNull(type);
		
		Map map = null;
		if (kind.equals(KIND_TEST)) {
			map = testExtensionMap;
		} else if (kind.equals(KIND_COMPONENT)) {
			map = componentExtensionMap;
		} else {
			throw new IllegalArgumentException("Invalid kind");
		}
		Assert.isNotNull(map);
		
		Object o = map.get(type);
		if (o == null) {
			o = new ExtensionDescriptor();
			map.put(type, o);
		}
		return (ExtensionDescriptor)o;
	}
	
	private ExtensionDescriptor getExtensionDescriptor(Object testElement) {
    	if (testElement != null) {
			if (testElement instanceof TPFTest) {
				TPFTest test = (TPFTest)testElement;
				if (test.getType() != null) {
					return getExtensionDescriptor(test.getType(), KIND_TEST);
				}
			} else if (testElement instanceof TPFTestComponent) {
				TPFTestComponent testComponent = (TPFTestComponent)testElement;
				if (testComponent.getType() != null) {
					return getExtensionDescriptor(testComponent.getType(), KIND_COMPONENT);
				}
			}
    	}
    	return null;
	}
	
	private class ExtensionDescriptor {
		public IConfigurationElement runHandlerConfigElem = null;
		private IRunHandler runHandler = null;
		public List supportedLaunchModes = new ArrayList(3);
		public List availableShortcutModes = new ArrayList(3);
		public String launchConfigurationType = null;
		public IConfigurationElement execHistoryDefaultsProviderConfigElem = null;
		private IExecutionHistoryDefaultsProvider execHistoryDefaultsProvider = null;
		public IConfigurationElement deploymentsProviderConfigElem = null;
		private IDeploymentsProvider deploymentsProvider = null;
		public IRunHandler getRunHandler() {
			if (this.runHandler == null) {
				if (this.runHandlerConfigElem != null) {
					try {
						this.runHandler = (IRunHandler)this.runHandlerConfigElem.createExecutableExtension("class"); //$NON-NLS-1$
					} catch (CoreException e) {
						TestUIPlugin.logError(e);
						this.runHandlerConfigElem = null;
					}
				}
			}
			return this.runHandler;
		}
		public IExecutionHistoryDefaultsProvider getExecutionHistoryDefaultsProvider() {
			if (this.execHistoryDefaultsProvider == null) {
				if (this.execHistoryDefaultsProviderConfigElem != null) {
					try {
						this.execHistoryDefaultsProvider = (IExecutionHistoryDefaultsProvider)this.execHistoryDefaultsProviderConfigElem.createExecutableExtension("class"); //$NON-NLS-1$
					} catch (CoreException e) {
						TestUIPlugin.logError(e);
						this.execHistoryDefaultsProviderConfigElem = null;
					}
				}
			}
			return this.execHistoryDefaultsProvider;
		}
		public IDeploymentsProvider getDeploymentsProvider() {
			if (this.deploymentsProvider == null) {
				if (this.deploymentsProviderConfigElem != null) {
					try {
						this.deploymentsProvider = (IDeploymentsProvider)this.deploymentsProviderConfigElem.createExecutableExtension("class"); //$NON-NLS-1$
					} catch (CoreException e) {
						TestUIPlugin.logError(e);
						this.deploymentsProviderConfigElem = null;
					}
				}
			}
			return this.deploymentsProvider;
		}
		public boolean isLaunchConfigurationTypeSupported(String lcType) {
			return (this.launchConfigurationType == lcType)
				|| (lcType != null && lcType.equals(this.launchConfigurationType));
		}
	}
	
	private IDeploymentsProvider defaultDeploymentsProvider = new DefaultDeploymentsProvider();
	private IExecutionHistoryDefaultsProvider defaultExecutionHistoryDefaultsProvider = new DefaultExecutionHistoryDefaultsProvider();
	
	private void registerRunHandlers() {
		IExtensionPoint extPoint = Platform.getPluginRegistry().getExtensionPoint(TestUIPlugin.getID() + ".runHandler"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] runHandlers = extPoint.getConfigurationElements();
			for (int i = 0; i < runHandlers.length; i++) {
				IConfigurationElement[] supportedTypes = runHandlers[i].getChildren();
				for (int j = 0; j < supportedTypes.length; j++) {
					String type = supportedTypes[j].getAttribute("type"); //$NON-NLS-1$
					String kind = supportedTypes[j].getAttribute("kind"); //$NON-NLS-1$
					try {
						ExtensionDescriptor ed = getExtensionDescriptor(type, kind);
						ed.runHandlerConfigElem = runHandlers[i];
					} catch (Exception e) {
						TestUIPlugin.logError("Extension " + runHandlers[i].getName() + " was ignored. See next messages for details.");
						TestUIPlugin.logError(e);
					}
				}
			}
		}
	}
	
	private void registerExecutionHistoryDefaultsProviders() {
		IExtensionPoint extPoint = Platform.getPluginRegistry().getExtensionPoint(TestUIPlugin.getID() + ".executionHistoryDefaults"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] defaultsProviders = extPoint.getConfigurationElements();
			for (int i = 0; i < defaultsProviders.length; i++) {
				IConfigurationElement[] supportedTypes = defaultsProviders[i].getChildren();
				for (int j = 0; j < supportedTypes.length; j++) {
					String type = supportedTypes[j].getAttribute("type"); //$NON-NLS-1$
					String kind = supportedTypes[j].getAttribute("kind"); //$NON-NLS-1$
					try {
						ExtensionDescriptor ed = getExtensionDescriptor(type, kind);
						ed.execHistoryDefaultsProviderConfigElem = defaultsProviders[i];
					} catch (Exception e) {
						TestUIPlugin.logError("Extension " + defaultsProviders[i].getName() + " was ignored. See next messages for details.");
						TestUIPlugin.logError(e);
					}
				}
			}
		}
	}
	
	private void registerDeploymentsProviders() {
		IExtensionPoint extPoint = Platform.getPluginRegistry().getExtensionPoint(TestUIPlugin.getID() + ".deploymentProvider"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] deploymentProviders = extPoint.getConfigurationElements();
			for (int i = 0; i < deploymentProviders.length; i++) {
				IConfigurationElement[] supportedTypes = deploymentProviders[i].getChildren();
				for (int j = 0; j < supportedTypes.length; j++) {
					String type = supportedTypes[j].getAttribute("type"); //$NON-NLS-1$
					String kind = supportedTypes[j].getAttribute("kind"); //$NON-NLS-1$
					try {
						ExtensionDescriptor ed = getExtensionDescriptor(type, kind);
						ed.deploymentsProviderConfigElem = deploymentProviders[i];
					} catch (Exception e) {
						TestUIPlugin.logError("Extension " + deploymentProviders[i].getName() + " was ignored. See next messages for details.");
						TestUIPlugin.logError(e);
					}
				}
			}
		}
	}
	
	private void registerSupportedTestTypes() {
		IExtensionPoint extPoint = Platform.getPluginRegistry().getExtensionPoint(TestUIPlugin.getID() + ".launchableType"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] supportedTypes = extPoint.getConfigurationElements();
			for (int i = 0; i < supportedTypes.length; i++) {
				String type = supportedTypes[i].getAttribute("type"); //$NON-NLS-1$
				String kind = supportedTypes[i].getAttribute("kind"); //$NON-NLS-1$
				String lcType = supportedTypes[i].getAttribute("launchConfigurationType"); //$NON-NLS-1$
				try {
					ExtensionDescriptor ed = getExtensionDescriptor(type, kind);
					ed.launchConfigurationType = lcType;
					IConfigurationElement[] supportedModes = supportedTypes[i].getChildren();
					for (int j = 0; j < supportedModes.length; j++) {
						String mode = supportedModes[j].getAttribute("mode"); //$NON-NLS-1$
						boolean shortcutAvailable = "true".equals(supportedModes[j].getAttribute("shortcutAvailable")); //$NON-NLS-1$ //$NON-NLS-2$
						ed.supportedLaunchModes.add(mode);
						if (shortcutAvailable) {
							ed.availableShortcutModes.add(mode);
						}
					}
				} catch (Exception e) {
					TestUIPlugin.logError("Extension " + supportedTypes[i].getName() + " was ignored. See next messages for details.");
					TestUIPlugin.logError(e);
				}
			}
		}
	}
	
	/**
	 * Returns a Run Handler for the specified testElement. This method may return
	 * null if the specified testElement has no Run Handler associated. In that case,
	 * no pre- or post-run handling is expected.
	 * @param testElement
	 * @return
	 */
    public IRunHandler getRunHandler(Object testElement) {
    	ExtensionDescriptor ed = getExtensionDescriptor(testElement);
    	if (ed != null) {
    		return ed.getRunHandler();
    	}
    	return null;
    }

    /**
     * Return an ExecutionHistoryDefaultsProvider for the specified testElement.
     * This method never returns null.
     * @param testElement
     * @return
     */
    public IExecutionHistoryDefaultsProvider getExecutionHistoryDefaultsProvider(Object testElement) {
    	IExecutionHistoryDefaultsProvider result = null;
    	ExtensionDescriptor ed = getExtensionDescriptor(testElement);
    	if (ed != null) {
	    	result = ed.getExecutionHistoryDefaultsProvider();
    	}
    	if (result == null) {
    		result = defaultExecutionHistoryDefaultsProvider;
    	}
    	return result;
    }

    /**
     * Return a DeploymentsProvider for the specified testElement.
     * This method never returns null.
     * @param testElement
     * @return
     */
    public IDeploymentsProvider getDeploymentsProvider(Object testElement) {
    	IDeploymentsProvider result = null;
    	ExtensionDescriptor ed = getExtensionDescriptor(testElement);
    	if (ed != null) {
    		result = ed.getDeploymentsProvider();
    	}
    	if (result == null) {
    		result = defaultDeploymentsProvider;
    	}
    	return result;
    }
    
    /**
     * Returns whether the specified testElement is launchable by a specific LaunchConfigurationType
     * in the given mode.
     * @param testElement The testElement
     * @param mode The run mode used for launching the testElement (usually run, debug or profile)
     * @param launchConfigurationType The launchConfigurationType. Use <code>null</code> for the 
     * Hyades Basic Launch Configuration Type.
     * @return
     */
    public boolean isLaunchableElement(Object testElement, String mode, String launchConfigurationType) {
    	ExtensionDescriptor ed = getExtensionDescriptor(testElement);
    	if (ed != null) {
    		return ed.isLaunchConfigurationTypeSupported(launchConfigurationType)
				&& ed.supportedLaunchModes.contains(mode);
    	}
    	return false;
    }
    
    /**
     * Returns whether a shortcut should be available for launching the specified testElement
     * in the given mode.
     * @param testElement The testElement
     * @param mode The run mode used for launching the testElement (usually run, debug or profile)
      * @return
     */
    public boolean isShortcutAvailable(Object testElement, String mode) {
    	ExtensionDescriptor ed = getExtensionDescriptor(testElement);
    	if (ed != null) {
    		return ed.launchConfigurationType == null // We support shortcuts for default LC only
				&& ed.availableShortcutModes.contains(mode);
    	}
    	return false;
    }
    
	/**
	 * @return Returns the defaultExecutionHistoryDefaultsProvider.
	 */
	public IExecutionHistoryDefaultsProvider getDefaultExecutionHistoryDefaultsProvider() {
		return defaultExecutionHistoryDefaultsProvider;
	}

	/**
	 * @return Returns the defaultDeploymentsProvider.
	 */
	public IDeploymentsProvider getDefaultDeploymentsProvider() {
		return defaultDeploymentsProvider;
	}
	
}
