/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.launch.configurations;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;

/**
 * This class provides a facade for getting/setting properties regarding the Test Execution
 * History file generated by a Test Execution. These properties are stored into a Launch
 * Configuration.
 * @author jcanches
 */
public class ExecutionHistoryLaunchConfigurationFacade extends AbstractLaunchConfigurationFacade {
	
	private static final String PROPERTY_TEH_NAME = PROPERTY_PREFIX + ".tehName"; //$NON-NLS-1$
	private static final String PROPERTY_TEH_OVERRIDE = PROPERTY_PREFIX + ".tehOverride"; //$NON-NLS-1$
	private static final String PROPERTY_TEH_LOCATION = PROPERTY_PREFIX + ".tehLocation"; //$NON-NLS-1$
	
	/**
	 * Get the Test Execution History name from the supplied Launch Configuration.
	 * @param configuration
	 * @return
	 * @throws CoreException
	 */
	public static String getExecutionHistoryName(ILaunchConfiguration configuration) throws CoreException {
		return configuration.getAttribute(PROPERTY_TEH_NAME, (String)null);
	}
	
	/**
	 * Set the Test Execution History name to the supplied Launch Configuration.
	 * @param configuration
	 * @param name
	 */
	public static void setExecutionHistoryName(ILaunchConfigurationWorkingCopy configuration, String name) {
		configuration.setAttribute(PROPERTY_TEH_NAME, name);
	}
	
	/**
	 * Returns whether the produced Test Execution History resource is to be overriden or not.
	 * If it is not, a unique name will be produced, starting with the one specified in the
	 * name property of the configuration.
	 * @param configuration
	 * @return
	 * @throws CoreException
	 */
	public static boolean isExecutionHistoryOverriden(ILaunchConfiguration configuration) throws CoreException {
		return configuration.getAttribute(PROPERTY_TEH_OVERRIDE, false);
	}
	
	/**
	 * Set whether the produced Test Execution History resource is to be overriden or not.
	 * @param configuration
	 * @param value
	 * @see ExecutionHistoryLaunchConfigurationFacade#isExecutionHistoryOverriden(ILaunchConfiguration)
	 */
	public static void setExecutionHistoryOverriden(ILaunchConfigurationWorkingCopy configuration, boolean value) {
		configuration.setAttribute(PROPERTY_TEH_OVERRIDE, value);
	}
	
	/**
	 * Get the Test Execution History location from the supplied Launch Configuration.
	 * @param configuration
	 * @return The Project or Folder Location of the Test Execution History file.
	 * @throws CoreException
	 */
	public static IContainer getExecutionHistoryLocation(ILaunchConfiguration configuration) throws CoreException {
		return getContainerProperty(configuration, PROPERTY_TEH_LOCATION);
	}
	
	/**
	 * Set the Test Execution History location. The value is stored into the supplied Launch Configuration.
	 * @param configuration
	 * @param location This location must be either a project or a folder resource. The Workspace root
	 * is not acceptable.
	 */
	public static void setExecutionHistoryLocation(ILaunchConfigurationWorkingCopy configuration, IContainer location) {
		if (location != null && location.getType() == IResource.ROOT) {
			throw new IllegalArgumentException("location argument cannot be the Workspace Root"); //$NON-NLS-1$
		}
		setContainerProperty(configuration, PROPERTY_TEH_LOCATION, location);
	}
	
}
