package org.eclipse.hyades.logging.events;

import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.SimpleTimeZone;
import java.util.StringTokenizer;

import org.eclipse.hyades.logging.core.XmlUtility;
import org.eclipse.hyades.logging.events.exceptions.InvalidXMLAnyElementException;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * The CbeFormatter class defines a utility object used for 
 * converting, serializing and deserializing CommonBaseEvent
 * and related objects.  It is mostly used by other objects
 * in this API to facilitate shared functionality, but can
 * be used by anyone using the API.
 * 
 * @author Richard K. Duggan
 * @author Paul E. Slauenwhite
 * @author John K. Gerken
 * @version 1.0.1
 * @since 1.0
 * @see CommonBaseEventImpl
 * @see AssociationEngineImpl
 */
public final class CbeFormatter {

    private final static String LINE_SEPARATOR = System.getProperty("line.separator");

    private static ISimpleEventFactory factory = SimpleEventFactoryImpl.getInstance();

    /**
     * Public static string version of the Common Base Event XML schema.
     * 
     * @deprecated Use ICommonBaseEvent#getVersion() as of 1.0.1.
     */
    public final static String xml_version = ICommonBaseEvent.VERSION_1_0_1;

    /**
     * Converts an IAssociationEngine object into a serialized
     * XML fragment.
     * 
     * @param engine The IAssociationEngine to serialize.
     * @return An XML fragment String representing the parameter IAssociationEngine object.
     * @since 1.0
     */
    public static synchronized String toCanonicalXMLString(IAssociationEngine associationEngine) {

        StringBuffer buffer = new StringBuffer(256);

        buffer.append("<AssociationEngine");

        String attributeValue = associationEngine.getId();

        if (attributeValue != null) {
            buffer.append(" id=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = associationEngine.getName();

        if (attributeValue != null) {
            buffer.append(" name=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = associationEngine.getType();

        if (attributeValue != null) {
            buffer.append(" type=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        buffer.append("/>");

        return (buffer.toString());
    }

    /**
     * Internalizes a passed XML fragment String into a passed IAssociationEngine object.
     * 
     * @param engine     The IAssociationEngine object to populate.
     * @param xmlString
     *                   The XML fragment containing the information used to populate the Association Engine object.
     * @since V 1.0
     */
    public static synchronized void fromCanonicalXMLString(IAssociationEngine engine, String xmlString) {

        int index = xmlString.indexOf(" id=\"");

        if (index != -1) {
            engine.setId(XmlUtility.denormalize(xmlString.substring(index + 5, xmlString.indexOf("\"", index + 5)))); //5 is the length of " id=\"".
        }

        index = xmlString.indexOf(" name=\"");

        if (index != -1) {
            engine.setName(XmlUtility.denormalize(xmlString.substring(index + 7, xmlString.indexOf("\"", index + 7)))); //7 is the length of " name=\"".
        }

        index = xmlString.indexOf(" type=\"");

        if (index != -1) {
            engine.setType(XmlUtility.denormalize(xmlString.substring(index + 7, xmlString.indexOf("\"", index + 7)))); //7 is the length of " type=\"".
        }
    }

    /**
     * Internalizes a passed XML document String into a passed IAssociationEngine object.
     * 
     * @param engine     The IAssociationEngine object to populate.
     * @param aXMLDocString
     *                   The XML document containing the information used to populate the Association Engine object.
     * @since 1.0
     */
    public static synchronized void fromCanonicalXMLDocString(IAssociationEngine engine, String aXMLDocString) {
        fromCanonicalXMLString(engine, aXMLDocString);
    }

    /**
     * Converts an IAssociationEngine object into a formatted serialized
     * XML document.
     * 
     * @param engine The IAssociationEngine to serialize.
     * @return An XML document String representing the parameter IAssociationEngine object.
     * @since 1.0
     */
    public static synchronized String toCanonicalXMLDocString(AssociationEngineImpl engine) {
        return (toCanonicalXMLDocString(engine, true));
    }

    /**
     * Converts an IAssociationEngine object into a serialized
     * XML document.  Formatting is based on the format parameter.
     * 
     * @param engine The IAssociationEngine to serialize.
     * @param format If the serialized XML document is formatted with platform-dependent new line(s) and tab(s).
     * @return An XML document String representing the parameter IAssociationEngine object.
     * @since 1.0
     */
    public static synchronized String toCanonicalXMLDocString(AssociationEngineImpl engine, boolean format) {

        StringBuffer buffer = new StringBuffer(256);
        buffer.append("<CommonBaseEvents xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:noNamespaceSchemaLocation=\"commonbaseevent1_0.xsd\">");

        if (format) {
            buffer.append(LINE_SEPARATOR);
            buffer.append("\t");
        }

        buffer.append(toCanonicalXMLString(engine));

        if (format) {
            buffer.append(LINE_SEPARATOR);
        }

        buffer.append("</CommonBaseEvents>");

        return (buffer.toString());
    }

    /**
     * Converts an ICommonBaseEvent object into a formatted serialized
     * XML fragment.  The structure of the serialized XML fragment is
     * based on the parameter ICommonBaseEvent's version.
     * 
     * @param commonBaseEvent The ICommonBaseEvent to serialize.
     * @return An XML fragment String representing the parameter ICommonBaseEvent object.
     * @since 1.0
     */
    public static synchronized String toCanonicalXMLString(ICommonBaseEvent commonBaseEvent) {
        return (toCanonicalXMLString(commonBaseEvent, true));
    }

    /**
     * Converts an ICommonBaseEvent object into a serialized
     * XML fragment.  The format of the serialized XML fragment is
     * based on the parameter ICommonBaseEvent's version.  Formatting 
     * is based on the format parameter.
     * 
     * @param commonBaseEvent The ICommonBaseEvent to serialize.
     * @param format If the serialized XML fragment is formatted with platform-dependent new line(s) and tab(s).
     * @return An XML fragment String representing the parameter ICommonBaseEvent object.
     * @since 1.0
     */
    public static synchronized String toCanonicalXMLString(ICommonBaseEvent commonBaseEvent, boolean format) {

        String cbeVersion = commonBaseEvent.getVersion();

        if ((cbeVersion != null) && (cbeVersion.trim().equals(ICommonBaseEvent.VERSION_1_0))) {
            return (toCanonicalVersion_1_0_XMLString(commonBaseEvent, format));
        }

        //Force the parser for the most current Common Base Event XML schema:
        else {
            return (toCanonicalVersion_1_0_1_XMLString(commonBaseEvent, format));
        }
    }

    private static synchronized String toCanonicalVersion_1_0_XMLString(ICommonBaseEvent commonBaseEvent, boolean format) {

        /* Start with a 1k buffer to load the XML String into */
        StringBuffer buffer = new StringBuffer(1024);

        buffer.append("<CommonBaseEvent");

        if (commonBaseEvent.isSetCreationTime()) {
            buffer.append(" creationTime=\"");
            buffer.append(XmlUtility.normalize(commonBaseEvent.getCreationTime()));
            buffer.append("\"");
        }

        String attributeValue = commonBaseEvent.getExtensionName();

        if (attributeValue != null) {
            buffer.append(" extensionName=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = commonBaseEvent.getGlobalInstanceId();

        if (attributeValue != null) {
            buffer.append(" globalInstanceId=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = commonBaseEvent.getLocalInstanceId();

        if (attributeValue != null) {
            buffer.append(" localInstanceId=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = commonBaseEvent.getMsg();

        if (attributeValue != null) {
            buffer.append(" msg=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = commonBaseEvent.getSituationType();

        if (attributeValue != null) {
            buffer.append(" situationType=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        if (commonBaseEvent.isSetElapsedTime()) {
            buffer.append(" elapsedTime=\"");
            buffer.append(commonBaseEvent.getElapsedTime());
            buffer.append("\"");
        }

        if (commonBaseEvent.isSetPriority()) {
            buffer.append(" priority=\"");
            buffer.append(commonBaseEvent.getPriority());
            buffer.append("\"");
        }

        if (commonBaseEvent.isSetRepeatCount()) {
            buffer.append(" repeatCount=\"");
            buffer.append(commonBaseEvent.getRepeatCount());
            buffer.append("\"");
        }

        if (commonBaseEvent.isSetSequenceNumber()) {
            buffer.append(" sequenceNumber=\"");
            buffer.append(commonBaseEvent.getSequenceNumber());
            buffer.append("\"");
        }

        if (commonBaseEvent.isSetSeverity()) {
            buffer.append(" severity=\"");
            buffer.append(commonBaseEvent.getSeverity());
            buffer.append("\"");
        }

        attributeValue = commonBaseEvent.getVersion();

        if (attributeValue != null) {
            buffer.append(" version=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        buffer.append(">");

        List contextDataElements = commonBaseEvent.getContextDataElements();

        if (contextDataElements != null) {

            for (int i = 0; i < contextDataElements.size(); i++) {

                if (contextDataElements.get(i) != null) {

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t");
                    }

                    buffer.append("<contextDataElements");

                    attributeValue = ((IContextDataElement) (contextDataElements.get(i))).getName();

                    if (attributeValue != null) {
                        buffer.append(" name=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IContextDataElement) (contextDataElements.get(i))).getType();

                    if (attributeValue != null) {
                        buffer.append(" type=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append(">");

                    attributeValue = ((IContextDataElement) (contextDataElements.get(i))).getContextId();

                    if (attributeValue != null) {

                        if (format) {
                            buffer.append(LINE_SEPARATOR);
                            buffer.append("\t\t");
                        }

                        buffer.append("<contextId>");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("</contextId>");
                    }
                    else {

                        attributeValue = ((IContextDataElement) (contextDataElements.get(i))).getContextValue();

                        if (attributeValue != null) {

                            if (format) {
                                buffer.append(LINE_SEPARATOR);
                                buffer.append("\t\t");
                            }

                            buffer.append("<contextValue>");
                            buffer.append(XmlUtility.normalize(attributeValue));
                            buffer.append("</contextValue>");
                        }
                    }

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t");
                    }

                    buffer.append("</contextDataElements>");
                }
            }
        }

        List extendedDataElements = commonBaseEvent.getExtendedDataElements();

        if (extendedDataElements != null) {

            for (int i = 0; i < extendedDataElements.size(); i++) {

                buffer.append(getIExtendedDataElementXML(((IExtendedDataElement) (extendedDataElements.get(i))), "extendedDataElements", format, (format ? 1 : 0)));
            }
        }

        List associatedEvents = commonBaseEvent.getAssociatedEvents();

        if (associatedEvents != null) {

            for (int i = 0; i < associatedEvents.size(); i++) {

                if (associatedEvents.get(i) != null) {

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t");
                    }

                    buffer.append("<associatedEvents");

                    if (((IAssociatedEvent) (associatedEvents.get(i))).getAssociationEngine() != null) {

                        attributeValue = ((IAssociatedEvent) (associatedEvents.get(i))).getAssociationEngine();

                        if (attributeValue != null) {
                            buffer.append(" associationEngine=\"");
                            buffer.append(XmlUtility.normalize(attributeValue));
                            buffer.append("\"");
                        }
                    }

                    String resolvedEvents = ((IAssociatedEvent) (associatedEvents.get(i))).getResolvedEvents();

                    if (resolvedEvents != null) {

                        buffer.append(" resolvedEvents=\"");
                        buffer.append(resolvedEvents);
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }
            }
        }

        IComponentIdentification componentId = commonBaseEvent.getReporterComponentId();

        if (componentId != null) {

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("<reporterComponentId");

            attributeValue = componentId.getApplication();

            if (attributeValue != null) {
                buffer.append(" application=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponent();

            if (attributeValue != null) {
                buffer.append(" component=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponentIdType();

            if (attributeValue != null) {
                buffer.append(" componentIdType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getExecutionEnvironment();

            if (attributeValue != null) {
                buffer.append(" executionEnvironment=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getInstanceId();

            if (attributeValue != null) {
                buffer.append(" instanceId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocation();

            if (attributeValue != null) {
                buffer.append(" location=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocationType();

            if (attributeValue != null) {
                buffer.append(" locationType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getProcessId();

            if (attributeValue != null) {
                buffer.append(" processId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getSubComponent();

            if (attributeValue != null) {
                buffer.append(" subComponent=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getThreadId();

            if (attributeValue != null) {
                buffer.append(" threadId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append("/>");
        }

        componentId = commonBaseEvent.getSourceComponentId();

        if (componentId != null) {

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("<sourceComponentId");

            attributeValue = componentId.getApplication();

            if (attributeValue != null) {
                buffer.append(" application=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponent();

            if (attributeValue != null) {
                buffer.append(" component=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponentIdType();

            if (attributeValue != null) {
                buffer.append(" componentIdType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getExecutionEnvironment();

            if (attributeValue != null) {
                buffer.append(" executionEnvironment=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getInstanceId();

            if (attributeValue != null) {
                buffer.append(" instanceId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocation();

            if (attributeValue != null) {
                buffer.append(" location=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocationType();

            if (attributeValue != null) {
                buffer.append(" locationType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getProcessId();

            if (attributeValue != null) {
                buffer.append(" processId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getSubComponent();

            if (attributeValue != null) {
                buffer.append(" subComponent=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getThreadId();

            if (attributeValue != null) {
                buffer.append(" threadId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append("/>");
        }

        IMsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();

        if (msgDataElement != null) {

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("<msgDataElement");

            attributeValue = msgDataElement.getMsgLocale();

            if (attributeValue != null) {
                buffer.append(" msgLocale=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append(">");

            List tokens = msgDataElement.getMsgCatalogTokens();

            if (tokens != null) {

                for (int i = 0; i < tokens.size(); i++) {

                    if ((tokens.get(i) != null) && (((IMsgCatalogToken) (tokens.get(i))).getValue() != null)) {

                        if (format) {
                            buffer.append(LINE_SEPARATOR);
                            buffer.append("\t\t");
                        }

                        buffer.append("<msgCatalogTokens value=\"");
                        buffer.append(XmlUtility.normalize(((IMsgCatalogToken) (tokens.get(i))).getValue()));
                        buffer.append("\"/>");
                    }
                }
            }

            attributeValue = msgDataElement.getMsgId();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<msgId>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgId>");
            }

            attributeValue = msgDataElement.getMsgIdType();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<msgIdType>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgIdType>");
            }

            attributeValue = msgDataElement.getMsgCatalogId();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<msgCatalogId>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgCatalogId>");
            }

            attributeValue = msgDataElement.getMsgCatalogType();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<msgCatalogType>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgCatalogType>");
            }

            attributeValue = msgDataElement.getMsgCatalog();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<msgCatalog>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgCatalog>");
            }

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("</msgDataElement>");
        }

        List anyElements = commonBaseEvent.getAnyElements();

        if (anyElements != null) {
            for (int i = 0; i < anyElements.size(); i++) {
                if (anyElements.get(i) != null) {

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t");
                    }

                    buffer.append(((ICommonBaseEventAnyElement) (anyElements.get(i))).getXmlAnyElement());
                }
            }
        }

        if (format) {
            buffer.append(LINE_SEPARATOR);
        }

        buffer.append("</CommonBaseEvent>");

        return (buffer.toString());
    }

    private static synchronized String toCanonicalVersion_1_0_1_XMLString(ICommonBaseEvent commonBaseEvent, boolean format) {

        /* Start with a 1k buffer to load the XML String into */
        StringBuffer buffer = new StringBuffer(1024);

        buffer.append("<CommonBaseEvent");

        if (commonBaseEvent.isSetCreationTime()) {
            buffer.append(" creationTime=\"");
            buffer.append(XmlUtility.normalize(commonBaseEvent.getCreationTime()));
            buffer.append("\"");
        }

        String attributeValue = commonBaseEvent.getExtensionName();

        if (attributeValue != null) {
            buffer.append(" extensionName=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = commonBaseEvent.getGlobalInstanceId();

        if (attributeValue != null) {
            buffer.append(" globalInstanceId=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = commonBaseEvent.getLocalInstanceId();

        if (attributeValue != null) {
            buffer.append(" localInstanceId=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = commonBaseEvent.getMsg();

        if (attributeValue != null) {
            buffer.append(" msg=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        if (commonBaseEvent.isSetElapsedTime()) {
            buffer.append(" elapsedTime=\"");
            buffer.append(commonBaseEvent.getElapsedTime());
            buffer.append("\"");
        }

        if (commonBaseEvent.isSetPriority()) {
            buffer.append(" priority=\"");
            buffer.append(commonBaseEvent.getPriority());
            buffer.append("\"");
        }

        if (commonBaseEvent.isSetRepeatCount()) {
            buffer.append(" repeatCount=\"");
            buffer.append(commonBaseEvent.getRepeatCount());
            buffer.append("\"");
        }

        if (commonBaseEvent.isSetSequenceNumber()) {
            buffer.append(" sequenceNumber=\"");
            buffer.append(commonBaseEvent.getSequenceNumber());
            buffer.append("\"");
        }

        if (commonBaseEvent.isSetSeverity()) {
            buffer.append(" severity=\"");
            buffer.append(commonBaseEvent.getSeverity());
            buffer.append("\"");
        }

        attributeValue = commonBaseEvent.getVersion();

        if (attributeValue != null) {
            buffer.append(" version=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        buffer.append(">");

        List contextDataElements = commonBaseEvent.getContextDataElements();

        if (contextDataElements != null) {

            for (int i = 0; i < contextDataElements.size(); i++) {

                if (contextDataElements.get(i) != null) {

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t");
                    }

                    buffer.append("<contextDataElements");

                    attributeValue = ((IContextDataElement) (contextDataElements.get(i))).getName();

                    if (attributeValue != null) {
                        buffer.append(" name=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IContextDataElement) (contextDataElements.get(i))).getType();

                    if (attributeValue != null) {
                        buffer.append(" type=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append(">");

                    attributeValue = ((IContextDataElement) (contextDataElements.get(i))).getContextId();

                    if (attributeValue != null) {

                        if (format) {
                            buffer.append(LINE_SEPARATOR);
                            buffer.append("\t\t");
                        }

                        buffer.append("<contextId>");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("</contextId>");
                    }
                    else {

                        attributeValue = ((IContextDataElement) (contextDataElements.get(i))).getContextValue();

                        if (attributeValue != null) {

                            if (format) {
                                buffer.append(LINE_SEPARATOR);
                                buffer.append("\t\t");
                            }

                            buffer.append("<contextValue>");
                            buffer.append(XmlUtility.normalize(attributeValue));
                            buffer.append("</contextValue>");
                        }
                    }

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t");
                    }

                    buffer.append("</contextDataElements>");
                }
            }
        }

        List extendedDataElements = commonBaseEvent.getExtendedDataElements();

        if (extendedDataElements != null) {

            for (int i = 0; i < extendedDataElements.size(); i++) {

                buffer.append(getIExtendedDataElementXML(((IExtendedDataElement) (extendedDataElements.get(i))), "extendedDataElements", format, (format ? 1 : 0)));
            }
        }

        List associatedEvents = commonBaseEvent.getAssociatedEvents();

        if (associatedEvents != null) {

            for (int i = 0; i < associatedEvents.size(); i++) {

                if (associatedEvents.get(i) != null) {

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t");
                    }

                    buffer.append("<associatedEvents");

                    String resolvedEvents = ((IAssociatedEvent) (associatedEvents.get(i))).getResolvedEvents();

                    if (resolvedEvents != null) {

                        buffer.append(" resolvedEvents=\"");
                        buffer.append(resolvedEvents);
                        buffer.append("\"");
                    }

                    buffer.append(">");

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t\t");
                    }

                    IAssociationEngine associationEngineInfo = ((IAssociatedEvent) (associatedEvents.get(i))).getAssociationEngineInfo();

                    if (associationEngineInfo != null) {

                        String associationEngineInfoXML = toCanonicalXMLString(associationEngineInfo).trim();
                        buffer.append("<associationEngineInfo"); //Add the 'Info' to '<AssociationEngine'.
                        buffer.append(associationEngineInfoXML.substring(18)); //18 is the length of '<AssociationEngine'.
                    }
                    else {

                        attributeValue = ((IAssociatedEvent) (associatedEvents.get(i))).getAssociationEngine();

                        if (attributeValue != null) {

                            buffer.append("<associationEngine>");
                            buffer.append(XmlUtility.normalize(attributeValue));
                            buffer.append("</associationEngine>");
                        }

                    }

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t");
                    }

                    buffer.append("</associatedEvents>");
                }
            }
        }

        IComponentIdentification componentId = commonBaseEvent.getReporterComponentId();

        if (componentId != null) {

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("<reporterComponentId");

            attributeValue = componentId.getApplication();

            if (attributeValue != null) {
                buffer.append(" application=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponent();

            if (attributeValue != null) {
                buffer.append(" component=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponentIdType();

            if (attributeValue != null) {
                buffer.append(" componentIdType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getExecutionEnvironment();

            if (attributeValue != null) {
                buffer.append(" executionEnvironment=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getInstanceId();

            if (attributeValue != null) {
                buffer.append(" instanceId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocation();

            if (attributeValue != null) {
                buffer.append(" location=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocationType();

            if (attributeValue != null) {
                buffer.append(" locationType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getProcessId();

            if (attributeValue != null) {
                buffer.append(" processId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getSubComponent();

            if (attributeValue != null) {
                buffer.append(" subComponent=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getThreadId();

            if (attributeValue != null) {
                buffer.append(" threadId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponentType();

            if (attributeValue != null) {
                buffer.append(" componentType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append("/>");
        }

        componentId = commonBaseEvent.getSourceComponentId();

        if (componentId != null) {

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("<sourceComponentId");

            attributeValue = componentId.getApplication();

            if (attributeValue != null) {
                buffer.append(" application=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponent();

            if (attributeValue != null) {
                buffer.append(" component=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponentIdType();

            if (attributeValue != null) {
                buffer.append(" componentIdType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getExecutionEnvironment();

            if (attributeValue != null) {
                buffer.append(" executionEnvironment=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getInstanceId();

            if (attributeValue != null) {
                buffer.append(" instanceId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocation();

            if (attributeValue != null) {
                buffer.append(" location=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocationType();

            if (attributeValue != null) {
                buffer.append(" locationType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getProcessId();

            if (attributeValue != null) {
                buffer.append(" processId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getSubComponent();

            if (attributeValue != null) {
                buffer.append(" subComponent=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getThreadId();

            if (attributeValue != null) {
                buffer.append(" threadId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponentType();

            if (attributeValue != null) {
                buffer.append(" componentType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append("/>");
        }

        IMsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();

        if (msgDataElement != null) {

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("<msgDataElement");

            attributeValue = msgDataElement.getMsgLocale();

            if (attributeValue != null) {
                buffer.append(" msgLocale=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append(">");

            List tokens = msgDataElement.getMsgCatalogTokens();

            if (tokens != null) {

                for (int i = 0; i < tokens.size(); i++) {

                    if ((tokens.get(i) != null) && (((IMsgCatalogToken) (tokens.get(i))).getValue() != null)) {

                        if (format) {
                            buffer.append(LINE_SEPARATOR);
                            buffer.append("\t\t");
                        }

                        buffer.append("<msgCatalogTokens value=\"");
                        buffer.append(XmlUtility.normalize(((IMsgCatalogToken) (tokens.get(i))).getValue()));
                        buffer.append("\"/>");
                    }
                }
            }

            attributeValue = msgDataElement.getMsgId();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<msgId>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgId>");
            }

            attributeValue = msgDataElement.getMsgIdType();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<msgIdType>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgIdType>");
            }

            attributeValue = msgDataElement.getMsgCatalogId();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<msgCatalogId>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgCatalogId>");
            }

            attributeValue = msgDataElement.getMsgCatalogType();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<msgCatalogType>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgCatalogType>");
            }

            attributeValue = msgDataElement.getMsgCatalog();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<msgCatalog>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgCatalog>");
            }

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("</msgDataElement>");
        }

        ISituation situation = commonBaseEvent.getSituation();

        if (situation == null) {

            attributeValue = commonBaseEvent.getSituationType();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t");
                }

                buffer.append("<situation categoryName=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\">");

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<reportSituation reasoningScope=\"INTERNAL\" reportCategory=\"LOG\"/>");

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t");
                }

                buffer.append("</situation>");
            }
        }
        else {

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("<situation");

            attributeValue = situation.getCategoryName();

            if (attributeValue != null) {
                buffer.append(" categoryName=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append(">");

            ISituationType situationType = situation.getSituationType();

            if (situationType != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<situationType");

                attributeValue = situationType.getReasoningScope();

                if (situationType instanceof IStartSituation) {

                    buffer.append(" xsi:type=\"StartSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IStartSituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IStartSituation) (situationType)).getSituationQualifier();

                    if (attributeValue != null) {

                        buffer.append(" situationQualifier=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }
                else if (situationType instanceof IStopSituation) {

                    buffer.append(" xsi:type=\"StopSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IStopSituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IStopSituation) (situationType)).getSituationQualifier();

                    if (attributeValue != null) {

                        buffer.append(" situationQualifier=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }
                else if (situationType instanceof IConnectSituation) {

                    buffer.append(" xsi:type=\"ConnectSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IConnectSituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IConnectSituation) (situationType)).getSituationDisposition();

                    if (attributeValue != null) {

                        buffer.append(" situationDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }
                else if (situationType instanceof IReportSituation) {

                    buffer.append(" xsi:type=\"ReportSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IReportSituation) (situationType)).getReportCategory();

                    if (attributeValue != null) {

                        buffer.append(" reportCategory=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }

                else if (situationType instanceof IFeatureSituation) {

                    buffer.append(" xsi:type=\"FeatureSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IFeatureSituation) (situationType)).getFeatureDisposition();

                    if (attributeValue != null) {

                        buffer.append(" featureDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }
                else if (situationType instanceof IConfigureSituation) {

                    buffer.append(" xsi:type=\"ConfigureSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IConfigureSituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }
                else if (situationType instanceof IDependencySituation) {

                    buffer.append(" xsi:type=\"DependencySituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IDependencySituation) (situationType)).getDependencyDisposition();

                    if (attributeValue != null) {

                        buffer.append(" dependencyDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }
                else if (situationType instanceof ICreateSituation) {

                    buffer.append(" xsi:type=\"CreateSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((ICreateSituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }

                else if (situationType instanceof IDestroySituation) {

                    buffer.append(" xsi:type=\"DestroySituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IDestroySituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }

                else if (situationType instanceof IAvailableSituation) {

                    buffer.append(" xsi:type=\"AvailableSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IAvailableSituation) (situationType)).getOperationDisposition();

                    if (attributeValue != null) {

                        buffer.append(" operationDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IAvailableSituation) (situationType)).getProcessingDisposition();

                    if (attributeValue != null) {

                        buffer.append(" processingDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IAvailableSituation) (situationType)).getAvailabilityDisposition();

                    if (attributeValue != null) {

                        buffer.append(" availabilityDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }

                else if (situationType instanceof IRequestSituation) {

                    buffer.append(" xsi:type=\"RequestSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IRequestSituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IRequestSituation) (situationType)).getSituationQualifier();

                    if (attributeValue != null) {

                        buffer.append(" situationQualifier=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }

                else if (situationType instanceof IOtherSituation) {

                    buffer.append(" xsi:type=\"OtherSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append(">");

                    String[] anyData = ((IOtherSituation) (situationType)).getAnyData();

                    if (anyData != null) {

                        for (int i = 0; i < anyData.length; i++) {

                            if ((anyData[i] != null) && (anyData[i].length() > 0)) {

                                if (format) {
                                    buffer.append(LINE_SEPARATOR);
                                    buffer.append("\t\t\t");
                                }

                                //Do NOT normalize since this string is assumed to be a valid XML fragment:
                                buffer.append(anyData[i]);
                            }
                        }
                    }

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t\t");
                    }

                    buffer.append("</situationType>");
                }
                else {
                    buffer.append("/>");
                }
            }

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("</situation>");
        }

        String[] otherData = commonBaseEvent.getOtherData();

        if (otherData == null) {

            List anyElements = commonBaseEvent.getAnyElements();

            if (anyElements != null) {

                for (int counter = 0; counter < anyElements.size(); counter++) {

                    if (anyElements.get(counter) != null) {

                        if (format) {
                            buffer.append(LINE_SEPARATOR);
                            buffer.append("\t");
                        }

                        //Do NOT normalize since this string is assumed to be a valid XML fragment:
                        buffer.append(((ICommonBaseEventAnyElement) (anyElements.get(counter))).getXmlAnyElement());
                    }
                }
            }
        }
        else {

            for (int i = 0; i < otherData.length; i++) {

                if ((otherData[i] != null) && (otherData[i].length() > 0)) {

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t");
                    }

                    //Do NOT normalize since this string is assumed to be a valid XML fragment:
                    buffer.append(otherData[i]);
                }
            }
        }

        if (format) {
            buffer.append(LINE_SEPARATOR);
        }

        buffer.append("</CommonBaseEvent>");

        return (buffer.toString());
    }

    private static String getIExtendedDataElementXML(IExtendedDataElement extendedDataElement, String tagName, boolean format, int indent) {

        if (extendedDataElement != null) {

            StringBuffer buffer = new StringBuffer(512);

            if (format) {

                buffer.append(LINE_SEPARATOR);

                for (int counter = 0; counter < indent; counter++)
                    buffer.append("\t");
            }

            buffer.append("<");
            buffer.append(tagName);

            String attributeValue = extendedDataElement.getName();

            if (attributeValue != null) {
                buffer.append(" name=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = extendedDataElement.getType();

            if (attributeValue != null) {
                buffer.append(" type=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append(">");

            if (extendedDataElement.getTypeValue() == IExtendedDataElement.TYPE_HEX_BINARY_VALUE) {

                char[] values = extendedDataElement.getHexValues();

                if ((values != null) && (values.length > 0)) {

                    if (format) {

                        buffer.append(LINE_SEPARATOR);

                        for (int counter = 0; counter < (indent + 1); counter++)
                            buffer.append("\t");
                    }

                    buffer.append("<hexValue>");

                    String hexValue = null;

                    for (int counter = 0; counter < values.length; counter++) {

                        hexValue = Integer.toHexString((int) values[counter]).toUpperCase();

                        if ((hexValue.length() == 1) || (hexValue.length() == 3))
                            buffer.append('0');

                        buffer.append(hexValue);
                    }

                    buffer.append("</hexValue>");
                }
            }
            else {

                List values = extendedDataElement.getValues();

                if ((values != null) && (values.size() > 0)) {

                    for (int counter = 0; counter < values.size(); counter++) {

                        if ((values.get(counter) != null) && (((String) (values.get(counter))).length() > 0)) {

                            if (format) {

                                buffer.append(LINE_SEPARATOR);

                                for (int index = 0; index < (indent + 1); index++)
                                    buffer.append("\t");
                            }

                            buffer.append("<values>");
                            buffer.append(XmlUtility.normalize(((String) (values.get(counter)))));
                            buffer.append("</values>");
                        }
                    }
                }
            }

            List childDataElements = extendedDataElement.getChildDataElements();

            if (childDataElements != null) {

                for (int counter = 0; counter < childDataElements.size(); counter++) {

                    buffer.append(getIExtendedDataElementXML(((IExtendedDataElement) childDataElements.get(counter)), "children", format, (format ? (indent + 1) : 0)));
                }
            }

            if (format) {

                buffer.append(LINE_SEPARATOR);

                for (int counter = 0; counter < indent; counter++)
                    buffer.append("\t");
            }

            buffer.append("</");
            buffer.append(tagName);
            buffer.append(">");

            return (buffer.toString());
        }
        else {
            return "";
        }
    }

    /**
     * Converts an ICommonBaseEvent object into a formatted serialized
     * XML document.  The structure of the serialized XML document is
     * based on the parameter ICommonBaseEvent's version.
     * 
     * @param commonBaseEvent The ICommonBaseEvent to serialize.
     * @return An XML fragment String representing the parameter ICommonBaseEvent object.
     * @since 1.0
     */
    public static synchronized String toCanonicalXMLDocString(ICommonBaseEvent commonBaseEvent) {
        return (toCanonicalXMLDocString(commonBaseEvent, true));
    }

    /**
     * Converts an ICommonBaseEvent object into a serialized
     * XML document.  Formatting is based on the format parameter.
     * The format of the serialized XML document is based on the 
     * parameter ICommonBaseEvent's version.
     * 
     * @param commonBaseEvent The ICommonBaseEvent to serialize.
     * @param format If the serialized XML document is formatted with platform-dependent new line(s) and tab(s).
     * @return An XML fragment String representing the parameter ICommonBaseEvent object.
     * @since 1.0
     */
    public static synchronized String toCanonicalXMLDocString(ICommonBaseEvent commonBaseEvent, boolean format) {

        String cbeVersion = commonBaseEvent.getVersion();

        if ((cbeVersion != null) && (cbeVersion.trim().equals(ICommonBaseEvent.VERSION_1_0))) {
            return (toCanonicalVersion_1_0_XMLDocString(commonBaseEvent, format));
        }

        //Force the parser for the most current Common Base Event XML schema:
        else {
            return (toCanonicalVersion_1_0_1_XMLDocString(commonBaseEvent, format));
        }
    }

    private static synchronized String toCanonicalVersion_1_0_XMLDocString(ICommonBaseEvent commonBaseEvent, boolean format) {

        // Create a buffer containing the top level element and namespace info...
        StringBuffer buffer = new StringBuffer(1024);

        buffer.append("<CommonBaseEvents xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:noNamespaceSchemaLocation=\"commonbaseevent1_0.xsd\">");

        // Add the CommonBaseEvent element fragment
        buffer.append(toCanonicalVersion_1_0_XMLString(commonBaseEvent, format));

        if (format) {
            buffer.append(LINE_SEPARATOR);
        }

        // Close and return the top level element...
        buffer.append("</CommonBaseEvents>");

        return (buffer.toString());
    }

    private static synchronized String toCanonicalVersion_1_0_1_XMLDocString(ICommonBaseEvent commonBaseEvent, boolean format) {

        // Create a buffer containing the top level element and namespace info...
        StringBuffer buffer = new StringBuffer(1024);

        buffer.append("<CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd\">");

        if (format) {
            buffer.append(LINE_SEPARATOR);
        }

        List associatedEvents = commonBaseEvent.getAssociatedEvents();

        // If associatedEvents exist, get the serialized association engine...
        if ((associatedEvents != null) && (associatedEvents.size() > 0)) {

            IAssociationEngine associationEngine = null;

            for (int counter = 0; counter < associatedEvents.size(); counter++) {

                associationEngine = ((IAssociatedEvent) (associatedEvents.get(counter))).getAssociationEngineInfo();

                if (associationEngine != null) {

                    if (format) {
                        buffer.append("\t");
                    }

                    buffer.append(toCanonicalXMLString(associationEngine));

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                    }
                }
            }
        }

        if (format) {

            //Add the CommonBaseEvent element fragment with an extra tab after every platform dependent new line character to indent all CommonBaseEvents child elements:
            StringTokenizer tokens = new StringTokenizer(toCanonicalVersion_1_0_1_XMLString(commonBaseEvent, format).trim(), LINE_SEPARATOR);

            while (tokens.hasMoreTokens()) {

                buffer.append("\t");
                buffer.append(tokens.nextToken());
                buffer.append(LINE_SEPARATOR);
            }
        }
        else {

            //Add the CommonBaseEvent element fragment:
            buffer.append(toCanonicalVersion_1_0_1_XMLString(commonBaseEvent, format));
        }

        // Close and return the top level element:
        buffer.append("</CommonBaseEvents>");

        return buffer.toString();
    }

    /**
     * Converts an XML fragment to an ICommonBaseEvent object.  The format of the 
     * XML fragment is based on the parameter ICommonBaseEvent's version.
     * 
     * @param commonBaseEvent The ICommonBaseEvent to be populated by the XML fragment.
     * @param xmlString The XML fragment to populate the ICommonBaseEvent object.
     * @since 1.0
     */
    public static synchronized void fromCanonicalXMLString(ICommonBaseEvent commonBaseEvent, String xmlString) {

        String cbeVersion = commonBaseEvent.getVersion();

        if ((cbeVersion != null) && (cbeVersion.trim().equals(ICommonBaseEvent.VERSION_1_0))) {
            fromCanonical_1_0_XMLString(commonBaseEvent, xmlString);
        }

        //Force the parser for the most current Common Base Event XML schema:
        else {
            fromCanonical_1_0_1_XMLString(commonBaseEvent, xmlString);
        }
    }

    private static synchronized void fromCanonical_1_0_XMLString(ICommonBaseEvent commonBaseEvent, String xmlString) {

        /* setup the pointers and our initial work area */
        int masterIndex = xmlString.indexOf("<CommonBaseEvent "); // tracks location in passed string
        int index = 0; // tracks location in temp work area
        String tempString = xmlString.substring(masterIndex, xmlString.indexOf(">", masterIndex + 1));
        masterIndex = xmlString.indexOf(">", masterIndex + 1) + 1; // push pointer after '>' character

        /* populate the attributes for the root element of this fragment */
        index = tempString.indexOf(" creationTime=\"");
        if (index != -1) {
            commonBaseEvent.setCreationTime(XmlUtility.denormalize(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)))); //15 is the length of " creationTime=\"".
        }

        index = tempString.indexOf(" extensionName=\"");
        if (index != -1) {
            commonBaseEvent.setExtensionName(XmlUtility.denormalize(tempString.substring(index + 16, tempString.indexOf("\"", index + 16)))); //16 is the length of " extensionName=\"".
        }

        index = tempString.indexOf(" globalInstanceId=\"");
        if (index != -1) {
            commonBaseEvent.setGlobalInstanceId(XmlUtility.denormalize(tempString.substring(index + 19, tempString.indexOf("\"", index + 19)))); //19 is the length of " globalInstanceId=\"".
        }

        index = tempString.indexOf(" localInstanceId=\"");
        if (index != -1) {
            commonBaseEvent.setLocalInstanceId(XmlUtility.denormalize(tempString.substring(index + 18, tempString.indexOf("\"", index + 18)))); //18 is the length of " localInstanceId=\"".
        }

        index = tempString.indexOf(" msg=\"");
        if (index != -1) {
            commonBaseEvent.setMsg(XmlUtility.denormalize(tempString.substring(index + 6, tempString.indexOf("\"", index + 6)))); //6 is the length of " msg=\"".
        }

        index = tempString.indexOf(" situationType=\"");
        if (index != -1) {
            commonBaseEvent.setSituationType(XmlUtility.denormalize(tempString.substring(index + 16, tempString.indexOf("\"", index + 16)))); //16 is the length of " situationType=\"".
        }

        index = tempString.indexOf(" elapsedTime=\"");
        if (index != -1) {
            commonBaseEvent.setElapsedTime(Long.valueOf(tempString.substring(index + 14, tempString.indexOf("\"", index + 14))).longValue()); //14 is the length of " elapsedTime=\"".
        }

        index = tempString.indexOf(" priority=\"");
        if (index != -1) {
            commonBaseEvent.setPriority(Short.valueOf(tempString.substring(index + 11, tempString.indexOf("\"", index + 11))).shortValue()); //11 is the length of " priority=\"".
        }

        index = tempString.indexOf(" repeatCount=\"");
        if (index != -1) {
            commonBaseEvent.setRepeatCount(Short.valueOf(tempString.substring(index + 14, tempString.indexOf("\"", index + 14))).shortValue()); //14 is the length of " repeatCount=\"".
        }

        index = tempString.indexOf(" sequenceNumber=\"");
        if (index != -1) {
            commonBaseEvent.setSequenceNumber(Long.valueOf(tempString.substring(index + 17, tempString.indexOf("\"", index + 17))).longValue()); //17 is the length of " sequenceNumber=\"".
        }

        index = tempString.indexOf(" severity=\"");
        if (index != -1) {
            commonBaseEvent.setSeverity(Short.valueOf(tempString.substring(index + 11, tempString.indexOf("\"", index + 11))).shortValue()); //11 is the length of " severity=\"".
        }

        /* Begin processing of ContextDataEements */
        while (xmlString.indexOf("<contextDataElements ", masterIndex) != -1) {

            /* update pointers and build work area */
            masterIndex = xmlString.indexOf("<contextDataElements ", masterIndex);
            tempString = xmlString.substring(masterIndex, xmlString.indexOf("</contextDataElements>", masterIndex + 1));

            IContextDataElement cde = factory.createContextDataElement();
            /* process the attributes for this element */
            index = tempString.indexOf(" name=\"");
            if (index != -1) {
                cde.setName(XmlUtility.denormalize(tempString.substring(index + 7, tempString.indexOf("\"", index + 7)))); //7 is the length of " name=\"".
            }

            index = tempString.indexOf(" type=\"");
            if (index != -1) {
                cde.setType(XmlUtility.denormalize(tempString.substring(index + 7, tempString.indexOf("\"", index + 7)))); //7 is the length of " type=\"".
            }

            /**
             * contextValue and contextId are mutually exclusive.  Therefore, only 
             * one of these two properties can be defined.  contextValue takes 
             * precidence, such that if contextValue is set then contextId is ignored. 
             */
            index = tempString.indexOf("<contextValue>");
            if (index != -1) {
                cde.setContextValue(XmlUtility.denormalize(tempString.substring(index + 14, tempString.indexOf("</contextValue>", index + 14))));

            }
            else if (tempString.indexOf("<contextId>") != -1) {
                index = tempString.indexOf("<contextId>");
                cde.setContextId(XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("</contextId>", index + 11))));
            }

            commonBaseEvent.addContextDataElement(cde);
            masterIndex = xmlString.indexOf("</contextDataElements>", masterIndex + 1) + 22;
        }

		int edeBeginIndex = -1;
		int edeEndIndex = masterIndex;
		int edeEndTestIndex = -1;

		/* Begin processing of ExtendedDataEements */
		while ((edeBeginIndex = xmlString.indexOf("<extendedDataElements ", edeEndIndex)) != -1) {

			edeEndTestIndex = xmlString.indexOf("</extendedDataElements>", edeBeginIndex);

			if (edeEndTestIndex != -1) {

				edeEndIndex = edeEndTestIndex + 23; //23 is the length of "</extendedDataElements>".

				edeEndTestIndex = xmlString.indexOf("/>", edeBeginIndex);

				if ((edeEndTestIndex != -1) && (edeEndTestIndex < edeEndIndex)) {
					edeEndIndex = edeEndTestIndex + 2; //2 is the length of "/>".
				}
			}
			else {
				edeEndIndex = xmlString.indexOf("/>", edeBeginIndex) + 2; //2 is the length of "/>".            	
			}

			commonBaseEvent.addExtendedDataElement(loadExtendedDataElement(xmlString.substring(edeBeginIndex, edeEndIndex)));
		}

		masterIndex = edeEndIndex;

        //Workaround since CommonBaseEventImpl stores associated events by name and not ID:
        StringBuffer spaces = new StringBuffer();

        /* Begin processing of AssociatedEvents */
        while (xmlString.indexOf("<associatedEvents ", masterIndex) != -1) {

            /* update pointers and build work area */
            masterIndex = xmlString.indexOf("<associatedEvents ", masterIndex);
            tempString = xmlString.substring(masterIndex, xmlString.indexOf("/>", masterIndex + 1));

            IAssociatedEvent ae = factory.createAssociatedEvent();

            // If there is an associationEngine
            index = tempString.indexOf(" associationEngine=\"");

            if (index != -1)
                ae.setAssociationEngine(XmlUtility.denormalize(tempString.substring(index + 20, tempString.indexOf("\"", index + 20)))); //20 is the length of " associationEngine=\"".

            index = tempString.indexOf(" resolvedEvents=\"");

            if (index != -1) {

                String eventList = new String(tempString.substring(index + 17, tempString.indexOf("\"", index + 17))); //17 is the length of " resolvedEvents=\"".
                StringTokenizer st = new StringTokenizer(eventList, " ");

                while (st.hasMoreTokens()) {
                    ICommonBaseEvent aCBE = factory.createCommonBaseEvent();
                    aCBE.setGlobalInstanceId(XmlUtility.denormalize(st.nextToken()));
                    ae.addResolvedEvent(aCBE.getGlobalInstanceId());
                }

                index += eventList.length() + 17; //17 is the length of " resolvedEvents=\"".
            }

            commonBaseEvent.addAssociatedEvent(ae);
            masterIndex = xmlString.indexOf("/>", masterIndex + 1) + 2;
        }

        /* if reporterComponentId is included... */
        if (xmlString.indexOf("<reporterComponentId") != -1) {

            /* setup the pointers and our initial work area */
            masterIndex = xmlString.indexOf("<reporterComponentId");
            tempString = xmlString.substring(masterIndex, xmlString.indexOf("/>", masterIndex + 1));
            masterIndex = xmlString.indexOf("/>", masterIndex + 1) + 2;

            IComponentIdentification rci = factory.createComponentIdentification();

            /* populate the attributes for this element... */
            index = tempString.indexOf(" application=\"");
            if (index != -1) {
                rci.setApplication(XmlUtility.denormalize(tempString.substring(index + 14, tempString.indexOf("\"", index + 14)))); //14 is the length of " application=\"".
            }

            index = tempString.indexOf(" component=\"");
            if (index != -1) {
                rci.setComponent(XmlUtility.denormalize(tempString.substring(index + 12, tempString.indexOf("\"", index + 12)))); //12 is the length of " component=\"".
            }

            index = tempString.indexOf(" componentIdType=\"");
            if (index != -1) {
                rci.setComponentIdType(XmlUtility.denormalize(tempString.substring(index + 18, tempString.indexOf("\"", index + 18)))); //18 is the length of " componentIdType=\"".
            }

            index = tempString.indexOf(" executionEnvironment=\"");
            if (index != -1) {
                rci.setExecutionEnvironment(XmlUtility.denormalize(tempString.substring(index + 23, tempString.indexOf("\"", index + 23)))); //23 is the length of " executionEnvironment=\"".
            }

            index = tempString.indexOf(" instanceId=\"");
            if (index != -1) {
                rci.setInstanceId(XmlUtility.denormalize(tempString.substring(index + 13, tempString.indexOf("\"", index + 13)))); //13 is the length of " instanceId=\"".
            }

            index = tempString.indexOf(" location=\"");
            if (index != -1) {
                rci.setLocation(XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)))); //11 is the length of " location=\"".
            }

            index = tempString.indexOf(" locationType=\"");
            if (index != -1) {
                rci.setLocationType(XmlUtility.denormalize(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)))); //15 is the length of " locationType=\"".
            }

            index = tempString.indexOf(" processId=\"");
            if (index != -1) {
                rci.setProcessId(XmlUtility.denormalize(tempString.substring(index + 12, tempString.indexOf("\"", index + 12)))); //12 is the length of " processId=\"".
            }

            index = tempString.indexOf(" subComponent=\"");
            if (index != -1) {
                rci.setSubComponent(XmlUtility.denormalize(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)))); //15 is the length of " subComponent=\"".
            }

            index = tempString.indexOf(" threadId=\"");
            if (index != -1) {
                rci.setThreadId(XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)))); //11 is the length of " threadId=\"".
            }

            commonBaseEvent.setReporterComponentId(rci);
        }

        /* if sourceComponentId is included... */
        if (xmlString.indexOf("<sourceComponentId") != -1) {

            /* setup the pointers and our initial work area */
            masterIndex = xmlString.indexOf("<sourceComponentId");
            tempString = xmlString.substring(masterIndex, xmlString.indexOf("/>", masterIndex + 1));
            masterIndex = xmlString.indexOf("/>", masterIndex + 1) + 2;

            IComponentIdentification sci = factory.createComponentIdentification();

            /* populate the attributes for this element... */
            index = tempString.indexOf(" application=\"");
            if (index != -1) {
                sci.setApplication(XmlUtility.denormalize(tempString.substring(index + 14, tempString.indexOf("\"", index + 14)))); //14 is the length of " application=\"".
            }

            index = tempString.indexOf(" component=\"");
            if (index != -1) {
                sci.setComponent(XmlUtility.denormalize(tempString.substring(index + 12, tempString.indexOf("\"", index + 12)))); //12 is the length of " component=\"".
            }

            index = tempString.indexOf(" componentIdType=\"");
            if (index != -1) {
                sci.setComponentIdType(XmlUtility.denormalize(tempString.substring(index + 18, tempString.indexOf("\"", index + 18)))); //18 is the length of " componentIdType=\"".
            }

            index = tempString.indexOf(" executionEnvironment=\"");
            if (index != -1) {
                sci.setExecutionEnvironment(XmlUtility.denormalize(tempString.substring(index + 23, tempString.indexOf("\"", index + 23)))); //23 is the length of " executionEnvironment=\"".
            }

            index = tempString.indexOf(" instanceId=\"");
            if (index != -1) {
                sci.setInstanceId(XmlUtility.denormalize(tempString.substring(index + 13, tempString.indexOf("\"", index + 13)))); //13 is the length of " instanceId=\"".
            }

            index = tempString.indexOf(" location=\"");
            if (index != -1) {
                sci.setLocation(XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)))); //11 is the length of " location=\"".
            }

            index = tempString.indexOf(" locationType=\"");
            if (index != -1) {
                sci.setLocationType(XmlUtility.denormalize(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)))); //15 is the length of " locationType=\"".
            }

            index = tempString.indexOf(" processId=\"");
            if (index != -1) {
                sci.setProcessId(XmlUtility.denormalize(tempString.substring(index + 12, tempString.indexOf("\"", index + 12)))); //12 is the length of " processId=\"".
            }

            index = tempString.indexOf(" subComponent=\"");
            if (index != -1) {
                sci.setSubComponent(XmlUtility.denormalize(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)))); //15 is the length of " subComponent=\"".
            }

            index = tempString.indexOf(" threadId=\"");
            if (index != -1) {
                sci.setThreadId(XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)))); //11 is the length of " threadId=\"".
            }

            commonBaseEvent.setSourceComponentId(sci);
        }

        /* if msgDataElement is included... */
        if (xmlString.indexOf("<msgDataElement") != -1) {

            /* setup the pointers and our initial work area */
            masterIndex = xmlString.indexOf("<msgDataElement");
            tempString = xmlString.substring(masterIndex, xmlString.indexOf("</msgDataElement>", masterIndex + 1));
            masterIndex = xmlString.indexOf("</msgDataElement>", masterIndex + 1) + 17;

            IMsgDataElement mde = factory.createMsgDataElement();

            /* populate the attributes for this element... */
            index = tempString.indexOf(" msgLocale=\"");
            if (index != -1) {
                mde.setMsgLocale(XmlUtility.denormalize(tempString.substring(index + 12, tempString.indexOf("\"", index + 12)))); //12 is the length of " msgLocale=\"".
            }

            // If there are tokens to process...
            if (tempString.indexOf("<msgCatalogTokens ", index) != -1) {

                IMsgCatalogToken msgCatalogToken = null;

                // For each token...
                while (tempString.indexOf("<msgCatalogTokens ", index) != -1) {

                    /* process the attribute for this subelement */
                    index = tempString.indexOf(" value=\"", index);

                    if (index != -1) {

                        msgCatalogToken = factory.createMsgCatalogToken();
                        msgCatalogToken.setValue(XmlUtility.denormalize(new String(tempString.substring(index + 8, tempString.indexOf("\"", index + 8))))); //8 is the length of " value=\"".
                        mde.addMsgCatalogToken(msgCatalogToken);
                    }

                    index = tempString.indexOf("/>", index + 1) + 2;
                }
            }

            index = tempString.indexOf("<msgCatalogType>");
            if (index != -1) {
                mde.setMsgCatalogType(XmlUtility.denormalize(tempString.substring(index + 16, tempString.indexOf("</msgCatalogType>", index + 16))));
            }

            index = tempString.indexOf("<msgId>");
            if (index != -1) {
                mde.setMsgId(XmlUtility.denormalize(tempString.substring(index + 7, tempString.indexOf("</msgId>", index + 7))));
            }

            index = tempString.indexOf("<msgIdType>");
            if (index != -1) {
                mde.setMsgIdType(XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("</msgIdType>", index + 11))));
            }

            index = tempString.indexOf("<msgCatalogId>");
            if (index != -1) {
                mde.setMsgCatalogId(XmlUtility.denormalize(tempString.substring(index + 14, tempString.indexOf("</msgCatalogId>", index + 14))));
            }

            index = tempString.indexOf("<msgCatalog>");
            if (index != -1) {
                mde.setMsgCatalog(XmlUtility.denormalize(tempString.substring(index + 12, tempString.indexOf("</msgCatalog>", index + 12))));
            }

            commonBaseEvent.setMsgDataElement(mde);
        }

        index = xmlString.indexOf("xmlns:");
        tempString = xmlString.substring(0);
        while (index > 0) {
            int equalsIndex = tempString.indexOf("=", index);
            if (equalsIndex > 0) {
                String nameSpaceIdentifier = tempString.substring((index + 6), (equalsIndex));

                int beginningIndex = tempString.indexOf("<" + nameSpaceIdentifier + ":");
                if (beginningIndex > -1) {
                    String elementName = tempString.substring(beginningIndex + 1, tempString.indexOf(" ", beginningIndex));
                    int endHeaderPosition = tempString.indexOf(">", beginningIndex);
                    int endingIndex = -1;
                    if (tempString.charAt(endHeaderPosition - 1) == '/') {
                        endingIndex = endHeaderPosition + 1;
                    }
                    else {
                        endingIndex = tempString.indexOf("</" + elementName + ">");
                        endingIndex = endingIndex + elementName.length() + 3;
                    }

                    if (endingIndex > 0) {
                        String elementString = tempString.substring(beginningIndex, endingIndex);
                        tempString = tempString.substring(endingIndex);
                        index = tempString.indexOf("xmlns:");

                        ICommonBaseEventAnyElement anyElement = factory.createAnyElement();
                        try {
                            anyElement.setXmlAnyElement(elementString);
                            commonBaseEvent.addAnyElement(anyElement);
                        }
                        catch (InvalidXMLAnyElementException exp) {
                            // ther eis no logic in this method if the event is bad
                            // so I will just drop this element
                            tempString = tempString.substring(index + 1);
                            index = tempString.indexOf("xmlns:");
                        }
                    }
                    else {
                        tempString = tempString.substring(index + 1);
                        index = tempString.indexOf("xmlns:");
                    }
                }
                else {
                    tempString = tempString.substring(index + 1);
                    index = tempString.indexOf("xmlns:");
                }
            }
            else {
                index = -1;
            }
        }
    }

    private static synchronized void fromCanonical_1_0_1_XMLString(ICommonBaseEvent commonBaseEvent, String xmlString) {

        /* setup the pointers and our initial work area */
        int masterIndex = xmlString.indexOf("<CommonBaseEvent "); // tracks location in passed string
        int index = 0; // tracks location in temp work area
        String tempString = xmlString.substring(masterIndex, xmlString.indexOf(">", masterIndex + 1));
        masterIndex = xmlString.indexOf(">", masterIndex + 1) + 1; // push pointer after '>' character

        /* populate the attributes for the root element of this fragment */
        index = tempString.indexOf(" creationTime=\"");
        if (index != -1) {
            commonBaseEvent.setCreationTime(XmlUtility.denormalize(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)))); //15 is the length of " creationTime=\"".
        }

        index = tempString.indexOf(" extensionName=\"");
        if (index != -1) {
            commonBaseEvent.setExtensionName(XmlUtility.denormalize(tempString.substring(index + 16, tempString.indexOf("\"", index + 16)))); //16 is the length of " extensionName=\"".
        }

        index = tempString.indexOf(" globalInstanceId=\"");
        if (index != -1) {
            commonBaseEvent.setGlobalInstanceId(XmlUtility.denormalize(tempString.substring(index + 19, tempString.indexOf("\"", index + 19)))); //19 is the length of " globalInstanceId=\"".
        }

        index = tempString.indexOf(" localInstanceId=\"");
        if (index != -1) {
            commonBaseEvent.setLocalInstanceId(XmlUtility.denormalize(tempString.substring(index + 18, tempString.indexOf("\"", index + 18)))); //18 is the length of " localInstanceId=\"".
        }

        index = tempString.indexOf(" msg=\"");
        if (index != -1) {
            commonBaseEvent.setMsg(XmlUtility.denormalize(tempString.substring(index + 6, tempString.indexOf("\"", index + 6)))); //6 is the length of " msg=\"".
        }

        index = tempString.indexOf(" situationType=\"");
        String situationTypeAttributeValue = null;
        if (index != -1) {
            situationTypeAttributeValue = (XmlUtility.denormalize(tempString.substring(index + 16, tempString.indexOf("\"", index + 16)))); //16 is the length of " situationType=\"".
        }

        index = tempString.indexOf(" elapsedTime=\"");
        if (index != -1) {
            commonBaseEvent.setElapsedTime(Long.valueOf(tempString.substring(index + 14, tempString.indexOf("\"", index + 14))).longValue()); //14 is the length of " elapsedTime=\"".
        }

        index = tempString.indexOf(" priority=\"");
        if (index != -1) {
            commonBaseEvent.setPriority(Short.valueOf(tempString.substring(index + 11, tempString.indexOf("\"", index + 11))).shortValue()); //11 is the length of " priority=\"".
        }

        index = tempString.indexOf(" repeatCount=\"");
        if (index != -1) {
            commonBaseEvent.setRepeatCount(Short.valueOf(tempString.substring(index + 14, tempString.indexOf("\"", index + 14))).shortValue()); //14 is the length of " repeatCount=\"".
        }

        index = tempString.indexOf(" sequenceNumber=\"");
        if (index != -1) {
            commonBaseEvent.setSequenceNumber(Long.valueOf(tempString.substring(index + 17, tempString.indexOf("\"", index + 17))).longValue()); //17 is the length of " sequenceNumber=\"".
        }

        index = tempString.indexOf(" severity=\"");
        if (index != -1) {
            commonBaseEvent.setSeverity(Short.valueOf(tempString.substring(index + 11, tempString.indexOf("\"", index + 11))).shortValue()); //11 is the length of " severity=\"".
        }

        /* Begin processing of ContextDataEements */
        while (xmlString.indexOf("<contextDataElements ", masterIndex) != -1) {

            /* update pointers and build work area */
            masterIndex = xmlString.indexOf("<contextDataElements ", masterIndex);
            tempString = xmlString.substring(masterIndex, xmlString.indexOf("</contextDataElements>", masterIndex + 1));

            IContextDataElement cde = factory.createContextDataElement();
            /* process the attributes for this element */
            index = tempString.indexOf(" name=\"");
            if (index != -1) {
                cde.setName(XmlUtility.denormalize(tempString.substring(index + 7, tempString.indexOf("\"", index + 7)))); //7 is the length of " name=\"".
            }

            index = tempString.indexOf(" type=\"");
            if (index != -1) {
                cde.setType(XmlUtility.denormalize(tempString.substring(index + 7, tempString.indexOf("\"", index + 7)))); //7 is the length of " type=\"".
            }

            /**
             * contextValue and contextId are mutually exclusive.  Therefore, only 
             * one of these two properties can be defined.  contextValue takes 
             * precidence, such that if contextValue is set then contextId is ignored. 
             */
            index = tempString.indexOf("<contextValue>");
            if (index != -1) {
                cde.setContextValue(XmlUtility.denormalize(tempString.substring(index + 14, tempString.indexOf("</contextValue>", index + 14))));

            }
            else if (tempString.indexOf("<contextId>") != -1) {
                index = tempString.indexOf("<contextId>");
                cde.setContextId(XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("</contextId>", index + 11))));
            }

            commonBaseEvent.addContextDataElement(cde);
            masterIndex = xmlString.indexOf("</contextDataElements>", masterIndex + 1) + 22;
        }

        int edeBeginIndex = -1;
        int edeEndIndex = masterIndex;
        int edeEndTestIndex = -1;

        /* Begin processing of ExtendedDataEements */
        while ((edeBeginIndex = xmlString.indexOf("<extendedDataElements ", edeEndIndex)) != -1) {

            edeEndTestIndex = xmlString.indexOf("</extendedDataElements>", edeBeginIndex);

            if (edeEndTestIndex != -1) {

                edeEndIndex = edeEndTestIndex + 23; //23 is the length of "</extendedDataElements>".

                edeEndTestIndex = xmlString.indexOf("/>", edeBeginIndex);

                if ((edeEndTestIndex != -1) && (edeEndTestIndex < edeEndIndex)) {
                    edeEndIndex = edeEndTestIndex + 2; //2 is the length of "/>".
                }
            }
            else {
                edeEndIndex = xmlString.indexOf("/>", edeBeginIndex) + 2; //2 is the length of "/>".            	
            }

            commonBaseEvent.addExtendedDataElement(loadExtendedDataElement(xmlString.substring(edeBeginIndex, edeEndIndex)));
        }

        masterIndex = edeEndIndex;

        //Workaround since CommonBaseEventImpl stores associated events by name and not ID:
        StringBuffer spaces = new StringBuffer();

        /* Begin processing of AssociatedEvents */
        while (xmlString.indexOf("<associatedEvents ", masterIndex) != -1) {

            /* update pointers and build work area */
            masterIndex = xmlString.indexOf("<associatedEvents ", masterIndex);
            tempString = xmlString.substring(masterIndex, xmlString.indexOf("/>", masterIndex + 1));

            IAssociatedEvent associatedEvent = factory.createAssociatedEvent();

            index = tempString.indexOf(" resolvedEvents=\""); //Start from the beginning of the new temporary string.

            if (index != -1) {

                StringTokenizer tokens = new StringTokenizer(tempString.substring(index + 17, tempString.indexOf("\"", index + 17)), " "); //17 is the length of " resolvedEvents=\"".

                while (tokens.hasMoreTokens())
                    associatedEvent.addResolvedEvent(XmlUtility.denormalize(tokens.nextToken()));
            }

            // If there is an associationEngine
            index = tempString.indexOf("<associationEngineInfo ", index);

            if (index == -1) {

                //If there is an associationEngine:
                index = tempString.indexOf("<associationEngine>");

                if (index != -1)
                    associatedEvent.setAssociationEngine(XmlUtility.denormalize(tempString.substring(index + 19, tempString.indexOf("</associationEngine>", index + 19)))); //19 is the length of "<associationEngine>".
            }
            else {

                IAssociationEngine associationEngine = factory.createAssociationEngine();

                fromCanonicalXMLString(associationEngine, tempString);

                associatedEvent.setAssociationEngineInfo(associationEngine);
            }

            commonBaseEvent.addAssociatedEvent(associatedEvent);

            masterIndex = xmlString.indexOf("/>", masterIndex) + 2; //2 is the length of "/>".
        }

        /* if reporterComponentId is included... */
        if (xmlString.indexOf("<reporterComponentId") != -1) {

            /* setup the pointers and our initial work area */
            masterIndex = xmlString.indexOf("<reporterComponentId");
            tempString = xmlString.substring(masterIndex, xmlString.indexOf("/>", masterIndex + 1));
            masterIndex = xmlString.indexOf("/>", masterIndex + 1) + 2;

            IComponentIdentification rci = factory.createComponentIdentification();

            /* populate the attributes for this element... */
            index = tempString.indexOf(" application=\"");
            if (index != -1) {
                rci.setApplication(XmlUtility.denormalize(tempString.substring(index + 14, tempString.indexOf("\"", index + 14)))); //14 is the length of " application=\"".
            }

            index = tempString.indexOf(" component=\"");
            if (index != -1) {
                rci.setComponent(XmlUtility.denormalize(tempString.substring(index + 12, tempString.indexOf("\"", index + 12)))); //12 is the length of " component=\"".
            }

            index = tempString.indexOf(" componentIdType=\"");
            if (index != -1) {
                rci.setComponentIdType(XmlUtility.denormalize(tempString.substring(index + 18, tempString.indexOf("\"", index + 18)))); //18 is the length of " componentIdType=\"".
            }

            index = tempString.indexOf(" executionEnvironment=\"");
            if (index != -1) {
                rci.setExecutionEnvironment(XmlUtility.denormalize(tempString.substring(index + 23, tempString.indexOf("\"", index + 23)))); //23 is the length of " executionEnvironment=\"".
            }

            index = tempString.indexOf(" instanceId=\"");
            if (index != -1) {
                rci.setInstanceId(XmlUtility.denormalize(tempString.substring(index + 13, tempString.indexOf("\"", index + 13)))); //13 is the length of " instanceId=\"".
            }

            index = tempString.indexOf(" location=\"");
            if (index != -1) {
                rci.setLocation(XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)))); //11 is the length of " location=\"".
            }

            index = tempString.indexOf(" locationType=\"");
            if (index != -1) {
                rci.setLocationType(XmlUtility.denormalize(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)))); //15 is the length of " locationType=\"".
            }

            index = tempString.indexOf(" processId=\"");
            if (index != -1) {
                rci.setProcessId(XmlUtility.denormalize(tempString.substring(index + 12, tempString.indexOf("\"", index + 12)))); //12 is the length of " processId=\"".
            }

            index = tempString.indexOf(" subComponent=\"");
            if (index != -1) {
                rci.setSubComponent(XmlUtility.denormalize(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)))); //15 is the length of " subComponent=\"".
            }

            index = tempString.indexOf(" threadId=\"");
            if (index != -1) {
                rci.setThreadId(XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)))); //11 is the length of " threadId=\"".
            }

            index = tempString.indexOf(" componentType=\"");
            if (index != -1) {
                rci.setComponentType(XmlUtility.denormalize(tempString.substring(index + 16, tempString.indexOf("\"", index + 16)))); //16 is the length of " componentType=\"".
            }

            commonBaseEvent.setReporterComponentId(rci);
        }

        /* if sourceComponentId is included... */
        if (xmlString.indexOf("<sourceComponentId") != -1) {

            /* setup the pointers and our initial work area */
            masterIndex = xmlString.indexOf("<sourceComponentId");
            tempString = xmlString.substring(masterIndex, xmlString.indexOf("/>", masterIndex + 1));
            masterIndex = xmlString.indexOf("/>", masterIndex + 1) + 2;

            IComponentIdentification sci = factory.createComponentIdentification();

            /* populate the attributes for this element... */
            index = tempString.indexOf(" application=\"");
            if (index != -1) {
                sci.setApplication(XmlUtility.denormalize(tempString.substring(index + 14, tempString.indexOf("\"", index + 14)))); //14 is the length of " application=\"".
            }

            index = tempString.indexOf(" component=\"");
            if (index != -1) {
                sci.setComponent(XmlUtility.denormalize(tempString.substring(index + 12, tempString.indexOf("\"", index + 12)))); //12 is the length of " component=\"".
            }

            index = tempString.indexOf(" componentIdType=\"");
            if (index != -1) {
                sci.setComponentIdType(XmlUtility.denormalize(tempString.substring(index + 18, tempString.indexOf("\"", index + 18)))); //18 is the length of " componentIdType=\"".
            }

            index = tempString.indexOf(" executionEnvironment=\"");
            if (index != -1) {
                sci.setExecutionEnvironment(XmlUtility.denormalize(tempString.substring(index + 23, tempString.indexOf("\"", index + 23)))); //23 is the length of " executionEnvironment=\"".
            }

            index = tempString.indexOf(" instanceId=\"");
            if (index != -1) {
                sci.setInstanceId(XmlUtility.denormalize(tempString.substring(index + 13, tempString.indexOf("\"", index + 13)))); //13 is the length of " instanceId=\"".
            }

            index = tempString.indexOf(" location=\"");
            if (index != -1) {
                sci.setLocation(XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)))); //11 is the length of " location=\"".
            }

            index = tempString.indexOf(" locationType=\"");
            if (index != -1) {
                sci.setLocationType(XmlUtility.denormalize(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)))); //15 is the length of " locationType=\"".
            }

            index = tempString.indexOf(" processId=\"");
            if (index != -1) {
                sci.setProcessId(XmlUtility.denormalize(tempString.substring(index + 12, tempString.indexOf("\"", index + 12)))); //12 is the length of " processId=\"".
            }

            index = tempString.indexOf(" subComponent=\"");
            if (index != -1) {
                sci.setSubComponent(XmlUtility.denormalize(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)))); //15 is the length of " subComponent=\"".
            }

            index = tempString.indexOf(" threadId=\"");
            if (index != -1) {
                sci.setThreadId(XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)))); //11 is the length of " threadId=\"".
            }

            index = tempString.indexOf(" componentType=\"");
            if (index != -1) {
                sci.setComponentType(XmlUtility.denormalize(tempString.substring(index + 16, tempString.indexOf("\"", index + 16)))); //16 is the length of " componentType=\"".
            }

            commonBaseEvent.setSourceComponentId(sci);
        }

        /* if msgDataElement is included... */
        if (xmlString.indexOf("<msgDataElement") != -1) {

            /* setup the pointers and our initial work area */
            masterIndex = xmlString.indexOf("<msgDataElement");
            tempString = xmlString.substring(masterIndex, xmlString.indexOf("</msgDataElement>", masterIndex + 1));
            masterIndex = xmlString.indexOf("</msgDataElement>", masterIndex + 1) + 17;

            IMsgDataElement mde = factory.createMsgDataElement();

            /* populate the attributes for this element... */
            index = tempString.indexOf(" msgLocale=\"");
            if (index != -1) {
                mde.setMsgLocale(XmlUtility.denormalize(tempString.substring(index + 12, tempString.indexOf("\"", index + 12)))); //12 is the length of " msgLocale=\"".
            }

            // If there are tokens to process...
            if (tempString.indexOf("<msgCatalogTokens ", index) != -1) {

                IMsgCatalogToken msgCatalogToken = null;

                // For each token...
                while (tempString.indexOf("<msgCatalogTokens ", index) != -1) {

                    /* process the attribute for this subelement */
                    index = tempString.indexOf(" value=\"", index);

                    if (index != -1) {

                        msgCatalogToken = factory.createMsgCatalogToken();
                        msgCatalogToken.setValue(XmlUtility.denormalize(new String(tempString.substring(index + 8, tempString.indexOf("\"", index + 8))))); //8 is the length of " value=\"".
                        mde.addMsgCatalogToken(msgCatalogToken);
                    }

                    index = tempString.indexOf("/>", index + 1) + 2;
                }
            }

            index = tempString.indexOf("<msgCatalogType>");
            if (index != -1) {
                mde.setMsgCatalogType(XmlUtility.denormalize(tempString.substring(index + 16, tempString.indexOf("</msgCatalogType>", index + 16))));
            }

            index = tempString.indexOf("<msgId>");
            if (index != -1) {
                mde.setMsgId(XmlUtility.denormalize(tempString.substring(index + 7, tempString.indexOf("</msgId>", index + 7))));
            }

            index = tempString.indexOf("<msgIdType>");
            if (index != -1) {
                mde.setMsgIdType(XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("</msgIdType>", index + 11))));
            }

            index = tempString.indexOf("<msgCatalogId>");
            if (index != -1) {
                mde.setMsgCatalogId(XmlUtility.denormalize(tempString.substring(index + 14, tempString.indexOf("</msgCatalogId>", index + 14))));
            }

            index = tempString.indexOf("<msgCatalog>");
            if (index != -1) {
                mde.setMsgCatalog(XmlUtility.denormalize(tempString.substring(index + 12, tempString.indexOf("</msgCatalog>", index + 12))));
            }

            commonBaseEvent.setMsgDataElement(mde);
        }

        //Check for the situation element:
        if (xmlString.indexOf("<situation") == -1) {

            if (situationTypeAttributeValue != null) {

                IReportSituation reportSituation = factory.createReportSituation();
                reportSituation.setReasoningScope("INTERNAL");
                reportSituation.setReportCategory("LOG");

                ISituation situation = factory.createSituation();
                situation.setCategoryName(situationTypeAttributeValue);
                situation.setSituationType(reportSituation);

                commonBaseEvent.setSituation(situation);
            }
        }
        else {

            /* setup the pointers and our initial work area */
            masterIndex = xmlString.indexOf("<situation");
            tempString = xmlString.substring(masterIndex, xmlString.indexOf("</situation>", masterIndex));
            masterIndex = xmlString.indexOf("</situation>", masterIndex) + 12; //12 is the length of "</situation>".

            ISituation situation = factory.createSituation();

            //Find the categoryName attribute:
            index = tempString.indexOf(" categoryName=\"");

            if (index != -1) {
                situation.setCategoryName(XmlUtility.denormalize(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)))); //15 is the length of " categoryName=\"".
            }

            //Check for the SituationType element:
            index = tempString.indexOf("<situationType ");

            if (index != -1) {

                int endIndex = tempString.indexOf("</situationType>", index);

                if (endIndex == -1) {
                    endIndex = tempString.indexOf("/>", index);
                }

                tempString = tempString.substring(index, endIndex);

                //Find the xsi:type attribute:
                index = tempString.indexOf(" xsi:type=\""); //Start from the beginning of the new temporary string.

                if (index != -1) {

                    String situationType = (XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)))); //11 is the length of " xsi:type=\"".
                    String reasoningScope = null;

                    //Find the reasoningScope attribute:
                    index = tempString.indexOf(" reasoningScope=\""); //Start from the beginning of the new temporary string.

                    if (index != -1) {
                        reasoningScope = (XmlUtility.denormalize(tempString.substring(index + 17, tempString.indexOf("\"", index + 17)))); //17 is the length of " reasoningScope=\"".
                    }

                    if (situationType.equals("StartSituation")) {

                        IStartSituation startSituation = factory.createStartSituation();

                        index = tempString.indexOf(" successDisposition=\"");

                        if (index != -1)
                            startSituation.setSuccessDisposition(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " successDisposition=\"".

                        index = tempString.indexOf(" situationQualifier=\"");

                        if (index != -1)
                            startSituation.setSituationQualifier(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " situationQualifier=\"".

                        startSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(startSituation);
                    }
                    else if (situationType.equals("StopSituation")) {

                        IStopSituation stopSituation = factory.createStopSituation();

                        index = tempString.indexOf(" successDisposition=\"");

                        if (index != -1)
                            stopSituation.setSuccessDisposition(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " successDisposition=\"".

                        index = tempString.indexOf(" situationQualifier=\"");

                        if (index != -1)
                            stopSituation.setSituationQualifier(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " situationQualifier=\"".

                        stopSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(stopSituation);
                    }
                    else if (situationType.equals("ConnectSituation")) {

                        IConnectSituation connectSituation = factory.createConnectSituation();

                        index = tempString.indexOf(" successDisposition=\"");

                        if (index != -1)
                            connectSituation.setSuccessDisposition(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " successDisposition=\"".

                        index = tempString.indexOf(" situationDisposition=\"");

                        if (index != -1)
                            connectSituation.setSituationDisposition(XmlUtility.denormalize(tempString.substring(index + 23, tempString.indexOf("\"", index + 23)))); //23 is the length of " situationDisposition=\"".

                        connectSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(connectSituation);
                    }
                    else if (situationType.equals("ReportSituation")) {

                        IReportSituation reportSituation = factory.createReportSituation();

                        index = tempString.indexOf(" reportCategory=\"");

                        if (index != -1)
                            reportSituation.setReportCategory(XmlUtility.denormalize(tempString.substring(index + 17, tempString.indexOf("\"", index + 17)))); //17 is the length of " reportCategory=\"".

                        reportSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(reportSituation);
                    }
                    else if (situationType.equals("FeatureSituation")) {

                        IFeatureSituation featureSituation = factory.createFeatureSituation();

                        index = tempString.indexOf(" featureDisposition=\"");

                        if (index != -1)
                            featureSituation.setFeatureDisposition(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " featureDisposition=\"".

                        featureSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(featureSituation);
                    }
                    else if (situationType.equals("ConfigureSituation")) {

                        IConfigureSituation configureSituation = factory.createConfigureSituation();

                        index = tempString.indexOf(" successDisposition=\"");

                        if (index != -1)
                            configureSituation.setSuccessDisposition(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " successDisposition=\"".

                        configureSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(configureSituation);
                    }
                    else if (situationType.equals("DependencySituation")) {

                        IDependencySituation dependencySituation = factory.createDependencySituation();

                        index = tempString.indexOf(" dependencyDisposition=\"");

                        if (index != -1)
                            dependencySituation.setDependencyDisposition(XmlUtility.denormalize(tempString.substring(index + 24, tempString.indexOf("\"", index + 24)))); //24 is the length of " dependencyDisposition=\"".

                        dependencySituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(dependencySituation);
                    }
                    else if (situationType.equals("CreateSituation")) {

                        ICreateSituation createSituation = factory.createCreateSituation();

                        index = tempString.indexOf(" successDisposition=\"");

                        if (index != -1)
                            createSituation.setSuccessDisposition(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " successDisposition=\"".

                        createSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(createSituation);
                    }
                    else if (situationType.equals("DestroySituation")) {

                        IDestroySituation destroySituation = factory.createDestroySituation();

                        index = tempString.indexOf(" successDisposition=\"");

                        if (index != -1)
                            destroySituation.setSuccessDisposition(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " successDisposition=\"".

                        destroySituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(destroySituation);
                    }
                    else if (situationType.equals("AvailableSituation")) {

                        IAvailableSituation availableSituation = factory.createAvailableSituation();

                        index = tempString.indexOf(" operationDisposition=\"");

                        if (index != -1)
                            availableSituation.setOperationDisposition(XmlUtility.denormalize(tempString.substring(index + 23, tempString.indexOf("\"", index + 23)))); //23 is the length of " operationDisposition=\"".

                        index = tempString.indexOf(" processingDisposition=\"");

                        if (index != -1)
                            availableSituation.setProcessingDisposition(XmlUtility.denormalize(tempString.substring(index + 24, tempString.indexOf("\"", index + 24)))); //24 is the length of " processingDisposition=\"".

                        index = tempString.indexOf(" availabilityDisposition=\"");

                        if (index != -1)
                            availableSituation.setAvailabilityDisposition(XmlUtility.denormalize(tempString.substring(index + 26, tempString.indexOf("\"", index + 26)))); //26 is the length of " availabilityDisposition=\"".

                        availableSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(availableSituation);
                    }
                    else if (situationType.equals("RequestSituation")) {

                        IRequestSituation requestSituation = factory.createRequestSituation();

                        index = tempString.indexOf(" successDisposition=\"");

                        if (index != -1)
                            requestSituation.setSuccessDisposition(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " successDisposition=\"".

                        index = tempString.indexOf(" situationQualifier=\"");

                        if (index != -1)
                            requestSituation.setSituationQualifier(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " situationQualifier=\"".

                        requestSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(requestSituation);
                    }
                    else if (situationType.equals("OtherSituation")) {

                        IOtherSituation otherSituation = factory.createOtherSituation();

                        String[] anyData = getXMLFragments(tempString.substring((tempString.indexOf(">") + 1)));

                        if ((anyData != null) && (anyData.length > 0)) {
                            otherSituation.setAnyData(anyData);
                        }

                        otherSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(otherSituation);
                    }
                }
            }

            commonBaseEvent.setSituation(situation);
        }

        String[] otherData = getXMLFragments(xmlString.substring(masterIndex, xmlString.indexOf("</CommonBaseEvent>", masterIndex)));

        if ((otherData != null) && (otherData.length > 0)) {
            commonBaseEvent.setOtherData(otherData);
        }
    }

    /**
     * Converts an XML document to an ICommonBaseEvent object.  The format of the 
     * XML document is based on the parameter ICommonBaseEvent's version.
     * 
     * @param commonBaseEvent The ICommonBaseEvent to be populated by the XML document.
     * @param xmlString The XML document to populate the ICommonBaseEvent object.
     * @since 1.0
     */
    public static synchronized void fromCanonicalXMLDocString(ICommonBaseEvent event, String aXMLDocString) {
        fromCanonicalXMLString(event, aXMLDocString);
    }

    /**
     * 3.2.7.1 Lexical representation
     *
     * A single lexical representation, which is a subset of the lexical representations allowed by
     * [ISO 8601], is allowed for dateTime. This lexical representation is the [ISO 8601] extended
     * format CCYY-MM-DDThh:mm:ss where "CC" represents the century, "YY" the year, "MM" the month
     * and "DD" the day, preceded by an optional leading "-" sign to indicate a negative number. If
     * the sign is omitted, "+" is assumed. The letter "T" is the date/time separator and "hh",
     * "mm", "ss" represent hour, minute and second respectively. Additional digits can be used
     * to increase the precision of fractional seconds if desired i.e the format ss.ss... with any
     * number of digits after the decimal point is supported. The fractional seconds part is optional;
     * other parts of the lexical form are not optional. To accommodate year values greater than 9999
     * additional digits can be added to the left of this representation. Leading zeros are required
     * if the year value would otherwise have fewer than four digits; otherwise they are forbidden.
     * The year 0000 is prohibited.
     *
     * The CCYY field must have at least four digits, the MM, DD, SS, hh, mm and ss fields exactly
     * two digits each (not counting fractional seconds); leading zeroes must be used if the field
     * would otherwise have too few digits.
     *
     * This representation may be immediately followed by a "Z" to indicate Coordinated Universal
     * Time (UTC) or, to indicate the time zone, i.e. the difference between the local time and
     * Coordinated Universal Time, immediately followed by a sign, + or -, followed by the difference
     * from UTC represented as hh:mm (note: the minutes part is required). See ISO 8601 Date and
     * Time Formats (D) for details about legal values in the various fields. If the time zone is
     * included, both hours and minutes must be present.
     *
     * For example, to indicate 1:20 pm on May the 31st, 1999 for Eastern Standard Time which is
     * 5 hours behind Coordinated Universal Time (UTC), one would write: 1999-05-31T13:20:00-05:00.
     */
    public static long convertXmlSchemaDateTimeToDate(String dateTime) {
        String dateString = null;
        String subSecondsString = null;
        String utcOffsetString = null;
        char utcOffsetDirection = '\0';
        Date date = null;
        long milliseconds = 0;
        int utcOffsetStringIndex = -1;
        int subSecondsStringIndex;

        // define a simple date format object
        String dateFormat = "yyyy-MM-dd'T'HH:mm:ss";
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat(dateFormat);
        int dateFormatLength = dateFormat.length() - 2;

        try {

            if (dateTime.length() < dateFormatLength) {
                milliseconds = 0;
            }
            else {

                if (dateTime.indexOf('Z') > dateFormatLength) {
                    utcOffsetStringIndex = dateTime.indexOf('Z');

                    // adjust dateTime to remove timezone information
                    dateString = dateTime.substring(0, utcOffsetStringIndex);

                    utcOffsetDirection = 'Z';
                }
                else if (dateTime.indexOf('+') > dateFormatLength) {
                    utcOffsetStringIndex = dateTime.indexOf('+');

                    // adjust dateTime to remove timezone information
                    dateString = dateTime.substring(0, utcOffsetStringIndex);

                    // get the offset string
                    utcOffsetDirection = dateTime.charAt(utcOffsetStringIndex);
                    utcOffsetString = dateTime.substring(utcOffsetStringIndex + 1);
                }
                else if (dateTime.indexOf('-', dateFormatLength) > dateFormatLength) {
                    utcOffsetStringIndex = dateTime.indexOf('-', dateFormatLength);

                    // adjust dateTime to remove timezone information
                    dateString = dateTime.substring(0, utcOffsetStringIndex);

                    // get the offset string
                    utcOffsetDirection = dateTime.charAt(utcOffsetStringIndex);
                    utcOffsetString = dateTime.substring(utcOffsetStringIndex + 1);
                }
                else {
                    dateString = dateTime;
                }

                subSecondsStringIndex = dateString.indexOf('.');

                if (subSecondsStringIndex > 0) {
                    // adjust dateTime to remove timezone information
                    if (utcOffsetStringIndex != -1) {
                        dateString = dateTime.substring(0, utcOffsetStringIndex);
                    }

                    // get the subSeconds string
                    // make sure to add 1 to the offset so we can skip the .
                    subSecondsString = dateString.substring(subSecondsStringIndex + 1);
                }

                if (dateString != null) {
                    String year = dateString.substring(0, 4);

                    if (Integer.parseInt(year) > 1969) {
                        // create a the GMT timezone 
                        SimpleTimeZone gmt = new SimpleTimeZone(0, "UTC");

                        // create a GregorianCalendar with the GMT time zone
                        Calendar calendar = new GregorianCalendar(gmt);
                        ParsePosition pos = new ParsePosition(0);
                        simpleDateFormat.setCalendar(calendar);
                        date = simpleDateFormat.parse(dateString, pos);
                    }

                }

                if (date != null) {
                    // now get the milliseconds from the date
                    milliseconds = date.getTime();

                    // now convert microseconds string into a long and add it to the milliseconds in the date
                    if ((subSecondsString != null) && (subSecondsString.length() > 0)) {
                        long subSeconds = Long.parseLong(subSecondsString);
                        int subSecondsLength = subSecondsString.length();
                        if (subSecondsLength < 4) {

                            for (int i = 1; i < (4 - subSecondsLength); i++) {
                                subSeconds = subSeconds * 10;
                            }
                        }
                        else {
                            for (int i = 1; i < (4 - subSecondsLength); i++) {
                                subSeconds = subSeconds / 10;
                            }
                        }

                        milliseconds = milliseconds + subSeconds;
                    }

                    // now depending on the offset adjust the time to UTC   
                    if ((utcOffsetDirection != '\0') && (utcOffsetString != null) && (utcOffsetString.length() == 5)) {
                        long offsetMinutes = Long.parseLong(utcOffsetString.substring(0, 2));
                        long offsetSeconds = Long.parseLong(utcOffsetString.substring(3));

                        if (utcOffsetDirection == '+') {
                            milliseconds = milliseconds + (offsetMinutes * 60 * 1000) + (offsetSeconds * 1000);
                        }
                        else if (utcOffsetDirection == '-') {
                            milliseconds = milliseconds - (offsetMinutes * 60 * 1000) - (offsetSeconds * 1000);
                        }
                    }
                }
            }
        }
        catch (Exception exp) {
            milliseconds = 0;
        }

        return milliseconds;
    }

    /**
     * Converts a long representing UTC in milliseconds to the XML Schema datetime format
     *
     * @param date the date in milliseconds
     *
     * @return String the date in the XML Schema datetime format
     */
    public static String convertDateToXmlSchemaDateTime(long date) {
        String dateTime = null;

        // create a the GMT timezone 
        SimpleTimeZone gmt = new SimpleTimeZone(0, "UTC");

        // create a GregorianCalendar with the GMT time zone
        Calendar calendar = new GregorianCalendar(gmt);

        // set the date in the calendar
        calendar.setTime(new Date(date));

        // get the interger representation for all the fields
        int year = calendar.get(Calendar.YEAR);
        int month = calendar.get(Calendar.MONTH);

        // java has January as month 0 so need add 1 to the month
        month++;

        int day = calendar.get(Calendar.DAY_OF_MONTH);
        int hour = calendar.get(Calendar.HOUR_OF_DAY);
        int minute = calendar.get(Calendar.MINUTE);
        int second = calendar.get(Calendar.SECOND);
        int milliseconds = calendar.get(Calendar.MILLISECOND);

        // now create a string buffer to build the string
        // if the fields are not the correct size they must be padded with 0's
        StringBuffer stringBuffer = new StringBuffer(35);
        stringBuffer.append(year);
        stringBuffer.append('-');

        if (month < 10) {
            stringBuffer.append("0");
        }

        stringBuffer.append(month);
        stringBuffer.append('-');

        if (day < 10) {
            stringBuffer.append("0");
        }

        stringBuffer.append(day);
        stringBuffer.append('T');

        if (hour < 10) {
            stringBuffer.append("0");
        }

        stringBuffer.append(hour);
        stringBuffer.append(':');

        if (minute < 10) {
            stringBuffer.append("0");
        }

        stringBuffer.append(minute);
        stringBuffer.append(':');

        if (second < 10) {
            stringBuffer.append("0");
        }

        stringBuffer.append(second);

        stringBuffer.append(".");

        stringBuffer.append(milliseconds);

        // are times are always UTC so append a Z
        stringBuffer.append("Z");
        dateTime = stringBuffer.toString();

        return dateTime;
    }

    /**
     * Returns the version of the Common Base Event XML schema.
     * 
     * @return String The version of the Common Base Event XML schema.
     * @deprecated Use ICommonBaseEvent#getVersion() as of 1.0.1.
     */
    public final static String getEventFormatterVersion() {
        return xml_version;
    }

	private static IExtendedDataElement loadExtendedDataElement(String xml) {

		IExtendedDataElement extendedDataElement = factory.createExtendedDataElement();

		//Retrieve and set the 'name' property of the ExtendedDataElement:
		int index = xml.indexOf(" name=\"");

		if (index != -1) {
			extendedDataElement.setName(XmlUtility.denormalize(xml.substring(index + 7, xml.indexOf("\"", index + 7)))); //7 is the length of " name=\"".
		}

		//Retrieve and set the 'type' property of the ExtendedDataElement:
		index = xml.indexOf(" type=\"");

		if (index != -1) {
			extendedDataElement.setType(XmlUtility.denormalize(xml.substring(index + 7, xml.indexOf("\"", index + 7)))); //7 is the length of " type=\"".
		}

		String[] childFragments = getChildXMLFragments(xml);

		if (childFragments != null) {

			ArrayList values = new ArrayList();

			for (int counter = 0; counter < childFragments.length; counter++) {

				//Process each <values> element(s):
				if (childFragments[counter].startsWith("<values>")) {
					values.add(XmlUtility.denormalize(childFragments[counter].substring(8, childFragments[counter].indexOf("</values>")))); //8 is the length of "<values>".
				}
				else if (childFragments[counter].startsWith("<hexValue>")) {

					//The hex string is a contiguous series of zero padded hex tuples (e.g. 0F):
					String hexString = XmlUtility.denormalize(childFragments[counter].substring(10, childFragments[counter].indexOf("</hexValue>"))); //10 is the length of "<hexValue>".

					//Each byte is one zero padded (if necessary) hex tuple (e.g. 0F or A4):
					int numberOfBytes = ((int) (hexString.length() / 2));
					char[] charArray = new char[numberOfBytes];

					for (int i = 0; i < numberOfBytes; i++)
						charArray[i] = ((char) (Short.parseShort(hexString.substring((i * 2), ((i * 2) + 2)), 16)));

					extendedDataElement.setHexValues(charArray);
				}
				else if (childFragments[counter].startsWith("<children ")) {
					extendedDataElement.addChildDataElement(loadExtendedDataElement(childFragments[counter]));
				}
			}

			//Retrieve and set the 'values' properties of the ExtendedDataElement:
		    extendedDataElement.setValues(((String[]) (values.toArray(new String[values.size()]))));
		}

		return extendedDataElement;
	}

	/**
	 * Separates all the child XML fragments in the parameter XML fragment and returns the child 
	 * fragments as a String array.  Child non-fragments (e.g. PCDATA), if any, are concatenated and 
	 * returned in the last index of the String array. 
	 * 
	 * @param xmlFragment The root level XML fragment containing child elements. 
	 * @return The resultant String array of separated child XML fragments.
	 */
	private static String[] getChildXMLFragments(String xmlFragment) {

		int firstGreaterThanIndex = xmlFragment.indexOf('>');
		int lastLessThanIndex = xmlFragment.lastIndexOf('<');

		if ((firstGreaterThanIndex != -1) && (lastLessThanIndex != -1) && (firstGreaterThanIndex < lastLessThanIndex)) {
			return (getXMLFragments(xmlFragment.substring((firstGreaterThanIndex + 1), lastLessThanIndex)));
		}

		return null;
	}

	/**
	 * Separates all the root level XML fragments in the parameter XML fragment(s) and returns the 
	 * fragments as a String array.  Non-fragments (e.g. PCDATA), if any, are concatenated and 
	 * returned in the last index of the String array. 
	 * 
	 * @param xmlFragments The root level XML fragments to be separated. 
	 * @return The resultant String array of separated XML fragments.
	 */
	private static String[] getXMLFragments(String xmlFragments) {

		if ((xmlFragments != null) && (xmlFragments.trim().length() > 0)) {

			//String array of separated XML fragment(s) and/or PCDATA:
			ArrayList xmlFragmentsArray = new ArrayList();

			//PCDATA buffer:
			StringBuffer pcdata = new StringBuffer();

			//Properties for locating complete XML fragments in parameter XML fragment(s):
			char previousChar = 0;
			char currentChar = 0;
			char currentXMLMarkupChar = 0;
			int xmlMarkupCharsCount = 0;
			int startTagIndex = -1;

			//NOTE:  XML fragments are determined by maintaining a count of the XML markup characters (e.g. <, > and /).  For every '<' character, the count 
			//in incremented.  For every '/>' set of characters, the count is decremented.  Finally, for every '</' set of characters, the count is decremented 
			//by 2.  The count is checked for every '>' character to determine if it is the end of a complete XML fragment (e.g. count is 0).

			//Check the parameter XML fragment(s) for a complete XML fragment(s) character by character while persisting the previous character and XML markup character: 						
			for (int counter = 0; counter < xmlFragments.length(); counter++) {

				//Retrive the current character:
				currentChar = xmlFragments.charAt(counter);

				//If the current character is '<', increment the count of XML markup characters.  This is the start of a complete XML fragment, if one has not been previously located:   
				if (currentChar == '<') {

					if (startTagIndex == -1)
						startTagIndex = counter;

					xmlMarkupCharsCount++;
				}

				//If the current character is '/', decrement the count of XML markup characters by 2 if the previous character is '<'.  This is a close tag and the end of a nested level, if the previous character is '<':
				else if (currentChar == '/') {

					if (previousChar == '<')
						xmlMarkupCharsCount -= 2;
				}

				//If the current character is '>', decrement the count of XML markup characters if the previous character is '/'.  This is a open tag and the start of a nested level, if the previous character is '<', otherwise a complete XML fragment if the start of the complete XML fragment has been located and no nested levels:
				else if (currentChar == '>') {

					if (previousChar == '/')
						xmlMarkupCharsCount--;

					if ((startTagIndex != -1) && (xmlMarkupCharsCount == 0)) {

						//Add the complete XML fragment to the String array of separated XML fragment(s) and/or PCDATA:
						xmlFragmentsArray.add(xmlFragments.substring(startTagIndex, (counter + 1)).trim());

						startTagIndex = -1;
					}
				}

				//Otherwise the current character is PCDATA and is added to the PCDATA buffer:
				else if (startTagIndex == -1)
					pcdata.append(currentChar);

				//Presist the previous non-whitespace character:
				if (!Character.isWhitespace(currentChar))
					previousChar = currentChar;
			}

			//If any PCADTA was found, add it to the last index of the String array of separated XML fragments:
			if (pcdata.length() > 0) {
				xmlFragmentsArray.add(pcdata.toString().trim());
			}

			if (xmlFragmentsArray.size() > 0) {
				//Return the String array of separated XML fragments:
				return (((String[]) (xmlFragmentsArray.toArray(new String[xmlFragmentsArray.size()]))));
			}
		}

		return null;
	}
}
