package org.eclipse.hyades.logging.java;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.util.List;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.LogRecord;

import org.eclipse.hyades.logging.core.IExternalizableToXml;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.MsgCatalogToken;
import org.eclipse.hyades.logging.events.cbe.MsgDataElement;
import org.eclipse.hyades.logging.events.cbe.impl.EventFactoryContext;
import org.eclipse.hyades.logging.events.cbe.util.EventFormatter;

/** 
 * Extension of the Java Logging <code>java.util.logging.LogRecord</code>
 * class used to encapsulate a Common Base Event (CBE) v1.0.1 or 
 * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code> 
 * within a <code>java.util.logging.LogRecord</code>.
 * <p>
 * A log record instance is created via the single arguement constructor 
 * that accepts a <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>
 * reference.
 * <p>
 *  
 * 
 * @author  Paul Slauenwhite
 * @version April 15, 2004
 * @see     java.util.logging.LogRecord
 * @see     org.eclipse.hyades.logging.events.cbe.CommonBaseEvent
 */
public final class CommonBaseEventLogRecord extends LogRecord implements IExternalizableToXml {

    /**
     * Instance variable which holds a reference to a <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>.
     */
    private CommonBaseEvent commonBaseEvent = null;

    /**
     * Instance variable which holds a reference to a <code>org.eclipse.hyades.logging.events.cbe.EventFactory</code> for generating event instances.
     */
    private static EventFactory eventFactory = EventFactoryContext.getInstance().getEventFactoryHome("org.eclipse.hyades.logging.events.cbe.impl.SimpleEventFactoryHomeImpl").getEventFactory("org.eclipse.hyades.logging.java.CommonBaseEventLogRecord");

    /**
     * Instance variable which holds the name property of the <code>commonBaseEvent</code>'s <code>ExtendedDataElement</code> which contains the <code>LogRecord</code>'s level.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_LEVEL = "level";

    /**
     * Instance variable which holds the name property of the <code>commonBaseEvent</code>'s <code>ExtendedDataElement</code>'s child which contains the <code>LogRecord</code>'s level's name.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_LEVEL_NAME = "name";

    /**
     * Instance variable which holds the name property of the <code>commonBaseEvent</code>'s <code>ExtendedDataElement</code>'s child which contains the <code>LogRecord</code>'s level's value.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_LEVEL_VALUE = "value";

    /**
     * Instance variable which holds the name property of the <code>commonBaseEvent</code>'s <code>ExtendedDataElement</code> which contains the <code>LogRecord</code>'s logger name.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME = "loggerName";

    /**
     * Instance variable which holds the name property of the <code>commonBaseEvent</code>'s <code>ExtendedDataElement</code> which contains the <code>LogRecord</code>'s sequence number.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER = "sequenceNumber";
    
    /**
     * Instance variable which holds the name property of the <code>commonBaseEvent</code>'s <code>ExtendedDataElement</code> which contains the <code>LogRecord</code>'s source class name.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME = "sourceClassName";

    /**
     * Instance variable which holds the name property of the <code>commonBaseEvent</code>'s <code>ExtendedDataElement</code> which contains the <code>LogRecord</code>'s source method name.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME = "sourceMethodName";

    /**
     * Instance variable which holds the name property of the <code>commonBaseEvent</code>'s <code>ExtendedDataElement</code> which contains the <code>LogRecord</code>'s thread ID.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_THREAD_ID = "threadID";

    /**
     * Instance variable which holds the name property of the <code>commonBaseEvent</code>'s <code>ExtendedDataElement</code> which contains the <code>LogRecord</code>'s thrown.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_THROWN = "stackTrace";

    /**
     * Constructor to create a CBE log record with the parameter CBE.
     * <p>
     * 
     * @param commonBaseEvent The CBE of the newly created log record.
     */
    public CommonBaseEventLogRecord(CommonBaseEvent commonBaseEvent) {

        super(Level.OFF,null);
        
        if(commonBaseEvent != null){

            this.commonBaseEvent = commonBaseEvent;

	        short severity = commonBaseEvent.getSeverity();
	        
	        if(severity >= 50){
	            setLevel(Level.SEVERE);
	        }
	        else if((severity >= 30) && (severity <= 49)){            
	            setLevel(Level.WARNING);
	        }
	        else{
	            setLevel(Level.INFO);            
	        }
	        
	        setSequenceNumber(super.getSequenceNumber());
	
	        setThreadID(super.getThreadID());
	        
	        if(!commonBaseEvent.isSetCreationTime()){
	            commonBaseEvent.setCreationTimeAsLong(System.currentTimeMillis());
	        }
        }
    }
    
    /**
     * Constructor to create a CBE log record with the parameter logging level and message.
     * <p>
     * Using this constructor will result in the construction of a <code>LogRecord</code> with 
     * no Common Base Event capabilities.
     * <p>
     * 
     * @param level The logging level of the newly created log record.
     * @param message The log message of the newly created log record.
     * @deprecated Use <code>CommonBaseEventLogRecord(CommonBaseEvent commonBaseEvent)</code>.
     */
    public CommonBaseEventLogRecord(Level level, String message) {        
        super(level,message);
    }

    /**
     * Constructor to create a CBE log record with the parameter logging level.
     * <p>
     * Using this constructor will result in the construction of a <code>LogRecord</code> with 
     * no Common Base Event capabilities.
     * <p>
     * 
     * @param level The logging level of the newly created log record.
     * @deprecated Use <code>CommonBaseEventLogRecord(CommonBaseEvent commonBaseEvent)</code>.
     */
    public CommonBaseEventLogRecord(Level level) {
        super(level,null);
    }
    
    /**
     * Constructor to create a CBE log record with the parameter <code>LogRecord</code>.
     * <p>
     * Using this constructor will result in the construction of a <code>LogRecord</code> with 
     * no Common Base Event capabilities.
     * <p>
     * 
     * @param logRecord The <code>LogRecord</code> of the newly created log record.
     * @deprecated Use <code>CommonBaseEventLogRecord(CommonBaseEvent commonBaseEvent)</code>.
     */
    public CommonBaseEventLogRecord(LogRecord logRecord) {

        super(logRecord.getLevel(),logRecord.getMessage());

        super.setMillis(logRecord.getMillis());
        super.setSequenceNumber(logRecord.getSequenceNumber());
        super.setThreadID(logRecord.getThreadID());
        super.setLoggerName(logRecord.getLoggerName());
        super.setParameters(logRecord.getParameters());
        super.setResourceBundle(logRecord.getResourceBundle());
        super.setResourceBundleName(logRecord.getResourceBundleName());
        super.setSourceClassName(logRecord.getSourceClassName());
        super.setSourceMethodName(logRecord.getSourceMethodName());
        super.setThrown(logRecord.getThrown());
    }

    /**
     * Sets value of the commonBaseEvent instance variable to the specified value.
     * <p>
     * 
     * @param commonBaseEvent The new value of the commonBaseEvent instance variable.
     * @deprecated Use the <code>CommonBaseEventLogRecord(CommonBaseEvent commonBaseEvent)</code> constructor.
     */
    public void setCommonBaseEvent(CommonBaseEvent commonBaseEvent) {
        this.commonBaseEvent = commonBaseEvent;
    }

    /**
     * @see java.util.logging.LogRecord#setLevel(java.util.logging.Level)
     */
    public void setLevel(Level level) {

        if(commonBaseEvent != null){
            
            commonBaseEvent.removeExtendedDataElements(EXTENDED_DATA_ELEMENT_NAME_LEVEL);
            
	        ExtendedDataElement levelExtendedDataElement = eventFactory.createExtendedDataElement();
	        levelExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_LEVEL);
	 
	        levelExtendedDataElement.addChild(EXTENDED_DATA_ELEMENT_NAME_LEVEL_NAME,level.getName());
	        levelExtendedDataElement.addChildWithIntValue(EXTENDED_DATA_ELEMENT_NAME_LEVEL_VALUE,level.intValue());
	        
	        commonBaseEvent.addExtendedDataElement(levelExtendedDataElement);
        }
        else{
            super.setLevel(level);
        }
    }

    /**
     * @see java.util.logging.LogRecord#setLoggerName(java.lang.String)
     */
    public void setLoggerName(String name) {

        if(commonBaseEvent != null){

            commonBaseEvent.removeExtendedDataElements(EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME);
            
	        ExtendedDataElement loggerNameExtendedDataElement = eventFactory.createExtendedDataElement();
	        loggerNameExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME);
	        loggerNameExtendedDataElement.setValuesAsString(name);
	 
	        commonBaseEvent.addExtendedDataElement(loggerNameExtendedDataElement);
        }
        else{
            super.setLoggerName(name);
        }
    }

    /**
     * @see java.util.logging.LogRecord#setMessage(java.lang.String)
     */
    public void setMessage(String message) {

        if(commonBaseEvent != null){

            MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();
	        
	        if(msgDataElement == null){
	            
	            msgDataElement = eventFactory.createMsgDataElement();
	            
	            commonBaseEvent.setMsgDataElement(msgDataElement);
	        }
	        
	        msgDataElement.setMsgCatalogId(message);
        }
        else{
            super.setMessage(message);
        }
    }

    /**
     * @see java.util.logging.LogRecord#setMillis(long)
     */
    public void setMillis(long millis) {

        if(commonBaseEvent != null){
            commonBaseEvent.setCreationTimeAsLong(millis);
        }
        else{
            super.setMillis(millis);
        }
    }

    /**
     * @see java.util.logging.LogRecord#setParameters(java.lang.Object[])
     */
    public void setParameters(Object[] parameters) {

        super.setParameters(parameters);

        if(commonBaseEvent != null){

            MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();
       
	        if(parameters != null){

		        if(msgDataElement == null){
		            
		            msgDataElement = eventFactory.createMsgDataElement();
		            
		            commonBaseEvent.setMsgDataElement(msgDataElement);
		        }

	            MsgCatalogToken[] msgCatalogTokens = new MsgCatalogToken[parameters.length];
		        
		        for (int counter = 0; counter < parameters.length; counter++) {
		            
		            msgCatalogTokens[counter] = eventFactory.createMsgCatalogToken();
		            msgCatalogTokens[counter].setValue(String.valueOf(parameters[counter]));
		        }
		
		        msgDataElement.setMsgCatalogTokens(msgCatalogTokens);
	        }
	        else{
	            msgDataElement.clearMsgCatalogTokens();
	        }
        }
    }

    /**
     * @see java.util.logging.LogRecord#setResourceBundle(java.util.ResourceBundle)
     */
    public void setResourceBundle(ResourceBundle bundle) {
        super.setResourceBundle(bundle);
    }

    /**
     * @see java.util.logging.LogRecord#setResourceBundleName(java.lang.String)
     */
    public void setResourceBundleName(String name) {

        if(commonBaseEvent != null){

            MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();
	        
	        if(msgDataElement == null){
	            
	            msgDataElement = eventFactory.createMsgDataElement();
	            
	            commonBaseEvent.setMsgDataElement(msgDataElement);
	        }
	        
	        msgDataElement.setMsgCatalog(name);
	        msgDataElement.setMsgCatalogType("Java");
        }
        else{
            super.setResourceBundleName(name);
        }
    }

    /**
     * @see java.util.logging.LogRecord#setSequenceNumber(long)
     */
    public void setSequenceNumber(long seq) {

        if(commonBaseEvent != null){
	
            commonBaseEvent.removeExtendedDataElements(EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER);

	        ExtendedDataElement sequenceNumberExtendedDataElement = eventFactory.createExtendedDataElement();
	        sequenceNumberExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER);
	        sequenceNumberExtendedDataElement.setValuesAsLong(seq);
	 
	        commonBaseEvent.addExtendedDataElement(sequenceNumberExtendedDataElement);
        }
        else{
            super.setSequenceNumber(seq);
        }
    }

    /**
     * @see java.util.logging.LogRecord#setSourceClassName(java.lang.String)
     */
    public void setSourceClassName(String sourceClassName) {

        if(commonBaseEvent != null){

            commonBaseEvent.removeExtendedDataElements(EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME);

	        ExtendedDataElement sourceClassNameExtendedDataElement = eventFactory.createExtendedDataElement();
	        sourceClassNameExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME);
	        sourceClassNameExtendedDataElement.setValuesAsString(sourceClassName);
	 
	        commonBaseEvent.addExtendedDataElement(sourceClassNameExtendedDataElement);
        }
        else{
            super.setSourceClassName(sourceClassName);
        }
    }

    /**
     * @see java.util.logging.LogRecord#setSourceMethodName(java.lang.String)
     */
    public void setSourceMethodName(String sourceMethodName) {

        if(commonBaseEvent != null){

            commonBaseEvent.removeExtendedDataElements(EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME);

	        ExtendedDataElement sourceMethodNameExtendedDataElement = eventFactory.createExtendedDataElement();
	        sourceMethodNameExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME);
	        sourceMethodNameExtendedDataElement.setValuesAsString(sourceMethodName);
	 
	        commonBaseEvent.addExtendedDataElement(sourceMethodNameExtendedDataElement);
        }
        else{
            super.setSourceMethodName(sourceMethodName);
        }
    }

    /**
     * @see java.util.logging.LogRecord#setThreadID(int)
     */
    public void setThreadID(int threadID) {

        if(commonBaseEvent != null){

            commonBaseEvent.removeExtendedDataElements(EXTENDED_DATA_ELEMENT_NAME_THREAD_ID);
            
	        ExtendedDataElement threadIDExtendedDataElement = eventFactory.createExtendedDataElement();
	        threadIDExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_THREAD_ID);
	        threadIDExtendedDataElement.setValuesAsInt(threadID);
	 
	        commonBaseEvent.addExtendedDataElement(threadIDExtendedDataElement);
        }
        else{
            super.setThreadID(threadID);
        }
    }

    /**
     * @see java.util.logging.LogRecord#setThrown(java.lang.Throwable)
     */
    public void setThrown(Throwable thrown) {
        
        super.setThrown(thrown);
        
        if(commonBaseEvent != null){

            commonBaseEvent.removeExtendedDataElements(EXTENDED_DATA_ELEMENT_NAME_THROWN);

            ExtendedDataElement thrownExtendedDataElement = eventFactory.createExtendedDataElement();
	        thrownExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_THROWN);
	        thrownExtendedDataElement.setValuesAsString(String.valueOf(thrown));
	 
	        commonBaseEvent.addExtendedDataElement(thrownExtendedDataElement);
        }
    }
    
    /**
     * Gets the value of the commonBaseEvent instance variable.
     * <p>
     * 
     * @return The value of the commonBaseEvent instance variable.
     */
    public CommonBaseEvent getCommonBaseEvent() {
        return commonBaseEvent;
    }
    
    /**
     * @see java.util.logging.LogRecord#getLevel()
     */
    public Level getLevel() {

        if(commonBaseEvent != null){
        
	        List levelExtendedDataElements = commonBaseEvent.getExtendedDataElements(EXTENDED_DATA_ELEMENT_NAME_LEVEL);
	        
	        if((levelExtendedDataElements != null) && (levelExtendedDataElements.size() > 0)){
	            
	            ExtendedDataElement levelExtendedDataElement = ((ExtendedDataElement)(levelExtendedDataElements.get(0)));
	            
	            List nameExtendedDataElements = levelExtendedDataElement.getChildren(EXTENDED_DATA_ELEMENT_NAME_LEVEL_NAME);
	            List valueExtendedDataElements = levelExtendedDataElement.getChildren(EXTENDED_DATA_ELEMENT_NAME_LEVEL_VALUE);
	            
	            if((nameExtendedDataElements != null) && (nameExtendedDataElements.size() > 0) && (valueExtendedDataElements != null) && (valueExtendedDataElements.size() > 0)){
	
	                ExtendedDataElement nameExtendedDataElement = ((ExtendedDataElement)(nameExtendedDataElements.get(0)));
	                ExtendedDataElement valueExtendedDataElement = ((ExtendedDataElement)(valueExtendedDataElements.get(0)));
	
					String name = nameExtendedDataElement.getValuesAsString();
					
					if(name != null){
					    
						try {
						    return (Level.parse(name));	
						} 
						catch (IllegalArgumentException i) {
							//Ignore since level is a customized level.
						}
					}
					
					return (new CommonBaseEventLevel(name,valueExtendedDataElement.getValuesAsInt()));
	            }    
	        }

	        return (Level.INFO);
        }

        return (super.getLevel());
    }
    
    /**
     * @see java.util.logging.LogRecord#getLoggerName()
     */
    public String getLoggerName() {

        if(commonBaseEvent != null){

	        List loggerNameExtendedDataElements = commonBaseEvent.getExtendedDataElements(EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME);
	        
	        if((loggerNameExtendedDataElements != null) && (loggerNameExtendedDataElements.size() > 0)){
	            
	            ExtendedDataElement loggerNameExtendedDataElement = ((ExtendedDataElement)(loggerNameExtendedDataElements.get(0)));
	            
	            return (loggerNameExtendedDataElement.getValuesAsString());            
	        }
        }
        else{
            return (super.getLoggerName());
        }
        
        return null;
    }
    
    /**
     * @see java.util.logging.LogRecord#getMessage()
     */
    public String getMessage() {

        if(commonBaseEvent != null){

            MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();
        
	        if(msgDataElement != null){
	            
	            String msgCatalogId = msgDataElement.getMsgCatalogId();
	            
	            if(msgCatalogId != null){
	                return msgCatalogId;
	            }
	        }
        }
        else{
            return (super.getMessage());
        }

        return (commonBaseEvent.getMsg());
    }
    
    /**
     * @see java.util.logging.LogRecord#getMillis()
     */
    public long getMillis() {
        
        if(commonBaseEvent != null){
            return (commonBaseEvent.getCreationTimeAsLong());
        }

        return (super.getMillis());
    }
    
    /**
     * @see java.util.logging.LogRecord#getParameters()
     */
    public Object[] getParameters() {
        
        Object[] parameters = super.getParameters();
		
        if (parameters != null) {
            return parameters;
        }
        
        if(commonBaseEvent != null){

            MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();
	        
	        if(msgDataElement != null){
	            
	            String[] msgCatalogTokens = msgDataElement.getMsgCatalogTokensAsStrings();
	            
	            if((msgCatalogTokens != null) && (msgCatalogTokens.length > 0)){
	                return msgCatalogTokens;
	            }
	        }
        }
        
        return null;
    }
    
    /**
     * @see java.util.logging.LogRecord#getResourceBundle()
     */
    public ResourceBundle getResourceBundle() {
        return (super.getResourceBundle());
    }
    
    /**
     * @see java.util.logging.LogRecord#getResourceBundleName()
     */
    public String getResourceBundleName() {

        if(commonBaseEvent != null){
	
	        MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();
	        
	        if(msgDataElement != null){            
	            return (msgDataElement.getMsgCatalog());
	        }
        }
        else{
            return (super.getResourceBundleName());
        }

        return null;
    }
    
    /**
     * @see java.util.logging.LogRecord#getSequenceNumber()
     */
    public long getSequenceNumber() {

        if(commonBaseEvent != null){

            List sequenceNumberExtendedDataElements = commonBaseEvent.getExtendedDataElements(EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER);
	        
	        if((sequenceNumberExtendedDataElements != null) && (sequenceNumberExtendedDataElements.size() > 0)){
	            
	            ExtendedDataElement sequenceNumberExtendedDataElement = ((ExtendedDataElement)(sequenceNumberExtendedDataElements.get(0)));
	            
	            return (sequenceNumberExtendedDataElement.getValuesAsLong());            
	        }
        }
        else{
            return (super.getSequenceNumber());
        }
        
        return 0;
    }
    
    /**
     * @see java.util.logging.LogRecord#getSourceClassName()
     */
    public String getSourceClassName() {

        if(commonBaseEvent != null){

            List sourceClassNameExtendedDataElements = commonBaseEvent.getExtendedDataElements(EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME);
	        
	        if((sourceClassNameExtendedDataElements != null) && (sourceClassNameExtendedDataElements.size() > 0)){
	            
	            ExtendedDataElement sourceClassNameExtendedDataElement = ((ExtendedDataElement)(sourceClassNameExtendedDataElements.get(0)));
	            
	            return (sourceClassNameExtendedDataElement.getValuesAsString());            
	        }
        }
        else{
            return (super.getSourceClassName());
        }
        
        return null;
    }
    
    /**
     * @see java.util.logging.LogRecord#getSourceMethodName()
     */
    public String getSourceMethodName() {

        if(commonBaseEvent != null){

	        List sourceMethodNameExtendedDataElements = commonBaseEvent.getExtendedDataElements(EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME);
	        
	        if((sourceMethodNameExtendedDataElements != null) && (sourceMethodNameExtendedDataElements.size() > 0)){
	            
	            ExtendedDataElement sourceMethodNameExtendedDataElement = ((ExtendedDataElement)(sourceMethodNameExtendedDataElements.get(0)));
	            
	            return (sourceMethodNameExtendedDataElement.getValuesAsString());            
	        }
        }
        else{
            return (super.getSourceMethodName());
        }
        
        return null;
    }
    
    /**
     * @see java.util.logging.LogRecord#getThreadID()
     */
    public int getThreadID() {

        if(commonBaseEvent != null){

	        List threadIDExtendedDataElements = commonBaseEvent.getExtendedDataElements(EXTENDED_DATA_ELEMENT_NAME_THREAD_ID);
	        
	        if((threadIDExtendedDataElements != null) && (threadIDExtendedDataElements.size() > 0)){
	            
	            ExtendedDataElement threadIDExtendedDataElement = ((ExtendedDataElement)(threadIDExtendedDataElements.get(0)));
	            
	            return (threadIDExtendedDataElement.getValuesAsInt());            
	        }
        }
        else{
            return (super.getThreadID());
        }
        
        return 0;
    }
    
    /**
     * @see java.util.logging.LogRecord#getThrown()
     */
    public Throwable getThrown() {
        return (super.getThrown());
    }

    /**
     * Generates the XML fragment representation of the CBE in this log record.
     * <p>
     * The XML fragment representation of the commonBaseEvent instance variable is generated and not the 
     * XML fragment representation of the complete <code>java.util.logging.LogRecord</code>.
     * <p>
     * 
     * @return The XML fragment representation of the CBE in this log record, or <code>null</code> if the value of the commonBaseEvent instance variable is set to <code>null</code>.
     */
    public String externalizeCanonicalXmlString() {
        
        if(commonBaseEvent != null){
            return (EventFormatter.toCanonicalXMLString(commonBaseEvent));
        }
        
        return null;
    }

    /**
     * Generates the XML document representation of the CBE in this log record.
     * <p>
     * The XML document representation of the commonBaseEvent instance variable is generated and not the 
     * XML document representation of the complete <code>java.util.logging.LogRecord</code>.
     * <p>
     * 
     * @return The XML document representation of the CBE in this log record, or <code>null</code> if the value of the commonBaseEvent instance variable is set to <code>null</code>.
     */
    public String externalizeCanonicalXmlDocString() {
        
        if(commonBaseEvent != null){
            return (EventFormatter.toCanonicalXMLDocString(commonBaseEvent));
        }
        
        return null;
    }

    /**
     * Returns the textual representation of this <code>CommonBaseEventLogRecord</code>.
     * <p>
     * 
     * @return Textual representation of this <code>CommonBaseEventLogRecord</code>.
     * @see java.lang.Object#toString()
     */
    public String toString() {
        
        if(commonBaseEvent != null){
            return (commonBaseEvent.toString());
        }
        
        return (super.toString());
    }
}
