/**********************************************************************
Copyright (c) 2003 Hyades project.
All rights reserved.  This program and the accompanying materials
are made available under the terms of the Common Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/cpl-v10.html

Contributors:
 IBM Corporation - initial implementation
**********************************************************************/
package org.eclipse.hyades.execution.local;

import org.eclipse.hyades.execution.core.ExecutionComponentStateChangeEvent;
import org.eclipse.hyades.execution.core.ExecutionComponentStateException;
import org.eclipse.hyades.execution.core.IControlMessage;
import org.eclipse.hyades.execution.core.IExecutableObject;
import org.eclipse.hyades.execution.core.IExecutionComponent;
import org.eclipse.hyades.execution.core.MessageDeliveryException;
import org.eclipse.hyades.execution.core.impl.IJavaTaskExecutableObject;
import org.eclipse.hyades.execution.core.impl.JavaTaskExecutorImpl;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentFactory;
import org.eclipse.hyades.internal.execution.local.control.AgentImpl;
import org.eclipse.hyades.internal.execution.local.control.AgentListener;
import org.eclipse.hyades.internal.execution.local.control.InactiveAgentException;

/**
 * In addition to providing remote execution semantics for the methods of
 * the "java task executor" (<code>JavaTaskExecutorImpl</code>), instances
 * of this class establish the local portion of the test monitoring and
 * control agent.
 */
public class JavaTaskExecutorStub extends ExecutorStub {
	
	private AgentImpl agent;
	private boolean isAgentActive;

	public JavaTaskExecutorStub() {
		super();
	}

	public JavaTaskExecutorStub(IExecutionComponent delegate){
		super(delegate);
	}
	
	/**
	 * @throws ExecutionComponentStateException if the test monitoring and
	 *         control agent is not active
	 * 
	 * @see org.eclipse.hyades.execution.core.IExecutor#launch()
	 */
	public void launch() throws ExecutionComponentStateException {
		if ( !isAgentActive() ) {
			StringBuffer buf = new StringBuffer();
			buf.append("The remote agent \"");
			buf.append(agent.getName());
			buf.append("\" is not active");
			throw new ExecutionComponentStateException(NOT_CONFIGURED, buf.toString());
		}
			
		super.launch();
	}

	/**
	 * @return an <code>IJavaTaskExecutableObjectStub</code>
	 * 
	 * @see org.eclipse.hyades.execution.core.IExecutor#getCompatibleExecutableObject(java.lang.String)
	 */
	public IExecutableObject getCompatibleExecutableObject(String classname)
		throws ClassNotFoundException {
		IExecutableObject exeObj = super.getCompatibleExecutableObject(classname);
		if ( !(exeObj instanceof IJavaTaskExecutableObject) )
			throw new ClassCastException("Component factory returned an incorrect type; expected an IJavaTaskExecutableObject");
		return exeObj;
	}

	/**
	 * Set up communication with the remote test agent. The local portion
	 * of the agent is created here.
	 * 
	 * @see org.eclipse.hyades.execution.invocation.IRemoteObject#init()
	 */
	public void init() {
		super.init();
		
		// Create the local side of the test agent.
		SessionImpl session = (SessionImpl)((SessionStub)getSessionContext()).getDelegate();
		agent = (AgentImpl)AgentFactory.createAgent(
			session.getAgent().getProcess(),
			"Executor$"+getUniqueId(),
			"tester");
		agent.setAutoAttach(true);
		
		agent.addAgentListener(new AgentListener() {
			public void agentActive(Agent agent) {
				setAgentActive(true);
				fireStateChangeEvent(new ExecutionComponentStateChangeEvent(
					JavaTaskExecutorStub.this, READY));
			}

			public void agentInactive(Agent agent) {
				setAgentActive(false);
				fireStateChangeEvent(new ExecutionComponentStateChangeEvent(
					JavaTaskExecutorStub.this, DEAD));
			}

			public void error(Agent agent, String errorId, String errorMessage) {
				StringBuffer buf = new StringBuffer();
				buf.append("ERROR: Agent \"");
				buf.append(agent.getName());
				buf.append("\"\nERROR ID: \"");
				buf.append(errorId);
				buf.append("\"\nERROR MESSAGE: \"");
				buf.append(errorMessage);
				buf.append("\"\n");
				System.err.println(buf.toString());
			}

			public void handleCommand(Agent agent, CommandElement command) {
				System.out.println(agent.getName() + " handleCommand() called");
			}
		});
		
		((JavaTaskExecutorImpl)delegate).init();	}

	/**
	 * This method is implemented at the stub level. Currently, the only
	 * message handled is the START_TEST message.
	 * 
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#sendMessage(org.eclipse.hyades.execution.core.IControlMessage)
	 */
	public void sendMessage(IControlMessage message)
		throws MessageDeliveryException {
		
		CustomCommand startCmd = new CustomCommand();
		startCmd.setData(message.getMessageData());
		try {
			agent.invokeCustomCommand(startCmd);
		}
		catch ( InactiveAgentException e ){
			throw new MessageDeliveryException(new ExecutionComponentStateException(NOT_CONFIGURED, "The monitoring agent is not active"));
		}
	}
	
	private synchronized void setAgentActive(boolean active) {
		isAgentActive = active;
	}
	
	private synchronized boolean isAgentActive() {
		return isAgentActive;
	}

	/**
	 * Add special semantics for <code>JavaTaskRemoteHyadesComponent</code>s that
	 * are added as children. When a <code>JavaTaskRemoteHyadesComponent</code> is
	 * added as a child, a reference the test monitoring and control agent is
	 * passed to the new child, enabling it to fulfill monitoring requests.
	 * 
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#addChild(org.eclipse.hyades.execution.core.IExecutionComponent)
	 */
	public void addChild(IExecutionComponent child) {
		if ( child instanceof JavaTaskRemoteHyadesComponentStub ) {
			((JavaTaskRemoteHyadesComponentStub)child).setAgent(agent);
		}
		super.addChild(child);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.core.IExecutor#supportsControlEvent(java.lang.String)
	 */
	public boolean supportsControlEvent(String controlEvent) {
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.core.IExecutor#performControlEvent(java.lang.String)
	 */
	public String performControlEvent(String controlEvent, String[] params) {
		return "";
	}

}
